/* Copyright (C) 1999 Hans Petter K. Jansson
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * You can contact the library's author by sending e-mail to <hpj@styx.net>.
 */

/* -----------------------------
   Includes for memory functions
   ----------------------------- */

#ifndef _MEM_H
#define _MEM_H 1

#ifndef _STDIO_H
#include <stdio.h>
#endif

/*
#ifndef _TYPES_H
#include <types.h>
#endif
*/

#define MEM_CLEAR  (1 << 0)
#define MEM_SECURE (1 << 1)
#define MEM_SHARED (1 << 2)
#define MEM_DEBUG  (1 << 3)
#define MEM_LOG    (1 << 4)
#define MEM_ABORT  (1 << 5)

/* --- Strictly internal --- */

struct mem_info_entry
{
  struct mem_info_entry *next;
  unsigned count; /* call count */
  const char *info; /* the reference to the info string */
};

struct mem_table_entry
{
  const void *user_p;  /* for reference: the pointer given to the user */
  size_t user_n;  /* length requested by the user */
  struct mem_table_entry *next; /* to build a list of unused entries */
  const struct mem_info_entry *info; /* points into the table with */
                                     /* the info strings */
  unsigned inuse:1; /* this entry is in use */
  unsigned count:31;
};

/* void *mem_alloc_secure(size_t n, const char *info); */
/* struct mem_table_entry *mem_check(const void *a, const char *info); */
/* void _mem_free(void *a, const char *info); */
/* size_t _mem_size(const void *a); */

/* --- Public functions --- */

int mem_init(size_t shared_size, size_t secure_size, unsigned long options);
#define mem_alloc(n) _mem_alloc((n), __FILE__ "->" __FUNCTION__, __LINE__)
#define mem_alloc_opt(n, options) _mem_allloc_opt((n), (options), __FILE__ "->" __FUNCTION__, __LINE__)
#define mem_free(p) _mem_free((p), __FILE__ "->" __FUNCTION__, __LINE__)
#define mem_realloc(p, n) _mem_realloc((p), (n), __FILE__ "->" __FUNCTION__, __LINE__)

/* --- Public function shortcuts --- */

#define mem_alloc_secure(n) _mem_alloc_secure((n), __FILE__ "->" __FUNCTION__, __LINE__)

/* --- Internal functions --- */

/* memory.c */
const void mem_bug(const char *fmt, ...);
void mem_stats_print(const char *prefix);
void mem_table_dump(const char *prefix);
void out_of_core(size_t n, int secure);
void *_mem_alloc(size_t n, const char *info_text, const unsigned int line);
void *_mem_alloc_secure(size_t n, const char *info_text, const unsigned int line);
void *_mem_alloc_clear(size_t n, const char *info, const unsigned int line);
void *_mem_alloc_secure_clear(size_t n, const char *info, const unsigned int line);
void *_mem_realloc(void *a, size_t n, const char *info, const unsigned int line);
void _mem_free(void *a, const char *info_text, const unsigned int line);
void _mem_check(const void *a, const char *info);
size_t mem_size(const void *a);
char *_mem_strdup(const char *a, const char *info, const unsigned int line);
void *_mem_alloc_opt(unsigned long size, unsigned long options, const char *info_text, const unsigned int line);

/* debug.c */
void _mem_debug_add_entry(unsigned char *p, unsigned n, int mode, const char *info, const char *by);
void mem_entry_free(unsigned char *p, const char *info);
struct mem_table_entry *mem_check(const void *a, const char *info);
void mem_entry_dump(struct mem_table_entry *e);
void mem_table_dump_on_exit(void);
void _mem_table_dump(void);
void mem_check_all(const char *info);

/* secure.c */
void mem_warning_print(void);
void mem_pool_lock(void *p, size_t n);
void mem_pool_init(size_t n);
void mem_sec_pool_compress(void);
void mem_sec_flags_set(unsigned flags);
unsigned mem_sec_flags_get(void);
void mem_sec_init(size_t n);
void *mem_sec_alloc(size_t size);
void *mem_sec_realloc(void *p, size_t newsize);
void mem_sec_free(void *a);
int m_is_secure(const void *p);
void mem_sec_term(void);
void mem_sec_stats_dump(void);

#endif  /* _MEM_H */
