/* Copyright (C) 1999 Hans Petter K. Jansson
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * You can contact the library's author by sending e-mail to <hpj@styx.net>.
 */

#ifndef _PROXY_H
#define _PROXY_H 1

#include <sys/time.h>
#include <unistd.h>

#ifdef LIBFLUX_BUILD
# include "tt.h"
# include "sock.h"
# include "comm.h"
#else
# include <flux/tt.h>
# include <flux/sock.h>
# include <flux/comm.h>
#endif

typedef struct proxy
{
  u32 exiting : 1;
  u32 changed : 1;

  TT *socks;
  TT *comms;
  TT *timers;
  TT *signals;

  fd_set fd_read, fd_write;
  int fd_max;
}
PROXY;


/* Nonblocking:
 * 
 * When connecting, select for write indicates connection done/error.
 * When listening, select for read indicates incoming connection/error.
 * When peer disconnects, select for read returns, but read bytes = 0. */

/* Allocation */

PROXY *proxy_new();
void proxy_del(PROXY *p);

/* Operation */

void proxy_loop(PROXY *p);
void proxy_break(PROXY *p);

/* Adding hooks */

void proxy_add_sock(PROXY *p, SOCK *s, void *priv,
                    int (*read)(SOCK *, void *),
                    int (*write)(SOCK *, void *),
                    int (*connect)(SOCK *, SOCK *, void *),
                    int (*close)(SOCK *, void *));

void proxy_add_sock_read(PROXY *p, SOCK *s, void *priv, int (*read)(SOCK *, void *));
void proxy_add_sock_write(PROXY *p, SOCK *s, void *priv, int (*write)(SOCK *, void *));
void proxy_add_sock_connect(PROXY *p, SOCK *s, void *priv, int (*connect)(SOCK *, void *));
void proxy_add_sock_close(PROXY *p, SOCK *s, void *priv, int (*close)(SOCK *, void *));

void proxy_add_comm(PROXY *p, COMM *c, char *root, ushort trans, int incomplete,
                    void *priv, int (*recv)(COMM *, TT *, ushort, int, void *));

void proxy_add_comm_connect(PROXY *p, COMM *c, void *priv, int (*connect)(COMM *, COMM *, void *));
void proxy_add_comm_close(PROXY *p, COMM *c, void *priv, int (*close)(COMM *, void *));

void proxy_add_timer(PROXY *p, char *name, struct timeval *interval,
                     int recurrent, void *priv, int (*timeout)(char *, void *));

/* Removing hooks */

void proxy_del_sock(PROXY *p, SOCK *s);
void proxy_del_comm(PROXY *p, COMM *c);
void proxy_del_comm_trans(PROXY *p, COMM *c, u16 trans);
void proxy_del_comm_block(PROXY *p, COMM *c, char *root);
void proxy_del_timer(PROXY *p, char *name);

/* Special watch interfaces */

void proxy_reset_timer(PROXY *p, char *name);

/* Calling watches (internal) */

void proxy_call_sock(SOCK *s, TT *tt);
void proxy_call_sock_connect(SOCK *s, SOCK *s_child, TT *tt);
void proxy_call_comm(PROXY *p, COMM *c, TT *node, ushort trans, int complete);
void proxy_call_comm_connect(COMM *c, COMM *c_child, TT *tt);
void proxy_call_comm_close(COMM *c, TT *tt);
void proxy_call_timer(PROXY *p, TT *timer);

#endif
