/*******************************************************************************************************************************************
 cwidget.h
 
 CClass					CClass
 |--CSerialized				|--CSerialized
    |--CMetaModule			   |--CMetaModule
       |--CObject			      |--CObjectListener
          |--CComponent			         |
	     |--CControl			 |
	        |--CWidget			 |--CWidgetListener

 Gtkol Widget definition. The gtkol widget encapsulates the gtk widget and lets it accessible if desired. It manages all of the generic 
 allocation and deletion process coherence according to expected specific virtual functions PerformWidgetInstanciate and 
 PerformWidgetInitialize resolution. This process checks the protected attributes m_PerformContainerProcess and m_PerformSignalConnection
 to see if the definition must handle the gtk container encapsulation aspect of the associated gtk widget and the listening associated
 default functions. Those members are set to "true" by default and must be modified accordinaly if desired when launching the process.

 The gtkol widget handles drag & drop of widgets and owned controls and it calls the gtk container aspect functions to reduce programmer 
 code according to implicit hierarchy construction as widgets are instanciated one under each others.

 Typically, derived classes must call CWidget::CreateWidget in their constructors and CWidget::DestroyWidget in their destructors. They 
 have to overwrite OwnerMustBe and ChildMustBe functions too. If the derived class handles no windowed controls, it must overwrite the
 GetControlAtPoint and GetControlPixbuf functions in order to give the widget d&d engine the control it should work on and a friendly 
 pixbuf representation of the managed control.
*******************************************************************************************************************************************/

#ifndef __CWIDGET_H__
#define __CWIDGET_H__

#include "ccontrol.h"

//-----------------------------------------------------------------------------------------------------------------------------------------
// keyboard and mouse states definition
//-----------------------------------------------------------------------------------------------------------------------------------------
typedef enum
{
	SHIFTSTATE_LBUTTON	=(1L),		// mouse left button
	SHIFTSTATE_MBUTTON	=(1L<<1),	// mouse middle button
	SHIFTSTATE_RBUTTON	=(1L<<2),	// mouse right button

	SHIFTSTATE_DBUTTON	=(1L<<3),	// button double click
	SHIFTSTATE_TBUTTON	=(1L<<4),	// triple click

	SHIFTSTATE_SHIFT	=(1L<<5),	// shift key hold
	SHIFTSTATE_CONTROL	=(1L<<6),	// control key hold
	SHIFTSTATE_ALT		=(1L<<7),	// alt key hold
	SHIFTSTATE_LOCK		=(1L<<8)	// shift lock active
} TShiftState;

//-----------------------------------------------------------------------------------------------------------------------------------------
// cwidget xml serialization constant definition
//-----------------------------------------------------------------------------------------------------------------------------------------
// <cwidget shown="bool" enabled="bool">
//   ...
// </cwidget>
//-----------------------------------------------------------------------------------------------------------------------------------------
static CString XML_WIDGET_ELEMENT		("cwidget");
static CString XML_WIDGET_ATTR_ISSHOWN		("shown");
static CString XML_WIDGET_ATTR_ISENABLED	("enabled");

//-----------------------------------------------------------------------------------------------------------------------------------------
// CWidgetListener class
//-----------------------------------------------------------------------------------------------------------------------------------------
class CWidgetListener : public CObjectListener
{
	// instanciation section
	public :

		CWidgetListener			();
		virtual ~CWidgetListener	() =0;

	// specific widget listener functions
	public :

		// called when the gtk widget of inSender is about to be destroyed
		virtual void			OnQueryDestroy			(CObject *inSender, Bool &ioDoDestroy)			{ }

		// called when the gtk widget of inSender has been destroyed
		virtual void			OnDestroy			(CObject *inSender)					{ }


		// called when inSender is about to be shown
		virtual void			OnShow				(CObject *inSender)					{ }
	
		// called when inSender is about to be hidden
		virtual void			OnHide				(CObject *inSender)					{ }


		// called when inSender receives the focus
		virtual void			OnSetFocus			(CObject *inSender)					{ }

		// called when inSender looses the focus
		virtual void			OnKillFocus			(CObject *inSender)					{ }


		// called to perform inSender size calculation
		virtual void			OnQueryResize			(CObject *inSender, TSize &ioSize)			{ }

		// called when inSender has been [re]sized or moved
		virtual void			OnResize			(CObject *inSender)					{ }


		// called when a key has been pressed on inSender
		virtual void			OnKeyPress			(CObject *inSender, int inShiftStates, UInt16 &ioKey)	{ }

		// called when a key has been released on inSender
		virtual void			OnKeyRelease			(CObject *inSender, int inShiftStates, UInt16 inKey)	{ }


		// called when the mouse enters in inSender
		virtual void			OnMouseEnter			(CObject *inSender)					{ }

		// called when the mouse exists from inSender
		virtual void			OnMouseExit			(CObject *inSender)					{ }


		// called when a mouse button is pressed on inSender
		virtual void			OnMouseDown			(CObject *inSender, int inShiftStates,
										 TPoint inRelativePoint)				{ }
		// called when a mouse button is released on inSender
		virtual void			OnMouseUp			(CObject *inSender, int inShiftStates,
										 TPoint inRelativePoint)				{ }
		// called when mouse is moved over inSender
		virtual void			OnMouseMove			(CObject *inSender, int inShiftStates,
										 TPoint inRelativePoint)				{ }

		// called when a drag is started on ioSender, you may change the ioSender parameter to reflect wich control will 
		// effectively be managed by the dnd gtkol engine, the relative point to set your personnal pointer's hot spot on the 
		// pre-allocated drag representation of the control that you can change as well or delete and set to null if so
		virtual void			OnDragStart			(CObject *&ioSender, TPoint &ioRelativePoint, 
										 CPixbuf *&ioPixbuf)					{ }
		// called on inDragged when inDragged is moved over inReceiver; the gtkol inTargeted control reflects wich control would
		// own the inDragged control if a drop occured, may equal the inReceiver parameter or its parent layout if inReceiver is
		// not a drop site itself; inRelativePoint is relative to the inReceiver widget or inReceiver parent widget if not a
		// widget itself
		virtual void			OnDragOver			(CObject *inDragged, CObject *inReceiver, 
										 CObject *inTargeted, TPoint inRelativePoint, 
										 Bool &ioAccept)					{ }
		// called on inDragged when inDragged is dropped on inReceiver; the gtkol ioTargeted control reflects wich control would
		// own the inDragged control if the ioDoDrop parameter is set to true and the expected gtkol hierarchy rules are respected,
		// may equal the inReceiver parameter or its parent layout if inReceiver is not a drop site itself; the ioIndex parameter
		// specifies the index placement of the inDragged control in its new owner if droped on; inRelativePoint is relative to 
		// the inReceiver widget or inReceiver parent widget if not a widget itself
		virtual void			OnDragDrop			(CObject *inDragged, CObject *inReceiver,
										 CObject *&ioTargeted, SInt16 &ioIndex, 
										 TPoint inRelativePoint, Bool &ioDoDrop)		{ }
		// called when drag motion of inSender is ending
		virtual void			OnDragStop			(CObject *inSender)					{ }


		// called when inSender is about to be painted on graphics specified invalidated bounds
		virtual void			OnPaint				(CObject *inSender, CGraphics &ioGraphics)		{ }

		// called after the specified GDK event has been handled on inSender
		virtual void			OnEventAfter			(CObject *inSender, GdkEvent *inGdkEvent)		{ }

		// metaclass association
		SECTION_GENERIC_METACLASS;
};

// metaclass and class tag declaration
DECLARE_GENERIC_METACLASS ('_wdg', CWidgetListener, CObjectListener);

//-----------------------------------------------------------------------------------------------------------------------------------------
// what's defined here
//-----------------------------------------------------------------------------------------------------------------------------------------
class CWidget; typedef TBuffer <CWidget *> CWidgets;

//-----------------------------------------------------------------------------------------------------------------------------------------
// CWidget class
//-----------------------------------------------------------------------------------------------------------------------------------------
class CWidget : public CControl
{
	// instanciation section
	public :

		CWidget				(CComponent *inOwner, const CWidgetListener *inListener=NULL);
		virtual ~CWidget		();

	// protected services section
	protected :

		// the derived definitions should call the CWidget::CreateWidget function (typically in their constructors) in order to
		// correctly manage the process of associating the gtkol widget instance to the to be allocated gtk widget; it calls 
		// PerformWidgetInstanciate and PerformWidgetInitialize of the specified CWidget instance and performs the gtk container
		// process assignement if any (see CContainer abstraction in ccontainer.h). This function is based on the 
		// m_PerformContainerProcess and m_PerformSignalConnection of the specified instance that default to true and must be
		// set before calling the process as needed
		static Bool			CreateWidget			(CWidget *This);

		// function to call in order to destroy the associated gtk widget of the specified instance; CWidget::DestroyWidget 
		// performs recursive calls in order to properly delete the children instances too and preserve the gui coherence
		static Bool			DestroyWidget			(CWidget *This);

	// protected pure virtual functions definition
	protected :

		// this function should be defined in derived definition; it has to instanciate and return the gtk widget that is to be
		// assoaciated to this gtkol widget instance; it is automatically called by CWidget::CreateWidget
		virtual GtkWidget *		PerformWidgetInstanciate	() =0;
		
		// this function is called by CWidget::CreateWidget too and must perform all of the widget initialization if any
		virtual void			PerformWidgetInitialize		() =0;

	// GtkWidget / GTKOL CWidget relationship
	public :

		// get the gtkol instance associated to the specified gtk widget or gdk window
		static CWidget *		GetGtkolWidget			(const GtkWidget *);
		static CWidget *		GetGtkolWidget			(const GdkWindow *);

		// get the gtk widget handled by this gtkol instance
		GtkWidget *			GetGtkWidget			() const;

	// CObject redefintion
	protected :

		// gtkol widget listener affectation; should be derived from CWidgetListener
		virtual const CMetaClass *	ListenerMustBe			() const;

	// CComponent redefinition
	public :

		// check if the specified component could be assigned as an owner of this instance, do not forget that a gtkol
		// container may be restricted to only one child handling, so this version adds a check for the potential inCandidate 
		// container empty state acknowledgment or gtkol layout definition presence
		virtual Bool			CheckSetOwner			(const CComponent *inCandidate) const;

		// set the widget's owner, handles the gui container affectation process if it has to do so (see CContainer abstraction),
		// there is no direct handling of the specified index at the widget level, it is the component's job and containers
		// definitions to set the right gui place accordinaly to the gtkol hierarchy representation
		virtual Bool			SetOwner			(CComponent *inOwner, const SInt16 inIndex=-1);

		// default owner type and expected children type specifies __metaclasses(CWidget)
		virtual CMetaClasses		OwnerMustBe			() const;
		virtual CMetaClasses		ChildMustBe			() const;

	// CControl redefinition
	public :

		// bounds affectation, because the widget position depends on its container managed placement, only the size section of 
		// the bounds is handled at this level; specific overwritten definitions may handle the placement section anyway
		virtual void			SetBounds			(const TBounds &inBounds);

		// draggable and drop site attributes affectation
		virtual void			SetDraggable			(const Bool inDraggable);
		virtual void			SetDropSite			(const Bool inDropSite);

		// default widget pixbuf representation
		virtual CPixbuf *		GetControlPixbuf		() const;

		// get the widget's control at specified relative point, defaults to this instance; this function should be overwritten
		// whenever a derived widget handles specific non windowed controls as this function is used by the gtkol d&d engine layer
		virtual CControl *		GetControlAtPoint		(const TPoint &inRelativePoint) const;

	// general widget functions
	public :

		// widget show / hide
		virtual void			Show				(const Bool inAll=false);
		virtual void			Hide				(const Bool inAll=false);
		Bool				IsShown				() const;

		// sensitive widget
		virtual void			Enable				();
		virtual void			Disable				();
		Bool				IsEnabled			() const;

		// widget focus
		Bool				CanFocus			() const;
		virtual void			SetFocus			();
		virtual Bool			IsFocused			() const;

	// graphics basics
	public :

		// invalidate a portion of the widget i.e. a zone that must be redrawn
		virtual void			Invalidate			(const TBounds &inInvalidate=TBounds());

		// immediately validate the widget invalidation zone set i.e. performs the repaint pending requests if any
		virtual void			Validate			();

	// serialization specifics
	public :

		// cwidget xml serialization
		virtual void			Serialize			(CXMLElementNode *&ioXMLElementNode, const int inMode) 
										 THROWABLE;

	// private attributes section
	private :

		// handled gtk widget
		GtkWidget *			m_GtkWidget;

		// gtkol d&d engine attributes
		static CWidget *		m_DraggedWidget;
		static CControl *		m_DraggedControl;
		static TPoint 			m_DraggedPoint;

	// protected attributes section
	protected :

		// gtkol widget automatic process management for the handled gtk widget (used by the CWidget::CreateWidget function
		// and CWidget::SetOwner definition), if m_PerformContainerProcess is true the widget instanciation process calls the
		// owner container add request if it is a CContainer instance; if m_PerformSignalConnection is true, the process 
		// attaches all widget signals, if not, the process does not attach any of them
		bool				m_PerformContainerProcess;
		bool				m_PerformSignalConnection;

		// the widget serialization defaults to handling the shown and enabled attributes; change those parameters to false
		// if the derived instances should not handle either the shown state or the enabled state serialization attributes
		bool				m_SerializeShown;
		bool				m_SerializeEnabled;

	// protected static -> dynamic events listener resolution
	protected :

		static gboolean			OnQueryDestroy			(GtkWidget *, GdkEvent *, gpointer);
		static void			OnDestroy			(GtkWidget *, gpointer);
		static void			OnShow				(GtkWidget *, gpointer);
		static void			OnHide				(GtkWidget *, gpointer);
		static gboolean                 OnSetFocus              	(GtkWidget *, GdkEventFocus *, gpointer);
                static gboolean                 OnKillFocus             	(GtkWidget *, GdkEventFocus *, gpointer);
		static gboolean                 OnKeyPress              	(GtkWidget *, GdkEventKey *, gpointer);
                static gboolean                 OnKeyRelease            	(GtkWidget *, GdkEventKey *, gpointer);
		static gboolean                 OnMouseDown             	(GtkWidget *, GdkEventButton *, gpointer);
                static gboolean                 OnMouseUp               	(GtkWidget *, GdkEventButton *, gpointer);
                static gboolean                 OnMouseMove             	(GtkWidget *, GdkEventMotion *, gpointer);
                static gboolean                 OnMouseEnter            	(GtkWidget *, GdkEventCrossing *, gpointer);
                static gboolean                 OnMouseExit             	(GtkWidget *, GdkEventCrossing *, gpointer);
		static void			OnDragDataGet			(GtkWidget *, GdkDragContext *, GtkSelectionData *, guint,
										 guint, gpointer);
		static void			OnDragDataReceived		(GtkWidget *, GdkDragContext *, gint, gint,
										 GtkSelectionData *,
										 guint, guint, gpointer);
		static void			OnDragStart			(GtkWidget *, GdkDragContext *, gpointer);
		static gboolean			OnDragOver			(GtkWidget *, GdkDragContext *, gint, gint, guint, 
										 gpointer);
		static gboolean			OnDragDrop			(GtkWidget *, GdkDragContext *, gint, gint, guint, 
										 gpointer);
		static void			OnDragStop			(GtkWidget *, GdkDragContext *, gpointer);
		static gboolean			OnConfigure			(GtkWidget *, GdkEventConfigure	*, gpointer);
		static void			OnQueryResize			(GtkWidget *, GtkRequisition *, gpointer);
		static void			OnResize			(GtkWidget *, GtkAllocation *, gpointer);
		static void			OnEventAfter			(GtkWidget *, GdkEvent *, gpointer);
		static gboolean 		OnPaint  			(GtkWidget *, GdkEventExpose *, gpointer);

		// a reciproq friend of us that should check the m_PerformContainerProcess attribute of gtkol widgets instances
		friend class			CContainer;

		// metaclass association
		SECTION_GENERIC_METACLASS;
};

// metaclass and class tag declaration
DECLARE_GENERIC_METACLASS ('wdgt', CWidget, CControl);

#endif
