/*******************************************************************************************************************************************
 cboxlayout.c
*******************************************************************************************************************************************/

#include "cboxlayout.h"

//-----------------------------------------------------------------------------------------------------------------------------------------
// constructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CBoxLayout::TChildBoxPack::TChildBoxPack (const TBoxPack inBoxPack, const bool inExpand, const bool inFill, const UInt16 inPadding)
			  :m_BoxPack	 (inBoxPack),
			   m_Expand	 (inExpand),
			   m_Fill	 (inFill),
			   m_Padding	 (inPadding)
{ }

//-----------------------------------------------------------------------------------------------------------------------------------------
// =
//-----------------------------------------------------------------------------------------------------------------------------------------
CBoxLayout::TChildBoxPack & CBoxLayout::TChildBoxPack::operator = (const CBoxLayout::TChildBoxPack &inChildBoxPack)
{
	m_BoxPack = inChildBoxPack.m_BoxPack;
	m_Expand  = inChildBoxPack.m_Expand;
	m_Fill 	  = inChildBoxPack.m_Fill;
	m_Padding = inChildBoxPack.m_Padding;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// metaclass resolution
//-----------------------------------------------------------------------------------------------------------------------------------------
RESOLVE_GENERIC_METACLASS (CBoxLayout);

//-----------------------------------------------------------------------------------------------------------------------------------------
// constructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CBoxLayout::CBoxLayout	      (CContainer *inOwner, const CWidgetListener *inListener)
	   :CLayout	      (inOwner, inListener),
	    m_ChildrenBoxPack (),
	    m_Homogeneous     (true),
	    m_Spacing	      (0L)
{ }

//-----------------------------------------------------------------------------------------------------------------------------------------
// destructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CBoxLayout::~CBoxLayout ()
{ }

//-----------------------------------------------------------------------------------------------------------------------------------------
// widget initialization
//-----------------------------------------------------------------------------------------------------------------------------------------
void CBoxLayout::PerformWidgetInitialize ()
{
	// show our job
	Show ();
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// container child add
//-----------------------------------------------------------------------------------------------------------------------------------------
void CBoxLayout::PerformContainerAdd (CWidget *inChild)
{
	// check pointer...
	if (inChild == NULL) return;

	// the child gtk widget
	GtkWidget *inChildGtkWidget = inChild -> GetGtkWidget();

	// our gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget();

	// pointer check
	if (inGtkWidget == NULL || inChildGtkWidget == NULL) return;

	// box pack buffer length check
	if (m_ChildrenBoxPack.GetLength() == 0)
	{
		// get the actual layout children
		CWidgets inChildren (GetGtkChildren() - inChild);

		// are there some children handled yet ?
		if (inChildren.GetLength() > 0)
		{
			// get the last one... 
			CWidget *inWidget = *inChildren[inChildren.GetLength()-1];

			// ... ank keep the current pack attributes
			m_ChildrenBoxPack += CBoxLayout::TChildBoxPack (GetBoxPack(inWidget), GetExpand(inWidget), GetFill(inWidget),
									GetPadding(inWidget));
		}
		// there are no child yet
		else
			// default box pack attributes
			m_ChildrenBoxPack += CBoxLayout::TChildBoxPack ();
	}

	// get the child expected pack parameters, should never be NULL
	CBoxLayout::TChildBoxPack *inChildBoxPack = m_ChildrenBoxPack[0];

	// box pack type analyse
	switch (inChildBoxPack -> m_BoxPack)
	{
		case BOXPACK_START : 
			::gtk_box_pack_start (GTK_BOX(inGtkWidget), inChildGtkWidget, inChildBoxPack -> m_Expand, 
					      inChildBoxPack -> m_Fill, inChildBoxPack -> m_Padding); 
			break;
		case BOXPACK_END : 
			::gtk_box_pack_end   (GTK_BOX(inGtkWidget), inChildGtkWidget, inChildBoxPack -> m_Expand, 
					      inChildBoxPack -> m_Fill, inChildBoxPack -> m_Padding);
			break;
	}

	// remove the first box pack indexed struct from the structure
	m_ChildrenBoxPack.Delete (0, 1);

	// so now, let's see if we got to change the children sequence, get the gtkol widget child in owner index first
	SInt16 inIndex = CContainer::GetGtkInOwnerIndex (this, inChild);

	// then, get the gktol widget handled children
	CWidgets inChildren (GetGtkChildren());

	// if an insert is to be handled (i.e. if it is not the only one child and if it has been found in the gtkol components hierarchy
	// and if it is not the last child, modify the gtk gui children sequence accordinaly)
	if ((inChildren.GetLength() > 1) && (inIndex >= 0) && (inIndex < (inChildren.GetLength()-1)))

		// reorder the gtk children sequence
		::gtk_box_reorder_child (GTK_BOX(inGtkWidget), inChildGtkWidget, inIndex);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// container child remove
//-----------------------------------------------------------------------------------------------------------------------------------------
void CBoxLayout::PerformContainerRemove (CWidget *inChild)
{
	// clayout generic call...
	CLayout::PerformContainerRemove (inChild);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// box pack style affectation
//-----------------------------------------------------------------------------------------------------------------------------------------
void CBoxLayout::SetBoxPack (const TBoxPack inBoxPack, const bool inExpand, const bool inFill, const UInt16 inPadding)
{
	// box pack attribute append
	m_ChildrenBoxPack += CBoxLayout::TChildBoxPack ();

	// get the box pack attributes the box instance is intersted in
	CBoxLayout::TChildBoxPack *inChildBoxPack = m_ChildrenBoxPack[m_ChildrenBoxPack.GetLength()-1];

	// set its attributes for the next container add request
	inChildBoxPack -> m_BoxPack = inBoxPack;
	inChildBoxPack -> m_Expand  = inExpand;
	inChildBoxPack -> m_Fill    = inFill;
	inChildBoxPack -> m_Padding = inPadding;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// box pack access
//-----------------------------------------------------------------------------------------------------------------------------------------
TBoxPack CBoxLayout::GetBoxPack (const CWidget *inChild) const
{
	// our gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// check pointer
	if (inGtkWidget == NULL) return BOXPACK_START;

	// check pointer
	if (inChild == NULL) return BOXPACK_START;

	// get the child gtk widget
	GtkWidget *inChildGtkWidget = inChild -> GetGtkWidget ();

	// check pointer
	if (inChildGtkWidget == NULL) return BOXPACK_START;

	// get its attributes
	gboolean inExpand, inFill; guint inPadding; GtkPackType inPackType;
	::gtk_box_query_child_packing (GTK_BOX(inGtkWidget), inChildGtkWidget, &inExpand, &inFill, &inPadding, &inPackType);

	// ok
	return (inPackType == GTK_PACK_START) ? BOXPACK_START : BOXPACK_END;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// homogeneous writer
//-----------------------------------------------------------------------------------------------------------------------------------------
void CBoxLayout::SetHomogeneous (const bool inHomogeneous)
{
	// local copy
	m_Homogeneous = inHomogeneous;

	// our gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// check pointer
	if (inGtkWidget == NULL) return;

	// attribute affectation
	::gtk_box_set_homogeneous (GTK_BOX(inGtkWidget), inHomogeneous);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// homogeneous access
//-----------------------------------------------------------------------------------------------------------------------------------------
bool CBoxLayout::GetHomogeneous () const
{
	// our gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// ok
	return inGtkWidget != NULL ? ::gtk_box_get_homogeneous (GTK_BOX(inGtkWidget)) : false;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// expand access
//-----------------------------------------------------------------------------------------------------------------------------------------
bool CBoxLayout::GetExpand (const CWidget *inChild) const
{
	// our gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// check pointer
	if (inGtkWidget == NULL) return false;

	// check pointer
	if (inChild == NULL) return false;

	// get the child gtk widget
	GtkWidget *inChildGtkWidget = inChild -> GetGtkWidget ();

	// check pointer
	if (inChildGtkWidget == NULL) return false;

	// get its attributes
	gboolean inExpand, inFill; guint inPadding; GtkPackType inPackType;
	::gtk_box_query_child_packing (GTK_BOX(inGtkWidget), inChildGtkWidget, &inExpand, &inFill, &inPadding, &inPackType);

	// ok
	return inExpand;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// fill access
//-----------------------------------------------------------------------------------------------------------------------------------------
bool CBoxLayout::GetFill (const CWidget *inChild) const
{
	// our gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// check pointer
	if (inGtkWidget == NULL) return false;

	// check pointer
	if (inChild == NULL) return false;

	// get the child gtk widget
	GtkWidget *inChildGtkWidget = inChild -> GetGtkWidget ();

	// check pointer
	if (inChildGtkWidget == NULL) return false;

	// get its attributes
	gboolean inExpand, inFill; guint inPadding; GtkPackType inPackType;
	::gtk_box_query_child_packing (GTK_BOX(inGtkWidget), inChildGtkWidget, &inExpand, &inFill, &inPadding, &inPackType);

	// ok
	return inFill;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// spacing writer
//-----------------------------------------------------------------------------------------------------------------------------------------
void CBoxLayout::SetSpacing (const UInt16 inSpacing)
{
	// local copy
	m_Spacing = inSpacing;

	// our gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// check pointer
	if (inGtkWidget == NULL) return;

	// ok
	::gtk_box_set_spacing (GTK_BOX(inGtkWidget), inSpacing);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// spacing access
//-----------------------------------------------------------------------------------------------------------------------------------------
UInt16 CBoxLayout::GetSpacing () const
{
	// our gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// ok
	return inGtkWidget != NULL ? ::gtk_box_get_spacing (GTK_BOX(inGtkWidget)) : false;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// padding access
//-----------------------------------------------------------------------------------------------------------------------------------------
UInt16 CBoxLayout::GetPadding (const CWidget *inChild) const
{
	// our gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// check pointer
	if (inGtkWidget == NULL) return BOXPACK_START;

	// check pointer
	if (inChild == NULL) return BOXPACK_START;

	// get the child gtk widget
	GtkWidget *inChildGtkWidget = inChild -> GetGtkWidget ();

	// check pointer
	if (inChildGtkWidget == NULL) return BOXPACK_START;

	// get its attributes
	gboolean inExpand, inFill; guint inPadding; GtkPackType inPackType;
	::gtk_box_query_child_packing (GTK_BOX(inGtkWidget), inChildGtkWidget, &inExpand, &inFill, &inPadding, &inPackType);

	// ok
	return inPadding;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// xml serialization
//-----------------------------------------------------------------------------------------------------------------------------------------
void CBoxLayout::Serialize (CXMLElementNode *&ioXMLElementNode, const int inMode) THROWABLE
{
	// serialization request analyse
	switch (inMode)
	{
		// xml dump
		case XML_WRITE :
		{
			// generic serialization first in this case
			CLayout::Serialize (ioXMLElementNode, inMode);

			// instanciate a new xml element
			CXMLElement *newXMLElement = new CXMLElement (ioXMLElementNode, XML_BOXLAYOUT_ELEMENT);

			// set the node attributes
			newXMLElement -> AddAttribute (XML_BOXLAYOUT_ATTR_SPACING,     CString((UInt32)GetSpacing()));
			newXMLElement -> AddAttribute (XML_BOXLAYOUT_ATTR_HOMOGENEOUS, CString(GetHomogeneous() ? "true" : "false"));

			// modify the input/output xml node so that serialization process continues under the current node
			ioXMLElementNode = newXMLElement -> GetXMLElementNode ();

			// get the direct children of this instance, expected children type is CWidget, forget the others
			CComponents inChildren (GetChildren(__metaclass(CWidget)));

			// foreach child, create an xml node
			for (size_t i=inChildren.GetLength(), j=0; i>0; i--, j++)
			{
				// current child
				CWidget *inChild = static_cast <CWidget *> (*inChildren[j]);

				// instanciate a new xml element
				newXMLElement = new CXMLElement (ioXMLElementNode, XML_BOXLAYOUT_CHILD_ELEMENT);

				// add the attributes to the xml element
				newXMLElement -> AddAttribute (XML_BOXLAYOUT_ATTR_CHILD_BOX_PACK, 
							       CString(GetBoxPack(inChild) == BOXPACK_START ? "start" : "end"));
				newXMLElement -> AddAttribute (XML_BOXLAYOUT_ATTR_CHILD_EXPAND, 
							       CString(GetExpand(inChild) ? "true" : "false"));
				newXMLElement -> AddAttribute (XML_BOXLAYOUT_ATTR_CHILD_FILL, 
							       CString(GetFill(inChild) ? "true" : "false"));
				newXMLElement -> AddAttribute (XML_BOXLAYOUT_ATTR_CHILD_PADDING, 
							       CString((UInt32)GetPadding(inChild)));
			}
		}
		break;

		// xml load
		case XML_READ :
		{
			// search for the node this instance is interested in
			CXMLElementNode *inXMLNode = ::xml_node_search (ioXMLElementNode, XML_BOXLAYOUT_ELEMENT); 

			// check we got an expected cboxlayout node
			if (inXMLNode == NULL)
				throw new CException (CString("CBoxLayout::Serialize, specified xml node is not a \"") + 
							      XML_BOXLAYOUT_ELEMENT + CString("\" element one."), __exception(XMLPARSE));

			// read in the node attributes
			SetSpacing     (::xml_node_get_attribute (inXMLNode, XML_BOXLAYOUT_ATTR_SPACING    ).GetValue().ToULong());
			SetHomogeneous (::xml_node_get_attribute (inXMLNode, XML_BOXLAYOUT_ATTR_HOMOGENEOUS).GetValue().ToBool ());

			// foreach child of the current xml node
			for (size_t i=0; i<xml_node_get_children_number(inXMLNode); i++)
			{
				// get the child node
				CXMLElementNode *inXMLChild = ::xml_node_get_child (inXMLNode, i);

				// if the xml node is a child pack specification node
				if (::xml_node_get_name (inXMLChild) == XML_BOXLAYOUT_CHILD_ELEMENT)
				{
					// read in the attributes
					TBoxPack inBoxPack = ::xml_node_get_attribute 
						(inXMLChild, XML_BOXLAYOUT_ATTR_CHILD_BOX_PACK).GetValue() == 
						 CString("start") ? BOXPACK_START : BOXPACK_END;
					bool inExpand = ::xml_node_get_attribute 
						(inXMLChild, XML_BOXLAYOUT_ATTR_CHILD_EXPAND).GetValue().ToBool();
					bool inFill = ::xml_node_get_attribute 
						(inXMLChild, XML_BOXLAYOUT_ATTR_CHILD_FILL).GetValue().ToBool();
					UInt16 inPadding = ::xml_node_get_attribute 
						(inXMLChild, XML_BOXLAYOUT_ATTR_CHILD_PADDING).GetValue().ToULong();

					// add the attributes to the local buffer of packs so that when the children will be added, the
					// pack attributes will be applied
					m_ChildrenBoxPack += CBoxLayout::TChildBoxPack (inBoxPack, inExpand, inFill, inPadding);
				}
			}

			// now we got our box attributes and we are ready to handle container add requests, so call the generic
			// serialization process that will instanciate all of the components
			CLayout::Serialize (ioXMLElementNode, inMode);

			// modify the in/out element node pointer so that the serialization process will continue under the cwidget node
			ioXMLElementNode = inXMLNode;
		}
		break;
	}
}

