/*******************************************************************************************************************************************
 cbutton.c
*******************************************************************************************************************************************/

#include "cbutton.h"
#include "cimage.h"

//-----------------------------------------------------------------------------------------------------------------------------------------
// metaclass code resolution
//-----------------------------------------------------------------------------------------------------------------------------------------
RESOLVE_GENERIC_METACLASS (CButtonListener);

//-----------------------------------------------------------------------------------------------------------------------------------------
// constructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CButtonListener::CButtonListener ()
{ }

//-----------------------------------------------------------------------------------------------------------------------------------------
// destructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CButtonListener::~CButtonListener ()
{ }

//-----------------------------------------------------------------------------------------------------------------------------------------
// OnClick
//-----------------------------------------------------------------------------------------------------------------------------------------
void CButton::OnClick (GtkButton *, gpointer inData)
{
	// retreive the gtkol instance
	CWidget *theWidget = reinterpret_cast <CWidget *> (inData);

	// send the notification if any listener available
	if (theWidget != NULL)
		if (theWidget -> GetListener() != NULL)
			static_cast <CButtonListener *> (theWidget -> GetListener()) -> OnClick (theWidget);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// metaclass code resolution
//-----------------------------------------------------------------------------------------------------------------------------------------
RESOLVE_DYNAMIC_METACLASS (CButton);

//-----------------------------------------------------------------------------------------------------------------------------------------
// constructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CButton::CButton    (CContainer *inOwner, const CButtonListener *inListener)
	:CContainer (inOwner, inListener)
{
	// launch the gtkol instanciation process if the owner is specified
	if (inOwner != NULL) CWidget::CreateWidget (this);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// destructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CButton::~CButton ()
{
	// widget deletion coherence requested
	CWidget::DestroyWidget (this);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// gtk widget instanciation
//-----------------------------------------------------------------------------------------------------------------------------------------
GtkWidget * CButton::PerformWidgetInstanciate ()
{
	return ::gtk_button_new ();
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// gtk widget initialization
//-----------------------------------------------------------------------------------------------------------------------------------------
void CButton::PerformWidgetInitialize ()
{
	// our gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// check pointer
	if (inGtkWidget == NULL) return;

	// specific signal connection
	::g_signal_connect (G_OBJECT(inGtkWidget), "clicked", G_CALLBACK(CButton::OnClick), this);

	// default caption
	SetCaption ("id"+CString(GetId()));

	// show our job
	Show ();
}

//---------------------------------------------------------------------------------------------------------------------------------------
// container add request
//---------------------------------------------------------------------------------------------------------------------------------------
void CButton::PerformContainerAdd (CWidget *inChild)
{
	// check the child instance
	if (inChild && inChild -> ClassIs (__metaclass(CImage)))
	{
		// retreive our gtk widget instance
		GtkWidget *inGtkWidget (GetGtkWidget());

		// check pointer 
		if (inGtkWidget == NULL) return;

		// set the button's image
		::gtk_button_set_image (GTK_BUTTON(inGtkWidget), inChild->GetGtkWidget());
	}
	// "classic" child, generic call
	else CContainer::PerformContainerAdd (inChild);
}

//---------------------------------------------------------------------------------------------------------------------------------------
// container remove request
//---------------------------------------------------------------------------------------------------------------------------------------
void CButton::PerformContainerRemove (CWidget *inChild)
{
	// check the child instance
	if (inChild && inChild -> ClassIs (__metaclass(CImage)))
	{
		// retreive our gtk widget instance
		GtkWidget *inGtkWidget (GetGtkWidget());

		// check pointer 
		if (inGtkWidget == NULL) return;

		// set the button's image
		::gtk_button_set_image (GTK_BUTTON(inGtkWidget), NULL);
	}
	// "classic" child, generic call
	else CContainer::PerformContainerRemove (inChild);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// listener affectation
//-----------------------------------------------------------------------------------------------------------------------------------------
const CMetaClass * CButton::ListenerMustBe () const
{
	return __metaclass(CButtonListener);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// expected owner type
//-----------------------------------------------------------------------------------------------------------------------------------------
CMetaClasses CButton::OwnerMustBe () const
{
	return __metaclasses(CContainer);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// caption affectation
//-----------------------------------------------------------------------------------------------------------------------------------------
void CButton::SetCaption (const CString &inCaption)
{
	// our widget
	GtkWidget *inGtkWidget = GetGtkWidget();

	// gtk affectation
	if (inGtkWidget != NULL)
	{
		::gtk_button_set_label (GTK_BUTTON(inGtkWidget), inCaption.Get());
		if (inCaption.Find ('_')) ::gtk_button_set_use_underline (GTK_BUTTON(inGtkWidget), true);
	}
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// caption access
//-----------------------------------------------------------------------------------------------------------------------------------------
CString CButton::GetCaption () const
{
	// our widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// ok ?
	return inGtkWidget != NULL ? CString(::gtk_button_get_label (GTK_BUTTON(inGtkWidget))) : CString();
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// click event sender
//-----------------------------------------------------------------------------------------------------------------------------------------
void CButton::Click ()
{
	if (GetGtkWidget() != NULL) ::gtk_button_clicked (GTK_BUTTON(GetGtkWidget()));
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// xml serialization
//-----------------------------------------------------------------------------------------------------------------------------------------
void CButton::Serialize (CXMLElementNode *&ioXMLElementNode, const int inMode) THROWABLE
{
	// generic serialization
	CContainer::Serialize (ioXMLElementNode, inMode);

	// serialization request analyse
	switch (inMode)
	{
		// xml dump
		case XML_WRITE :
		{
			// instanciate a new xml element under the current one
			CXMLElement *newXMLElement = new CXMLElement (ioXMLElementNode, XML_BUTTON_ELEMENT);

			// add the cbutton attributes
			newXMLElement -> AddAttribute (XML_BUTTON_ATTR_CAPTION, GetCaption());

			// modify the input output pointer so that the overwritten serialization process continues under this element
			ioXMLElementNode = newXMLElement -> GetXMLElementNode ();
		}
		break;
	
		// xml load
		case XML_READ :
		{
			// get the child node this serialization process is interested in
			CXMLElementNode *inXMLNode = ::xml_node_get_child (ioXMLElementNode, XML_BUTTON_ELEMENT);

			// check we got an expected cbutton node
			if (inXMLNode == NULL)
				throw new CException (CString("CButton::Serialize, specified xml node is not a \"") + 
							      XML_BUTTON_ELEMENT + CString("\" element one."), __exception(XMLPARSE));

			// modify the in/out element node pointer so that the serialization process will continue under the cbutton node
			ioXMLElementNode = inXMLNode;

			// set the button caption from the expected attribute
			SetCaption (::xml_node_get_attribute (inXMLNode, XML_BUTTON_ATTR_CAPTION).GetValue());
		}
		break;
	}
}
