/*******************************************************************************************************************************************
 cdialog.c
*******************************************************************************************************************************************/

#include "cdialog.h"

//-----------------------------------------------------------------------------------------------------------------------------------------
// metaclass code resolution
//-----------------------------------------------------------------------------------------------------------------------------------------
RESOLVE_GENERIC_METACLASS (CDialogListener);

//-----------------------------------------------------------------------------------------------------------------------------------------
// constructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CDialogListener::CDialogListener ()
{ }

//-----------------------------------------------------------------------------------------------------------------------------------------
// destructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CDialogListener::~CDialogListener ()
{ }

//-----------------------------------------------------------------------------------------------------------------------------------------
// metaclass code resolution
//-----------------------------------------------------------------------------------------------------------------------------------------
RESOLVE_DYNAMIC_METACLASS (CDialog);

//-----------------------------------------------------------------------------------------------------------------------------------------
// constructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CDialog::CDialog   (CForm *inOwner, const CString &inCaption, const int inButtons, const bool inModal, const CDialogListener *inListener)
	:CForm	   (NULL, inListener),
	 m_Modal   (inModal),
	 m_Buttons (inButtons),
	 m_Caption (inCaption==CString()?CString(GetId()):inCaption)
{
	// do not perform the container process on this instance when considered as a child, i.e. do not perform any automatic gui
	// container add on handling with this dialog instance as child of a container or it would result in some unwanted gtk errors !
	CWidget::m_PerformContainerProcess = false;

	// launch the gtk widget instanciation process
	if (inOwner != NULL) SetOwner (inOwner);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// destructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CDialog::~CDialog ()
{
	// deletion coherence requested
	CWidget::DestroyWidget (this);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// gtk widget instanciation
//-----------------------------------------------------------------------------------------------------------------------------------------
GtkWidget * CDialog::PerformWidgetInstanciate ()
{
	// be simple...
	return ::gtk_dialog_new ();
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// gtk widget initialization
//-----------------------------------------------------------------------------------------------------------------------------------------
void CDialog::PerformWidgetInitialize ()
{
	// generic call first
	CForm::PerformWidgetInitialize ();

	// dialog title, standard buttons and modal state affectation
	SetCaption (m_Caption);
	SetModal   (m_Modal);
	if (m_Buttons != _DBNone_) SetButtons (m_Buttons);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// container add on
//-----------------------------------------------------------------------------------------------------------------------------------------
void CDialog::PerformContainerAdd (CWidget *inChild)
{
	// get our widget
	GtkWidget *inGtkWidget = GetGtkWidget();

	// get the child widget
	GtkWidget *inChildGtkWidget = inChild == NULL ? NULL : inChild -> GetGtkWidget();

	// pointer check
	if (inGtkWidget == NULL || inChildGtkWidget == NULL) return;

	// add the child to the box as it is the only function we got, not insert handled directly
	::gtk_box_pack_start (GTK_BOX(GTK_DIALOG(inGtkWidget)->vbox), inChildGtkWidget, true, true, 0);

	// so now, let's see if we got to change the children sequence, get the gtkol widget child in owner index first
	SInt16 inIndex = CContainer::GetGtkInOwnerIndex (this, inChild);

	// then, get the gktol widget handled children
	CWidgets inChildren (GetGtkChildren());

	// if an insert is to be handled (i.e. if it is not the only one child and if it has been found in the gtkol components hierarchy
	// and if it is not the last child, modify the gtk gui children sequence accordinaly)
	if ((inChildren.GetLength() > 1) && (inIndex >= 0) && (inIndex < (inChildren.GetLength()-1)))

		// reorder the gtk children sequence
		::gtk_box_reorder_child (GTK_BOX(GTK_DIALOG(inGtkWidget)->vbox), inChildGtkWidget, inIndex);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// container remove
//-----------------------------------------------------------------------------------------------------------------------------------------
void CDialog::PerformContainerRemove (CWidget *inChild)
{
	// get our widget
	GtkWidget *inGtkWidget = GetGtkWidget();

	// get the child widget
	GtkWidget *inChildGtkWidget = inChild == NULL ? NULL : inChild -> GetGtkWidget();

	// pointer check
	if (inGtkWidget == NULL || inChildGtkWidget == NULL) return;

	// remove the child from the child box
	::gtk_container_remove (GTK_CONTAINER(GTK_DIALOG(inGtkWidget)->vbox), inChildGtkWidget);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// expected listener
//-----------------------------------------------------------------------------------------------------------------------------------------
const CMetaClass * CDialog::ListenerMustBe () const
{
	return __metaclass(CDialogListener);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// expected owner type
//-----------------------------------------------------------------------------------------------------------------------------------------
CMetaClasses CDialog::OwnerMustBe () const
{
	return __metaclasses(CForm);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// owner affectation
//-----------------------------------------------------------------------------------------------------------------------------------------
bool CDialog::SetOwner (CComponent *inOwner, const SInt16 inIndex)
{
	// generic call first
	if (!CForm::SetOwner (inOwner, inIndex)) return false;

	// transient for affectation (we know there that inOwner is a CForm instance as the return value of OwnerMustBe has been checked
	// by the generic definition of SetOwner called above)
	if (GetGtkWidget() != NULL)
		::gtk_window_set_transient_for (GTK_WINDOW(GetGtkWidget()), 
						inOwner != NULL ? GTK_WINDOW(static_cast <CForm *> (inOwner) -> GetGtkWidget()) : NULL);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// modal state affectation
//-----------------------------------------------------------------------------------------------------------------------------------------
void CDialog::SetModal (const bool inModal)
{
	// local copy
	m_Modal = inModal;

	// our gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget();

	// pointer check
	if (inGtkWidget == NULL) return;

	// property affectation
	::gtk_window_set_modal (GTK_WINDOW(inGtkWidget), inModal);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// modal state read in
//-----------------------------------------------------------------------------------------------------------------------------------------
bool CDialog::GetModal () const
{
	return m_Modal;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// standard buttons affectation
//-----------------------------------------------------------------------------------------------------------------------------------------
void CDialog::SetButtons (const int inButtons)
{
	// local property copy
	m_Buttons = inButtons;

	// get our widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// pointer check
	if (inGtkWidget == NULL) return;

	// remove the action area children if any; so get the action area widget
	GList *inChildren = ::gtk_container_get_children (GTK_CONTAINER(GTK_DIALOG(inGtkWidget)->action_area));

	// go through the children
	while (inChildren != NULL)
	{
		// instance check
		if (GTK_IS_WIDGET(inChildren->data))

			// remove the child
			::gtk_container_remove (GTK_CONTAINER(GTK_DIALOG(inGtkWidget)->action_area), GTK_WIDGET(inChildren->data));

		// next one
		inChildren = inChildren -> next;
	}

	// free the list
	::g_list_free (inChildren);

	// add the buttons
	if (m_Buttons & DIALOG_OK)     ::gtk_dialog_add_button (GTK_DIALOG(inGtkWidget), "_Ok",     DIALOG_OK);
	if (m_Buttons & DIALOG_YES)    ::gtk_dialog_add_button (GTK_DIALOG(inGtkWidget), "_Yes",    DIALOG_YES);
	if (m_Buttons & DIALOG_NO)     ::gtk_dialog_add_button (GTK_DIALOG(inGtkWidget), "_No",     DIALOG_NO);
	if (m_Buttons & DIALOG_REJECT) ::gtk_dialog_add_button (GTK_DIALOG(inGtkWidget), "_Reject", DIALOG_REJECT);
	if (m_Buttons & DIALOG_ACCEPT) ::gtk_dialog_add_button (GTK_DIALOG(inGtkWidget), "_Accept", DIALOG_ACCEPT);
	if (m_Buttons & DIALOG_CANCEL) ::gtk_dialog_add_button (GTK_DIALOG(inGtkWidget), "_Cancel", DIALOG_CANCEL);
	if (m_Buttons & DIALOG_CLOSE)  ::gtk_dialog_add_button (GTK_DIALOG(inGtkWidget), "_Close",  DIALOG_CLOSE);
	if (m_Buttons & DIALOG_APPLY)  ::gtk_dialog_add_button (GTK_DIALOG(inGtkWidget), "_Apply",  DIALOG_APPLY);
	if (m_Buttons & DIALOG_HELP)   ::gtk_dialog_add_button (GTK_DIALOG(inGtkWidget), "_Help",   DIALOG_HELP);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// standard buttons reader
//-----------------------------------------------------------------------------------------------------------------------------------------
int CDialog::GetButtons () const
{
	return m_Buttons;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// dialog box run
//-----------------------------------------------------------------------------------------------------------------------------------------
TDialog CDialog::Run ()
{
	// get our gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// pointer check
	if (inGtkWidget == NULL) return DIALOG_NONE;

	// show the dialog
	Show ();

	// launch it
	TDialog outResult = DIALOG_NONE; switch (::gtk_dialog_run (GTK_DIALOG(inGtkWidget)))
	{
		case DIALOG_NONE	 :
		case GTK_RESPONSE_NONE 	 : outResult = DIALOG_NONE;   break;
		case DIALOG_OK		 :
        	case GTK_RESPONSE_OK	 : outResult = DIALOG_OK;     break;
		case DIALOG_YES		 :
        	case GTK_RESPONSE_YES	 : outResult = DIALOG_YES;    break;
		case DIALOG_NO		 :
		case GTK_RESPONSE_NO	 : outResult = DIALOG_NO;     break;
		case DIALOG_REJECT	 :
		case GTK_RESPONSE_REJECT : outResult = DIALOG_REJECT; break;
		case DIALOG_ACCEPT	 :
		case GTK_RESPONSE_ACCEPT : outResult = DIALOG_ACCEPT; break;
		case DIALOG_CANCEL	 :
		case GTK_RESPONSE_CANCEL : outResult = DIALOG_CANCEL; break;
		case DIALOG_CLOSE	 :
		case GTK_RESPONSE_CLOSE	 : outResult = DIALOG_CLOSE;  break;
		case DIALOG_APPLY	 :
		case GTK_RESPONSE_APPLY	 : outResult = DIALOG_APPLY;  break;
		case DIALOG_HELP	 :
        	case GTK_RESPONSE_HELP	 : outResult = DIALOG_HELP;   break;
	}

	// is there a listener ?
	if (m_Listener != NULL)
		static_cast <CDialogListener *> (m_Listener) -> OnResponse (this, outResult);

	// close the dialog
	Close ();

	// ok
	return outResult;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// response requested
//-----------------------------------------------------------------------------------------------------------------------------------------
void CDialog::Response (const TDialog inDialog)
{
	// get our gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// pointer check...
	if (inGtkWidget != NULL) ::gtk_dialog_response (GTK_DIALOG(inGtkWidget), inDialog);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// xml serialization
//-----------------------------------------------------------------------------------------------------------------------------------------
void CDialog::Serialize (CXMLElementNode *&ioXMLElementNode, const int inMode) THROWABLE
{
	// generic process call first
	CForm::Serialize (ioXMLElementNode, inMode);

	// serialization request analyse
	switch (inMode)
	{
		// xml dump
		case XML_WRITE :
		{
			// instanciate a new xml element to store the attributes this instance is supposed to handle
			CXMLElement *newXMLElement = new CXMLElement (ioXMLElementNode, XML_DIALOG_ELEMENT);

			// register the xml modal attribute
			newXMLElement -> AddAttribute (XML_DIALOG_ATTR_MODAL, CString(m_Modal ? "true" : "false"));
			
			// set the standard buttons serialization string to be registered
			CString outButtons; if (m_Buttons != DIALOG_NONE) 
			{
				if (m_Buttons & DIALOG_OK) 
					outButtons = CString("ok");
				if (m_Buttons & DIALOG_YES) 
					if (outButtons.GetLength()>0) outButtons+=CString(",yes"); else outButtons+=CString("yes");
				if (m_Buttons & DIALOG_NO) 
					if (outButtons.GetLength()>0) outButtons+=CString(",no");  else outButtons+=CString("no");
				if (m_Buttons & DIALOG_REJECT)
					if (outButtons.GetLength()>0) outButtons+=CString(",reject"); else outButtons+=CString("reject");
				if (m_Buttons & DIALOG_ACCEPT)
					if (outButtons.GetLength()>0) outButtons+=CString(",accept"); else outButtons+=CString("accept");
				if (m_Buttons & DIALOG_CANCEL)
					if (outButtons.GetLength()>0) outButtons+=CString(",cancel"); else outButtons+=CString("cancel");
				if (m_Buttons & DIALOG_CLOSE)
					if (outButtons.GetLength()>0) outButtons+=CString(",close"); else outButtons+=CString("close");
				if (m_Buttons & DIALOG_APPLY)
					if (outButtons.GetLength()>0) outButtons+=CString(",apply"); else outButtons+=CString("apply");
				if (m_Buttons & DIALOG_HELP)
					if (outButtons.GetLength()>0) outButtons+=CString(",help"); else outButtons+=CString("help");

				// serialize the attribute
				newXMLElement -> AddAttribute (XML_DIALOG_ATTR_BUTTONS, outButtons);
			}

			// modify the io xml element node so that overloaded definition will continue under the cdialog xml element
			ioXMLElementNode = newXMLElement -> GetXMLElementNode ();
		}
		break;

		// xml load
		case XML_READ :
		{
			// get the xml element node the cdialog is expected to read from
			CXMLElementNode *inXMLNode = ::xml_node_get_child (ioXMLElementNode, XML_DIALOG_ELEMENT);

			// check we got it !
			if (inXMLNode == NULL)
				throw new CException (CString("CDialog::Serialize, specified xml node is not a \"") + 
							      XML_DIALOG_ELEMENT + CString("\" element one."), __exception(XMLPARSE));

			// modify the io xml element node pointer so that overwritten definition continue
			ioXMLElementNode = inXMLNode;

			// get the modal and the buttons attributes
			bool    inModal  = ::xml_node_get_attribute (inXMLNode, XML_DIALOG_ATTR_MODAL).GetValue().ToBool();
			CString inButton = ::xml_node_get_attribute (inXMLNode, XML_DIALOG_ATTR_BUTTONS).GetValue();

			// explose the standard buttons string
			CStrings inButtons (inButton.Cut (CString(","), true)); 
			int outButtons=_DBNone_; for (size_t i=inButtons.GetLength(), j=0; i>0; j++, i--)
			{
				if ((*inButtons[j]) == CString("none"))
				{
					outButtons = DIALOG_NONE;
					break;
				}
				if ((*inButtons[j]) == CString("ok")) 	  outButtons |= DIALOG_OK;
				if ((*inButtons[j]) == CString("yes")) 	  outButtons |= DIALOG_YES;
				if ((*inButtons[j]) == CString("no")) 	  outButtons |= DIALOG_NO;
				if ((*inButtons[j]) == CString("reject")) outButtons |= DIALOG_REJECT;
				if ((*inButtons[j]) == CString("accept")) outButtons |= DIALOG_ACCEPT;
				if ((*inButtons[j]) == CString("cancel")) outButtons |= DIALOG_CANCEL;
				if ((*inButtons[j]) == CString("close"))  outButtons |= DIALOG_CLOSE;
				if ((*inButtons[j]) == CString("apply"))  outButtons |= DIALOG_APPLY;
				if ((*inButtons[j]) == CString("help"))	  outButtons |= DIALOG_HELP;
			}

			// set the modal state and the standard buttons
			SetModal (inModal);
			if (outButtons != _DBNone_) SetButtons (outButtons);
		}
		break;
	}
}



