/*
A library to allow applictions to provide simple indications of
information to be displayed to users of the application through the
interface shell.

Copyright 2009 Canonical Ltd.

Authors:
    Ted Gould <ted@canonical.com>

This program is free software: you can redistribute it and/or modify it 
under the terms of either or both of the following licenses:

1) the GNU Lesser General Public License version 3, as published by the 
Free Software Foundation; and/or
2) the GNU Lesser General Public License version 2.1, as published by 
the Free Software Foundation.

This program is distributed in the hope that it will be useful, but 
WITHOUT ANY WARRANTY; without even the implied warranties of 
MERCHANTABILITY, SATISFACTORY QUALITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the applicable version of the GNU Lesser General Public 
License for more details.

You should have received a copy of both the GNU Lesser General Public 
License version 3 and version 2.1 along with this program.  If not, see 
<http://www.gnu.org/licenses/>
*/

#include "glib.h"
#include "glib/gmessages.h"
#include "indicator.h"
#include "server.h"

/* Signals */
enum {
	HIDE,
	SHOW,
	USER_DISPLAY,
	MODIFIED,
	DISPLAYED,
	LAST_SIGNAL
};

static guint signals[LAST_SIGNAL] = { 0 };

typedef struct _IndicateIndicatorPrivate IndicateIndicatorPrivate;
struct _IndicateIndicatorPrivate
{
	guint id;
	gboolean is_visible;
	IndicateServer * server;
	GHashTable * properties;
	gboolean is_displayed;
};

#define INDICATE_INDICATOR_GET_PRIVATE(o) \
          (G_TYPE_INSTANCE_GET_PRIVATE ((o), INDICATE_TYPE_INDICATOR, IndicateIndicatorPrivate))

G_DEFINE_TYPE (IndicateIndicator, indicate_indicator, G_TYPE_OBJECT);

static void indicate_indicator_finalize (GObject * object);
static void set_property (IndicateIndicator * indicator, const gchar * key, const GValue * data);
static const GValue * get_property (IndicateIndicator * indicator, const gchar * key);
static GPtrArray * list_properties (IndicateIndicator * indicator);


/* Functions */
static void
indicate_indicator_class_init (IndicateIndicatorClass * class)
{
	/* g_debug("Indicator Class Initialized."); */

	GObjectClass * gobj;
	gobj = G_OBJECT_CLASS(class);

	g_type_class_add_private (class, sizeof (IndicateIndicatorPrivate));

	gobj->finalize = indicate_indicator_finalize;

	/** 
		IndicateIndicator::display:
		@arg0: The #IndicateIndicator object

		Emitted when the user has clicked on this indicator.  In the
		messaging indicator this would be when someone clicks on the
		menu item for the indicator.
	*/
	signals[USER_DISPLAY] = g_signal_new(INDICATE_INDICATOR_SIGNAL_DISPLAY,
	                                     G_TYPE_FROM_CLASS(class),
	                                     G_SIGNAL_RUN_LAST,
	                                     G_STRUCT_OFFSET(IndicateIndicatorClass, user_display),
	                                     NULL, NULL,
	                                     g_cclosure_marshal_VOID__UINT,
	                                     G_TYPE_NONE, 1, G_TYPE_UINT);
	/**
		IndicateIndicator::hide:
		@arg0: The #IndicateIndicator object

		Emitted every time this indicator is hidden.  This
		is mostly used by #IndicateServer.

		Typically this results in an emition of #IndicateServer::indicator-removed.
	*/
	signals[HIDE] = g_signal_new(INDICATE_INDICATOR_SIGNAL_HIDE,
	                                     G_TYPE_FROM_CLASS(class),
	                                     G_SIGNAL_RUN_LAST,
	                                     G_STRUCT_OFFSET(IndicateIndicatorClass, hide),
	                                     NULL, NULL,
	                                     g_cclosure_marshal_VOID__VOID,
	                                     G_TYPE_NONE, 0);
	/**
		IndicateIndicator::show:
		@arg0: The #IndicateIndicator object

		Emitted every time this indicator is shown.  This
		is mostly used by #IndicateServer.

		Typically this results in an emition of #IndicateServer::indicator-added.
	*/
	signals[SHOW] = g_signal_new(INDICATE_INDICATOR_SIGNAL_SHOW,
	                                     G_TYPE_FROM_CLASS(class),
	                                     G_SIGNAL_RUN_LAST,
	                                     G_STRUCT_OFFSET(IndicateIndicatorClass, show),
	                                     NULL, NULL,
	                                     g_cclosure_marshal_VOID__VOID,
	                                     G_TYPE_NONE, 0);
	/**
		IndicateIndicator::modified:
		@arg0: The #IndicateIndicator object
		@arg1: The name of the property that changed.

		Emitted every time an indicator property is changed.
		This is mostly used by #IndicateServer.

		Typically this results in an emition of #IndicateServer::indicator-modified.
	*/
	signals[MODIFIED] = g_signal_new(INDICATE_INDICATOR_SIGNAL_MODIFIED,
	                                     G_TYPE_FROM_CLASS(class),
	                                     G_SIGNAL_RUN_LAST,
	                                     G_STRUCT_OFFSET(IndicateIndicatorClass, modified),
	                                     NULL, NULL,
	                                     g_cclosure_marshal_VOID__STRING,
	                                     G_TYPE_NONE, 1, G_TYPE_STRING);
	/**
		IndicateIndicator::displayed:
		@arg0: The #IndicateIndicator object
		@arg1: Whether the indicator has been displayed

		This is the signal that the indicator has been displayed, or
		hidden by a listener.  In most cases, the signal will be that it
		has been displayed as most folks don't go hiding it later.
	*/
	signals[DISPLAYED] = g_signal_new(INDICATE_INDICATOR_SIGNAL_DISPLAYED,
	                                     G_TYPE_FROM_CLASS(class),
	                                     G_SIGNAL_RUN_LAST,
	                                     G_STRUCT_OFFSET(IndicateIndicatorClass, displayed),
	                                     NULL, NULL,
	                                     g_cclosure_marshal_VOID__BOOLEAN,
	                                     G_TYPE_NONE, 1, G_TYPE_BOOLEAN);

	class->set_property = set_property;
	class->get_property = get_property;
	class->list_properties = list_properties;

	return;
}

/* A small little function to both clear the insides of a 
   value as well as the memory it itself uses. */
static void 
_g_value_free (gpointer data)
{
	if (data == NULL) return;
	GValue * value = (GValue*)data;
	g_value_unset(value);
	g_free(data);
	return;
}

static void
indicate_indicator_init (IndicateIndicator * indicator)
{
	/* g_debug("Indicator Object Initialized."); */
	IndicateIndicatorPrivate * priv = INDICATE_INDICATOR_GET_PRIVATE(indicator);

	priv->is_visible = FALSE;
	priv->is_displayed = FALSE;

	priv->properties = g_hash_table_new_full(g_str_hash, g_str_equal, g_free, _g_value_free);

	priv->server = indicate_server_ref_default();
	priv->id = indicate_server_get_next_id(priv->server);

	indicate_server_add_indicator(priv->server, indicator);

	return;
}

static void
indicate_indicator_finalize (GObject * obj)
{
	IndicateIndicator * indicator = INDICATE_INDICATOR(obj);
	IndicateIndicatorPrivate * priv = INDICATE_INDICATOR_GET_PRIVATE(indicator);

	indicate_server_remove_indicator(priv->server, indicator);
	g_object_unref(priv->server);
	priv->server = NULL;

	G_OBJECT_CLASS (indicate_indicator_parent_class)->finalize (obj);
	return;
}

/**
	indicate_indicator_get_type:

	Gets a unique #GType for the #IndicateIndicator objects.

	Return value: A unique #GType value.
*/

/**
	indicate_indicator_new:

	Builds a new indicator object using g_object_new().

	Return value: A pointer to a new #IndicateIndicator object.
*/
IndicateIndicator *
indicate_indicator_new (void)
{
	IndicateIndicator * indicator = g_object_new(INDICATE_TYPE_INDICATOR, NULL);
	return indicator;
}

/**
	indicate_indicator_new_with_server:
	@server: The server that should be associated with this indicator.

	Builds a new indicator object using g_object_new() and sets
	the server to the specified server.  Also, adds a reference
	to the server.

	Return value: A pointer to a new #IndicateIndicator object.
*/
IndicateIndicator *
indicate_indicator_new_with_server (IndicateServer * server)
{
	g_return_val_if_fail(server != NULL, NULL);

	IndicateIndicator * indicator = g_object_new(INDICATE_TYPE_INDICATOR, NULL);

	indicate_indicator_set_server (indicator, server);

	return indicator;
}

/**
	indicate_indicator_set_server:
	@indicator: a #IndicateIndicator to act on
	@server: the #IndicateServer which should be associated with @indicator

	Defines which server this indicator must be associated to.  Also, adds a
	reference to the server.
*/
void
indicate_indicator_set_server (IndicateIndicator * indicator, IndicateServer * server)
{
	IndicateIndicatorPrivate * priv = INDICATE_INDICATOR_GET_PRIVATE(indicator);

	if (server != NULL) {
		g_object_ref(server);
	}

	if (priv->server != NULL) {
		indicate_server_remove_indicator (priv->server, indicator);
		g_object_unref(priv->server);
	}

	priv->server = server;
	if (server != NULL) {
		indicate_server_add_indicator (server, indicator);
	}

	return;
}


/**
	indicate_indicator_show:
	@indicator: a #IndicateIndicator to act on

	Shows this indicator on the bus.  If the #IndicateServer that it's
	connected to is not shown itself this function will show the server
	as well using #indicate_server_show.
*/
void
indicate_indicator_show (IndicateIndicator * indicator)
{
	IndicateIndicatorPrivate * priv = INDICATE_INDICATOR_GET_PRIVATE(indicator);

	if (priv->is_visible) {
		return;
	}

	if (priv->server) {
		indicate_server_show(priv->server);
	}

	priv->is_visible = TRUE;
	g_signal_emit(indicator, signals[SHOW], 0, TRUE);

	return;
}

/**
	indicate_indicator_hide:
	@indicator: a #IndicateIndicator to act on

	Hides the indicator from the bus.  Does not effect the
	indicator's #IndicateServer in any way.
*/
void
indicate_indicator_hide (IndicateIndicator * indicator)
{
	IndicateIndicatorPrivate * priv = INDICATE_INDICATOR_GET_PRIVATE(indicator);

	if (!priv->is_visible) {
		return;
	}

	priv->is_visible = FALSE;
	g_signal_emit(indicator, signals[HIDE], 0, TRUE);
	priv->is_displayed = FALSE;
	g_signal_emit(G_OBJECT(indicator), signals[DISPLAYED], 0, priv->is_displayed, TRUE);

	return;
}

/**
	indicate_indicator_is_visible:
	@indicator: a #IndicateIndicator to act on

	Checkes the visibility status of @indicator.

	Return value: %TRUE if the indicator is visible else %FALSE.
*/
gboolean
indicate_indicator_is_visible (IndicateIndicator * indicator)
{
	g_return_val_if_fail(INDICATE_IS_INDICATOR(indicator), FALSE);
	IndicateIndicatorPrivate * priv = INDICATE_INDICATOR_GET_PRIVATE(indicator);
	return priv->is_visible;
}

/**
	indicate_indicator_get_id:
	@indicator: a #IndicateIndicator to act on

	Gets the ID value of the @indicator.

	Return value: The ID of the indicator.  Can not be zero.
		Zero represents an error.
*/
guint
indicate_indicator_get_id (IndicateIndicator * indicator)
{
	g_return_val_if_fail(INDICATE_IS_INDICATOR(indicator), 0);
	IndicateIndicatorPrivate * priv = INDICATE_INDICATOR_GET_PRIVATE(indicator);
	return priv->id;
}

/**
	indicate_indicator_get_server:
	@indicator: a #IndicateIndicator to act on

	Gets the server of the @indicator.

	Return value: The server associated with the indicator
		or NULL if error.
*/
IndicateServer *
indicate_indicator_get_server (IndicateIndicator * indicator)
{
	g_return_val_if_fail(INDICATE_IS_INDICATOR(indicator), NULL);
	IndicateIndicatorPrivate * priv = INDICATE_INDICATOR_GET_PRIVATE(indicator);
	return priv->server;
}

/**
	indicate_indicator_user_display:
	@indicator: a #IndicateIndicator to act on
	@timestamp: The time that the event happened

	Emits the #IndicateIndicator::user-display signal simliar to a user
	clicking on @indicator over the bus.  Signal will not be sent if the
	@indicator is not visible.
*/
void
indicate_indicator_user_display (IndicateIndicator * indicator, guint timestamp)
{
	IndicateIndicatorPrivate * priv = INDICATE_INDICATOR_GET_PRIVATE(indicator);
	if (!priv->is_visible) {
		return;
	}

	g_signal_emit(indicator, signals[USER_DISPLAY], 0, timestamp, TRUE);
	return;
}

/**
	indicate_indicator_set_property:
	@indicator: a #IndicateIndicator to act on
	@key: name of the property
	@data: value of the property

	Sets a simple string property on @indicator.  If the property
	had previously been set it will replace it with the new value,
	otherwise it will create the property.  This will include an
	emition of #IndicateIndicator::modified if the property value
	was changed.
*/
void
indicate_indicator_set_property (IndicateIndicator * indicator, const gchar * key, const gchar * data)
{
	IndicateIndicatorClass * class = INDICATE_INDICATOR_GET_CLASS(indicator);
	if (class->set_property == NULL) {
		return;
	}

	GValue strval = {0};
	g_value_init(&strval, G_TYPE_STRING);
	g_value_set_static_string(&strval, data);

	return class->set_property(indicator, key, &strval);
}

/**
	indicate_indicator_set_property_time:
	@indicator: a #IndicateIndicator to act on
	@key: name of the property
	@time: time to set property with

	This is a helper function that wraps around #indicate_indicator_set_property
	but takes an #GTimeVal parameter.  It then takes the @data
	parameter converts it to an ISO 8601 time string and
	uses that data to call #indicate_indicator_set_property.
*/
void
indicate_indicator_set_property_time (IndicateIndicator * indicator, const gchar * key, GTimeVal * time)
{
	gchar * timestr = g_time_val_to_iso8601(time);
	if (timestr != NULL) {
		indicate_indicator_set_property(indicator, key, timestr);
		g_free(timestr);
	}
	return;
}

/**
	indicate_indicator_set_property_int:
	@indicator: a #IndicateIndicator to act on
	@key: name of the property
	@value: integer to set property with

	This is a helper function that wraps around #indicate_indicator_set_property
	but takes an integer property and turns into a string and
	uses that data to call #indicate_indicator_set_property.
*/
void
indicate_indicator_set_property_int (IndicateIndicator * indicator, const gchar * key, gint value)
{
	IndicateIndicatorClass * class = INDICATE_INDICATOR_GET_CLASS(indicator);
	if (class->set_property == NULL) {
		return;
	}

	GValue intval = {0};
	g_value_init(&intval, G_TYPE_INT);
	g_value_set_int(&intval, value);

	return class->set_property(indicator, key, &intval);
}

/**
	indicate_indicator_set_property_bool:
	@indicator: a #IndicateIndicator to act on
	@key: name of the property
	@value: integer to set property with

	This is a helper function that wraps around #indicate_indicator_set_property
	but takes a boolean property and turns into a string and
	uses that data to call #indicate_indicator_set_property.
*/
void
indicate_indicator_set_property_bool (IndicateIndicator * indicator, const gchar * key, gboolean value)
{
	IndicateIndicatorClass * class = INDICATE_INDICATOR_GET_CLASS(indicator);
	if (class->set_property == NULL) {
		return;
	}

	GValue boolval = {0};
	g_value_init(&boolval, G_TYPE_BOOLEAN);
	g_value_set_boolean(&boolval, value);

	return class->set_property(indicator, key, &boolval);
}

void
indicate_indicator_set_property_value (IndicateIndicator * indicator, const gchar * key, GValue * value)
{
	IndicateIndicatorClass * class = INDICATE_INDICATOR_GET_CLASS(indicator);
	if (class->set_property == NULL) {
		return;
	}

	return class->set_property(indicator, key, value);
}

/**
	indicate_indicator_get_property:
	@indicator: a #IndicateIndicator to act on
	@key: name of the property

	Returns the value that is set for a property or %NULL if that
	property is not set.

	Return value: A constant string or NULL.
*/
const gchar *
indicate_indicator_get_property (IndicateIndicator * indicator, const gchar * key)
{
	IndicateIndicatorClass * class = INDICATE_INDICATOR_GET_CLASS(indicator);
	if (class->get_property == NULL) {
		return NULL;
	}

	const GValue * val = class->get_property(indicator, key);
	return g_value_get_string(val);
}

const GValue *
indicate_indicator_get_property_value (IndicateIndicator * indicator, const gchar * key)
{
	IndicateIndicatorClass * class = INDICATE_INDICATOR_GET_CLASS(indicator);
	if (class->get_property == NULL) {
		return NULL;
	}

	return class->get_property(indicator, key);
}

GPtrArray *
indicate_indicator_list_properties (IndicateIndicator * indicator)
{
	IndicateIndicatorClass * class = INDICATE_INDICATOR_GET_CLASS(indicator);
	if (class->list_properties == NULL) {
		return g_ptr_array_new();
	}

	return class->list_properties(indicator);
}

static void
set_property (IndicateIndicator * indicator, const gchar * key, const GValue * data)
{
	g_return_if_fail(INDICATE_IS_INDICATOR(indicator));

	IndicateIndicatorPrivate * priv = INDICATE_INDICATOR_GET_PRIVATE(indicator);

	gchar * newkey = g_strdup(key);
	GValue * newval = g_new0(GValue, 1);
	g_value_init(newval, G_VALUE_TYPE(data));
	g_value_copy(data, newval);

	g_hash_table_replace(priv->properties, newkey, newval);

	if (indicate_indicator_is_visible(indicator)) {
		/* g_debug("Indicator property modified: %s", key); */
		g_signal_emit(indicator, signals[MODIFIED], 0, key, TRUE);
	}

	return;
}

static const GValue *
get_property (IndicateIndicator * indicator, const gchar * key)
{
	g_return_val_if_fail(INDICATE_IS_INDICATOR(indicator), NULL);

	IndicateIndicatorPrivate * priv = INDICATE_INDICATOR_GET_PRIVATE(indicator);

	return (const GValue *)g_hash_table_lookup(priv->properties, key);
}

static GPtrArray *
list_properties (IndicateIndicator * indicator)
{
	g_return_val_if_fail(INDICATE_IS_INDICATOR(indicator), g_ptr_array_new());
	IndicateIndicatorPrivate * priv = INDICATE_INDICATOR_GET_PRIVATE(indicator);

	GList * keys = g_hash_table_get_keys(priv->properties);
	GPtrArray * properties = g_ptr_array_sized_new(g_list_length(keys) + 1);

	for (; keys != NULL; keys = keys->next) {
		g_ptr_array_add(properties, g_strdup(keys->data));
	}

	return properties;
}

/**
	indicate_indicator_set_displayed:
	@indicator: The #IndicateIndicator to configure
	@displayed: Whether or not the indicator is visible to users

	Sets whether or not the indicator is visible to the user from
	a listener.  This does not include things like whether the menu
	is open, but more whether it's in the menu to be found.
*/
void
indicate_indicator_set_displayed (IndicateIndicator * indicator, gboolean displayed)
{
	g_return_if_fail(INDICATE_IS_INDICATOR(indicator));
	IndicateIndicatorPrivate * priv = INDICATE_INDICATOR_GET_PRIVATE(indicator);

	if (priv->is_displayed != displayed) {
		priv->is_displayed = displayed;
		g_signal_emit(G_OBJECT(indicator), signals[DISPLAYED], 0, displayed, TRUE);
	}

	return;
}

/**
	indicate_indicator_get_displayed:
	@indicator: The #IndicateIndicator to query

	Checks to see if the indicator is visible to users in some way.

	Return value: Whether or not this indicator can be seen by a user
		or not.
*/
gboolean
indicate_indicator_get_displayed (IndicateIndicator * indicator)
{
	g_return_val_if_fail(INDICATE_IS_INDICATOR(indicator), FALSE);
	IndicateIndicatorPrivate * priv = INDICATE_INDICATOR_GET_PRIVATE(indicator);
	return priv->is_visible && priv->is_displayed;
}

