#include <string.h>
#include "mb.h"

int
mb_call_getc(mb_info_t *info)
{
  if (info->in_n > 0) {
    int c = (unsigned char)info->inbuf[info->in_i % sizeof(info->inbuf)];

    ++(info->in_i);
    --(info->in_n);
    return c;
  }
  else
    return info->io_func.in(info->io_arg);
}

void
mb_fill_inbuf(mb_info_t *info, const char *s, size_t n)
{
  size_t b = (info->in_i + sizeof(info->inbuf) - n) % sizeof(info->inbuf);
  size_t m = sizeof(info->inbuf) - b;

  info->in_i = b;

  if (m < n) {
    memcpy(&info->inbuf[b], s, m);
    info->in_n += m;
    n -= m;
    s += m;
    b = 0;
  }

  memcpy(&info->inbuf[b], s, n);
  info->in_n += n;
}

int
mb_utf8_escenc(mb_info_t *info)
{
  char buf[ISO2MB_ESC_LEN_MAX];
  char *s = buf;
  int c, g, set;

  if ((c = mb_call_getc(info)) == EOF)
    goto failure;

  switch (*s++ = c) {
  case 0x24:
    if ((c = mb_call_getc(info)) == EOF)
      goto failure;

    set = mb_94x94;

    switch (*s++ = c) {
    case 0x40:
    case 0x41:
    case 0x42:
      g = mb_G0;
      goto desig;
    case 0x28:
    case 0x29:
    case 0x2A:
    case 0x2B:
      g = mb_G0 + c - 0x28;

      if ((c = mb_call_getc(info)) == EOF)
	goto failure;

      *s++ = c;

      if (c < 0x40 || c > 0x5F)
	goto failure;

      goto desig;
    default:
      goto failure;
    }
  case 0x25:
    if ((c = mb_call_getc(info)) == EOF)
      goto failure;

    switch (*s++ = c) {
    case 0x40:
      if (info->GR != mb_UTF8 || info->GRB4UTF8 == EOF)
	goto failure;

      info->GR = info->GRB4UTF8;
      info->GRB4UTF8 = EOF;
      return 0;
    case 0x47:
      if (info->GL == mb_UTF8)
	goto failure;

      info->GRB4UTF8 = info->GR;
      info->GR = mb_UTF8;
      return 0;
    default:
      goto failure;
    }
  case 0x28:
  case 0x29:
  case 0x2A:
  case 0x2B:
    g = mb_G0 + c - 0x28;

    if ((c = mb_call_getc(info)) == EOF)
      goto failure;

    switch (*s++ = c) {
    case 0x21:
      if ((c = mb_call_getc(info)) == EOF)
	goto failure;

      *s++ = c;

      if (c < 0x40 || c > 0x7E)
	goto failure;

      set = mb_94_2 + (c >= 0x60);
      goto desig;
    default:
      if (c < 0x40 || c > 0x7E)
	goto failure;

      set = mb_94_0 + (c >= 0x60);
      goto desig;
    }
  case 0x2C:
  case 0x2D:
  case 0x2E:
  case 0x2F:
    g = mb_G0 + c - 0x2C;

    if ((c = mb_call_getc(info)) == EOF)
      goto failure;

    *s++ = c;

    if (c < 0x40 || c > 0x7E)
      goto failure;

    set = mb_96_0 + (c >= 0x60);
  desig:
    info->G[g] = MB_ESC_ENC(set, c);
    return 0;
  case 0x4F:
  case 0x4E:
    if (info->flag & MB_FLAG_NOSSL)
      goto failure;

    return c;
  case 0x6E:
  case 0x6F:
    info->GL = mb_G2 + c - 0x6E;
    return 0;
  case 0x7E:
  case 0x7D:
  case 0x7C:
    info->GR = mb_G1 + 0x7E - c;
    return 0;
  default:
    goto failure;
  }

failure:
  mb_fill_inbuf(info, buf, s - buf);
  return EOF;
}

size_t
mb_utf8_enc(int c, char *buf)
{
  if (c < 0x80) {
    buf[0] = c;
    return 1;
  }
  else if (c < 0x800) {
    buf[0] = 0xC0 | ((c >> 6) & 0x1F);
    buf[1] = 0x80 | (c & 0x3F);
    return 2;
  }
  else if (c < 0x10000) {
    buf[0] = 0xE0 | ((c >> 12) & 0xF);
    buf[1] = 0x80 | ((c >> 6) & 0x3F);
    buf[2] = 0x80 | (c & 0x3F);
    return 3;
  }
  else if (c < 0x200000) {
    buf[0] = 0xF0 | ((c >> 18) & 0x7);
    buf[1] = 0x80 | ((c >> 12) & 0x3F);
    buf[2] = 0x80 | ((c >> 6) & 0x3F);
    buf[3] = 0x80 | (c & 0x3F);
    return 4;
  }
  else if (c < 0x4000000) {
    buf[0] = 0xF8 | ((c >> 24) & 0x3);
    buf[1] = 0x80 | ((c >> 18) & 0x3F);
    buf[2] = 0x80 | ((c >> 12) & 0x3F);
    buf[3] = 0x80 | ((c >> 6) & 0x3F);
    buf[4] = 0x80 | (c & 0x3F);
    return 5;
  }
  else {
    buf[0] = 0xFC | ((c >> 30) & 0x1);
    buf[1] = 0x80 | ((c >> 24) & 0x3F);
    buf[2] = 0x80 | ((c >> 18) & 0x3F);
    buf[3] = 0x80 | ((c >> 12) & 0x3F);
    buf[4] = 0x80 | ((c >> 6) & 0x3F);
    buf[5] = 0x80 | (c & 0x3F);
    return 6;
  }
}

static int
mb_utf8_prepare(int c, mb_info_t *info)
{
  size_t n = mb_utf8_enc(c, info->outbuf);

  if (n > 1) {
    info->out_n = n;
    info->out_i = 1;
  }

  return (unsigned char)info->outbuf[0];
}

static struct {
  char beg, end;
  mb_conv_t func;
} mb_convg0sl[] = {
#undef def_mb128
#define def_mb128(mac, fcb, fce, convg0sl, encoder, decoder, cwidth) {(fcb) & MB_ESC_FC_MASK, (fce) & MB_ESC_FC_MASK, convg0sl},
#include "mb128.h"
};

static unsigned int jis1flag_tab[] = {
#include "jis1flag.h"
};

int
mb_get_jis1flag(int c)
{
  unsigned int value;
  mb_bt_result_t res = mb_bt_search(c, jis1flag_tab, &value);

  return res != mb_bt_failure ? value : MB_IN_JIS1COMMON;
}

static unsigned int jis2flag_tab[] = {
#include "jis2flag.h"
};

int
mb_get_jis2flag(int c)
{
  unsigned int value;
  mb_bt_result_t res = mb_bt_search(c, jis2flag_tab, &value);

  return res != mb_bt_failure ? value : MB_IN_JIS2COMMON;
}

int
mb_conv_jisx0213(int fc, int c, mb_info_t *info)
{
  int f;

  switch (fc & MB_ESC_FC_MASK) {
  case 0x42 & MB_ESC_FC_MASK:
    if (!((f = mb_get_jis1flag(c)) & MB_IN_JISX0208) ||
	((info->flag & MB_FLAG_JISX_PREFERENCE) == MB_FLAG_PREFER_JISX0213 && (f & MB_IN_JISX0213_1)))
      fc = 0x4F & MB_ESC_FC_MASK;

    break;
  case 0x44 & MB_ESC_FC_MASK:
    if (!((f = mb_get_jis2flag(c)) & MB_IN_JISX0212) ||
	((info->flag & MB_FLAG_JISX_PREFERENCE) == MB_FLAG_PREFER_JISX0213 && (f & MB_IN_JISX0213_2)))
      fc = 0x50 & MB_ESC_FC_MASK;

    break;
  case 0x4F & MB_ESC_FC_MASK:
    if (!((f = mb_get_jis1flag(c)) & MB_IN_JISX0213_1) ||
	((info->flag & MB_FLAG_JISX_PREFERENCE) == MB_FLAG_DONTPREFER_JISX0213 && (f & MB_IN_JISX0208)))
      fc = 0x42 & MB_ESC_FC_MASK;

    break;
  case 0x50 & MB_ESC_FC_MASK:
    if (!((f = mb_get_jis2flag(c)) & MB_IN_JISX0213_2) ||
	((info->flag & MB_FLAG_JISX_PREFERENCE) == MB_FLAG_DONTPREFER_JISX0213 && (f & MB_IN_JISX0212)))
      fc = 0x44 & MB_ESC_FC_MASK;
  default:
    break;
  }

  return fc;
}

void
mb_conv_allg0sl(mb_char_t *ch, mb_info_t *info)
{
  int fc;
  size_t b, e;

  switch (ch->set) {
  case mb_96_0:
  case mb_96_1:
    return;
  case mb_128:
    for (fc = ch->fc & MB_ESC_FC_MASK, b = 0, e = sizeof(mb_convg0sl) / sizeof(mb_convg0sl[0]) ; b < e ;) {
      size_t i = (b + e) / 2;

      if (fc < (unsigned char)mb_convg0sl[i].beg)
	e = i;
      else if (fc > (unsigned char)mb_convg0sl[i].end)
	b = i + 1;
      else {
	mb_convg0sl[i].func(ch, info);
	return;
      }
    }

    return;
  default:
    break;
  }

  ch->gn = mb_G0;
  ch->sn = mb_SL;
}

static unsigned int default_notascii_tab[] = {
#include "notascii.h"
};

static unsigned int *notascii_tab = default_notascii_tab;

int *
mb_set_localized_ascii_table(int *new)
{
  int *old = notascii_tab;

  notascii_tab = new;
  return old;
}

static unsigned int default_domestic_ascii_tab[] = {
#include "domestic-ascii.h"
};

static unsigned int *domestic_ascii_tab = default_domestic_ascii_tab;

int *
mb_set_domestic_ascii_table(int *new)
{
  int *old = domestic_ascii_tab;

  domestic_ascii_tab = new;
  return old;
}

void
mb_conv_ascii(mb_char_t *ch, mb_info_t *info)
{
  switch (ch->set) {
  case mb_94_0:
    if ((ch->fc & MB_ESC_FC_MASK) == (0x42 & MB_ESC_FC_MASK))
      break;
  case mb_94_1:
  case mb_94_2:
  case mb_94_3:
    {
      mb_bt_result_t res = mb_bt_search(MB_ESC_ENC(ch->set, ch->fc), domestic_ascii_tab, NULL);

      if (res != mb_bt_failure) {
	ch->set = mb_94_0;
	ch->fc = 0x42 & MB_ESC_FC_MASK;
      }
    }
  default:
    break;
  }
}

size_t
mb_nonutf8_enc(mb_char_t *ch, char *buf)
{
  int c = ch->c;
  int set = ch->set & MB_ESC_SET_MASK;
  int fc = ch->fc & MB_ESC_FC_MASK;
  int gn = ch->gn & MB_ESC_Gn_MASK;
  int sn = ch->sn & MB_ESC_Sn_MASK;
  int gnsn;

  switch ((sn << MB_ESC_Gn_LEN) | gn) {
  default:
  case (mb_SL << MB_ESC_Gn_LEN) | mb_G0:
    gnsn = mb_G0SL;
    break;
  case (mb_SL << MB_ESC_Gn_LEN) | mb_G1:
    gnsn = mb_G1SL;
    break;
  case (mb_SL << MB_ESC_Gn_LEN) | mb_G2:
    gnsn = mb_G2SL;
    break;
  case (mb_SL << MB_ESC_Gn_LEN) | mb_G3:
    gnsn = mb_G3SL;
    break;
  case (mb_SR << MB_ESC_Gn_LEN) | mb_G0:
    gnsn = mb_G0SR;
    break;
  case (mb_SR << MB_ESC_Gn_LEN) | mb_G1:
    gnsn = mb_G1SR;
    break;
  case (mb_SR << MB_ESC_Gn_LEN) | mb_G2:
    gnsn = mb_G2SR;
    break;
  case (mb_SR << MB_ESC_Gn_LEN) | mb_G3:
    gnsn = mb_G3SR;
    break;
  case (mb_SSL << MB_ESC_Gn_LEN) | mb_G2:
    gnsn = mb_G2SL;
    break;
  case (mb_SSL << MB_ESC_Gn_LEN) | mb_G3:
    gnsn = mb_G3SL;
    break;
  case (mb_SSR << MB_ESC_Gn_LEN) | mb_G2:
    gnsn = mb_G2SR;
    break;
  case (mb_SSR << MB_ESC_Gn_LEN) | mb_G3:
    gnsn = mb_G3SR;
    break;
  }

  switch (set) {
  case mb_94x94:
    if (c >= (1 << 10))
      c = (1 << 0) | ((c - (1 << 10)) << (1 + 0)) | (MB_UTF8_BADENC(mb_94x94, fc, gnsn) << (1 + 0 + 13));
    else
      c = (1 << 1) | (c << (1 + 1)) | (MB_UTF8_BADENC(mb_94x94, fc, gnsn) << (1 + 1 + 10));

    break;
  case mb_128:
    if (fc == (0x47 & MB_ESC_FC_MASK))
      return mb_utf8_enc(c, buf);
    else if (c >= (1 << 10))
      c = (1 << 2) | ((c - (1 << 10)) << (1 + 2)) | (MB_UTF8_BADENC(mb_128, fc, 0) << (1 + 2 + 13));
    else
      c = (1 << 3) | (c << (1 + 3)) | (MB_UTF8_BADENC(mb_128, fc, 0) << (1 + 3 + 10));

    break;
  case mb_96_0:
    c = (1 << 1) | (c << (1 + 1)) | (MB_UTF8_BADENC(mb_96_0, fc, gnsn) << (1 + 1 + 10));
    break;
  case mb_96_1:
    c = (1 << 0) | ((0x5F * 0x60 + c - (1 << 10)) << (1 + 0)) | (MB_UTF8_BADENC(mb_96_1, fc, gnsn) << (1 + 0 + 13));
    break;
  case mb_94_0:
    if (fc == (0x42 & MB_ESC_FC_MASK) && gnsn == mb_G0SL) {
      buf[0] = c + 0x20;
      return 1;
    }
  case mb_94_1:
  case mb_94_2:
  case mb_94_3:
    if (gnsn == mb_G0SL) {
      mb_bt_result_t res = mb_bt_search(MB_WORD_ENC_3(c, set, fc), notascii_tab, NULL);

      if (res == mb_bt_failure) {
	buf[0] = c + 0x20;
	return 1;
      }
    }

    c = (c << (1 + 3)) | ((set - mb_94_0) << (1 + 3 + 7)) | (MB_UTF8_BADENC(set, fc, gnsn) << (1 + 3 + 7 + 2));
    break;
  }

  if (c < 0x80) {
    buf[0] = 0xC0 | ((c >> 6) & 0x3);
    buf[1] = 0x80 | (c & 0x3F);
    return 2;
  }
  else {
    c -= 0x80;

    if (c < 0x800) {
      buf[0] = 0xE0;
      buf[1] = 0x80 | ((c >> 6) & 0x1F);
      buf[2] = 0x80 | (c & 0x3F);
      return 3;
    }
    else {
      c -= 0x800;

      if (c < 0x10000) {
	buf[0] = 0xF0;
	buf[1] = 0x80 | ((c >> 12) & 0xF);
	buf[2] = 0x80 | ((c >> 6) & 0x3F);
	buf[3] = 0x80 | (c & 0x3F);
	return 4;
      }
      else {
	c -= 0x10000;

	if (c < 0x200000) {
	  buf[0] = 0xF8;
	  buf[1] = 0x80 | ((c >> 18) & 0x7);
	  buf[2] = 0x80 | ((c >> 12) & 0x3F);
	  buf[3] = 0x80 | ((c >> 6) & 0x3F);
	  buf[4] = 0x80 | (c & 0x3F);
	  return 5;
	}
	else {
	  c -= 0x200000;
	  buf[0] = 0xFC;
	  buf[1] = 0x80 | ((c >> 24) & 0x3);
	  buf[2] = 0x80 | ((c >> 18) & 0x3F);
	  buf[3] = 0x80 | ((c >> 12) & 0x3F);
	  buf[4] = 0x80 | ((c >> 6) & 0x3F);
	  buf[5] = 0x80 | (c & 0x3F);
	  return 6;
	}
      }
    }
  }
}

static void
mb_apply_convv_internal(mb_char_t *ch, mb_info_t *info, mb_conv_t *v)
{
  if (v)
    for (; *v ; ++v)
      (*v)(ch, info);
}

void
mb_conv_for_charset(mb_char_t *ch, mb_info_t *info)
{
  if (info->cs2esc)
    mb_apply_convv_internal(ch, info, info->cs2esc->convv);
}

void
mb_apply_convv(mb_char_t *ch, mb_info_t *info)
{
  mb_apply_convv_internal(ch, info, info->convv);

  if (ch->set == mb_94x94)
    ch->fc = mb_conv_jisx0213(ch->fc, ch->c, info);
}

int
mb_utf8_badenc(mb_char_t *ch, mb_info_t *info)
{
  size_t e;

  mb_apply_convv(ch, info);
  e = mb_nonutf8_enc(ch, info->outbuf);

  if (e > 1) {
    info->out_i = 1;
    info->out_n = e;
  }
  else
    info->out_i = info->out_n = 0;

  return (unsigned char)info->outbuf[0];
}

static int
mb_dbc_to_badutf8(int c1, int c2, int gn, int sn, mb_info_t *info)
{
  int esc = info->G[gn];
  mb_char_t ch;

  ch.c = ((c1 & 0x7F) - 0x20) * 0x60 + (c2 & 0x7F) - 0x20;
  ch.set = esc & MB_ESC_SET_MASK;
  ch.fc = (esc >> MB_ESC_SET_LEN) & MB_ESC_FC_MASK;

  if (ch.set == mb_94x94)
    ch.fc = mb_conv_jisx0213(ch.fc, ch.c, info);

  ch.gn = gn;
  ch.sn = sn;
  return mb_utf8_badenc(&ch, info);
}

static mb_128encoder_t mb_128encoderv[] = {
#undef def_mb128
#define def_mb128(mac, fcb, fce, convg0sl, encoder, decoder, cwidth) encoder,
#include "mb128.h"
};

int
mb_fetch_char(mb_info_t *info)
{
  int c, cc, ccc, mask, gn, sn;
  char temp[3];
  size_t i, n;

start:
  info->out_i = info->out_n = 0;

  switch (c = mb_call_getc(info)) {
  case EOF:
    return EOF;
  case 0x0F:
  case 0x0E:
    info->GL = mb_G0 + 0x0F - c;
    goto start;
  case 0x1B:
    switch (c = mb_utf8_escenc(info)) {
    case EOF:
    case 0x1B:
      return 0x1B;
    case 0x4E:
    case 0x4F:
      if (info->G[gn = mb_G2 + c - 0x4E] == EOF || (cc = mb_call_getc(info)) == EOF) {
	temp[0] = c;
	mb_fill_inbuf(info, temp, 1);
	return 0x1B;
      }

      sn = mb_SSL;

      switch (info->G[gn] & MB_ESC_SET_MASK) {
      case mb_94x94:
	if (cc >= 0x21 && cc <= 0x7E && (ccc = mb_call_getc(info)) != EOF)
	  if (ccc >= 0x21 && ccc <= 0x7E)
	    return mb_dbc_to_badutf8(cc, ccc, gn, sn, info);
	  else {
	    temp[0] = c;
	    temp[1] = cc;
	    temp[2] = ccc;
	    mb_fill_inbuf(info, temp, 3);
	    return 0x1B;
	  }
	else
	  goto fail_g23l;
      case mb_96_0:
      case mb_96_1:
	if (cc >= 0x20 && cc <= 0x7F)
	  return mb_dbc_to_badutf8(0x20, cc, gn, sn, info);
	else
	  goto fail_g23l;
      case mb_94_0:
      case mb_94_1:
      case mb_94_2:
      case mb_94_3:
	if (cc >= 0x21 && cc <= 0x7E)
	  return mb_dbc_to_badutf8(0x20, cc, gn, sn, info);
      default:
      fail_g23l:
	temp[0] = c;
	temp[1] = cc;
	mb_fill_inbuf(info, temp, 2);
	return 0x1B;
      }
    default:
      goto start;
    }
  default:
    if (c & 0x80) {
      if (info->GR == mb_UTF8 || info->GR == mb_FAKEUTF8) {
	for (mask = 0x40, n = 1 ; n <= 6 && mask & c ; mask >>= 1, ++n)
	  ;

	if (n < 2 || n > 6)
	  goto bad_gr;

	info->outbuf[0] = c;

	for (i = 1, ccc = c & (mask - 1) ; i < n ;)
	  if ((cc = mb_call_getc(info)) == EOF) {
	    mb_fill_inbuf(info, &info->outbuf[1], i - 1);
	    break;
	  }
	  else {
	    info->outbuf[i++] = cc;

	    if ((cc & 0xC0) != 0x80) {
	      mb_fill_inbuf(info, &info->outbuf[1], i - 1);
	      break;
	    }
	    else
	      ccc = ccc * 0x40 + (cc & 0x3F);
	  }

	if (i == n) {
	  mb_char_t ch;

	  if (info->GR == mb_FAKEUTF8) {
	    mb_chartype(ccc, i, &ch);
	  }
	  else {
	    ch.c = ccc;
	    ch.set = mb_128;
	    ch.fc = 0x47;
	    ch.gn = mb_G1;
	    ch.sn = mb_SR;
	  }

	  mb_apply_convv(&ch, info);

	  if (ch.set != mb_128 || (ch.fc & MB_ESC_FC_MASK) != (0x47 & MB_ESC_FC_MASK))
	    c = mb_utf8_badenc(&ch, info);
	  else
	    c = mb_utf8_prepare(ch.c, info);
	}

	return c;
      }
      else if ((info->GR & MB_ESC_MASK) >= mb_UTF8 + 1 &&
	       (info->GR & MB_ESC_MASK) < mb_UTF8 + 1 + sizeof(mb_128encoderv) / sizeof(mb_128encoderv[0])) {
	return mb_128encoderv[(info->GR & MB_ESC_MASK) - mb_UTF8 - 1](c, info);
      }
      else
	switch (c) {
	case 0x8E:
	case 0x8F:
	  if (info->G[gn = mb_G2 + c - 0x8E] == EOF || (cc = mb_call_getc(info)) == EOF)
	    goto bad_gr;

	  sn = mb_SSR;

	  switch (info->G[gn] & MB_ESC_SET_MASK) {
	  case mb_94x94:
	    if (cc >= 0xA1 && cc <= 0xFE && (ccc = mb_call_getc(info)) != EOF)
	      if (ccc >= 0xA1 && ccc <= 0xFE)
		return mb_dbc_to_badutf8(cc, ccc, gn, sn, info);
	      else {
		temp[0] = cc;
		temp[1] = ccc;
		mb_fill_inbuf(info, temp, 2);
		goto bad_gr;
	      }
	    else
	      goto fail_g23;
	  case mb_96_0:
	  case mb_96_1:
	    if (cc >= 0xA0)
	      return mb_dbc_to_badutf8(0x20, cc, gn, sn, info);
	    else
	      goto fail_g23;
	  case mb_94_0:
	  case mb_94_1:
	  case mb_94_2:
	  case mb_94_3:
	    if (cc >= 0xA1 && cc <= 0xFE)
	      return mb_dbc_to_badutf8(0x20, cc, gn, sn, info);
	  default:
	  fail_g23:
	    temp[0] = cc;
	    mb_fill_inbuf(info, temp, 1);
	    goto bad_gr;
	  }
	default:
	  if (c < 0xA0 || info->GR < mb_G0 || info->GR > mb_G3 || info->G[info->GR] == EOF)
	    goto bad_gr;

	  switch (info->G[info->GR] & MB_ESC_SET_MASK) {
	  case mb_94x94:
	    if (c >= 0xA1 && c <= 0xFE && (cc = mb_call_getc(info)) != EOF)
	      if (cc >= 0xA1 && cc <= 0xFE)
		return mb_dbc_to_badutf8(c, cc, info->GR, mb_SR, info);
	      else {
		temp[0] = cc;
		mb_fill_inbuf(info, temp, 1);
		goto bad_gr;
	      }
	    else
	      goto bad_gr;
	  case mb_96_0:
	  case mb_96_1:
	    if (c >= 0xA0)
	      return mb_dbc_to_badutf8(0x20, c, info->GR, mb_SR, info);
	    else
	      goto bad_gr;
	  case mb_94_0:
	  case mb_94_1:
	  case mb_94_2:
	  case mb_94_3:
	    if (c >= 0xA1 || c <= 0xFE)
	      return mb_dbc_to_badutf8(0x20, c, info->GR, mb_SR, info);
	  default:
	    goto bad_gr;
	  }
	}
    bad_gr:
      {
	mb_char_t ch;

	ch.c = c;
	ch.set = mb_128;
	ch.fc = MB_UNKNOWN_FC;
	ch.gn = mb_G1;
	ch.sn = mb_SR;
	return mb_utf8_badenc(&ch, info);
      }
    }
    else if (c < 0x20 || info->GL < mb_G0 || info->GL > mb_G3 || info->G[info->GL] == EOF)
      return c;
    else {
      switch (info->G[info->GL] & MB_ESC_SET_MASK) {
      case mb_94x94:
	if (c >= 0x21 && c <= 0x7E && (cc = mb_call_getc(info)) != EOF)
	  if (cc >= 0x21 && cc <= 0x7E)
	    return mb_dbc_to_badutf8(c, cc, info->GL, mb_SL, info);
	  else {
	    temp[0] = cc;
	    mb_fill_inbuf(info, temp, 1);
	    return c;
	  }
	else
	  return c;
      case mb_96_0:
      case mb_96_1:
	return mb_dbc_to_badutf8(0x20, c, info->GL, mb_SL, info);
      case mb_94_0:
      case mb_94_1:
      case mb_94_2:
      case mb_94_3:
	if (info->G[info->GL] == MB_ESC_ASCII || c < 0x21 || c > 0x7E)
	  return c;
	else
	  return mb_dbc_to_badutf8(0x20, c, info->GL, mb_SL, info);
      default:
	return c;
      }
    }
  }
}

int
mb_unfetch_char(int c, mb_info_t *info)
{
  if (c != EOF) {
    if (info->out_i > 0)
      info->outbuf[--(info->out_i)] = c;
    else {
      char temp[1];

      temp[0] = c;
      mb_fill_inbuf(info, temp, 1);
    }
  }

  return c;
}

int
mb_getc(mb_info_t *info)
{
  if (info->out_i < info->out_n)
    return (unsigned char)info->outbuf[(info->out_i)++];
  else
    return mb_fetch_char(info);
}

size_t
mb_getmbc(char *d, mb_info_t *info)
{
  size_t i = info->out_i;
  size_t n = info->out_n;

  if (i < n) {
    memcpy(d, &info->outbuf[i], n - i);
    info->out_i = n;
    return n - i;
  }
  else {
    int c = mb_fetch_char(info);

    if (c == EOF)
      return 0;
    else if (info->out_n > 0) {
      memcpy(d, info->outbuf, info->out_n);
      return info->out_i = info->out_n;
    }
    else {
      *d = c;
      return 1;
    }
  }
}

size_t
mb_getmem(char *d, size_t n, mb_info_t *info)
{
  size_t i;
  int c;

  for (i = 0 ; i < n && info->out_i < info->out_n ;)
    d[i++] = mb_getc(info);

  while (i < n) {
    if ((c = mb_fetch_char(info)) == EOF)
      return i;

    if (info->out_i < info->out_n) {
      if (i + info->out_n > n) {
	mb_unfetch_char(c, info);
	return i;
      }

      memcpy(d, info->outbuf, info->out_n);
      i += info->out_n;
    }
    else
      d[i++] = c;
  }

  return i;
}

char *
mb_getline(char *d, int n, mb_info_t *info)
{
  if (n > 0) {
    size_t i;
    int c;

    for (--n, i = 0 ; i < n && info->out_i < info->out_n ;)
      if ((c = mb_getc(info)) == '\n')
	goto end;
      else
	d[i++] = c;

    while (i < n) {
      if ((c = mb_fetch_char(info)) == EOF)
	goto end;

      if (info->out_i < info->out_n) {
	if (i + info->out_n > n) {
	  mb_unfetch_char(c, info);
	  goto end;
	}

	memcpy(d, info->outbuf, info->out_n);
	i += info->out_n;
      }
      else {
	d[i++] = c;

	if (c == '\n')
	  goto end;
      }
    }

  end:
    d[i++] = '\0';
  }

  return d;
}

char *
mb_info2mb(mb_info_t *info, size_t n)
{
  char *d = malloc(n);

  if (d) {
    size_t end, size, nsize;
    int c;

    for (end = 0, size = n ; (c = mb_fetch_char(info)) != EOF ;) {
      if (end + 1 + info->out_n - info->out_i >= size) {
	nsize = ((end + 1 + info->out_n - info->out_i) / MB_MEM_DELTA + 1) * MB_MEM_DELTA;

	if (!(d = realloc(d, nsize)))
	  break;

	size = nsize;
      }

      if (info->out_n) {
	memcpy(&d[end], info->outbuf, info->out_n);
	end += info->out_n;
      }
      else
	d[end++] = c;
    }
  }

  return d;
}

static int
mb_mem_getc(void *ap)
{
  mb_mem_r_t *mem = ap;

  if (mem->i < mem->n)
    return (unsigned char)mem->s[(mem->i)++];
  else
    return EOF;
}

void
mb_vmem2mb_setup(mb_info_t *info, mb_mem_r_t *mem, const char *s, size_t n, mb_setup_t *setup, const char *op, va_list ap)
{
  mem->s = s;
  mem->i = 0;
  mem->n = n;
  mb_vinit_r(info, mem, mb_mem_getc, setup, op, ap);
}

char *
mb_vmem2mb(const char *s, size_t n, mb_setup_t *setup, const char *op, va_list ap)
{
  mb_mem_r_t mem;
  mb_info_t info;

  mb_vmem2mb_setup(&info, &mem, s, n, setup, op, ap);
  return mb_info2mb(&info, n);
}

char *
mb_mem2mb(const char *s, size_t n, mb_setup_t *setup, const char *op, ...)
{
  va_list ap;
  char *mb;

  va_start(ap, op);
  mb = mb_vmem2mb(s, n, setup, op, ap);
  va_end(ap);
  return mb;
}

char *
mb_vstr2mb(const char *s, mb_setup_t *setup, const char *op, va_list ap)
{
  return mb_vmem2mb(s, strlen(s) + 1, setup, op, ap);
}

char *
mb_str2mb(const char *s, mb_setup_t *setup, const char *op, ...)
{
  va_list ap;
  char *mb;

  va_start(ap, op);
  mb = mb_vstr2mb(s, setup, op, ap);
  va_end(ap);
  return mb;
}

void
mb_cs_judge_utf8(mb_cs_detector_stat_t *p, const char *bag, size_t e)
{
  size_t i, j;
  int c;

  for (i = p->processed ; i < e ;)
    if (bag[i] & 0x80) {
      j = i;

      if ((c = mb_char_dec(bag, &j, e)) != EOF) {
	if (j > i) {
	  mb_char_t ch;

	  mb_chartype(c, j - i, &ch);

	  if (ch.set == mb_128 && (ch.fc & MB_ESC_FC_MASK) == (0x47 & MB_ESC_FC_MASK))
	    p->by_char += j - i;
	  else
	    p->by_encode -= 1;

	  i = j;
	}
	else if (c >= -0xFD && c <= -0xC0) {
	  if (i <= p->processed) {
	    p->by_encode -= 1;
	    ++i;
	  }

	  break;
	}
	else {
	  p->by_encode -= 1;
	  ++i;
	}
      }
      else
	break;
    }
    else
      ++i;

  p->processed = i;
}

void
mb_cs_judge_latin1(mb_cs_detector_stat_t *p, const char *bag, size_t e)
{
  size_t i;

  for (i = p->processed ; i < e ; ++i)
    if (bag[i] & 0x80) {
      if ((unsigned char)bag[i] >= 0xA0)
	p->by_char += 1;
      else
	p->by_encode -= 1;
    }

  p->processed = i;
}

void
mb_mkunbound_cs_detector(mb_cs_detector_t *p)
{
  p->orig->io_func.in = p->orig_reader;
  p->orig->io_arg = p->orig_arg;

  if (p->free_bag && p->bag)
    p->free_bag(p->bag);

  if (p->free_detector)
    p->free_detector(p);
}

size_t
mb_cs_detector_find_best(mb_cs_detector_t *p, size_t *p_same)
{
  size_t i, j, same;

  for (i = same = 0, j = 1 ; j < p->nstats ; ++j) {
    if (p->stat[i].by_encode < p->stat[j].by_encode)
      goto i_lt_j;
    else if (p->stat[i].by_encode > p->stat[j].by_encode)
      continue;
    else if (p->stat[i].by_char < p->stat[j].by_char)
      goto i_lt_j;
    else {
      if (p->stat[i].by_char == p->stat[j].by_char)
	p->samev[same++] = j;

      continue;
    }

  i_lt_j:
    i = j;
    same = 0;
  }

  *p_same = same;
  return i;
}

static int
mb_cs_detector_getc(void *ap)
{
  mb_cs_detector_t *p = ap;
  int c;

  if (p->beg > 0) {
    if (p->beg < p->end) {
      c = (unsigned char)p->bag[(p->beg)++];

      if (p->beg >= p->end && p->flag & MB_CS_DETECT_FLAG_MKUNBOUND)
	mb_mkunbound_cs_detector(p);
    }
    else
      c = p->orig_reader(p->orig_arg);
  }
  else if ((c = p->orig_reader(p->orig_arg)) == EOF) {
    if (p->flag & MB_CS_DETECT_FLAG_MKUNBOUND)
      mb_mkunbound_cs_detector(p);
  }
  else if (c & 0x80) {
    size_t i, nsize, same;
    char *nbag;

    for (same = 0 ;;) {
      do {
	p->bag[(p->end)++] = c;
      } while (p->end < p->size && (c = p->orig_reader(p->orig_arg)) != EOF);

      if ((i = p->judge(p)) < p->nstats)
	goto end;

      if (c == EOF || (p->limit && p->size >= p->limit) || !p->realloc_bag)
	break;

      nsize = (p->size / MB_MEM_BIG_DELTA + 1) * MB_MEM_BIG_DELTA;

      if (p->limit && nsize > p->limit)
	nsize = p->limit;

      if (!(nbag = p->realloc_bag(p->bag, nsize)))
	break;

      p->bag = nbag;
      p->size = nsize;

      if ((c = p->orig_reader(p->orig_arg)) == EOF)
	break;
    }

    i = mb_cs_detector_find_best(p, &same);
  end:
    p->setup(p, i, same);
    c = (unsigned char)p->bag[(p->beg)++];
  }

  return c;
}

void
mb_bind_cs_detector(mb_cs_detector_t *p, mb_info_t *info)
{
  size_t i;

  p->orig = info;
  p->orig_reader = info->io_func.in;
  p->orig_arg = info->io_arg;
  info->io_func.in = mb_cs_detector_getc;
  info->io_arg = p;

  for (i = 0 ; i < sizeof(p->stat) / sizeof(p->stat[0]) ; ++i) {
    p->stat[i].processed = 0;
    p->stat[i].by_encode = p->stat[i].by_char = 0;
  }
}

mb_cs_detector_t *
mb_alloc_cs_detector(mb_info_t *info, size_t init, size_t limit)
{
  mb_cs_detector_t *p = NULL;

  if (!limit || init < limit) {
    char *bag = malloc(init);

    if (bag) {
      if ((p = malloc(sizeof(mb_cs_detector_t)))) {
	p->bag = bag;
	p->realloc_bag = realloc;
	p->free_bag = free;
      }
      else
	free(bag);
    }
  }
  else if ((p = malloc(sizeof(mb_cs_detector_t) + init))) {
    p->bag = (char *)p + sizeof(mb_cs_detector_t);
    p->realloc_bag = NULL;
    p->free_bag = NULL;
  }

  if (p) {
    p->limit = limit;
    p->size = init;
    p->beg = p->end = 0;
    p->free_detector = free;
    mb_bind_cs_detector(p, info);
  }

  return p;
}

char *
mb_cs_detect_from_mem(mb_cs_detector_t *p, mb_info_t *info, const char *s, size_t n)
{
  size_t i, same = 0;
  char *cs;

  mb_bind_cs_detector(p, info);
  p->bag = (char *)s;
  p->beg = 0;
  p->end = p->size = p->limit = n;
  p->free_bag = NULL;

  if ((i = p->judge(p)) >= p->nstats)
    i = mb_cs_detector_find_best(p, &same);

  cs = p->setup(p, i, same);
  mb_mkunbound_cs_detector(p);
  return cs;
}

char *
mb_cs_detect_from_str(mb_cs_detector_t *p, mb_info_t *info, const char *s)
{
  return mb_cs_detect_from_mem(p, info, s, strlen(s) + 1);
}

char *
mb_cs_detect_vmem2mb(mb_cs_detector_t *p, const char *s, size_t n, mb_setup_t *setup, const char *op, va_list ap)
{
  mb_mem_r_t mem;
  mb_info_t info;

  mb_vmem2mb_setup(&info, &mem, s, n, setup, op, ap);
  mb_cs_detect_from_mem(p, &info, s, n);
  return mb_info2mb(&info, n);
}

char *
mb_cs_detect_mem2mb(mb_cs_detector_t *p, const char *s, size_t n, mb_setup_t *setup, const char *op, ...)
{
  va_list ap;
  char *d;

  va_start(ap, op);
  d = mb_cs_detect_vmem2mb(p, s, n, setup, op, ap);
  va_end(ap);
  return d;
}

char *
mb_cs_detect_vstr2mb(mb_cs_detector_t *p, const char *s, mb_setup_t *setup, const char *op, va_list ap)
{
  return mb_cs_detect_vmem2mb(p, s, strlen(s) + 1, setup, op, ap);
}

char *
mb_cs_detect_str2mb(mb_cs_detector_t *p, const char *s, mb_setup_t *setup, const char *op, ...)
{
  va_list ap;
  char *d;

  va_start(ap, op);
  d = mb_cs_detect_vstr2mb(p, s, setup, op, ap);
  va_end(ap);
  return d;
}
