/**************************************************************************

    begin                : June 30 2004
	version		 : 1.2 
    copyright            : (C) 2004 by Gleb Beliakov
    email                : gleb@deakin.edu.au


  SLipInt.cpp: implementation of the Simple Lipschitz interpolant class.

 *                                                                         *
 * This program is free software; you can redistribute it and/or modify it *
 * under the terms of the GNU General Public License as published by the   *
 * Free Software Foundation; either version 2 of the License, or (at your  *
 * option) any later version.                                              *
 *                                                                         *
 * This program is distributed in the hope that it will be useful, but     *
 * WITHOUT ANY WARRANTY; without even the implied warranty of              *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU       *
 * General Public License for more details.                                *
 *                                                                         *
 * You should have received a copy of the GNU General Public License       *
 * along with this program; if not, write to the Free Software Foundation, *
 * Inc., 59 Temple Place Suite 330, Boston, MA 02111-1307 USA.             *
 ***************************************************************************/


#include "slipint.h"

double sqr__(double a) {return a*a; }
double max__(double a, double b) { return((a>b)?a:b); }

double dist(int dim, double* x, double* xk)
{
	double d=0;
	dim--;
	for( ; dim>=0; dim--) d+=sqr__(x[dim] - xk[dim]);
	return sqrt(d);
}

double distCol(int dim, int npts, double* x, double* xk)
{
	double d=0;
	int i;
	for(i=0 ; i<dim; i++) {
		d+=sqr__(x[i*npts] - xk[i*npts]);
	}
	return sqrt(d);
}

double distInfDir(int dim, double* x, double* xk, int* dir)
{
	double dk,d=-1;
	int i=dim-1;
	for( ; i>=0; i--) {
		dk=(x[i] - xk[i]);
		if(fabs(dk)>d) {
			d=fabs(dk);
			if(dk>=0) *dir=i; else *dir=dim+i;
		}
	}
	return (d);
}

double distInf(int dim, double* x, double* xk, int* dir)
{
	double dk,d=-1;
	int i = dim-1;
	for( ; i>=0; i--) {
		dk=fabs(x[i] - xk[i]);
		if(dk>d) {
			d=dk;
			*dir=i; 
		}
	}
	return (d);
}

double distInfCol(int dim, int npts, double* x, double* xk, int* dir)
{
	double dk,d=-1;
	int i;
	for(i=0 ; i<dim; i++) {
		dk=fabs(x[i*npts] - xk[i*npts]);
		if(dk>d) {
			d=dk;
			*dir=i; 
		}
	}
	return (d);
}

double	SLipInt::Value2(int dim, int npts,   double* x, double* XData, double* YData)
{
	return Value2(dim,npts,x, XData, YData, MaxLipConst);
}

double	SLipInt::Value2(int dim, int npts,   double* x, double* XData, double* YData,   double Lipconst)
{
	g1=-10e20;
	g2=-g1;

	for(i=0;i<npts;i++) {
		j=i*dim;
		d2 = dist(dim, x, &(XData[j]));
		d1= YData[i] - Lipconst * d2;
		d2= YData[i] + Lipconst * d2;
		if(g1<d1) g1=d1;
		if(g2>d2) g2=d2;
	}
	g2=0.5*(g1+g2);
	return g2;
}

double	SLipInt::ValueInf(int dim, int npts,  double* x, double* XData, double* YData)
{
	return ValueInf(dim,npts,x, XData, YData, LipConst);
}


double	SLipInt::ValueInf(int dim, int npts, double* x, double* XData, double* YData,   double* Lipconst)
{ 
	g1=-10e20;
	g2=-g1;

	for(i=0;i<npts;i++) {
		j=i*dim;
		d2 = distInf(dim, x, &(XData[j]),&j);
		d1= YData[i] - Lipconst[j] * d2;
		d2= YData[i] + Lipconst[j] * d2;
		if(g1<d1) g1=d1;
		if(g2>d2) g2=d2;
	}
	g2=0.5*(g1+g2);
	return g2;
}

double	SLipInt::ValueInf(int dim, int npts, double* x, double* XData, double* YData,   double Lipconst)
{ 
	g1=-10e20;
	g2=-g1;

	for(i=0;i<npts;i++) {
		j=i*dim;
		d2 = distInf(dim, x, &(XData[j]),&j);
		d1= YData[i] - Lipconst * d2;
		d2= YData[i] + Lipconst * d2;
		if(g1<d1) g1=d1;
		if(g2>d2) g2=d2;
	}
	g2=0.5*(g1+g2);
	return g2;
}


void	SLipInt::ComputeLipschitz2(int dim, int npts, double* XData, double* YData)
{
	int k1,k2;
	MaxLipConst=0;
// for all pairs
	for(i=0;i<npts;i++) {
		k1=dim*i; 
		for(j=i+1;j<npts;j++) {
				k2=dim*j;
				d1=dist(dim,&(XData[k2]),&(XData[k1]));
				g1=fabs(YData[j] - YData[i]);

				if(d1>0) 
					MaxLipConst = max__(MaxLipConst, g1/d1);		
		}
	}
}


void	SLipInt::ComputeLipschitzInf(int dim, int npts, double* XData, double* YData)
{
	if(LipConst!=NULL) {
		if(dim!=Dim) {
			free(LipConst);
			LipConst=(double*) malloc(sizeof(double)*dim);
		}
	} else
		LipConst=(double*) malloc(sizeof(double)*dim);

	for(j=0;j<dim;j++) LipConst[j]=0;

	Dim=dim;

	int dir;
	int k1,k2;

// for all pairs

	for(i=0;i<npts;i++) {
		k1=dim*i; 
		for(j=i+1;j<npts;j++) {
				k2=dim*j;
				d1=distInf(dim,&(XData[k2]),&(XData[k1]),&dir);
				g1=fabs(YData[j] - YData[i]);

				if(d1>0) 
					LipConst[dir] = max__(LipConst[dir], g1/d1);
		}
	}

	MaxLipConst=0;
	for(i=0;i<dim;i++) MaxLipConst=max__(MaxLipConst,LipConst[i]);

}


// same as above, but data stored in columns (Fortran-like)

double	SLipInt::Value2Col(int dim, int npts,   double* x, double* XData, double* YData)
{
	return Value2Col(dim,npts,x, XData, YData, MaxLipConst);
}

double	SLipInt::Value2Col(int dim, int npts,   double* x, double* XData, double* YData,   double Lipconst)
{
	g1=-10e20;
	g2=-g1;

	for(i=0;i<npts;i++) {
		d2 = distCol(dim,npts,  x, &(XData[i]));
		d1= YData[i] - Lipconst * d2;
		d2= YData[i] + Lipconst * d2;
		if(g1<d1) g1=d1;
		if(g2>d2) g2=d2;
	}
	g2=0.5*(g1+g2);
	return g2;
}

double	SLipInt::ValueInfCol(int dim, int npts,  double* x, double* XData, double* YData)
{
	return ValueInfCol(dim,npts,x, XData, YData, LipConst);
}


double	SLipInt::ValueInfCol(int dim, int npts, double* x, double* XData, double* YData,   double* Lipconst)
{ 
	g1=-10e20;
	g2=-g1;

	for(i=0;i<npts;i++) {
		d2 = distInfCol(dim,npts,  x, &(XData[i]),&j);
		d1= YData[i] - Lipconst[j] * d2;
		d2= YData[i] + Lipconst[j] * d2;
		if(g1<d1) g1=d1;
		if(g2>d2) g2=d2;
	}
	g2=0.5*(g1+g2);
	return g2;
}

double	SLipInt::ValueInfCol(int dim, int npts, double* x, double* XData, double* YData,   double Lipconst)
{ 
	g1=-10e20;
	g2=-g1;

	for(i=0;i<npts;i++) {
		d2 = distInfCol(dim, npts, x, &(XData[i]),&j);
		d1= YData[i] - Lipconst * d2;
		d2= YData[i] + Lipconst * d2;
		if(g1<d1) g1=d1;
		if(g2>d2) g2=d2;
	}
	g2=0.5*(g1+g2);
	return g2;
}


void	SLipInt::ComputeLipschitz2Col(int dim, int npts, double* XData, double* YData)
{
	MaxLipConst=0;
// for all pairs
	for(i=0;i<npts;i++) {
		for(j=i+1;j<npts;j++) {
				d1=distCol(dim,npts, &(XData[i]),&(XData[j]));
				g1=fabs(YData[j] - YData[i]);

				if(d1>0) 
					MaxLipConst = max__(MaxLipConst, g1/d1);
		}
	}
}


void	SLipInt::ComputeLipschitzInfCol(int dim, int npts, double* XData, double* YData)
{
	if(LipConst!=NULL) {
		if(dim!=Dim) {
			free(LipConst);
			LipConst=(double*) malloc(sizeof(double)*dim);
		}
	} else
		LipConst=(double*) malloc(sizeof(double)*dim);

	for(j=0;j<dim;j++) LipConst[j]=0;

	Dim=dim;

	int dir;

// for all pairs

	for(i=0;i<npts;i++) {
		for(j=i+1;j<npts;j++) {
				d1=distInfCol(dim,npts, &(XData[i]),&(XData[j]),&dir);
				g1=fabs(YData[j] - YData[i]);

				if(d1>0) 
					LipConst[dir] = max__(LipConst[dir], g1/d1);	
		}
	}

	MaxLipConst=0;
	for(i=0;i<dim;i++) MaxLipConst=max__(MaxLipConst,LipConst[i]);

}




