(**
   A list gadget. Is prepared for handling multible columns. Can be
   used as multi-selection listgadget.
**)

MODULE VO:List;

(*
    Implements a list gadget.
    Copyright (C) 1997  Tim Teulings (rael@edge.ping.de)

    This module is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    This module is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with VisualOberon. If not, write to the Free Software Foundation,
    59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)

IMPORT A  := VO:Base:Adjustment,
       D  := VO:Base:Display,
       E  := VO:Base:Event,
       F  := VO:Base:Frame,
       O  := VO:Base:Object,

       LM := VO:Model:List,

       G  := VO:Object,
       S  := VO:Scroller,

       U  := VO:Base:Util;

CONST
  selectedMsg* = 0;

TYPE
  Prefs*     = POINTER TO PrefsDesc;

  (**
    In this class all preferences stuff of the button is stored.
  **)

  PrefsDesc* = RECORD (G.PrefsDesc)
                 rFrame*,
                 wFrame* : LONGINT; (* the frame to use for the button *)
               END;

  List*         = POINTER TO ListDesc;
  ListDesc*     = RECORD (G.GadgetDesc)
                    model-      : LM.ListModel;
                    top,
                    bottom      : LM.ListEntry;

                    adjustment  : A.Adjustment;
                    scroller    : S.Scroller;

                    multi       : BOOLEAN;
                    readonly    : BOOLEAN;    (* this is a read-only list *)
                    useScroller : BOOLEAN;    (* generate a scroller *)
                  END;

  (* Some messages *)

  SelectedMsg*      = POINTER TO SelectedMsgDesc;
  SelectedMsgDesc*  = RECORD (O.MessageDesc)
                        entry* : LM.ListEntry;
                      END;

VAR
  prefs* : Prefs;

  PROCEDURE (p : Prefs) Init*;

  BEGIN
    p.Init^;

    p.rFrame:=F.double3DOut;
    p.wFrame:=F.double3DIn;
  END Init;

  PROCEDURE (l : List) Init*;

  BEGIN
    l.Init^;

    l.SetBackground(D.tableBackgroundColor);

    l.SetPrefs(prefs);

    l.SetFlags({G.canFocus});

    l.model:=NIL;
    l.top:=NIL;
    l.bottom:=NIL;

    l.scroller:=NIL;

    l.multi:=FALSE;
    l.readonly:=FALSE;
    l.useScroller:=TRUE;

    NEW(l.adjustment);
    l.adjustment.Init;
    l.AttachModel(l.adjustment.GetTopModel());
  END Init;

  PROCEDURE (l : List) SetModel*(model : O.Model);

  BEGIN
    IF l.model#NIL THEN
      l.UnattachModel(l.model);
    END;
    IF (model#NIL) & (model IS LM.ListModel) THEN
      l.model:=model(LM.ListModel);
      l.top:=l.model.Get(1);
      l.adjustment.SetTop(1);
      l.AttachModel(l.model);
    ELSE
      l.model:=NIL;
    END;
  END SetModel;

  (**
    This function is used to check if an argument to SetModel
    was successfully accepted.
   **)

  PROCEDURE (l : List) ModelAccepted * (m : O.Model):BOOLEAN;

  BEGIN
    RETURN m=l.model
  END ModelAccepted;


  PROCEDURE (l: List) SetReadOnly*(readonly : BOOLEAN);

  BEGIN
    l.readonly:=readonly;
  END SetReadOnly;

  PROCEDURE (l : List) UseScroller*(use : BOOLEAN);

  BEGIN
    l.useScroller:=use;
  END UseScroller;

  PROCEDURE (l : List) CalcSize*;

  BEGIN
    IF l.readonly THEN
      l.SetBackground(D.backgroundColor);
      l.SetObjectFrame(l.prefs(Prefs).rFrame);
    ELSE
      l.SetBackground(D.tableBackgroundColor);
      l.SetObjectFrame(l.prefs(Prefs).wFrame);
    END;

    l.width:=0;
    l.height:=0;

    IF l.useScroller THEN
      NEW(l.scroller);
      l.scroller.Init;
      l.scroller.SetParent(l);
      l.scroller.SetFlags({G.verticalFlex});
      l.scroller.SetModel(l.adjustment);
      l.scroller.CalcSize;
      INC(l.width,U.MaxLong(3*D.display.spaceWidth,l.scroller.oWidth));
      INC(l.height,U.MaxLong(3*D.display.spaceHeight,l.scroller.oHeight));
    ELSE
      INC(l.width,3*D.display.spaceWidth);
      INC(l.height,3*D.display.spaceHeight);
    END;

    l.minWidth:=l.width;
    l.minHeight:=l.height;

    l.CalcSize^;
  END CalcSize;

  PROCEDURE (l : List) GetEntry(y : LONGINT):LM.ListEntry;

  VAR
    entry  : LM.ListEntry;
    object : G.Object;

  BEGIN
    IF l.top=NIL THEN
      RETURN NIL;
    END;

    entry:=l.top;
    WHILE entry#l.bottom.next DO
      object:=entry.GetObject(1);
      IF (y>=object.y) & (y<=object.y+object.height-1) THEN
        RETURN entry;
      END;
      entry:=entry.next;
    END;
    RETURN NIL;
  END GetEntry;

  PROCEDURE (l : List) HandleMouseEvent*(event : E.MouseEvent;
                                         VAR grab : G.Object):BOOLEAN;
  VAR
    entry    : LM.ListEntry;
    selected : SelectedMsg;

  BEGIN
    IF ~l.visible OR l.disabled OR l.readonly THEN
      IF l.useScroller THEN
        RETURN l.scroller.HandleMouseEvent(event,grab);
      ELSE
        RETURN FALSE;
      END;
    END;

    WITH event : E.ButtonEvent DO
      IF (event.type=E.mouseDown)
      & l.PointIsIn(event.x,event.y) & (event.button=E.button1) THEN
        IF l.useScroller & l.scroller.PointIsIn(event.x,event.y) THEN
          RETURN l.scroller.HandleMouseEvent(event,grab);
        ELSE
          entry:=l.GetEntry(event.y);
          IF entry#NIL THEN
            IF l.multi THEN
              entry.Toggle;
              (* Action: State changed *)
            ELSE
              IF l.model.lastSelected#NIL THEN
                l.model.lastSelected.Deselect;
              END;
              entry.Select;
              (* Action: Selected *)
              NEW(selected);
              selected.entry:=entry;
              l.Send(selected,selectedMsg);
            END;

            RETURN TRUE;
          END;
        END;
      END;
    ELSE
    END;

    IF l.useScroller THEN
      RETURN l.scroller.HandleMouseEvent(event,grab);
    ELSE
      RETURN FALSE;
    END;
  END HandleMouseEvent;

  PROCEDURE (l : List) HandleKeyEvent*(event : E.KeyEvent):BOOLEAN;

  VAR
    keysym : LONGINT;

  BEGIN
    IF event.type=E.keyDown THEN
      keysym:=event.GetKey();
      IF keysym=E.up THEN
        l.adjustment.DecTop;
        RETURN TRUE;
      ELSIF keysym=E.down THEN
        l.adjustment.IncTop;
        RETURN TRUE;
      END;
    END;
    RETURN FALSE;
  END HandleKeyEvent;

  PROCEDURE (l : List) RedrawEntry(entry : LM.ListEntry);

  VAR
    help    : LM.ListEntry;
    current : G.Object;
    draw    : D.DrawInfo;

  BEGIN
    IF ~l.visible THEN
      RETURN;
    END;

    draw:=l.GetDrawInfo();

    help:=l.top;
    WHILE (help#NIL) & (help#l.bottom.next) DO
      IF help=entry THEN
        IF l.useScroller THEN
          draw.InstallClip(l.x,l.y,l.width-l.scroller.oWidth,l.height);
        ELSE
          draw.InstallClip(l.x,l.y,l.width,l.height);
        END;
        current:=entry.GetObject(1);

        IF entry.selected THEN
          draw.mode:={D.selected};
        END;
        current.SetParent(l);
        current.Draw(current.oX,current.oY,current.oWidth,current.oHeight);
        draw.mode:={};
        draw.FreeLastClip;
        RETURN;
      END;
      help:=help.next;
    END;

    IF l.disabled THEN (* This is suboptimal *)
      l.DrawDisabled;
    END;
  END RedrawEntry;


  PROCEDURE (l : List) Draw*(x,y,w,h : LONGINT);

  VAR
    pos,vis : LONGINT;
    current : G.Object;
    entry   : LM.ListEntry;
    exit    : BOOLEAN;
    draw    : D.DrawInfo;

  BEGIN
    l.Draw^(x,y,w,h);

    draw:=l.GetDrawInfo();

    IF l.useScroller THEN
      l.scroller.Resize(-1,l.height);
      l.scroller.Move(l.x+l.width-l.scroller.oWidth,l.y);
      l.scroller.Draw(x,y,w,h);
    END;

    IF l.useScroller THEN
      draw.InstallClip(l.x,l.y,l.width-l.scroller.oWidth,l.height);

      l.DrawBackground(l.x,l.y,l.width-l.scroller.oWidth,l.height);
    ELSE
      draw.InstallClip(l.x,l.y,l.width,l.height);

      l.DrawBackground(l.x,l.y,l.width,l.height);
    END;

    l.top:=l.model.Get(l.adjustment.GetTop());

    pos:=0;
    entry:=l.top;
    l.bottom:=l.top;
    vis:=0;
    exit:=FALSE;
    WHILE ~exit DO
      IF entry#NIL THEN
        entry.CalcSize;
        current:=entry.GetObject(1);
        IF pos+entry.height<=l.height THEN
          IF l.useScroller THEN
            current.Resize(l.width-l.scroller.oWidth,-1);
          ELSE
            current.Resize(l.width,-1);
          END;
          IF entry.selected THEN
            draw.mode:={D.selected};
          END;
          l.CopyBackground(current);
          current.SetParent(l);
          current.Move(l.x,l.y+pos);
          current.Draw(x,y,w,h);
          draw.mode:={};
          INC(pos,entry.height);
          INC(vis);
          entry:=entry.next;
          IF entry#NIL THEN
            l.bottom:=entry;
          END;
        ELSE
          exit:=TRUE;
        END;

      ELSE
        exit:=TRUE;
      END;
    END;

    draw.FreeLastClip;

    l.adjustment.SetDimension(vis,l.model.size);

    IF l.disabled THEN
      l.DrawDisabled;
    END;

  END Draw;

  PROCEDURE (l : List) Resync*(model : O.Model; msg : O.ResyncMsg);

  BEGIN
    IF (l.model#NIL) THEN
      l.adjustment.SetDimension(l.adjustment.GetVisible(),l.model.size);
      IF l.visible & ~l.disabled THEN
        IF msg#NIL THEN
          WITH
            msg : LM.EntryChgdMsg DO
              l.RedrawEntry(msg.entry);
              RETURN;
          ELSE

            l.Redraw;
          END;
        ELSE
          l.Redraw;
        END;
      END;
    END;

  END Resync;

  PROCEDURE (l : List) Hide*;

  BEGIN
    IF l.visible THEN
      l.DrawHide;
      IF l.useScroller THEN
        l.scroller.Hide;
      END;
      l.Hide^;
    END;
  END Hide;

  PROCEDURE (l : List) SetTop*(top : LONGINT);

  BEGIN
    l.adjustment.SetTop(top);
  END SetTop;

  PROCEDURE CreateList*():List;

  VAR
    list : List;

  BEGIN
    NEW(list);
    list.Init;

    RETURN list;
  END CreateList;

BEGIN
  NEW(prefs);
  prefs.Init;
END VO:List.