/* oslConfig.h
 */
#ifndef OSL_OSLCONFIG_H
#define OSL_OSLCONFIG_H

#include "osl/config.h"
#include <boost/thread/mutex.hpp>
#include <stdexcept>
#include <string>

namespace osl
{
  /** osl の実行環境に関する指定 */
  struct OslConfig
  {
    static const int MaxThreads=16;
    /** compile時に指定されたディレクトリを返す. 実行時の指定は環境変数で */
    static const std::string& home();
    static const char * home_c_str();
    static const std::string gpsusiConf();

    /** テストケースのデータ */
    static const std::string test();
    static const char *testFile(const std::string& filename);
    static const char *testCsaFile(const std::string& filename);

    /** 標準の定跡ファイルを返す
     * @param filename specify to use non-standard file
     * (absolute path, or home()/data/filename otherwise)
     */
    static const char *openingBook(const std::string& filenamme="");

    static void setVerbose(bool verbose);
    static bool verbose();

    static void showOslHome();
    static void setNumCPUs(int ncpu);
    static int numCPUs();

    static bool usiMode();
    static void setUsiMode(bool enable=true);
    static bool usiModeInSilent();
    static void setUsiSilent(bool silent=true);

    static size_t residentMemoryUse();
    static size_t memoryUseLimit() { return memory_use_limit; }
    static void setMemoryUseLimit(size_t limit) { memory_use_limit = limit; }
    static double memoryUseRatio() 
    {
      return residentMemoryUse() * 1.0 / memoryUseLimit();
    }
    static bool isMemoryLimitEffective() 
    {
      return memory_use_limit != memory_use_limit_system_max
	&& residentMemoryUse() > 0;
    }
    /** @return standard deviation of normal distribution */
    static unsigned int evalRandom() { return eval_random; }
    static void setEvalRandom(unsigned int sigma) { eval_random = sigma; }

    static bool useLogLinearProbability();
    /** only effective before the first call of useLogLinearProbability() */
    static void enableLogLinearProbability(bool enabled);
    static void setUsiOutputPawnValue(int new_value) { usi_output_pawn_value = new_value; }
    static int usiOutputPawnValue() { return usi_output_pawn_value; }
    static bool forceRootNullWindow() { return force_root_null_window; }
    static int rootNullWindow() { return root_null_window; }
    static void setRootNullWindow(int value) 
    {
      root_null_window = value; 
      force_root_null_window = true;
    }
    static void resetRootNullWindow() { force_root_null_window = false; }
  private:
    static const std::string makeHome();
    static const std::string makeTest();
    static bool isGoodDir(const std::string&);
    static void trySetDir(std::string&, const std::string&);
    static void showOslHome(const std::string&);
    static size_t memory_use_limit;
    static const size_t memory_use_limit_system_max;
    static unsigned int eval_random;
    static bool is_verbose;
    static const int default_ncpus;
    static int num_cpu;
    static volatile bool usi_mode, usi_mode_silent, 
      use_log_linear_probability;
    static int usi_output_pawn_value;
    static volatile bool force_root_null_window;
    static volatile int root_null_window;
  public:
    static boost::mutex lock_io;
  };

  struct NoMoreMemory : std::runtime_error
  {
    NoMoreMemory() : std::runtime_error("memory exhausted")
    {
    }
  };
}

#endif /* OSL_OSLCONFIG_H */
// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
