#include "osl/move_generator/allMoves.h"
#include "osl/move_action/store.h"
#include "osl/move_classifier/safeMove.h"
#include "osl/move_generator/addEffect8.h"
#include "osl/record/record.h"
#include "osl/record/csaRecord.h"
#include "osl/record/csaString.h"
#include "osl/container/pieceVector.h"
#include "osl/container/moveVector.h"
#include "osl/apply_move/applyMove.h"
#include "osl/effect_util/effectUtil.h"
#include "osl/effect_util/additionalEffect.h"
#include "osl/oslConfig.h"

#include <cppunit/TestCase.h>
#include <cppunit/extensions/HelperMacros.h>
#include <iostream>
#include <fstream>


class AddEffect8Test : public CppUnit::TestFixture 
{
  CPPUNIT_TEST_SUITE(AddEffect8Test);
  CPPUNIT_TEST(testOne);
  CPPUNIT_TEST(testAdditional);
  CPPUNIT_TEST(testIsAddEffect8);
  CPPUNIT_TEST(testLong);
  CPPUNIT_TEST(testDropLong);
  CPPUNIT_TEST(testDropShort);
  CPPUNIT_TEST(testDropMember);
  CPPUNIT_TEST(testShort);
  CPPUNIT_TEST(testMoveMember);
  CPPUNIT_TEST(testBug_r3527);
  CPPUNIT_TEST_SUITE_END();
public:
  void testOne();
  void testIsAddEffect8();
  void testDropShort();
  void testDropLong();
  void testDropMember();
  void testShort();
  void testLong();
  void testAdditional();
  void testMoveMember();
  void testBug_r3527();
};

CPPUNIT_TEST_SUITE_REGISTRATION(AddEffect8Test);
using namespace osl;
using namespace osl::state;
using namespace osl::move_generator;
using namespace osl::move_action;

static bool isAddEffect8Move(const NumEffectState& state_org,Move move);
static bool mayNotGeneratedMove(const NumEffectState& savedState,Move move);
void AddEffect8Test::testOne()
{
  {
    NumEffectState state(CsaString(
			   "P1-KY-KE-OU-KI *  *  *  * +RY\n"
			   "P2 *  *  * -KI *  *  *  *  * \n"
			   "P3 * +GI-FU-FU * -FU * -FU * \n"
			   "P4-FU *  *  * -FU+FU *  * +FU\n"
			   "P5 * +KE+KE-KA *  * -FU+FU * \n"
			   "P6+FU * -KY * +FU *  *  *  * \n"
			   "P7 *  *  * +FU+KA * +FU *  * \n"
			   "P8 *  *  * +GI *  *  *  *  * \n"
			   "P9+KY+OU *  * +KI *  *  * -RY\n"
			   "P+00KI00GI00KY\n"
			   "P-00GI00KE00FU00FU00FU00FU\n"
			   "-\n"
			   ).getInitialState());
    CPPUNIT_ASSERT(mayNotGeneratedMove(state,Move(Position(7,6),Position(7,8),LANCE,PTYPE_EMPTY,false,WHITE)));
    MoveVector moves;
    {
      move_action::Store store(moves);
      GenerateAddEffect8::generate(state.getTurn(),state,store);
    }
    CPPUNIT_ASSERT(!moves.isMember(Move(Position(7,6),Position(7,8),LANCE,PTYPE_EMPTY,false,WHITE)) ||
		   (std::cerr << moves << std::endl,0));
  }
  {
    NumEffectState state(CsaString(
			   "P1-KY *  *  *  *  *  * -KE-KY\n"
			   "P2-HI *  *  *  *  * -KI-OU * \n"
			   "P3 *  * +UM-FU *  * -KI-FU * \n"
			   "P4-FU * -FU *  * -GI-FU * -FU\n"
			   "P5 *  *  *  * -FU-FU * +FU * \n"
			   "P6+FU * +FU+FU *  * +FU * +FU\n"
			   "P7 * +FU+KE+GI+GI+FU * -UM * \n"
			   "P8 *  * +KI * +OU * -KI *  * \n"
			   "P9+KY+HI *  *  *  *  *  * +KY\n"
			   "P+00GI00KE00KE00FU00FU\n"
			   "-\n"
			   ).getInitialState());
    CPPUNIT_ASSERT(mayNotGeneratedMove(state,Move(Position(3,8),Position(3,9),GOLD,PTYPE_EMPTY,false,WHITE)));
    MoveVector moves;
    {
      move_action::Store store(moves);
      GenerateAddEffect8::generate(state.getTurn(),state,store);
    }
    CPPUNIT_ASSERT(!moves.isMember(Move(Position(3,8),Position(3,9),GOLD,PTYPE_EMPTY,false,WHITE)) ||
		   (std::cerr << moves << std::endl,0));
  }
  {
    NumEffectState state(CsaString(
			   "P1-KY-KE * -KI * -KA-FU * +RY\n"
			   "P2 * +GI-GI * -FU *  *  *  * \n"
			   "P3 * -OU * -FU * -FU *  * -FU\n"
			   "P4-FU-RY+KE-KY *  *  *  *  * \n"
			   "P5 *  * +KI * +UM *  *  *  * \n"
			   "P6+FU * +FU *  *  *  *  *  * \n"
			   "P7 *  *  *  * +OU+FU+FU * +FU\n"
			   "P8 *  *  *  *  *  *  *  *  * \n"
			   "P9+KY *  *  *  * +KI+GI+KE+KY\n"
			   "P+00KI00GI00FU00FU00FU00FU\n"
			   "P-00KE00FU00FU00FU\n"
			   "-\n").getInitialState());
    MoveVector moves;
    {
      move_action::Store store(moves);
      GenerateAddEffect8::generate(state.getTurn(),state,store);
    }
    CPPUNIT_ASSERT(!moves.isMember(Move(Position(6,4),Position(6,7),LANCE,PTYPE_EMPTY,false,WHITE)) ||
		   (std::cerr << moves << std::endl,0));
  }
}
void AddEffect8Test::testDropShort()
{
  {
    SimpleState state=
      CsaString(
		"P1+NY+TO *  *  *  * -OU-KE-KY\n"
		"P2 *  *  *  *  * -GI-KI *  *\n"
		"P3 * +RY *  * +UM * -KI-FU-FU\n"
		"P4 *  * +FU-FU *  *  *  *  *\n"
		"P5 *  * -KE * +FU *  * +FU *\n"
		"P6+KE *  * +FU+GI-FU *  * +FU\n"
		"P7 *  * -UM *  *  *  *  *  *\n"
		"P8 *  *  *  *  *  *  *  *  * \n"
		"P9 * +OU * -GI *  *  *  * -NG\n"
		"P+00HI00KI00KE00KY00FU00FU00FU00FU00FU00FU\n"
		"P-00KI00KY00FU00FU\n"
		"P-00AL\n"
		"+\n"
		).getInitialState();
    NumEffectState eState(state);
    MoveVector moves;
    {
      move_action::Store store(moves);
      GenerateAddEffect8::generate(BLACK,eState,store);
    }
    CPPUNIT_ASSERT(moves.isMember(Move(Position(4,3),PAWN,BLACK)));
    CPPUNIT_ASSERT(moves.isMember(Move(Position(4,4),KNIGHT,BLACK)));
    CPPUNIT_ASSERT(moves.isMember(Move(Position(3,4),KNIGHT,BLACK)));
    CPPUNIT_ASSERT(moves.isMember(Move(Position(1,4),KNIGHT,BLACK)));
    CPPUNIT_ASSERT(moves.isMember(Move(Position(2,4),KNIGHT,BLACK)));
    // 王手はproduceしない
    CPPUNIT_ASSERT(!moves.isMember(Move(Position(2,2),GOLD,BLACK)));
    // 二歩
    CPPUNIT_ASSERT(!moves.isMember(Move(Position(2,2),PAWN,BLACK)));
  }
}

void AddEffect8Test::testDropLong()
{
  {
    SimpleState state=
      CsaString(
		"P1-KY *  *  *  * -OU * -KE-KY\n"
		"P2 * -HI *  *  *  * -KI *  * \n"
		"P3-FU *  *  * +KI-GI * -FU-FU\n"
		"P4 *  * -FU-FU *  * -FU *  * \n"
		"P5 *  *  * -KE *  *  * +FU * \n"
		"P6+GI-FU+FU *  *  * +FU *  * \n"
		"P7+FU *  * +FU-KI *  *  * +FU\n"
		"P8 * +OU+KI * -FU *  *  *  * \n"
		"P9+KY+KE *  * -RY *  *  * +KY\n"
		"P+00KA00GI00GI00KE00FU00FU00FU00FU\n"
		"P-00KA\n"
		"-\n"
		).getInitialState();
    NumEffectState eState(state);
    MoveVector moves;
    {
      move_action::Store store(moves);
      GenerateAddEffect8::generate(WHITE,eState,store);
    }
    CPPUNIT_ASSERT(!moves.isMember(Move(Position(4,6),BISHOP,WHITE)) ||
		   (std::cerr << moves << std::endl,0)
		   );
  }
  {
    SimpleState state=
      CsaString(
		"P1+NY+TO *  *  *  * -OU-KE *\n"
		"P2 *  *  *  *  * -GI-KI * -KY\n"
		"P3 * +RY *  *  *  *  * -FU-FU\n"
		"P4 *  * +FU-FU *  *  *  *  *\n"
		"P5 *  * -KE * +FU *  * +FU *\n"
		"P6+KE *  * +FU+GI-FU *  * +FU\n"
		"P7 *  * -UM *  *  *  *  *  *\n"
		"P8 *  *  *  *  *  *  *  *  * \n"
		"P9 * +OU * -GI *  *  *  * -NG\n"
		"P+00HI00KA00KI00KE00KY00FU00FU00FU00FU00FU00FU\n"
		"P-00KI00KI00KY00FU00FU\n"
		"P-00AL\n"
		"+\n"
		).getInitialState();
    NumEffectState eState(state);
    MoveVector moves;
    {
      move_action::Store store(moves);
      GenerateAddEffect8::generate(BLACK,eState,store);
    }
    CPPUNIT_ASSERT(!moves.isMember(Move(Position(2,2),ROOK,BLACK)));
    CPPUNIT_ASSERT(moves.isMember(Move(Position(2,2),LANCE,BLACK)));

    // 飛車のただ捨て
    CPPUNIT_ASSERT(!moves.isMember(Move(Position(4,3),ROOK,BLACK)));
    CPPUNIT_ASSERT(moves.isMember(Move(Position(4,3),LANCE,BLACK)));
    CPPUNIT_ASSERT(moves.isMember(Move(Position(4,4),ROOK,BLACK)));
    CPPUNIT_ASSERT(moves.isMember(Move(Position(4,4),LANCE,BLACK)));
    CPPUNIT_ASSERT(moves.isMember(Move(Position(4,5),ROOK,BLACK)));
    CPPUNIT_ASSERT(moves.isMember(Move(Position(4,5),LANCE,BLACK)));

    CPPUNIT_ASSERT(moves.isMember(Move(Position(5,2),ROOK,BLACK)));
    CPPUNIT_ASSERT(moves.isMember(Move(Position(6,2),ROOK,BLACK)));

    CPPUNIT_ASSERT(!moves.isMember(Move(Position(7,2),ROOK,BLACK)));
    CPPUNIT_ASSERT(!moves.isMember(Move(Position(8,2),ROOK,BLACK)));
    CPPUNIT_ASSERT(!moves.isMember(Move(Position(9,2),ROOK,BLACK)));

    CPPUNIT_ASSERT(moves.isMember(Move(Position(1,1),BISHOP,BLACK)));
    //
    CPPUNIT_ASSERT(!moves.isMember(Move(Position(3,3),BISHOP,BLACK)));
    CPPUNIT_ASSERT(moves.isMember(Move(Position(4,4),BISHOP,BLACK)));

    CPPUNIT_ASSERT(!moves.isMember(Move(Position(4,3),BISHOP,BLACK)));

    CPPUNIT_ASSERT(moves.isMember(Move(Position(5,4),BISHOP,BLACK)));
    CPPUNIT_ASSERT(!moves.isMember(Move(Position(6,5),BISHOP,BLACK)));
    CPPUNIT_ASSERT(!moves.isMember(Move(Position(7,6),BISHOP,BLACK)));
    CPPUNIT_ASSERT(!moves.isMember(Move(Position(8,7),BISHOP,BLACK)));
    CPPUNIT_ASSERT(moves.isMember(Move(Position(9,8),BISHOP,BLACK)));
    CPPUNIT_ASSERT(!moves.isMember(Move(Position(4,1),BISHOP,BLACK)));

    CPPUNIT_ASSERT(!moves.isMember(Move(Position(2,4),BISHOP,BLACK)));
    CPPUNIT_ASSERT(moves.isMember(Move(Position(1,5),BISHOP,BLACK)));
    CPPUNIT_ASSERT(!moves.isMember(Move(Position(5,1),BISHOP,BLACK)));

    CPPUNIT_ASSERT(moves.isMember(Move(Position(5,2),BISHOP,BLACK)));
    CPPUNIT_ASSERT(moves.isMember(Move(Position(6,3),BISHOP,BLACK)));

    // direct系
    CPPUNIT_ASSERT(moves.isMember(Move(Position(1,1),ROOK,BLACK)));
    CPPUNIT_ASSERT(!moves.isMember(Move(Position(3,3),ROOK,BLACK)));
    CPPUNIT_ASSERT(moves.isMember(Move(Position(3,3),LANCE,BLACK)));
    CPPUNIT_ASSERT(moves.isMember(Move(Position(3,4),ROOK,BLACK)));
    CPPUNIT_ASSERT(moves.isMember(Move(Position(3,4),LANCE,BLACK)));
    CPPUNIT_ASSERT(moves.isMember(Move(Position(3,5),ROOK,BLACK)));
    CPPUNIT_ASSERT(moves.isMember(Move(Position(3,5),LANCE,BLACK)));
    // 2つ以上は生成しない
    CPPUNIT_ASSERT(!moves.isMember(Move(Position(3,6),ROOK,BLACK)));
    CPPUNIT_ASSERT(!moves.isMember(Move(Position(3,7),ROOK,BLACK)));
    CPPUNIT_ASSERT(!moves.isMember(Move(Position(3,8),ROOK,BLACK)));
    CPPUNIT_ASSERT(!moves.isMember(Move(Position(3,9),ROOK,BLACK)));

    CPPUNIT_ASSERT(moves.isMember(Move(Position(3,6),LANCE,BLACK)));
    CPPUNIT_ASSERT(moves.isMember(Move(Position(3,7),LANCE,BLACK)));
    CPPUNIT_ASSERT(moves.isMember(Move(Position(3,8),LANCE,BLACK)));
    CPPUNIT_ASSERT(moves.isMember(Move(Position(3,9),LANCE,BLACK)));
    CPPUNIT_ASSERT(!moves.isMember(Move(Position(5,3),BISHOP,BLACK)));


    // 王手はproduceしない
    CPPUNIT_ASSERT(!moves.isMember(Move(Position(2,2),BISHOP,BLACK)));
    CPPUNIT_ASSERT(!moves.isMember(Move(Position(4,1),ROOK,BLACK)));
    CPPUNIT_ASSERT(!moves.isMember(Move(Position(5,1),ROOK,BLACK)));
    CPPUNIT_ASSERT(!moves.isMember(Move(Position(6,1),ROOK,BLACK)));
    CPPUNIT_ASSERT(!moves.isMember(Move(Position(7,1),ROOK,BLACK)));

    // 盤外への利き
    CPPUNIT_ASSERT(!moves.isMember(Move(Position(4,1),LANCE,BLACK)));
    CPPUNIT_ASSERT(!moves.isMember(Move(Position(6,2),BISHOP,BLACK)));
  }
}

void AddEffect8Test::testShort()
{
  {
    SimpleState state=
      CsaString(
		"P1+NY+TO *  *  *  * -OU-KE-KY\n"
		"P2 *  *  *  *  * -GI-KI *  *\n"
		"P3 * +RY *  * +UM * -KI-FU-FU\n"
		"P4 *  * +FU-FU *  *  *  *  *\n"
		"P5 *  * -KE * +FU *  * +FU *\n"
		"P6+KE *  * +FU+GI-FU *  * +FU\n"
		"P7 *  * -UM *  *  *  *  *  *\n"
		"P8 *  *  *  *  *  *  *  *  * \n"
		"P9 * +OU * -GI *  *  *  * -NG\n"
		"P+00HI00KI00KE00KY00FU00FU00FU00FU00FU00FU\n"
		"P-00KI00KY00FU00FU\n"
		"P-00AL\n"
		"+\n"
		).getInitialState();
    NumEffectState eState(state);
    MoveVector moves;
    {
      move_action::Store store(moves);
      GenerateAddEffect8::generate(BLACK,eState,store);
    }
    CPPUNIT_ASSERT(moves.isMember(Move(Position(4,4),KNIGHT,BLACK)));
    CPPUNIT_ASSERT(moves.isMember(Move(Position(3,4),KNIGHT,BLACK)));
    CPPUNIT_ASSERT(moves.isMember(Move(Position(1,4),KNIGHT,BLACK)));
    CPPUNIT_ASSERT(moves.isMember(Move(Position(2,4),KNIGHT,BLACK)));
    CPPUNIT_ASSERT(moves.isMember(Move(Position(4,3),PAWN,BLACK)));
    // 王手はproduceしない
    CPPUNIT_ASSERT(!moves.isMember(Move(Position(4,3),KNIGHT,BLACK)));
    CPPUNIT_ASSERT(!moves.isMember(Move(Position(2,2),GOLD,BLACK)));
  }
  {
    SimpleState state=
      CsaString(
		"P1+NY *  *  *  *  * -OU-KE-KY\n"
		"P2 * +HI *  * -FU-GI-KI-FU *\n"
		"P3 * +RY *  * +UM-FU-KI * -FU\n"
		"P4 *  * +FU-FU *  *  * +FU *\n"
		"P5 *  * -KE * +FU+KY *  *  *\n"
		"P6+KE * +KA+FU+GI *  *  * +FU\n"
		"P7 *  *  *  *  *  *  *  *  *\n"
		"P8 *  *  *  *  *  *  *  *  * \n"
		"P9 * +OU * -GI *  *  *  * -NG\n"
		"P+00KI00KE00FU00FU00FU00FU00FU00FU\n"
		"P-00KI00KY00FU\n"
		"P-00AL\n"
		"+\n"
		).getInitialState();
    NumEffectState eState(state);
    MoveVector moves;
    {
      move_action::Store store(moves);
      GenerateAddEffect8::generate(BLACK,eState,store);
    }
    CPPUNIT_ASSERT(!moves.isMember(Move(Position(2,4),Position(2,3),PAWN,PTYPE_EMPTY,false,BLACK)));
  }
}
void AddEffect8Test::testLong()
{
  {
    SimpleState state=
      CsaString(
"P1 *  *  *  * +RY *  *  * -KY\n"
"P2 *  *  * -HI *  *  *  *  * \n"
"P3-KY *  * -OU-KA * +KI *  * \n"
"P4 * -FU-FU-GI-KI *  *  *  * \n"
"P5 *  *  * -KE-FU-FU-FU *  * \n"
"P6-KY * +FU+GI *  *  *  *  * \n"
"P7+KE+FU * +KI *  *  *  * +FU\n"
"P8+FU+OU+KI+KA-GI *  *  *  * \n"
"P9 * +KE *  *  *  *  *  * +KY\n"
"P+00GI00KE00FU00FU00FU00FU00FU00FU00FU\n"
"P-00FU00FU\n"
"-\n"
		).getInitialState();
    NumEffectState eState(state);
    MoveVector moves;
    {
      move_action::Store store(moves);
      GenerateAddEffect8::generate(WHITE,eState,store);
    }
    // 2重の追加利き
    CPPUNIT_ASSERT(moves.isMember(Move(Position(6,2),Position(9,2),ROOK,PTYPE_EMPTY,false,WHITE)));
  }
  {
    SimpleState state=
      CsaString(
"P1 *  *  *  *  *  *  * +TO+RY\n"
"P2 *  *  *  *  *  *  *  *  * \n"
"P3+OU+NK * +TO *  *  *  *  * \n"
"P4+GI+KI *  *  *  *  *  *  * \n"
"P5-KI *  *  *  * +GI *  *  * \n"
"P6 *  * -UM * +FU * +TO-KI * \n"
"P7 *  *  *  *  *  * -NK-NY * \n"
"P8 *  *  *  *  *  * -NY-OU-KI\n"
"P9-RY *  *  *  *  *  *  *  * \n"
"P+00KA00KY00KY00FU00FU00FU00FU\n"
"P-00GI00GI00KE00KE00FU00FU00FU00FU00FU00FU00FU00FU00FU00FU\n"
"-\n"
		).getInitialState();
    NumEffectState eState(state);
    MoveVector moves;
    {
      move_action::Store store(moves);
      GenerateAddEffect8::generate(WHITE,eState,store);
    }
    // openだが，直接利きを減らしている．生成しない
    CPPUNIT_ASSERT(!moves.isMember(Move(Position(9,5),Position(8,6),GOLD,PTYPE_EMPTY,false,WHITE)));
  }
  {
    SimpleState state=
      CsaString(
"P1 *  *  *  *  *  *  * +TO+RY\n"
"P2 *  *  *  *  *  *  *  *  * \n"
"P3+OU+NK * +TO *  *  *  *  * \n"
"P4+GI+KI *  *  *  *  *  *  * \n"
"P5-GI *  *  *  * +GI *  *  * \n"
"P6 *  * -UM * +FU * +TO-KI * \n"
"P7 *  *  *  *  *  * -NK-NY * \n"
"P8 *  *  *  *  *  * -NY-OU-KI\n"
"P9-RY *  *  *  *  *  *  *  * \n"
"P+00KA00KY00KY00FU00FU00FU00FU\n"
"P-00KI00GI00KE00KE00FU00FU00FU00FU00FU00FU00FU00FU00FU00FU\n"
"-\n"
		).getInitialState();
    NumEffectState eState(state);
    MoveVector moves;
    {
      move_action::Store store(moves);
      GenerateAddEffect8::generate(WHITE,eState,store);
    }
    // open
    CPPUNIT_ASSERT(moves.isMember(Move(Position(9,5),Position(8,6),SILVER,PTYPE_EMPTY,false,WHITE)));
  }
  {
    SimpleState state=
      CsaString(
"P1-KY-KE *  *  *  *  *  * -KY\n"
"P2 *  *  *  *  *  * -KI-OU * \n"
"P3 *  * -FU *  * -KI *  *  * \n"
"P4-FU-HI *  * -FU * -FU+GI-FU\n"
"P5 * -FU+FU * +GI-FU * -FU+KY\n"
"P6+FU *  *  *  *  * +KE *  * \n"
"P7 * +FU * -TO * +FU+FU+FU * \n"
"P8+KY *  *  *  * +KI+KI+GI * \n"
"P9 * +KE-UM * +KA *  * +KE+OU\n"
"P+00FU\n"
"P-00HI00GI00FU00FU\n"
"+\n"
		).getInitialState();
    NumEffectState eState(state);
    MoveVector moves;
    {
      move_action::Store store(moves);
      GenerateAddEffect8::generate(BLACK,eState,store);
    }
    // 飛車角の追加利きは取られるところには移動しても良い
    CPPUNIT_ASSERT(moves.isMember(Move(Position(5,9),Position(6,8),BISHOP,PTYPE_EMPTY,false,BLACK)));
  }
  {
    SimpleState state=
      CsaString(
"P1 * +RY *  * -FU-OU * -KE-KY\n"
"P2 *  *  *  * +FU-GI-KI *  * \n"
"P3-FU * -FU-FU * -KI *  *  * \n"
"P4 *  *  *  *  * -FU+FU * -FU\n"
"P5 *  *  *  *  *  *  *  *  * \n"
"P6 *  * +FU+FU *  * +KY *  * \n"
"P7+FU+FU *  *  * +FU-UM * +FU\n"
"P8 * +KI * +GI+OU *  *  *  * \n"
"P9+KY+KE *  * +KE *  * -RY+KY\n"
"P+00KA00GI00FU\n"
"P-00KI00GI00KE00FU00FU00FU\n"
"-\n"
		).getInitialState();
    NumEffectState eState(state);
    MoveVector moves;
    {
      move_action::Store store(moves);
      GenerateAddEffect8::generate(WHITE,eState,store);
    }
    CPPUNIT_ASSERT(moves.isMember(Move(Position(2,9),Position(2,7),PROOK,PTYPE_EMPTY,false,WHITE)));
  }
  {
    SimpleState state=
      CsaString(
		"P1-KY-KE * -KI *  * -GI+RY-KY\n"
		"P2 *  *  * -OU *  * -KI *  * \n"
		"P3-FU * -GI-FU-FU-FU *  * -FU\n"
		"P4 *  * -FU *  *  *  *  *  * \n"
		"P5 *  *  *  * +KA *  *  *  * \n"
		"P6 *  *  *  *  *  *  *  *  * \n"
		"P7+FU * +FU+FU+FU+FU+FU * +FU\n"
		"P8 * +GI+KI *  *  *  *  *  * \n"
		"P9+KY+KE *  * +OU+KI+GI * +KY\n"
		"P+00KE00FU00FU\n"
		"P-00HI00KA00KE00FU00FU00FU\n"
		"+\n"
		).getInitialState();
    NumEffectState eState(state);
    MoveVector moves;
    {
      move_action::Store store(moves);
      GenerateAddEffect8::generate(BLACK,eState,store);
    }
    CPPUNIT_ASSERT(moves.isMember(Move(Position(5,5),Position(6,4),BISHOP,PTYPE_EMPTY,false,BLACK)));
    CPPUNIT_ASSERT(moves.isMember(Move(Position(5,5),Position(4,4),BISHOP,PTYPE_EMPTY,false,BLACK)));
    // 飛車角は利きのあるところには移動しても良い
  }

  {
    SimpleState state=
      CsaString(
		"P1+NY+TO *  *  *  * -OU-KE-KY\n"
		"P2 *  *  *  *  * -GI-KI *  *\n"
		"P3 * +RY *  * +UM * -KI-FU-FU\n"
		"P4 *  * +FU-FU *  *  *  *  *\n"
		"P5 *  * -KE * +FU *  * +FU *\n"
		"P6+KE *  * +FU+GI-FU *  * +FU\n"
		"P7 *  * -UM *  *  *  *  *  *\n"
		"P8 *  *  *  *  *  *  *  *  * \n"
		"P9 * +OU * -GI *  *  *  * -NG\n"
		"P+00HI00KI00KE00KY00FU00FU00FU00FU00FU00FU\n"
		"P-00KI00KY00FU00FU\n"
		"P-00AL\n"
		"+\n"
		).getInitialState();
    NumEffectState eState(state);
    MoveVector moves;
    {
      move_action::Store store(moves);
      GenerateAddEffect8::generate(BLACK,eState,store);
    }
    CPPUNIT_ASSERT(moves.isMember(Move(Position(5,3),Position(5,2),PBISHOP,PTYPE_EMPTY,false,BLACK))||
		   (std::cerr << moves << std::endl,0)
		   );
    CPPUNIT_ASSERT(moves.isMember(Move(Position(5,3),Position(4,3),PBISHOP,PTYPE_EMPTY,false,BLACK)));
    CPPUNIT_ASSERT(moves.isMember(Move(Position(8,3),Position(8,2),PROOK,PTYPE_EMPTY,false,BLACK)));
    CPPUNIT_ASSERT(moves.isMember(Move(Position(8,3),Position(7,2),PROOK,PTYPE_EMPTY,false,BLACK)));
    CPPUNIT_ASSERT(moves.isMember(Move(Position(8,3),Position(9,2),PROOK,PTYPE_EMPTY,false,BLACK)));
    CPPUNIT_ASSERT(!moves.isMember(Move(Position(2,2),ROOK,BLACK)));
    CPPUNIT_ASSERT(moves.isMember(Move(Position(2,2),LANCE,BLACK)));
    CPPUNIT_ASSERT(moves.isMember(Move(Position(4,5),LANCE,BLACK)));
    CPPUNIT_ASSERT(moves.isMember(Move(Position(4,4),LANCE,BLACK)));
    CPPUNIT_ASSERT(moves.isMember(Move(Position(4,3),LANCE,BLACK)));
    // 王手はproduceしない
    CPPUNIT_ASSERT(!moves.isMember(Move(Position(5,3),Position(4,2),PBISHOP,PTYPE_EMPTY,false,BLACK)));
    // 盤外に利きを付ける手は生成しない
    CPPUNIT_ASSERT(!moves.isMember(Move(Position(5,3),Position(6,2),PBISHOP,PTYPE_EMPTY,false,BLACK)));
    // 利きがblockされていることはチェックする
    CPPUNIT_ASSERT(!moves.isMember(Move(Position(5,3),Position(4,4),PBISHOP,PTYPE_EMPTY,false,BLACK)));
    CPPUNIT_ASSERT(!moves.isMember(Move(Position(3,4),LANCE,BLACK)));
  }
}
void AddEffect8Test::testAdditional()
{
  {
    SimpleState state=
      CsaString(
		"P1-KY-KE * -KI *  *  *  * +RY\n"
		"P2 * -OU-GI-KI *  *  *  *  * \n"
		"P3 *  * -FU-FU * -FU * -FU * \n"
		"P4-FU+FU *  * -FU-GI *  * +FU\n"
		"P5 *  *  *  *  *  * -FU+FU * \n"
		"P6+FU *  *  * +FU+FU *  *  * \n"
		"P7 * +GI+KE+FU+KA * +FU *  * \n"
		"P8 *  * +OU+GI *  *  *  *  * \n"
		"P9+KY *  *  * +KI *  *  * -RY\n"
		"P+00KE00KY\n"
		"P-00KA00KI00KE00KY00FU00FU00FU\n"
		"+\n"
		).getInitialState();
    NumEffectState eState(state);
    MoveVector moves;
    {
      move_action::Store store(moves);
      GenerateAddEffect8::generate(BLACK,eState,store);
    }
    CPPUNIT_ASSERT(moves.isMember(Move(Position(8,5),LANCE,BLACK)));
    CPPUNIT_ASSERT(moves.isMember(Move(Position(8,6),LANCE,BLACK)));
  }
  {
    SimpleState state=
      CsaString(
		"P1-KY-KE * -KI *  *  *  * +RY\n"
		"P2 * -OU-GI * -KI *  *  *  * \n"
		"P3 * -FU-FU-FU-FU-FU * -FU * \n"
		"P4-FU *  *  * +KE-GI *  * +FU\n"
		"P5 * +FU *  *  *  * -FU+FU * \n"
		"P6+FU * -KY * +FU+FU *  *  * \n"
		"P7 * +GI+KA+FU *  * +FU *  * \n"
		"P8 * +OU+KI+GI *  *  *  *  * \n"
		"P9+KY+KE *  * +KI *  *  * -RY\n"
		"P+00KA\n"
		"P-00KE00KY00FU00FU\n"
		"-\n"
		).getInitialState();
    NumEffectState eState(state);
    MoveVector moves;
    {
      move_action::Store store(moves);
      GenerateAddEffect8::generate(WHITE,eState,store);
    }
    CPPUNIT_ASSERT(moves.isMember(Move(Position(7,4),LANCE,WHITE)));
  }
  {
    SimpleState state=
      CsaString(
		"P1-KY-KE * -KI *  *  *  * +RY\n"
		"P2 * -OU-GI * -KI *  *  *  * \n"
		"P3 * -FU * -FU-FU-FU * -FU * \n"
		"P4 *  *  *  * +KE-GI *  * +FU\n"
		"P5 * +FU-FU *  *  * -FU+FU * \n"
		"P6-FU *  *  * +FU+FU *  *  * \n"
		"P7+FU+GI+KA+FU *  * +FU *  * \n"
		"P8 * +OU+KI+GI *  *  *  *  * \n"
		"P9+KY+KE *  * +KI *  *  * -RY\n"
		"P+00KA\n"
		"P-00KE00KY00KY00FU00FU\n"
		"-\n"
		).getInitialState();
    NumEffectState eState(state);
    MoveVector moves;
    {
      move_action::Store store(moves);
      GenerateAddEffect8::generate(WHITE,eState,store);
    }
    CPPUNIT_ASSERT(!moves.isMember(Move(Position(7,4),LANCE,WHITE)));
    CPPUNIT_ASSERT(moves.isMember(Move(Position(7,6),LANCE,WHITE)));
    CPPUNIT_ASSERT(moves.isMember(Move(Position(9,4),LANCE,WHITE)));
  }
}
static bool isCheckMove(NumEffectState& state,Move move){
  // 自分で動いて元のpositionに利きをつける手は含まない
  Player pl=state.getTurn();
  Position kingPosition=state.getKingPosition(alt(pl));
  NumEffectState next_state(state);
  ApplyMoveOfTurn::doMove(next_state, move);
  // なるべく王手は生成したくないが，生成してしまうこともある．
  return next_state.hasEffectBy(pl,kingPosition);
}

/**
 * 作られない可能性のあるmove
 * 作られる可能性もある．
 */
static bool mayNotGeneratedMove(const NumEffectState& savedState,Move move){
  NumEffectState state = savedState;
  Player pl=state.getTurn();
  Position kingPosition=state.getKingPosition(alt(pl));
  Ptype ptype=move.ptype();
  // 
  if(move.isPromote()) return true;
  if(move.capturePtype()!=PTYPE_EMPTY) return true;
  // 王が動いた時は生成しないことがある．
  if(ptype==KING) return true;
  // 飛車角のdropは2つしか生成しないので
  if((ptype==ROOK || ptype==BISHOP) &&
     !move.from().isOnBoard()) return true;
  // 飛車角歩がpromote可能なのにpromoteしない手は生成しない
  if(move.from().isOnBoard() &&
     (ptype==ROOK || ptype==BISHOP || ptype==PAWN) &&
     (move.from().canPromote(pl) || move.to().canPromote(pl))) return true;
  // 二段目の香車は必ず成る
  if(move.from().isOnBoard() &&
     ptype==LANCE &&
     ((pl==BLACK && move.to().y()==2) ||
      (pl==WHITE && move.to().y()==8))) return true;
  ApplyMoveOfTurn::doMove(state, move);
  // なるべく王手は生成したくないが，生成してしまうこともある．
  if(state.hasEffectBy(pl,kingPosition)) return true;
  for(int i=0;i<8;i++){
    Direction d=static_cast<Direction>(i);
    Position pos=kingPosition+Board_Table.getOffsetForBlack(d);
    if(!pos.isOnBoard()) continue;
    if(savedState.countEffect(pl,pos)>state.countEffect(pl,pos)) continue;
    if(effect_util::AdditionalEffect::count2(savedState,pos,pl)>
       effect_util::AdditionalEffect::count2(state,pos,pl)){
      // 元々の駒が利きのあったところに動いても利きを持つが，追加が変わる例
      // 元々の駒が直接利きを持っていたのを間接利きに変わる例も
      // これは未対応
      if(move.from().isOnBoard() &&
	 savedState.hasEffectByPiece(savedState.getPieceAt(move.from()),pos)){
	return true;
      }
      continue;
    }
    else if(effect_util::AdditionalEffect::count2(savedState,pos,pl)==
	    effect_util::AdditionalEffect::count2(state,pos,pl)) continue;
    else return false;
  }
  return true;
}
/**
 * 相手の8近傍に利きをつける手かどうかのチェック.
 * 欲しい仕様は8近傍のどこかにこれまで利きのなかった駒の利きが追加されること．
 * 全体としてそのマスへの利きが減っても可．
 * 王手は除外する．
 * 王によって8近傍に利きをつける手も除外する．
 * 自殺手は生成してしまうことはあってもよい．
 * @param state - move前の局面
 * @param move - チェックの対象の手
 */
static bool isAddEffect8Move(const NumEffectState& state_org,Move move){
  // 自分で動いて元のpositionに利きをつける手は含まない
  NumEffectState state = state_org;
  Player pl=state.getTurn();
  Position kingPosition=state.getKingPosition(alt(pl));
  NumEffectState savedState(state);
  Ptype ptype=move.ptype();
  if(move.isPromote()) return false;
  if(move.capturePtype()!=PTYPE_EMPTY) return false;
  if(isCheckMove(state,move)) return false;
  // 飛車角は相手の利きのあるところにはdropしない
  if(!move.from().isOnBoard() && (ptype==ROOK || ptype==BISHOP) &&
     state.hasEffectBy(alt(pl),move.to())) return false;
  ApplyMoveOfTurn::doMove(state, move);
  for(int i=0;i<8;i++){
    Direction d=static_cast<Direction>(i);
    Position pos=kingPosition+Board_Table.getOffsetForBlack(d);
    if(!pos.isOnBoard()){
      PtypeO ptypeO=newPtypeO(pl,ptype);
      // 盤外であっても短い利きが増えたらOK
      Position from=move.from();
      Position to=move.to();
      if(move.from().isOnBoard() &&
	 (abs(from.x()-pos.x())>1 ||
	  abs(from.y()-pos.y())>1 ||
	 !Ptype_Table.getEffect(ptypeO,Offset32(pos,move.from())).hasUnblockableEffect()) &&
	 abs(to.x()-pos.x())<=1 &&
	 abs(to.y()-pos.y())<=1 &&
	 Ptype_Table.getEffect(ptypeO,Offset32(pos,move.to())).hasUnblockableEffect() &&
	 (!(ptype==ROOK || ptype==BISHOP || ptype==PAWN) ||
	  !(move.from().canPromote(pl) || move.to().canPromote(pl)))){
	return true;
      }
      continue;
    }
    if(state.hasEffectBy(pl,pos) &&
       (effect_util::AdditionalEffect::count2(savedState,pos,pl)<
	effect_util::AdditionalEffect::count2(state,pos,pl))){
      return true;
    }
    for(int num=0;num<40;num++){
      Piece p=state.getPieceOf(num);
      if(p.ptype()==KING) continue;
      if(p.isOnBoardByOwner(pl) &&
	 state.hasEffectByPiece(p,pos)){
	if(!savedState.hasEffectByPiece(savedState.getPieceOf(num),pos) &&
	   ((p.position()!=move.to() && move.to() != pos)||
	    !move.from().isOnBoard() ||
	    !(ptype==ROOK || ptype==BISHOP || ptype==PAWN) ||
	    !(move.from().canPromote(pl) || move.to().canPromote(pl)))){
	  // ただ捨ての飛車，角のdropは生成しない
	  return true;
	}
	// PROOKが斜めに動いて斜めに利きをつける場合は
	// 元々長い利きがある場合もあるが，常にtrueにする．
	if(p.position()==move.to() && ptype==PROOK && 
	   abs(move.from().x()-move.to().x())==1 &&
	   abs(move.from().y()-move.to().y())==1 &&
	   abs(move.to().x()-pos.x())==1 &&
	   abs(move.to().y()-pos.y())==1){
	  return true;
	}
      }
    }
  }
  return false;
}

void AddEffect8Test::testIsAddEffect8()
{
  {
    SimpleState state=
      CsaString(
"P1-KY-KE-OU-KY+RY *  *  *  * \n"
"P2 *  * -KI-KI *  *  *  *  * \n"
"P3 * +GI-FU *  * +TO * -FU * \n"
"P4-FU *  *  * -FU *  *  * +FU\n"
"P5 * +KE * -KA *  * -FU+FU * \n"
"P6+FU * +KY * +FU *  *  *  * \n"
"P7 * +OU * +FU+KA * +FU *  * \n"
"P8 *  *  * +GI *  *  *  *  * \n"
"P9+KY *  *  * +KI *  *  * -RY\n"
"P+00GI00KE00FU00FU\n"
"P-00KI00GI00KE00FU00FU00FU00FU\n"
"+\n"
		).getInitialState();
    NumEffectState eState(state);
    CPPUNIT_ASSERT(!isAddEffect8Move(eState,Move(Position(5,1),Position(4,2),PROOK,PTYPE_EMPTY,false,BLACK)));
  }
  {
    SimpleState state=
      CsaString(
		"P1 *  * +RY-KE *  *  * -KE-KY\n"
		"P2 *  *  * +GI * -OU-KI *  * \n"
		"P3-FU * -FU-FU * -KI *  *  * \n"
		"P4 *  *  * -UM * -FU+FU * -FU\n"
		"P5+OU *  *  *  *  *  *  *  * \n"
		"P6 *  * +FU-KI *  * +KY *  * \n"
		"P7+FU+FU *  * +FU+FU *  * +FU\n"
		"P8 * +KI *  *  *  *  *  *  * \n"
		"P9+KY+KE *  * -RY *  *  * +KY\n"
		"P+00GI00KE00FU\n"
		"P-00KA00GI00GI00FU00FU00FU00FU00FU\n"
		"+\n"
		).getInitialState();
    NumEffectState eState(state);
    //    CPPUNIT_ASSERT(!isAddEffect8Move(eState,Move(Position(3,4),Position(3,3),PAWN,PTYPE_EMPTY,false,BLACK)));
  }
}


static bool isAddEffect8Drop(const NumEffectState& state_org,Move move){
  // 自分で動いて元のpositionに利きをつける手は含まない
  if(!move.isDrop()) return false;
  NumEffectState state = state_org;
  Player pl=state.getTurn();
  Position kingPosition=state.getKingPosition(alt(pl));
  NumEffectState savedState(state);
  ApplyMoveOfTurn::doMove(state, move);
  if(state.hasEffectBy(pl,kingPosition)) return false;
  // ただ捨ての飛車角
  if(!move.from().isOnBoard() && 
     (move.ptype()==ROOK || move.ptype()==BISHOP) &&
     state.hasEffectBy(alt(pl),move.to())) return false;
  for(int i=0;i<8;i++){
    Direction d=static_cast<Direction>(i);
    Position pos=kingPosition+Board_Table.getOffsetForBlack(d);
    if(!pos.isOnBoard()) continue;
    if(effect_util::AdditionalEffect::count2(savedState,pos,pl)<
       effect_util::AdditionalEffect::count2(state,pos,pl)) return true;
    for(int num=0;num<40;num++){
      Piece p=state.getPieceOf(num);
      if(p.ptype()==KING) continue;
      if(p.isOnBoardByOwner(pl) &&
	 state.hasEffectByPiece(p,pos) &&
	 !savedState.hasEffectByPiece(p,pos))
	return true;
    }
  }
  return false;
}


static void testDropFile(const std::string& fileName){
  Record rec=CsaFile(fileName).getRecord();
  NumEffectState state(rec.getInitialState());
  vector<osl::Move> moves=rec.getMoves();
  for(unsigned int i=0;i<moves.size();i++){
    if (! state.inCheck())
    {				// 王手以外の状況でテスト
      MoveVector allMoves;
      {
	Store store(allMoves);
	AllMoves<Store>::
	  generate(state.getTurn(),state,store);
      }
	  // 次の1行はaddEffect.hのバグの範囲を突き止めるためのテスト
	  //	if(state.hasEffectBy(state.getTurn(),pos)) continue;
      MoveVector effectMoves;
      {
	Store storeEffect(effectMoves);
	GenerateAddEffect8::
	  generate(state.getTurn(),state,storeEffect);
      }
      
      size_t count1=0;
      for(size_t j=0;j<effectMoves.size();j++){
	if(effectMoves[j].isDrop()){
	  count1++;
	  CPPUNIT_ASSERT((state.isValidMove(effectMoves[j],true)  
			 && isAddEffect8Drop(state,effectMoves[j])) ||
			 (std::cerr << std::endl << state  << std::endl << effectMoves[j],0)
			 );
	}
      }
      // 
      size_t count=0;
      MoveVector tmpMoves;
      for(size_t j=0;j<allMoves.size();j++){
	if(isAddEffect8Drop(state,allMoves[j])){
	  tmpMoves.push_back(allMoves[j]);
	  count++;
	  CPPUNIT_ASSERT(effectMoves.isMember(allMoves[j]) ||
			 ((allMoves[j].ptype()==ROOK ||allMoves[j].ptype()==BISHOP) &&
			  !allMoves[j].from().isOnBoard()) ||
			 (std::cerr << allMoves[j] << std::endl <<
			  state << std::endl,0)
			 );
	}
      }
      //      CPPUNIT_ASSERT(count == count1);
    }
    Move move=moves[i];
    ApplyMoveOfTurn::doMove(state, move);
  }
}

void AddEffect8Test::testDropMember(){
  extern bool isShortTest;
  std::ifstream ifs(OslConfig::testCsaFile("FILES"));
  CPPUNIT_ASSERT(ifs);
  int i=0;
  int count=20;
  if (isShortTest) 
    count=2;
  std::string fileName;
  while((ifs >> fileName) && ++i<count){
    if(fileName == "") 
      break;
    if (! isShortTest)
      std::cerr << fileName << " ";
    testDropFile(OslConfig::testCsaFile(fileName));
  }
}

static void testMoveFile(const std::string& fileName){
  Record rec=CsaFile(fileName).getRecord();
  NumEffectState state(rec.getInitialState());
  vector<osl::Move> moves=rec.getMoves();
  for(unsigned int i=0;i<moves.size();i++){
    if (! state.inCheck())
    {				// 王手以外の状況でテスト
      MoveVector allMoves;
      {
	Store store(allMoves);
	AllMoves<Store>::
	  generate(state.getTurn(),state,store);
      }
	  // 次の1行はaddEffect.hのバグの範囲を突き止めるためのテスト
	  //	if(state.hasEffectBy(state.getTurn(),pos)) continue;
      MoveVector effectMoves;
      {
	Store storeEffect(effectMoves);
	GenerateAddEffect8::
	  generate(state.getTurn(),state,storeEffect);
      }
      
      for(size_t j=0;j<effectMoves.size();j++){
	CPPUNIT_ASSERT(state.isValidMove(effectMoves[j],true) ||
		       (std::cerr << std::endl << state << std::endl << effectMoves[j] << std::endl,0));
	CPPUNIT_ASSERT(isSafeMove(state,effectMoves[j]) ||
		       (std::cerr << std::endl << state << std::endl << effectMoves[j] << std::endl,0));
	if(!isAddEffect8Move(state,effectMoves[j]) &&
	   !isCheckMove(state,effectMoves[j])){
	  std::cerr << std::endl << state << std::endl << effectMoves[j] << std::endl;
	}
      }
      // 
      for(size_t j=0;j<allMoves.size();j++){
	if(isAddEffect8Move(state,allMoves[j]))
	  CPPUNIT_ASSERT(effectMoves.isMember(allMoves[j]) ||
			 mayNotGeneratedMove(state,allMoves[j]) ||
			 (std::cerr << std::endl << state << std::endl << allMoves[j] << std::endl,0)
			 );
      }
    }
    Move move=moves[i];
    ApplyMoveOfTurn::doMove(state, move);
  }
}

void AddEffect8Test::testMoveMember(){
  extern bool isShortTest;
  std::ifstream ifs(OslConfig::testCsaFile("FILES"));
  CPPUNIT_ASSERT(ifs);
  int i=0;
  int count=200;
  if (isShortTest) 
    count=2;
  std::string fileName;
  while((ifs >> fileName) && ++i<count){
    if(fileName == "") 
      break;
    if (! isShortTest)
      std::cerr << fileName << " ";
    testMoveFile(OslConfig::testCsaFile(fileName));
  }
}

void AddEffect8Test::testBug_r3527()
{
  NumEffectState state(CsaString(
			 "P1 *  *  * +RY *  *  * -KE-KY\n"
			 "P2 *  *  *  *  * +GI *  *  * \n"
			 "P3-FU *  * +TO * -FU * -FU-OU\n"
			 "P4 *  *  *  * -FU-KI-FU * -FU\n"
			 "P5 * -FU *  *  *  *  *  *  * \n"
			 "P6 *  *  *  * +FU+FU * +FU+FU\n"
			 "P7+FU+FU * +KI *  * +FU * +KE\n"
			 "P8+KY *  * +FU+GI+KA *  * +OU\n"
			 "P9 * +KE-RY *  *  *  *  * -KA\n"
			 "P+00KI00KI00GI00KE00KY00KY\n"
			 "P-00GI00FU00FU\n"
			 "-\n").getInitialState());
  MoveVector moves;
  {
    Store store(moves);
    move_generator::AddEffect8<WHITE>::generate(state, store); // assersion failure here with r3527
  }
  CPPUNIT_ASSERT(! moves.empty());
}

// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
