/*
 * libsysactivity
 * http://sourceforge.net/projects/libsysactivity/
 * Copyright (c) 2009, 2010 Carlos Olmedo Escobar <carlos.olmedo.e@gmail.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <stdlib.h>
#include <sys/types.h>
#include <sys/sysctl.h>
#include <string.h>
#include <errno.h>
#include <vm/vm_param.h>
#include <sys/stat.h>
#include <limits.h>
#include <fcntl.h>
#include <kvm.h>

#include "libsysactivity.h"

#define MIB_SIZE 3

__thread int page_size;
__thread int m_mib[MIB_SIZE];

int sa_open_memory() {
	m_mib[0] = CTL_HW;
	m_mib[1] = HW_PAGESIZE;
	size_t len = sizeof page_size;
	if (sysctl(m_mib, 2, &page_size, &len, NULL, 0) == -1)
		return ENOSYS;

	size_t mib_size = MIB_SIZE;
	if (sysctlnametomib("vm.swap_info", m_mib, &mib_size) == -1)
		return ENOSYS;
	return 0;
}

int sa_get_memory(struct sa_memory* dst) {
	if (dst == NULL)
		return EINVAL;

	int tmp;
	size_t len = sizeof dst;
	if (sysctlbyname("hw.physmem", &tmp, &len, NULL, 0) == -1)
		return ENOSYS;
	dst->total = tmp;

	if (sysctlbyname("vm.stats.vm.v_free_count", &tmp, &len, NULL, 0) == -1)
		return ENOSYS;
	dst->free = tmp * page_size;

	if (sysctlbyname("vfs.bufspace", &tmp, &len, NULL, 0) == -1)
		return ENOSYS;
	dst->buffers = tmp;

	if (sysctlbyname("vm.stats.vm.v_cache_count", &tmp, &len, NULL, 0) == -1)
		return ENOSYS;
	dst->cached = tmp * page_size;

	if (sysctlbyname("vm.stats.vm.v_active_count", &tmp, &len, NULL, 0) == -1)
		return ENOSYS;
	dst->active = tmp * page_size;

	if (sysctlbyname("vm.stats.vm.v_inactive_count", &tmp, &len, NULL, 0) == -1)
		return ENOSYS;
	dst->inactive = tmp * page_size;

	if (sysctlbyname("vm.stats.vm.v_wire_count", &tmp, &len, NULL, 0) == -1)
		return ENOSYS;
	dst->wired = tmp * page_size;

	struct xswdev xsw;
	size_t size = sizeof xsw;
	dst->swap_total = 0;
	dst->swap_free = 0;
	for (m_mib[MIB_SIZE - 1] = 0; sysctl(m_mib, MIB_SIZE, &xsw, &size, NULL, 0) != -1; m_mib[MIB_SIZE - 1]++) {
		dst->swap_total += (uint64_t) xsw.xsw_nblks * page_size;
		dst->swap_free += (uint64_t)(xsw.xsw_nblks - xsw.xsw_used) * page_size;
	}
	return 0;
}

int sa_count_swaps(uint16_t* number) {
	if (number == NULL)
		return EINVAL;

	*number = 0;
	struct xswdev xsw;
	size_t size = sizeof xsw;
	for (m_mib[MIB_SIZE - 1] = 0; sysctl(m_mib, MIB_SIZE, &xsw, &size, NULL, 0) != -1; m_mib[MIB_SIZE - 1]++) {
		(*number)++;
	}
	return 0;
}

int sa_get_swap(uint16_t index, struct sa_swap* dst) {
	if (dst == NULL)
		return EINVAL;

	m_mib[MIB_SIZE - 1] = index;
	struct xswdev xsw;
	size_t size = sizeof xsw;
	if (sysctl(m_mib, MIB_SIZE, &xsw, &size, NULL, 0) == -1)
		return ENODEV;

	devname_r(xsw.xsw_dev, S_IFCHR, dst->name, SA_SWAP_NAME);
	dst->total = (uint64_t) xsw.xsw_nblks * page_size;
	dst->free = (uint64_t)(xsw.xsw_nblks - xsw.xsw_used) * page_size;
	return 0;
}

int sa_get_swaps(struct sa_swap* dst, uint16_t dst_size, uint16_t* written) {
	if (dst == NULL || dst_size == 0 || written == NULL)
		return EINVAL;

	*written = 0;
	int ret;
	uint16_t i;
	for (i = 0; i < dst_size; i++) {
		ret = sa_get_swap(i, &dst[i]);
		if (ret != 0)
			return ret;
		(*written)++;
	}
	return 0;
}
