/*
 * libsysactivity
 * http://sourceforge.net/projects/libsysactivity/
 * Copyright (c) 2009, 2010 Carlos Olmedo Escobar <carlos.olmedo.e@gmail.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <stdio.h>
#include <unistd.h>
#include <pthread.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>
#include <inttypes.h>

#include <libsysactivity/libsysactivity.h>

void print_data_storage_info(struct sa_data_storage* hd) {
#ifdef SA_DATA_STORAGE_NAME
	printf("%s: ", hd->name);
#endif
#ifdef SA_DATA_STORAGE_READS
	printf("%"PRIu64" reads, ", hd->reads);
#endif
#ifdef SA_DATA_STORAGE_READS_MERGED
	printf("%"PRIu64" reads merged, ", hd->reads_merged);
#endif
#ifdef SA_DATA_STORAGE_SECTORS_READ
	printf("%"PRIu64" sectors read, ", hd->sectors_read);
#endif
#ifdef SA_DATA_STORAGE_TIME_SPENT_READING
	printf("%"PRIu64" time spent reading, ", hd->time_spent_reading);
#endif
#ifdef SA_DATA_STORAGE_WRITES
	printf("%"PRIu64" writes, ", hd->writes);
#endif
#ifdef SA_DATA_STORAGE_SECTORS_WRITTEN
	printf("%"PRIu64" sectors written, ", hd->sectors_written);
#endif
#ifdef SA_DATA_STORAGE_TIME_SPENT_WRITING
	printf("%"PRIu64" time spent writing, ", hd->time_spent_writing);
#endif
#ifdef SA_DATA_STORAGE_BYTES_READ
	printf("%"PRIu64" bytes read, ", hd->bytes_read);
#endif
#ifdef SA_DATA_STORAGE_BYTES_WRITTEN
	printf("%"PRIu64" bytes written", hd->bytes_written);
#endif
	printf("\n");
}

void test_data_storage_info(struct sa_data_storage* hd) {
#ifdef SA_DATA_STORAGE_READS
#ifdef SA_DATA_STORAGE_SECTORS_READ
	if (hd->sectors_read < hd->reads) {
		printf("\nERROR: The number of sectors read is lower than the reads\n");
		exit(EXIT_FAILURE);
	}
#endif
	if (hd->reads != 0) {
#ifdef SA_DATA_STORAGE_TIME_SPENT_READING
		if (hd->time_spent_reading == 0) {
			printf("\nERROR: The time spent reading is zero\n");
			exit(EXIT_FAILURE);
		}
#endif
#ifdef SA_DATA_STORAGE_BYTES_READ
		if (hd->bytes_read == 0) {
			printf("\nERROR: The number of bytes read is zero\n");
			exit(EXIT_FAILURE);
		}
#endif
	}
#endif

#ifdef SA_DATA_STORAGE_WRITES
#ifdef SA_DATA_STORAGE_SECTORS_WRITTEN
	if (hd->sectors_written < hd->writes) {
		printf("\nERROR: The number of sectors read is lower than the reads\n");
		exit(EXIT_FAILURE);
	}
#endif
	if (hd->writes != 0) {
#ifdef SA_DATA_STORAGE_TIME_SPENT_WRITING
		if (hd->time_spent_writing == 0) {
			printf("\nERROR: The time spent writing is zero\n");
			exit(EXIT_FAILURE);
		}
#endif
#ifdef SA_DATA_STORAGE_BYTES_WRITTEN
		if (hd->bytes_written == 0) {
			printf("\nERROR: The number of bytes written is zero\n");
			exit(EXIT_FAILURE);
		}
#endif
	}
#endif
}

void* get_data_storage_info(void* arg) {
	int ret;
#ifdef SA_OPEN_DATA_STORAGE
	ret = sa_open_data_storage();
	if (ret != 0) {
		printf("sa_open_data_storage(): %s\n", strerror(ret));
		exit(EXIT_FAILURE);
	}
#endif

	struct sa_data_storage dev;
#ifdef __FreeBSD_kernel__ // gcc -E -dM -x c /dev/null
	ret = sa_get_data_storage("ad0", &dev);
#else
	ret = sa_get_data_storage("sda", &dev);
#endif
	if (ret != 0) {
		printf("sa_get_data_storage(): %s\n", strerror(ret));
		exit(EXIT_FAILURE);
	}
	print_data_storage_info(&dev);
	printf("\n");

	struct sa_data_storage* hds = NULL;
	uint16_t number, written, i;
	for (i = 0; i < 6; i++) {
		ret = sa_count_data_storages(&number);
		if (ret != 0 || number == 0) {
			printf("sa_count_data_storages(): %s\n", strerror(ret));
			exit(EXIT_FAILURE);
		}
		printf("there are %d data storage devices\n", number);

		uint16_t j;
		hds = (struct sa_data_storage*) realloc(hds, number * sizeof(struct sa_data_storage));
		written = 0;
		ret = sa_get_data_storages(hds, number, &written);
		if (ret != 0 || written == 0) {
			printf("sa_get_data_storages(): %s\n", strerror(ret));
			exit(EXIT_FAILURE);
		}
		for (j = 0; j < written; j++) {
			print_data_storage_info(&hds[j]);
		}

		printf("\n");
		sleep(1);
	}
	if (hds != NULL)
		free(hds);
#ifdef SA_CLOSE_DATA_STORAGE
	ret = sa_close_data_storage();
	if (ret != 0) {
		printf("sa_close_data_storage(): %s\n", strerror(ret));
		exit(EXIT_FAILURE);
	}
#endif
#ifdef SA_OPEN_DATA_STORAGE
	ret = sa_open_data_storage();
	if (ret != 0) {
		printf("sa_open_data_storage(): %s\n", strerror(ret));
		exit(EXIT_FAILURE);
	}
#endif
#ifdef SA_CLOSE_DATA_STORAGE
	ret = sa_close_data_storage();
	if (ret != 0) {
		printf("sa_close_data_storage(): %s\n", strerror(ret));
		exit(EXIT_FAILURE);
	}
#endif
	return NULL;
}

int main() {
	pthread_t thread1;
	struct timespec delay;
	delay.tv_sec = 0;
	delay.tv_nsec = 500000000;

	pthread_create(&thread1, NULL, get_data_storage_info, NULL);
	nanosleep(&delay, NULL);
	get_data_storage_info(NULL);

	return EXIT_SUCCESS;
}
