/************************************************************************************
TerraLib - a library for developing GIS applications.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*************************************************************************************/

/*! \file TeRaster.h
    This file contains structures and definitions to deal with raster structures
*/
#ifndef  __TERRALIB_INTERNAL_RASTER_H
#define  __TERRALIB_INTERNAL_RASTER_H

#include "TeDecoder.h"
#include "TeDataTypes.h"
#include "TeCoord2D.h"
#include "TeAttribute.h"
#include "TeGeometry.h"
#include "TeGeometryAlgorithms.h"

class TeRasterTransform;

#include <map>

//! Strategy used to decide if a pixel is inside/outside a polygon
enum TeStrategicIterator
{	TeBoxPixelIn,			//!< center of pixel is inside the geometry
	TeBBoxPixelInters,		//!< bounding box of pixel intercepts the geometry
	TeBoxPixelOut,			//!< center of pixel is outside the geometry
	TeBBoxPixelNotInters	//!< bounding box of pixel do not intercept the geometry
};

/*! \fn bool TeCoordPairVect applyStrategic(double& y, double ymin, double xmin, TeStrategicIterator st, TeRaster* raster, TePolygon& poly);
    \brief Returns  
	\param y         
	\param ymin         
	\param xmin         
	\param st pixel in polygon strategy       
    \param raster pointer to raster
	\param poly polygon        
	*/
TeCoordPairVect 
applyStrategic(double& y, double ymin, double xmin, 
			   TeStrategicIterator st, TeRaster* raster, TePolygon& poly);

//! Support for a raster geometry 
/*
	\par A Raster Geometry is made of a matrix of ncolumns x nlines elements
	with nBands dimension. Each element, or pixel, of the raster its is
	indexed by a column, line and band and has a double value associated.
	\par Each Raster geometry has a decoder associated to it. Its decoder 
	knows how to get return the value (as a double) of each pixel.
	\par All information about the data is stored in a TeRasterParams structure
	that is duplicated in its decoder.
	\sa TeDecoder TeRasterParams
*/
class TeRaster : public TeGeometry
{
	TeRasterParams  params_;	//!< structure that describes all information about a raster 
	TeDecoder		*decoder_;	//!< pointer to a decoder of values
public:

	//! Constructor from file
	/*!
	  This method tries to instatiate a decoder for this data from the 
	  file extension of the filename. 
      \param filename the name of a file that contains the raster data
      \param mode file opening mode ('r', 'w' or 'c')
	  \par Obs: the method init should be called before any attempt to read or write this raster data 
	*/
	TeRaster(const string& filename, const char& mode = 'r');

	//! Constructor from parameters
	/*!
	  This method tries to instatiate a decoder for this data. It uses the identification
	  explicitly described or the extension of a file name specified in the parameters.
      \param pars a raster parameters structure that contains information about the raster
	  \par Obs: the method init should be called before any attempt to read or write data 
	  to the raster.
	*/
	TeRaster(TeRasterParams& pars);

	//! Contructor from commom parameters
	/*
		This method creates a raste data in memory, using only the most basic
		parameters:
		\par ncols number of columns
		\par nlines number of lines
		\par nbands number of bands
		\par elemType size of each element
	*/
	TeRaster(int ncols, int nlines, int nbands, TeDataType elemType);

	//! Empty constructor
	TeRaster() : decoder_(0) {}

	//! Destructor
	~TeRaster ();

	//! Returns the parameters of the raster
	TeRasterParams& params()
	{ return params_; }
	
	//! Returns the number of lines of the raster
	int nBands()
	{ return params_.nBands(); }

	//! Update the parametes of a raster file
	/*!
      \param par a new raster parameters structure
	*/
	void updateParams( TeRasterParams& par)
	{
		params_ = par;	// update raster parameters
		if (decoder_)
			decoder_->updateParams(par);		// update its decoder parameters
		box_ = params_.box();
	}

	//! Returns the type of the geometry
	TeGeomRep elemType() { return TeRASTER; }

	//! Associate a decoder to a raster
	/*!
      \param dec pointer to a raster decoder
	*/
	void setDecoder( TeDecoder* dec )
	{ 
		if (dec)
		{
			decoder_ = dec;					// set the decoder
			params_ = decoder_->params();	// copy decoder parameters to raster
			box_ = params_.box();
		}
	}

	//! Returns status of the raster
	int status() 
	{ return params_.status_; }

	//! Returns a pointer to the decoder associated to this raster
	TeDecoder* decoder( )
	{ return decoder_; }

	//! Returns a pointer to the raster projection
	TeProjection* projection () 
	{ return params_.projection(); }

	//! Sets the value of a element of the raster 
	/*!
      \param col column number
	  \param lin line number
	  \param val element value
	  \param band band number
	*/
	bool setElement (int col, int lin, double val, int band=0);

	//! Gets the value of a element of the raster 
	/*!
      \param col column number
	  \param lin line number
	  \param val element value
	  \param band band number
	*/
	bool getElement (int col, int lin, double& val, int band=0);

	//! Fills a destination raster with the raster elements
	/*
		\param dstRaster destination raster
		\param transf pixel evaluator to be used
		\param bestRes flag to indicate that it should use the best resolution available
	*/
	bool fillRaster (TeRaster* dstRaster, TeRasterTransform* transf=0, bool bestRes = true);

	//! Initalize the raster decoding tool from its parameters
	/*! The result of this method depends on the mode_ specified in the parameters. 
		\par 'r' : initializes a raster with reading only permission. If the source
		of the data doesnt exist it will fail.
		\par 'w' : initializes a raster with reading and writting permission. If the source
		of the data doesnt exist will fail.
		\par 'c' : creates a new raster data with default values. Initializes it with 
		reading and writting permision. Fails if creation fail.
	*/
	bool init();

	//! Initalize the raster decoding tool from a raster parameters structure
	/*!
		\return true if raster is read to be read/written and false otherwise.
	*/
	bool init(TeRasterParams& pars);
  
	//! Clear internal structures and disable the raster decoding tool
	void clear()
	{
		if (decoder_)
			decoder_->clear();
	}

	//! Transform a coordinate from line/column domain to projection domain
	TeCoord2D index2Coord(TeCoord2D pt)
	{ return decoder_->index2Coord(pt); }

	//! Transform a coordinate from projection domain domain to line/column
	TeCoord2D coord2Index(TeCoord2D pt)
	{ return decoder_->coord2Index(pt); }

	//----------------
	//! A raster iterator
	/*! 
		An strucuture that allows the traversal over the raster elements 
		(pixels) in a similar way as the STL iterators. 
	*/
	class iterator
	{
	public:

		//! Constructor
		/*!
		  \param c0 initial column
		  \param l0 initial line
		  \param nc number of columns
		  \param nl number of lines
		  \param nb number of bands
		  \param pt pointer to the raster to iterate
		*/
		 iterator(int c0=0, int l0=0, int nc=0, int nl=0, int nb=0, TeRaster* pt=0) : 
		  linCurrent_(l0),
		  colCurrent_(c0),
		  nLines_(nl),
		  nCols_(nc),
		  nBands_(nb),
		  raster_(pt)
		  {}  

		//! Move iterator forward
		virtual void moveForward()
		{
			this->operator++();
		}

		//! Prefix move forward operator
		iterator& operator++()
		{
			if (++colCurrent_ == nCols_)
			{
				if (linCurrent_ < nLines_-1)
				{
					linCurrent_++;
					colCurrent_ = 0;
				}
			}
			return *this;
		}

		//! Posfix move forward operator
		iterator operator++(int)	
		{
			iterator temp = *this;
			++(*this);
			return temp;
		}

		//! Returns iterator current line 
		int currentLine()
		{ return linCurrent_;}

		//! Returns iterator current column
		int currentColumn()
		{return colCurrent_; }

		//! Returns the number of bands
		int nBands(void)
		{	return nBands_; }

		//! Iterator de-referencing operator
		/*!
			\return a vector (with nbands_ size) with the values in
			each band, of the element pointed by the iterator
		*/
		vector<double> operator*()
		{
			vector<double> vt(nBands_);
			if ((linCurrent_ < nLines_) && (colCurrent_ < nCols_))
			{
				for (int n=0; n<nBands_; n++)
					raster_->getElement(colCurrent_,linCurrent_,vt[n],n);
			}
			return vt;
		}
		
		//! Returns the value in a given band of a element pointed by the iterator 
		double operator*(int band)
		{
			double val = 0.0;
			if ((linCurrent_ < nLines_) && (colCurrent_ < nCols_))
			{
				if (raster_->getElement(colCurrent_,linCurrent_,val,band))
					return val;
			}
			return 0.0;
		}

		//! Returns whether two iterators point to the same element
		bool operator==(const iterator& rhs) const
		{
			return (this->colCurrent_ == rhs.colCurrent_ && this->linCurrent_== rhs.linCurrent_);
		}

		//! Returns whether two iterators point to the different elements
		bool operator!=(const iterator& rhs) const
		{
			return (this->colCurrent_ != rhs.colCurrent_ || this->linCurrent_!= rhs.linCurrent_);
		}

	protected:
		int linCurrent_, colCurrent_;
		int nLines_, nCols_;
		int nBands_;
		TeRaster* raster_;
	};   

	// ---------------------  End of class raster Iterator

	//----------------
	//! A raster iterator with a restriction of area 
	/*! 
		A strucuture that allows to cover the raster elements 
		(pixels) that are IN or OUT a specific region (polygon)  
	*/
	class iteratorPoly : public iterator
	{
		public:
			//! Constructor
			/*!
			\param colCurr	initial column
			\param linCurr	initial line
			\param nc		number of columns
			\param nl		number of lines
			\param nb		number of bands
			\param pt		pointer to the raster that will be covered
			\param poly		polygon that defines a specific region 
			\param str		strategic of the iterator, IN or OUT the polygon 
			\param linMin	minimal line of the raster that will be covered
			\param linMax	maximal line of the raster that will be covered
			\param colMin	minimal column of the raster that will be covered
			\param colMax	maximal column of the raster that will be covered
			\param seg		the segments of intersection of the current line 
			\param posSeg	the current segment 
			\param nlInPoly		number of lines IN the polygon
			\param nColsInPoly	number of columns IN the polygon 
			\param e			iterator end
			\param minLinCM		minimal line (in world coordinates)
			\param band			band to iterate
			*/
			iteratorPoly(int colCurr, int linCurr, int nc, int nl, int nb, TeRaster* pt, TePolygon& poly,
						 TeStrategicIterator str, double linMin=0.0, double linMax=0.0, double colMin=0.0, 
						 double colMax=0.0, TeCoordPairVect seg=TeCoordPairVect(),  
						 int posSeg=0, int nlInPoly=0, int nColsInPoly=0, bool e=true, double minLinCM=0.0, int band = 0):
				iterator(colCurr,linCurr,nc,nl,nb,pt),
				linMin_(linMin),
				linMax_(linMax),
				colMin_(colMin),
				colMax_(colMax),
				segments_(seg),
				posSegments_(posSeg),
				poly_(poly),
				end_(e),
				strategy_(str),
				nLinesInPoly_(nlInPoly),
				nColsInPoly_(nColsInPoly),
				linMinCM_(minLinCM),
				band_(band)
				{}
			
			//! empty contructor
			iteratorPoly(TeRaster* raster = 0): 
				iterator(0,0,0,0,0,raster),
				linMin_(0.0),
				linMax_(0.0),
				colMin_(0.0), 
				colMax_(0.0),
				posSegments_(0),
				end_(true),
				nLinesInPoly_(0),
				nColsInPoly_(0),
				band_(0)
				{}

			//! destructor
			virtual ~iteratorPoly() {}

			//! calculate the segment of the current line that intersect the polygon  					
			void getNewSegment(int linCurr)
			{
				//change to world coordinates
				TeCoord2D coord(colMin_,linCurr);
				TeCoord2D colLinCM = raster_->index2Coord(coord);
				
				double linCM = colLinCM.y();
				double colMinCM = colLinCM.x();
				
				//applyStrategic: return the segments 
				segments_ = applyStrategic(linCM, linMinCM_, colMinCM, strategy_, raster_, poly_);
				
				colMin_ = segments_[0].pt1.x();
				colMax_ = segments_[0].pt2.x();
				
				colCurrent_=(int)colMin_;
				posSegments_ = 0;
				end_ = false;
			}

			void moveForward()
			{
				this->operator++();
			}
			
			//! Prefix move forward operator
			iteratorPoly& operator++() //prefix
			{
				if (++colCurrent_>colMax_)
				{
					if((++posSegments_ > (int)(segments_.size()-1)) || (segments_.size()<1))
					{
						if(++linCurrent_>linMax_)
						{
							end_ = true;
							*this = raster_->end(poly_,strategy_);
						}
						else
						{	
							segments_.clear();
							do
							{
								getNewSegment(linCurrent_);
								if (segments_.empty())
									linCurrent_++;
								else
									break;
							} while (linCurrent_<=linMax_);
							if (linCurrent_ > linMax_)
							{
								end_ = true;
								*this = raster_->end(poly_,strategy_);
							}
						}
					}
					else
					{
						colMin_ = segments_[posSegments_].pt1.x();
						colMax_ = segments_[posSegments_].pt2.x();
						colCurrent_=(int)colMin_;
					}
				}
				
				return *this;
			}

			//! Prefix move orward operator
				iteratorPoly operator++(int)
				{
					iteratorPoly temp = *this;
					++(*this);
					return temp;
				}

			//! Dereferencing operator
			/*!
				This operator was designed for compatibility reasons with standard
				containers of STL. 
				If there is a band restriction defined it will return the value associated to that band.
				Otherwise will return the value associated to band 0.
			*/
			double operator*() 
			{
				double val = 0.0;
				if ((linCurrent_ < nLines_) && (colCurrent_ < nCols_))
				{
					if (raster_->getElement(colCurrent_,linCurrent_,val,band_))
						return val;
				}
				return 0.0;
			}

			//! Returns the value of a given band of the element pointed by the iterator 
			double operator*(int band)
			{
				double val = 0.0;
				if ((linCurrent_ < nLines_) && (colCurrent_ < nCols_))
				{
					if (raster_->getElement(colCurrent_,linCurrent_,val,band))
						return val;
				}
				return 0.0;
			}
		
		    //! Returns the value of of the element pointed by the iterator as TeProperty structure
			/*!
				If there is a band restriction defined it will return the value associated to that band.
				Otherwise will return the value associated to band 0.
			*/
			bool getProperty(TeProperty& prop) 
			{
				double val = 0.0;
				prop.attr_.rep_.type_ = TeREAL;
				if (!raster_) prop.attr_.rep_.name_ = raster_->params().fileName_ + Te2String(band_);
				prop.value_ = Te2String (val);
				
				if ((linCurrent_ < nLines_) && (colCurrent_ < nCols_))
				{
					if (raster_->getElement(colCurrent_,linCurrent_,val,band_))
					{
						prop.value_ = Te2String (val);
						return true;
					}
				}
				return false;
			}

			//! Returns the last past one element in the polygon
			bool end()
			{	return end_; }

			//! Returns the number of lines covered by the iterator
			int nLinesInPoly()
			{ return nLinesInPoly_;}

			//! Returns the number of columns covered by the iterator
			int nColsInPoly()
			{ return nColsInPoly_;}

		private:
			double				linMin_, linMax_;  
			double				colMin_, colMax_;
			TeCoordPairVect		segments_;
			int					posSegments_;
			TePolygon			poly_;
			bool				end_;
			TeStrategicIterator strategy_;
			int					nLinesInPoly_, nColsInPoly_;
			double				linMinCM_;	//minimal line in world coordinates 
			int					band_;		// band restriction
	};

	//! Returns  an iterator to the first element of the raster
	iterator begin() 
	{
		return iterator(0,0, params_.ncols_,params_.nlines_, params_.nBands(), this);
	}

	//! Returns an iterator to the first element of the raster IN or OUT the polygon 
	/*!
		\param poly polygon that delimitates the iterator traversal
		\param st stragetegy used to decide if a pixel is in or out a polygon
		\param band optional band restriction
	*/
	iteratorPoly begin(TePolygon& poly, TeStrategicIterator st, int band=0);

	
	//! Returns the end past one position of the elements of the raster
    iterator end() 
	{
		return iterator(params_.ncols_,params_.nlines_-1,params_.ncols_,params_.nlines_, params_.nBands(), this);
	}

	//! Returns an iterator to the end element of the raster 
	/*!
		\param poly polygon that delimitates the iterator traversal
		\param st stragetegy used to decide if a pixel is in or out a polygon
		\param band optional band restriction
	*/
	iteratorPoly end(TePolygon& poly, TeStrategicIterator st, int band=0) 
	{
		return iteratorPoly(params_.ncols_,params_.nlines_-1,params_.ncols_,params_.nlines_, params_.nBands(), this, poly, st, band);
	}

	//! Select all blocks of raster, in a certain resolution factor that intercepts a given bounding box
	bool selectBlocks(TeBox& bb, int resFac, TeRasterParams& parBlock) 
	{
		if (!decoder_)
			return false;
		return decoder_->selectBlocks(bb,resFac,parBlock);
	}
	//! Returns the number of blocks selected by the last block selection
	int numberOfSelectedBlocks()
	{
		if (!decoder_)
			return 0;
		return decoder_->numberOfSelectedBlocks();
	}

	//! Returns the current block of a set selected by the last block selection
	bool fetchRasterBlock(TeDecoderMemory* memDec)
	{
		if (!decoder_)
			return false;
		return decoder_->getSelectedRasterBlock(memDec);
	}

	//! Clears the current selection of a set selected by the last block selection
	void clearBlockSelection()
	{
		if (!decoder_)
			return ;
		decoder_->clearBlockSelection();
	}
};

#endif


