/*
TerraLib - a library for developing GIS applications.
Copyright  2001, 2002, 2003 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular
purpose. The library provided hereunder is on an "as is" basis, and the
authors have no obligation to provide maintenance, support, updates,
enhancements, or modifications.
In no event shall INPE be held liable to any party
for direct, indirect, special, incidental, or consequential damages arising
out of the use of this library and its documentation.
*/

#ifndef TEPDIBATCHGEOMOSAIC_HPP
  #define TEPDIBATCHGEOMOSAIC_HPP

  #include "TePDIAlgorithm.hpp"
  #include "TePDIParameters.hpp"
  #include <TeSharedPtr.h>

  /**
   * @brief This is the class for Batch Geographic Mosaic
   * ( geo-referenced rasters ) strategy.
   * @author Emiliano F. Castejon <castejon@dpi.inpe.br>
   *
   * @note The required parameters are:
   * @param blending_type ( std::string ) - Blending type.
   * type ( see each blending strategy documentation for other required
   * parameters ).   
   * @param input_rasters ( TePDITypes::TePDIRasterVectorType ) - 
   * Input rasters vector (i.e if using two bands from each
   * raster, the vector will follow the order [ raster0 raster0
   * raster1 raster1 ...... ] ).
   * @param bands ( std::vector< int > ) - The bands from each input_raster 
   * following the order [Raster0 Band 0 - Raster0 Band 1 - ....
   * RasterN Band 0 - RasterN Band 1 - .... - RasterN Band N ].
   * @param output_raster ( TePDITypes::TePDIRasterPtrType ) - Output raster.
   *
   * @note The optional parameters are:
   *
   * @param dummy_value (double) - A dummy pixel value for use 
   * in pixels where no data is available.
   * @param auto_equalize ( int ) - If present ( any value ), input_raster2
   * auto-equalizing will be made ( using overlap area reference ).    
   *
   * @note The output_raster pixel resolution will follow the best pixel 
   * resolution found.
   * @note Please check each blending strategy for other specific required
   * parameters.
   * @ingroup TePDIMosaicGroup
   */
  class PDI_DLL TePDIBatchGeoMosaic : public TePDIAlgorithm {
    public :
      
      /**
       * @brief Default Constructor.
       *
       */
      TePDIBatchGeoMosaic();

      /**
       * @brief Default Destructor
       */
      ~TePDIBatchGeoMosaic();
      
      /**
       * @brief Checks if the supplied parameters fits the requirements of 
       * each PDI algorithm implementation.
       *
       * @note Error log messages must be generated. No exceptions generated.
       *
       * @param parameters The parameters to be checked.
       * @return true if the parameters are OK. false if not.
       */
      bool CheckParameters( const TePDIParameters& parameters ) const;      

    protected :
    
      /**
       * @brief Reset the internal state to the initial state.
       *
       * @param params The new parameters referente at initial state.
       */
      void ResetState( const TePDIParameters& params );    
     
      /**
       * @brief Runs the current algorithm implementation.
       *
       * @return true if OK. false on error.
       */
      bool RunImplementation();     
      
      /**
       * @brief Locates the raster with the closest box ( relative boxes centers ).
       *
       * @param input_rasters A list with input rasters.
       * @param reference_raster_box The reference raster box.
       * @param reference_raster_projection Reference raster projection.
       * @param closest_raster The returned closest raster.
       */      
      void FindClosestRaster( 
        std::list< TePDITypes::TePDIRasterPtrType >& input_rasters,
        const TeBox& reference_raster_box,
        TeProjection* reference_raster_projection,
        TePDITypes::TePDIRasterPtrType& closest_raster );      
      
      /**
       * @brief Locates the two closest rasters ( relative boxes centers ).
       *
       * @param input_rasters A list with input rasters.
       * @param raster1 The returned first raster.
       * @param raster2 The returned second raster.
       */      
      void FindClosestRasters( 
        std::list< TePDITypes::TePDIRasterPtrType >& input_rasters,
        TePDITypes::TePDIRasterPtrType& raster1,
        TePDITypes::TePDIRasterPtrType& raster2 );
        
      /**
       * @brief Find the required raster channels inside the current parameters.
       *
       * @param raster_ptr The raster pointer.
       * @param params A reference to the current parameters.
       * @param channels The found channels.
       */          
      void findRastersChannels( 
        const TePDITypes::TePDIRasterPtrType& raster_ptr,
        const TePDIParameters& params,
        std::vector< int >& channels );       
  };
  
/** @example TePDIMosaic_test.cpp
 *    Shows how to use this class.
 */    

#endif
