/*  gtktiemu - a TI89/92/92+ emulator
 *  (c) Copyright 2000-2001, Romain Lievin and Thomas Corvazier
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "intl.h"

#include <stdlib.h>
#include "sysdeps.h"
#include <assert.h>
#include <stdio.h>
#include <ctype.h>
#include <sys/types.h>
#ifndef __WIN32__
#include <dirent.h>
#else
#include <glib.h> // for readdir and some others
#endif
#include <sys/stat.h>
#include <unistd.h>
#include "str.h"

#include "lib68k.h"
#include "config.h"
#include "options.h"
#include "memory.h"
#include "newcpu.h"
#include "packets.h"
#include "debug.h"
#include "hardware.h"
#include "keyboard.h"
#include "globinfo.h"
#include "callbacks.h"
#include "intlist.h"
#include "errcodes.h"
#include "cmdinterface.h"
#include "rom.h"
#include "platform.h"

#define DEFAULT_ROMFILE "ti92.rom"

extern struct intlist *listBkptAddress;
extern int nBkptAddress;
extern int rom_loaded;
extern int tib_loaded;
extern ROM_INFO current_rom_info;
extern TIB_INFO current_tib_info;

#define is_num(c)   isdigit(c)
#define is_alnum(c) isalnum(c)


/*****************/
/* ROM functions */
/*****************/

/*
  Search the ROM version string in the ROM
  Arguments:
  - ptr: a ROM or update image
  - size: the size of the buffer
  - version: the returned string version
*/
static int get_rom_version(UBYTE *ptr, int size, char *version)
{
  int i;

  for (i=0x12000; i<size-16; i+=2)
    {
      if (is_num(ptr[i])&&(ptr[i+1]=='.')&&is_num(ptr[i+2])&&
	  (ptr[i+3]==0)&&is_alnum(ptr[i+4])&&is_alnum(ptr[i+5])&&
	  (ptr[i+6]=='/')&&is_alnum(ptr[i+7])&&is_alnum(ptr[i+8])&&
	  (ptr[i+9]=='/')&&is_alnum(ptr[i+10])&&is_alnum(ptr[i+11]))
	{
	  break;
	}
      if (is_num(ptr[i])&&(ptr[i+1]=='.')&&is_num(ptr[i+2])&&
	  is_num(ptr[i+3])&&(ptr[i+4]==0)&&is_alnum(ptr[i+5])&&
	  is_alnum(ptr[i+6])&&(ptr[i+7]=='/')&&is_alnum(ptr[i+8])&&
	  is_alnum(ptr[i+9])&&(ptr[i+10]=='/')&&is_alnum(ptr[i+11])&&
	  is_alnum(ptr[i+12]))
	{
	  break;
	}
      if (is_num(ptr[i])&&(ptr[i+1]=='.')&&is_num(ptr[i+2])&&
	  (ptr[i+3]==0)&&is_alnum(ptr[i+4])&&is_alnum(ptr[i+5])&&
	  is_alnum(ptr[i+6])&&is_alnum(ptr[i+7])&&is_alnum(ptr[i+8])&&
	  is_alnum(ptr[i+9])&&is_alnum(ptr[i+10])&&is_alnum(ptr[i+11]))
	{
	  break;
	}
      if (is_num(ptr[i])&&(ptr[i+1]=='.')&&is_num(ptr[i+2])&&
	  is_alnum(ptr[i+3])&&(ptr[i+4]==0)&&is_alnum(ptr[i+5])&&
	  is_alnum(ptr[i+6])&&is_alnum(ptr[i+7])&&is_alnum(ptr[i+8])&&
	  is_alnum(ptr[i+9])&&is_alnum(ptr[i+10])&&is_alnum(ptr[i+11]))
	{
	  break;
	}
    }
  if (i<size-16) 
    {
      int n;
      for (n=i; n<i+16; n++) 
	{
	  if (ptr[n]==0) 
	    {
	      strcpy(version, ptr+i);
	      (version)[n-i]=0;
	      //free(ptr);
	      return ERR_NONE;
	    }
	}
      strcpy(version, _("Unknown"));
    }
  return ERR_NONE;
}

/*
  Convert a TIB file into a ROM image (without boot block)
*/
int ti68k_convertTibToRom(char *f_in, char *f_out)
{
  char *filename = f_in;
  char *filename2 = f_out;
  FILE *file, *fo;
  byte data;
  char str[128];
  longword flash_size;
  int i, j;
  int num_blocks;
  word last_block;
  byte str_size;
  char date[5];
  char *ext;
  char *signature = "Advanced Mathematics Software";
  int tib = 0;
  int calc_type;
  long offset;

  file = fopen(filename, "rb");
  if(file == NULL)
    {
      fprintf(stderr, "Unable to open this file: <%s>\n", filename);
      return ERR_CANT_OPEN;
    }
  
  ext = strrchr(filename, '.');
  if(ext != NULL)
    {
      strncpy(filename2, filename, strlen(filename)-strlen(ext));
      filename2[strlen(filename)-strlen(ext)] = '\0';
    }  
  else
    strcpy(filename2, filename);
  strcat(filename2, ".rom");

  fo = fopen(filename2, "wb");
  if(fo == NULL)
    {
      fprintf(stderr, "Unable to open this file: <%s>\n", filename2);
      return ERR_CANT_OPEN;
    }

  /* Check whether we have a .89u/.9xu or a .tib file */
  fgets(str, 128, file);
  if(strstr(str, "**TIFL**") == NULL) // is a .89u file
    {
      for(i=0, j=0; i<127; i++) // is a .tib file
	{
	  if(str[i] == signature[j])
	    {
	      j++;
	      if(j==strlen(signature))
		{
		  tib = 1;
		  break;
		}
	    }
	}
      if(j < strlen(signature))
	return ERR_TI_FILE; // not a FLASH file
    }

  /* Now, we convert it */
  rewind(file);
  if(!tib)
    {
      /* If a .89u/.9xu file, we skip the licence header */
      fgets(str, 9, file);
      if(strcmp(str, "**TIFL**")) 
	return ERR_INVALID_FLASH;
      for(i=0; i<4; i++) 
	fgetc(file);
      
      for(i=0; i<4; i++)
	date[i] = fgetc(file);
      DISPLAY("Date of the FLASHapp or License: %02X/%02X/%02X%02X\n", 
	      date[0], date[1], date[2], 0xff & date[3]);
      str_size=fgetc(file);
      for(i=0; i<str_size; i++)
	str[i]=fgetc(file);
      str[i]='\0';
      for(i=16+str_size+1; i<0x4A; i++)
	fgetc(file);
      flash_size = fgetc(file);
      flash_size += (fgetc(file) << 8);
      flash_size += (fgetc(file) << 16);
      flash_size += (fgetc(file) << 24);
      
      if(!strcmp(str, "License"))
	{
	  DISPLAY("There is a license header: skipped.\n");
	  for(i=0; i<flash_size; i++)
	    fgetc(file);
	  
	  fgets(str, 9, file);
	  if(strcmp(str, "**TIFL**"))
	    return ERR_INVALID_FILE;
	  for(i=0; i<4; i++) fgetc(file);
	  for(i=0; i<4; i++)
	    date[i] = 0xff & fgetc(file);
	  DISPLAY("Date of the FLASHapp or License: %02X/%02X/%02X%02X\n", 
		  date[0], date[1], date[2], 0xff & date[3]);
	  str_size=fgetc(file);
	  for(i=0; i<str_size; i++)
	    str[i]=fgetc(file);
	  str[i]='\0';
	  for(i=16+str_size+1; i<0x4A; i++)
	    fgetc(file);
      flash_size = fgetc(file);
      flash_size += (fgetc(file) << 8);
      flash_size += (fgetc(file) << 16);
      flash_size += (fgetc(file) << 24);
	}
    }
  else
    {
      /* else, we can read it directly */
      fseek(file, 0, SEEK_END);
      flash_size = ftell(file);
      fseek(file, 0, SEEK_SET);
      strcpy(str, "basecode");
    }       
  
  DISPLAY("FLASH application name: \"%s\"\n", str);
  DISPLAY("FLASH application/Operating System size: %i bytes.\n", flash_size);

  /* Now, read data from the file and convert it */
  num_blocks = flash_size/65536;
  DISPLAY("Number of blocks: %i\n", num_blocks + 1);
  DISPLAY("Processing: ");

  /* Boot block */
  for(i=0; i<0x05; i++)
    fputc(0xff, fo);

  offset = ftell(file);
  fseek(file, 0x8d, SEEK_CUR); // MSB of the PC reset vector
  data = fgetc(file);
  if((data&0x60)==0x20) // internal (0x200000)  or external (0x400000)
    calc_type=TI89; 
  else
    calc_type=TI92;
  fseek(file, offset, SEEK_SET);

  if(calc_type == TI89)
    fputc(0x20, fo); // internal
  else
    fputc(0x40, fo); // external

  for(i=0x06; i<0x65; i++)
    fputc(0xff, fo);

  fputc(0xf0, fo); // FLASH ROM

  for(i=0x66; i<0x12000; i++)
    fputc(0xff, fo);
  
  /* FLASH upgrade */
  for(i=0; i<num_blocks; i++ )
    {
      DISPLAY(".", i);
      fflush(stdout);

      for(j=0; j<65536; j++)
 	{
	  data=fgetc(file);
	  fputc(data, fo);
	}
    }

  DISPLAY(".");
  fflush(stdout);
  last_block=flash_size % 65536;
  for(j=0; j<last_block; j++)
    {
      data=fgetc(file);
      fputc(data, fo);
    }
  DISPLAY("\n");
  DISPLAY("Completing to 2MB size\n");
  for(j=0x12000+flash_size; j<2*1024*1024; j++)
  fputc(0xff, fo);
  fclose(file);
  fclose(fo);

  return ERR_NONE;
}

int ti68k_getRomFileInfo(const char *filename, ROM_INFO *ri);

/*
  Get informations on the ROM to load and next, load it
*/
int ti68k_loadRom(char *filename)
{
  FILE *fp; 
  ROM_INFO *cri = &current_rom_info;  
  int i;
  int err;

  DISPLAY(_("Trying to load this ROM: <%s>\n"), filename);
  if (rom_loaded)
    { 
      exit_hardware();
      //cb_exit_specific();
      mem_exit();
    }

  rom_loaded = 0;
  nBkptAddress = 0;
  listBkptAddress = 0;
  cycleInstr = glob_inf.itick;

  if((err=ti68k_getRomFileInfo(filename, cri)))
    {
      DISPLAY(_("Unable to get ROM informations.\n"));
      return err;
    }  
  DISPLAY(_("ROM informations:\n"));
  DISPLAY(_("  Calc: %s\n"), (cri->calc_type & TI92) ? "TI92(+)" : "TI89");
  DISPLAY(_("  Version: %s\n"), cri->version);
  DISPLAY(_("  Type: %s "), (cri->internal) ? "internal" : "external");
  DISPLAY(_("%s\n"), (cri->flash) ? "FLASH" : "PROM");
  DISPLAY(_("  Size: %i bytes (%1.2f MB)\n"), cri->size, 
	  (float)(cri->size >> 10)/1024.0);
  
  fp = fopen(filename, "rb");
  if(fp == NULL)
    {
      DISPLAY(_("Unable to open this file: <%s>\n"), filename);
      return ERR_CANT_OPEN;
    }

  glob_inf.romSize = cri->size;
  glob_inf.ramSize = (cri->size == 1024*1024) ? 128 : 256;

  if(!(mem_init())) return ERR_INVALID_ROM;   
  init_hardware();
  init_m68k(); // build CPU table
  if(cri->internal) ti_rom = ti_int_rom; else ti_rom = ti_ext_rom;
  fread(ti_rom, 1, cri->size, fp);
  if(cri->size <= 1024*1024) // if ROM <= 1MB, wrap it
    {
      for(i=0; i<cri->size; i++)
	ti_rom[i+0x100000] = ti_rom[i];
    }
  fclose(fp);
  
  if (!find_pc())
    {
      DISPLAY(_("Unable to find PC in ROM.\n"));
      return ERR_INVALID_ROM;
    }

  printf(_("Calculator loaded : %s"), (cri->calc_type & TI92) ? "TI92" : "TI89");
  printf(_("%c\n"), (cri->calc_type & MODULEPLUS) ? '+' : '\0');

  if (cri->calc_type & TI92)
    keyRow = (int*)keyRow92;
  else 
    keyRow = (int*)keyRow89;

  rom_loaded = 1;

  return ERR_NONE;
}

/*
  This function tries to load an update (.tib/.9xu/.89u).
  A ROM must have been loaded before...
*/
int ti68k_loadTib(char *filename)
{
  FILE *file;
  char str[128];
  longword flash_size;
  int i, j;
  byte str_size;
  char date[5];
  char *signature = "Advanced Mathematics Software";
  int tib = 0;
  UBYTE *buf = ti_rom + 0x12000;
  ROM_INFO *cri = &current_rom_info;

  DISPLAY(_("Trying to load this FLASH upgrade: <%s>\n"), filename);
  if(!rom_loaded) 
    return ERR_ROM_NOT_LOADED;

  file = fopen(filename, "rb");
  if(file == NULL)
    {
      fprintf(stderr, _("Unable to open this file: <%s>\n"), filename);
      return ERR_CANT_OPEN;
    }

  /* Check whether we have a .89u/.9xu or a .tib file */
  fgets(str, 128, file);
  if(strstr(str, "**TIFL**") == NULL) // is a .89u file
    {
      for(i=0, j=0; i<127; i++) // is a .tib file
        {
          if(str[i] == signature[j])
            {
              j++;
              if(j==strlen(signature))
                {
                  //DISPLAY(_("TIB file.\n"));
                  tib = 1;
                  break;
                }
            }
        }
      if(j < strlen(signature))
        return ERR_INVALID_FLASH; // not a FLASH file
    }

  /* Now, we read it */
  rewind(file);
  if(!tib)
    {
      /* If a .89u/.9xu file, we skip the licence header */
      fgets(str, 9, file);
      if(strcmp(str, "**TIFL**"))
        return ERR_INVALID_FLASH;
      for(i=0; i<4; i++)
        fgetc(file);

      for(i=0; i<4; i++)
        date[i] = fgetc(file);
      //DISPLAY("Date of the FLASHapp or License: %02X/%02X/%02X%02X\n",
      //        date[0], date[1], date[2], 0xff & date[3]);
      str_size=fgetc(file);
      for(i=0; i<str_size; i++)
        str[i]=fgetc(file);
      str[i]='\0';
      for(i=16+str_size+1; i<0x4A; i++)
        fgetc(file);
      flash_size  =  LSB(fgetc(file));
      flash_size += (LSB(fgetc(file)) << 8);
      flash_size += (LSB(fgetc(file)) << 16);
      flash_size += (LSB(fgetc(file)) << 24);

      if(!strcmp(str, "License"))
        {
          //DISPLAY("There is a license header: skipped.\n");
          for(i=0; i<flash_size; i++)
            fgetc(file);

          fgets(str, 9, file);
          if(strcmp(str, "**TIFL**"))
            return ERR_INVALID_FLASH;
          for(i=0; i<4; i++) fgetc(file);
          for(i=0; i<4; i++)
            date[i] = 0xff & fgetc(file);
          //DISPLAY("Date of the FLASHapp or License: %02X/%02X/%02X%02X\n",
          //        date[0], date[1], date[2], 0xff & date[3]);
          str_size=fgetc(file);
	  for(i=0; i<str_size; i++)
            str[i]=fgetc(file);
          str[i]='\0';
          for(i=16+str_size+1; i<0x4A; i++)
            fgetc(file);
	  flash_size = fgetc(file);
	  flash_size += (fgetc(file) << 8);
	  flash_size += (fgetc(file) << 16);
	  flash_size += (fgetc(file) << 24);
        }
    }
  else
    {
      /* else, we can read it directly */
      fseek(file, 0, SEEK_END);
      flash_size = ftell(file);
      fseek(file, 0, SEEK_SET);
      strcpy(str, "basecode");
    }

//DISPLAY("FLASH application name: \"%s\"\n", str);
  DISPLAY(_("FLASH application/Operating System size: %i bytes.\n"), flash_size);

  /* Now, read data from the file */
  fread(ti_rom+0x12000, 1, flash_size, file);

  /* Update current ROM infos */
  if ((buf[0x8d] & 0x60) == 0x20)
    {
      cri->internal = 1;
      cri->flash = 1;
    }
  else
    {
      cri->internal = 0;
      cri->flash = 1;
    }
  
  get_rom_version(ti_rom, 0x12000+flash_size, cri->version);

  tib_loaded = 1;

  /* Reboot calc */
  if (!find_pc()) return ERR_INVALID_ROM;
  ti68k_resetLib68k();

  return ERR_NONE;
}

/*
  This function load a ROM image or a TIB file.
  If a TIB file is loaded, it may be converted in a ROM image if necessary.
*/
int ti68k_loadImage(char *filename, char *file_loaded)
{
  char *ext = NULL;

  ext = strrchr(filename, '.');
  if(!str_cmp(ext, ".rom"))
    return ti68k_loadRom(filename); // a ROM image
  else 
    {
      if(rom_loaded) // FLASH upgrade
	{
	  if(!str_cmp(ext, ".tib"))
	    return ti68k_loadTib(filename);
	  else if(!str_cmp(ext, ".89u"))
	    return ti68k_loadTib(filename);
	  else if(!str_cmp(ext, ".9xu"))
	    return ti68k_loadTib(filename);
	  else
	    return ERR_INVALID_ROM;
	}
      else // FLASH image
	{
	  ti68k_convertTibToRom(filename, file_loaded);
	  ti68k_loadRom(file_loaded);
	}
    }
  
  return ERR_NONE;
}

/*
  Return ROM type: internal/external, EPROM/FLASH
*/
int ti68k_getRomType(void)
{
  int internal, flashrom;

  internal = ((ti_rom[5] & 0x60) == 0x20) ? INTERNAL : 0;
  flashrom = (ti_rom[0x65] & 0xf) ? 0 : FLASH_ROM;
  
  return internal | flashrom;
}

/*
  Try to get some informations on the ROM image:
  - size
  - internal/external
  - FLASH/EPROM
  - version
  - TIB
  - calc type
*/
int ti68k_getRomFileInfo(const char *filename, ROM_INFO *ri)
{
  FILE *fp;
  unsigned char *rom;

  fp = fopen(filename, "rb");
  if(fp == NULL)
    {
      DISPLAY(_("Unable to open this file: <%s>\n"), filename);
      return ERR_CANT_OPEN;
    }

  // Retrieve ROM size
  fseek(fp, 0, SEEK_END);
  ri->size = ftell(fp);
  fseek(fp, 0, SEEK_SET);

  if(ri->size < 256) return ERR_INVALID_SIZE;

  // Check if TIB
  if( !((ri->size == 1024*1024) || (ri->size == 1024*1024*2)) )
    ri->tib = 1;
  else
    {
      if (ri->size > 2*1024*1024)
	{
	  DISPLAY(_("Invalid ROM size.\n"));
	  return ERR_INVALID_SIZE;
	}
      else
        ri->tib = 0;
    }
  
  // Load ROM into memory (a bit boring since a ROM is 2MB wide)
  rom = malloc(ri->size + 4);
  memset(rom, 0xff, ri->size);
  if (ri->tib)
    fread(rom + 0x12000, 1, ri->size, fp);
  else
    fread(rom, 1, ri->size, fp);

  fclose(fp);

  ri->internal = ((rom[5]&0x60) == 0x20) ? 1 : 0;
  ri->flash = (rom[0x65]&0xf) ? 0 : 1;
  
  if(ri->internal && ri->flash)
    ri->calc_type = TI89;
  else if (ri->flash)
    ri->calc_type = TI92 | MODULEPLUS;
  else 
    ri->calc_type = TI92;
  
  get_rom_version(rom, ri->size, ri->version);
  free(rom);

  return ERR_NONE;
}

/*
  Try to get some informations on the FLASH upgrade file:
  - size
  - internal/external
  - version
*/
int ti68k_getTibFileInfo(const char *filename, TIB_INFO *ri)
{
  FILE *fp;
  char str[128];
  longword flash_size;
  int i, j;
  byte str_size;
  char date[5];
  char *signature = "Advanced Mathematics Software";
  int tib = 0;
  UBYTE *buf;
  char *ext = NULL;

  ext = strrchr(filename, '.');
  printf("ext=<%s>\n", ext);
  if(ext==NULL)
    return ERR_TI_FILE;
#ifdef __WIN32__
  if(!stricmp(ext, ".89u"))
#else
  if(!strcasecmp(ext, ".89u"))
#endif
    ri->calc_type = TI89;
  else
    ri->calc_type = TI92;

  fp = fopen(filename, "rb");
  if(fp == NULL)
    {
      fprintf(stderr, _("Unable to open this file: <%s>\n"), filename);
      return ERR_CANT_OPEN;
    }

  /* Check whether we have a .89u/.9xu or a .tib file */
  fgets(str, 128, fp);
  if(strstr(str, "**TIFL**") == NULL) // is a .89u file
    {
      for(i=0, j=0; i<127; i++) // is a .tib file
        {
          if(str[i] == signature[j])
            {
              j++;
              if(j==strlen(signature))
                {
                  //DISPLAY(_("TIB file.\n"));
                  tib = 1;
                  break;
                }
            }
        }
      if(j < strlen(signature))
        return ERR_INVALID_FLASH; // not a FLASH file
    }

  /* Now, we read it */
  rewind(fp);
  if(!tib)
    {
      /* If a .89u/.9xu file, we skip the licence header */
      fgets(str, 9, fp);
      if(strcmp(str, "**TIFL**"))
        return ERR_INVALID_FLASH;
      for(i=0; i<4; i++)
        fgetc(fp);

      for(i=0; i<4; i++)
        date[i] = fgetc(fp);
      //DISPLAY("Date of the FLASHapp or License: %02X/%02X/%02X%02X\n",
      //        date[0], date[1], date[2], 0xff & date[3]);
      str_size=fgetc(fp);
      for(i=0; i<str_size; i++)
        str[i]=fgetc(fp);
      str[i]='\0';
      for(i=16+str_size+1; i<0x4A; i++)
        fgetc(fp);
      flash_size = fgetc(fp);
      flash_size += (fgetc(fp) << 8);
      flash_size += (fgetc(fp) << 16);
      flash_size += (fgetc(fp) << 24);
      
      if(!strcmp(str, "License"))
        {
          //DISPLAY("There is a license header: skipped.\n");
          for(i=0; i<flash_size; i++)
            fgetc(fp);

          fgets(str, 9, fp);
          if(strcmp(str, "**TIFL**"))
            return ERR_INVALID_FLASH;
          for(i=0; i<4; i++) fgetc(fp);
          for(i=0; i<4; i++)
            date[i] = 0xff & fgetc(fp);
          //DISPLAY("Date of the FLASHapp or License: %02X/%02X/%02X%02X\n",
          //        date[0], date[1], date[2], 0xff & date[3]);
          str_size=fgetc(fp);
	  for(i=0; i<str_size; i++)
            str[i]=fgetc(fp);
          str[i]='\0';
          for(i=16+str_size+1; i<0x4A; i++)
            fgetc(fp);
	  flash_size = fgetc(fp);
	  flash_size += (fgetc(fp) << 8);
	  flash_size += (fgetc(fp) << 16);
	  flash_size += (fgetc(fp) << 24);
        }
    }
  else
    {
      /* else, we can read it directly */
      fseek(fp, 0, SEEK_END);
      flash_size = ftell(fp);
      fseek(fp, 0, SEEK_SET);
      strcpy(str, "basecode");
    }

  //DISPLAY("FLASH application name: \"%s\"\n", str);
  //  DISPLAY("FLASH application/Operating System size: %i bytes.\n", flash_size);

  // Load TIB into memory (a bit boring since a ROM is 2MB wide)
  buf = malloc(0x12000+flash_size + 4);
  memset(buf, 0xff, 0x12000+flash_size);

  /* Now, read data from the fp */
  fread(buf+0x12000, 1, flash_size, fp);

  /* Update current ROM infos */
  if ((buf[0x8d] & 0x60) == 0x20)
    {
      ri->internal = 1;
      ri->calc_type = TI89;
    }
  else
    {
      ri->internal = 0;
      ri->calc_type = TI92|MODULEPLUS;
    }
  
  get_rom_version(buf, 0x12000+flash_size, ri->version);

  free(buf);
  return ERR_NONE;
}


/*
  Scan ROM images in a given directory and build a cache file.
*/
int ti68k_scanRomImages(const char *dirname, const char *filename)
{
  DIR *dir;
  struct dirent *file;
  struct stat f_info;
  char *path, *path2;
  char *text[6]; // 6 columns
  ROM_INFO ri;
  char *p1, *p2, *p3, *p5;
  char buffer[MAXCHARS];
  char buffer2[20];
  int err;
  FILE *fp = NULL;
  char rom_names[256][MAXCHARS];
  int i=0, j;
  int res=-1;

  DISPLAY(_("Scanning ROM images... "));

  /* First, check if cache file exists */
  if(!access(filename, F_OK))
    {
      // if yes, read it
      fp = fopen(filename, "rt");
      if(fp == NULL)
	{
	  fprintf(stderr, _("Unable to open this file: <%s>\n"), filename);
	  return ERR_CANT_OPEN;
	}

      while(!feof(fp))
	{
	  fgets(buffer, MAXCHARS, fp);
	  sscanf(buffer, "%s\t", buffer2);
	  strcpy(rom_names[i], buffer2);
	  i++;
	}
      // and store ROM names for comparison
      strcpy(rom_names[i], "");
      fclose(fp);
      fp = fopen(filename, "at");
      if(fp == NULL)
        {
          fprintf(stderr, _("Unable to reopen this file: <%s>\n"), filename);
          return ERR_CANT_OPEN;
        }
    }
  else
    {
      DISPLAY(_("cache need to be built... "));
      fp = fopen(filename, "wt");
      if(fp == NULL)
        {
          fprintf(stderr, _("Unable to open this file: <%s>\n"), filename);
          return ERR_CANT_OPEN;
        }
    }  

  /* List all ROMs available in the ROM directory */
  path = (char *)dirname;
  if( (dir=opendir(dirname)) == NULL)
    {
      fprintf(stderr, _("Opendir error\n"));
      return ERR_CANT_OPEN_DIR;
    }
  
  while( (file=readdir(dir)) != NULL)
    {
      if(!strcmp(file->d_name, ".")) { continue; }
      if(!strcmp(file->d_name, "..")) { continue; }
      for(j=0; j<i; j++)
	{
	  if(!strcmp(rom_names[j], file->d_name))
	    break;
	}
      if(i==j)
	{
	  // we have a new ROM, we add it in the cache file
	  //strcpy(iupdate->label_text, file->d_name);
	  //iupdate_label();
	  path2 = (char *)malloc((strlen(path) + 1 +
				  strlen(file->d_name) + 1) *
				 sizeof(char));
	  strcpy(path2, path);
	  strcat(path2, file->d_name);
	  if( (err=stat(path2, &f_info)) != -1)
	    {
	      res = ti68k_getRomFileInfo(path2, &ri);
	      if(res)
		  res = ti68k_getTibFileInfo(path2, &ri);
	      if(!res)
		{
		  text[0] = file->d_name;
		  
		  switch(ri.calc_type)
		    {
		    case TI92: p1 = "TI92";
		      break;
		    case TI89: p1 = "TI89";
		      break;
		    case TI92 | MODULEPLUS: p1 = "TI92+";
		      break;
		    default: p1 = "TI??";
		      break;
		    }
		  text[1] = p1;
		  
		  text[2] = ri.version;
		  
		  if(ri.internal)
		    p2 = _("internal");
		  else
		    p2 = _("external");
		  if(ri.flash)
		    p3 = _("FLASH");
		  else
		    p3 = _("PROM");
		  sprintf(buffer, "%s(%s)", p3, p2);
		  text[3] = buffer;
		  
		  sprintf(buffer2, "%iKB", ri.size>>10);
		  text[4] = buffer2;
		  
		  if(ri.tib)
		    p5 = _("yes");
		  else
		    p5 = _("no");
		  text[5] = p5;
		  
		  fprintf(fp, "%s\t%s\t%s\t%s\t%s\t%s\n", text[0], text[1],
			  text[2], text[3], text[4], text[5]);
		}
	      else
		{
		  fprintf(stderr, _("Can not get ROM/update info: <%s>\n"), 
			  path2);
		}
	    }
	  else
	    {
	      fprintf(stderr, _("Can not stat: <%s> %i\n"), file->d_name, err);
	      perror("stat: ");
	    }
	  free(path2);
	}
    }      
  if(closedir(dir)==-1)
    {
      fprintf(stderr, _("Closedir error\n"));
      return ERR_CANT_CLOSE_DIR;
    }
  
  fclose(fp);
  DISPLAY(_("Done.\n"));
  
  return ERR_NONE;
}

/*
  Write ROM to file
*/
int ti68k_dumpRom(char *filename)
{
  FILE *f;
  int i;
  int j;
  int c;
  UBYTE *rom = ti_rom;

  f = fopen(filename, "wt");
  if(f == NULL)
    return ERR_CANT_OPEN;

  //iupdate->total = glob_inf.romSize;
  for(i=0; i<glob_inf.romSize; i+=16) // size in Bytes
    {
      //iupdate->count = i;
      //iupdate->percentage = (float)i/glob_inf.romSize;
      //iupdate_pbar();
      
      fprintf(f, "%06X: ", i);
      for(j=0; j<16; j++)
	{
	  fprintf(f, "%02X ", rom[i+j]);
	}
      fprintf(f, "| ");
      for(j=0; j<16; j++)
        {
	  if( (rom[i+j] > 31) && (rom[i+j] < 128) )
	    c = rom[i+j];
	  else
	    c = ' ';
          fprintf(f, "%c", c);
        }
      fprintf(f, "\n");
    }

  fclose(f);

  return ERR_NONE;
}
