/******************************** LICENSE ********************************

 Copyright 2007 European Centre for Medium-Range Weather Forecasts (ECMWF)

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at 

    http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 ******************************** LICENSE ********************************/


/*! \file SpotDecoder.h
    \brief Implementation of the Template class SpotDecoder.
    
    Magics Team - ECMWF 2005
    
    Started: Mon 19-Sep-2005
    
    Changes:
    
*/
 

#include <locale>
#include "EpsBufr.h"
#include "MetaData.h"
#include "CustomisedPoint.h"
#include "DateTime.h"
#include "TextVisitor.h"
#include "MvObs.h"

using namespace magics;

    
void EpsBufr::visit(Transformation& transformation)
{
	decode();
	transformation.setDataMinX((minstep_ - shift_) * 3600, base_);
	transformation.setDataMaxX((maxstep_ + 6) * 3600, base_);
	transformation.setMinY(miny_);
	transformation.setMaxY(maxy_);
}
		
void EpsBufr::decode()
{
	if ( !points_.empty()) return;
	
     Log::dev() << "EpsBufr::decode()-->" << *this << endl;		
     // Test
     MvObsSet set(path_.c_str());
     
     MvObsSetIterator   filter(set);
     MvLocation ll(latitude_-0.1, longitude_-0.1);
     MvLocation ur(latitude_+0.1, longitude_+0.1);
     filter.setArea(ll, ur);
     
     MvObs obs = filter();
     map<int, vector<float> > members;
     map<int, float > control; 
     
     minstep_ = std::numeric_limits<double>::max();
     maxstep_ = -std::numeric_limits<double>::max();
     miny_ = std::numeric_limits<double>::max();
     maxy_ = -std::numeric_limits<double>::max();

     while (obs)
     {
             //Log::dev()<< "obs---------------------------------------------------------------------------" << endl;		
     	MvLocation loc = obs.location();
//        int subsets = obs.msgSubsetCount();

        float value = obs.value( param_descriptor_ );
        	if ( value == kFortranBufrMissingValue )
		{
        		obs = filter(NR_returnMsg);         	// Were going to the next message!
        		continue;
        	}
        	
    		int i = 0;
    		 
    		while ( ++i  )
		{
    			float step =  obs.valueByOccurrence(i, 4024);   			
    			float value =  obs.valueByOccurrence(i, param_descriptor_);
    			value = (value * param_scaling_factor_ ) + param_offset_factor_;
    			base_ = DateTime(obs.obsTime().CharValue());
    			Log::dev() << "date--->" << base_ << endl; 
    			if ( value == kFortranBufrMissingValue ) {
    				obs = filter(); // We going to the next subset!
    				break;
    			}
    			if  ( minstep_ > step ) minstep_ = step;
    			if  ( maxstep_ < step ) maxstep_ = step;
    			if  ( miny_ > value ) miny_ = value;
    			if  ( maxy_ < value ) maxy_ = value;
    			if ( obs.subsetNumber() == 1 )
			{
    				control[step] = value;
    			}
    			else {
    				map<int, vector<float> >::iterator member = members.find(step);
    				if ( member == members.end() ) {
    					members.insert(make_pair(step, vector<float>()));
    					member = members.find(step);
    				}
    				member->second.push_back(value);      
    			}
    		}
     	}
     
     	for (map<int, vector<float> >::iterator member = members.begin(); member != members.end(); ++member) {
     	    std::sort( member->second.begin(), member->second.end());
     	}

	for (map<int, float>::const_iterator step = control.begin(); step != control.end(); ++step) {
		CustomisedPoint* point = new CustomisedPoint();		
		point->longitude(step->first * 3600);		 
		point->latitude(step->first * 3600);		 

		(*point)["step"]    = step->first * 3600;
		(*point)["shift"] = shift_ *3600;
		(*point)["year"]    = base_.date().year();
		(*point)["month"]   = base_.date().month();
		(*point)["day"]     = base_.date().day();
		(*point)["hours"]   = base_.time().hours();
		(*point)["minutes"] = base_.time().minutes();
		(*point)["seconds"] = base_.time().seconds();
		(*point)["width"]    = 1 * 3600;
		(*point)["control"] = step->second;	
		map<int, vector<float> >::iterator member = members.find(step->first);

		(*point)["min"]  =    	member->second.front();
		(*point)["ten"]  =  	member->second[4];
		(*point)["twentyfive"]  = member->second[12];
		(*point)["median"]  = 	member->second[24];
		(*point)["seventyfive"]  = 	 member->second[30];	
		(*point)["ninty"] = member->second[45];
		(*point)["max"]  = member->second.back();

		points_.push_back(point);
	}
}

void EpsBufr::customisedPoints(const std::set<string>&, CustomisedPointsList& out)
{
	Log::dev() << "EpsBufr::customisedPoints-->need to be implemented" <<endl;
	Log::dev() << *this << endl;
	decode();

	for (vector<CustomisedPoint*>::const_iterator point = points_.begin(); point != points_.end(); ++point)
			out.push_back(*point);	
}

PointsHandler<UserPoint>& EpsBufr::points()
{
}

void EpsBufr::visit(TextVisitor& title)
{
	bool long_title_ = true;
	bool short_title_ = true;
	if ( !long_title_ && !short_title_) return;
	decode();
	if ( long_title_ )
	{
		ostringstream out;
		tm convert = base_;	
		locale loc("");      
		out.imbue(loc);   
		const std::time_put<char>& tfac = use_facet<time_put<char> >(loc); 
		string format = "Forecast VT %A %e %B %Y %H UTC";
		tfac.put(out, out, ' ', &convert, format.c_str(), format.c_str()+format.length()); 
	    		
		ostringstream line;
		GeoPoint position(longitude_, latitude_);
		line << station_name_ << "(" << position.writeLatitude() << ", " << position.writeLongitude() << ")" << endl;
		title.add(new TextEntry(title_));
		title.add(new TextEntry(line.str()));
		title.add(new TextEntry(out.str()));
	}
	if ( short_title_ ) {
		title.add(new TextEntry(""));
		if ( param_title_.empty() ) 
			param_title_ = "" + param_descriptor_; 
		title.add(new TextEntry(param_title_));
	}
}

void EpsBufr::visit(MetaDataVisitor&) {
}
	
void EpsBufr::print(ostream& out) const
{
	out << "EpsBufr[";
	EpsBufrAttributes::print(out);
	out << "]";
}

EpsBufr::EpsBufr() : shift_(12)
{
}

EpsBufr::~EpsBufr()
{
}
