/*
Magpie - reference librarian for Debian systems
Copyright (C) 2000  Bear Giles <bgiles@coyotesong.com>

This program is free software; you may redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the license, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

static const char rcsid[] = "$Id$";

/*****
This module lists all installed packages.
*****/
#include <assert.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include "magpie.h"

extern int mkdir (const char *, mode_t);

#define OUTPUT_DIR	"installed"

static int cnt[CNT_CATEGORIES];

/*+
Comparison function for sorting by category, section, name, version
+*/
static int cmp_c_s_n (const void *p, const void *q)
{
	struct package_info *pp = *((struct package_info **) p);
	struct package_info *qq = *((struct package_info **) q);
	int r;
	
	assert (pp->name);
	assert (qq->name);

	r = pp->category - qq->category;
	if (r)
		return r;

	r = pp->section - qq->section;
	if (r)
		return r;

	r = strcoll (pp->name, qq->name);
	if (r)
		return r;

	return -strcoll (pp->version, qq->version);
}


/*+
Print the intermediate index.
+*/
static int installed_index (FILE *fp, int type)
{
	int cidx;
	char short_category[30], *s;

	if (type != MAGPIE_INSTALLED_PACKAGES)
		return 0;

	mp_title (fp, 3, "Grouped by Category");

	/*
	 *	Write the index files.
	 */
	mp_list_open (fp);
	for (cidx = 0; cidx < CNT_CATEGORIES; cidx++) {
		if (cnt[cidx] == 0)
			continue;
	
		strcpy (short_category, categories[cidx]);
		s = strchr (short_category, '/');
		if (s)
			*s = '\0';

		mp_item_open (fp);
		mp_url (fp, "%1$s/%2$s.html.gz", "%2$s", OUTPUT_DIR, short_category);
		fprintf (fp, " (%d)", cnt[cidx]);
		mp_item_close (fp);
	}
	mp_list_close (fp);

	return 0;
}


/*+
Print the *installed* packages ordered by category/section/priority.
+*/
static int installed_init (void)
{
	int sidx, cidx, i;
	FILE *fp = NULL;
	struct package_info *p;
	char short_category[30], *s;
	char pathname[256];

	sidx = -1;
	cidx = -1;
	memset (cnt, 0, sizeof cnt);

	if (mkdir (OUTPUT_DIR, 0755) == -1) {
		perror ("mkdir (installed)");
		return -1;
	}

	qsort (cache, cachecnt, sizeof (cache[0]), cmp_c_s_n);

	for (i = 0; i < cachecnt; i++) {
		p = cache[i];
		if (!p->installed && !p->unpacked)
			continue;

		if (cidx != p->category || sidx != p->section) {
			if (cidx != p->category) {
				if (fp != NULL) {
					mp_doc_close (fp);
					fclose (fp);
					gzip (pathname);
				}

				cidx = p->category;
				strcpy (short_category, categories[cidx]);
				s = strchr (short_category, '/');
				if (s)
					*s = '\0';
				sprintf (pathname, "%s/%s.html", OUTPUT_DIR, short_category);
				fp = fopen (pathname, "w");
				mp_doc_open (fp, "Installed Packages, Category '%s'", 
					categories[cidx]);
			}

			sidx = p->section;
			mp_title (fp, 2, "Section '%s'", sections[sidx]);
		}

		assert (fp);
		mp_package (fp, p, 1, 0);
		cnt[cidx]++;
	}
	if (fp != NULL) {
		mp_doc_close (fp);
		fclose (fp);
		gzip (pathname);
	}
	return 0;
}


struct magpie_module mod_installed = { 
	version           : MAGPIE_VERSION,
	description       : "group packages by category",
	init              : installed_init,
	annotated_index   : installed_index
};
