/*
Magpie - reference librarian for Debian systems
Copyright (C) 2000  Bear Giles <bgiles@coyotesong.com>

This program is free software; you may redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the license, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

static const char rcsid[] = "$Id$";

/*****
This module lists all packages, sorted by package maintainer.
*****/
#include <assert.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include "magpie.h"

#define OUTPUT_FILE	"maintainer.html"


/*+
Comparison function for sorting by maintainer.
We provide one list sorted by maintainer since that can suggest 
other packages we may be interested in.
+*/
static int cmp_maintainer (const void *p, const void *q)
{
	struct package_info *pp = *((struct package_info **) p);
	struct package_info *qq = *((struct package_info **) q);
	int r;
	
	assert (pp->maintainer);
	assert (qq->maintainer);

	r = strcoll (pp->maintainer, qq->maintainer);
	if (r)
		return r;

	return strcoll (pp->name, qq->name);
}


/*+
Index the packages by maintainer.
+*/
static int maintainers_init (void)
{
	int i, cnt;
	FILE *fp;
	struct package_info *p, *q;

#	define MAX_VALUES 200
	int histogram[MAX_VALUES];

	memset (histogram, 0, sizeof histogram);

	qsort (cache, cachecnt, sizeof (cache[0]), cmp_maintainer);

	fp = fopen (OUTPUT_FILE, "w");
	mp_doc_open (fp, "All Packages, Grouped by Maintainer");

	mp_abstract (fp, "\
This page lists all packages, grouped by maintainer.  It can be useful\n\
since maintainers often work on related tasks - if you find one package\n\
useful, you may find other packages maintained by the same person equally\n\
useful.");

	p = cache[0];
	mp_list_open (fp);
	for (i = 0; i < cachecnt; ) {

		/* count the number of packages with this maintainer. */
		for (cnt = 0; i + cnt < cachecnt; cnt++) {
			q = cache[i+cnt];
			if (strcoll (p->maintainer, q->maintainer) != 0)
				break;
		}
		if (cnt < MAX_VALUES)
			histogram[cnt]++;

		mp_name (fp, p->maintainer);
		mp_item_open (fp);
		mp_text (fp, p->maintainer);
		fprintf (fp, " (%d)", cnt);
		mp_item_close (fp);

		mp_list_open (fp);
		while (cnt--) {
			mp_item_open (fp);
			mp_url (fp, "details/%1$1.1s/%1$s.html.gz", "%1$s", p->name);
			mp_nbsp (fp);
			mp_text (fp, p->summary);
			mp_item_close (fp);
			p = cache[++i];
		}
		mp_list_close (fp);
		fprintf (fp, "\n");
	}
	mp_list_close (fp);

	mp_break (fp);
	mp_title (fp, 2, "Histogram");

	fprintf (fp, "\
This histogram shows the number of maintainers with the indicated\n\
number of packages.");

	mp_histogram (fp, histogram, MAX_VALUES);

	mp_doc_close (fp);
	fclose (fp);

	gzip (OUTPUT_FILE);

	return 0;
}


/*+
Index the packages by package maintainer.
+*/
static int maintainers_index (FILE *fp, int type)
{
	if (type != MAGPIE_ALL_PACKAGES)
		return 0;

	mp_item_open (fp);
	mp_url (fp, "%s.gz", "Grouped by package maintainer", 
		OUTPUT_FILE);
	mp_item_close (fp);

	return 0;
}


struct magpie_module mod_maintainers = { 
	version           : MAGPIE_VERSION,
	description       : "group packages by package name",
	init              : maintainers_init,
	unannotated_index : maintainers_index
};
