/* This file is part of the KDE project

   Copyright (C) 2006-2007 KovoKs <info@kovoks.nl>
   Copyright (C) 2007 Frode M. Døving <frode@lnix.net>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

#ifndef MESSAGEVIEW_H
#define MESSAGEVIEW_H

#include <qwidget.h>
#include <qscrollview.h>
#include <kactivelabel.h>
#include <khtml_part.h>

class QVBox;

namespace Mailody {

class MessageData;
class HeaderView;
class MessageView;
class AddressLabel;

/** @class TotalView
 * This class is responsible for displaying messages. It consits of three
 * classes. There is a TotalView, which displays two widgets: the HeaderView
 * for all the headers and the MessageView which contains the body of the
 * message
 * @author Tom Albers <tomalbers@kde.nl>
 */
class TotalView : public QWidget
{
    public:
        /**
         * Constructor
         */
        explicit TotalView( QWidget* );

        /**
         * Destructor
         */
        ~TotalView();

        /**
         * returns a pointer to the header widget
         */
        HeaderView*  headers() const     { return m_headers; };

        /**
         * returns a pointer to the body widgget
         */
        MessageView* body() const        { return m_body; };

        /**
         * returns a pointer to the currently shown message, or 0.
         */
        MessageData* msg() const         { return m_msg; };

        /**
         * this call asks to show @p msg in the view.
         */
        void setMsg(MessageData* msg);

        /**
         * this will hide the headers and clear the body.
         */
        void clearView();

        /**
         * guess what...
         */
        void print();

        /**
         * Guess again...
         */
        void save( const QString& filename);

    private:
        MessageData*         m_msg;
        HeaderView*          m_headers;
        MessageView*         m_body;
};

/**
 * @class HeaderView
 * This class will show the headers.
 * @author Tom Albers <tomalbers@kde.nl>
 */
class HeaderView : public QScrollView
{
    Q_OBJECT

    public:

        /**
         * Constructor
         */
        explicit HeaderView( QWidget* );

        /**
         * Destructor
         */
        ~HeaderView();

        /**
         * this will fill the widget with the info from @p msg
         * you probably dont want to call this, you probably want
         * TotalView::setMsg() so it also updates the body
         */
        void setMsg( MessageData* msg );

        /**
         * remove the existing info.
         */
        void clearView();

    private slots:
        void addattachment(const QString& filename, const QString& file);
        void slotLeftMouseClick(const QString&);
        void slotRightMouseClick(const QString&);
        void updateHeaders();

    signals:
        void openComposer( const QString&, const QString&, const QString& );

    private:
        QMap<QString, QString>      aMap;
        MessageData*                m_currentMessage;
        QGridLayout*                m_layout;
        AddressLabel*               m_label;
        KActiveLabel*               m_foto;
};

/**
 * @class MessageView
 * This class is responsible for displaying the body of the message
 * to achieve this, it inherits a KHTMLPart
 */
class MessageView : public KHTMLPart
{
    Q_OBJECT

    public:
        /**
        * Constructor
        */
        explicit MessageView( QWidget* );

        /**
        * Destructor
        */
        ~MessageView();

        /**
         * Scrolls to the top of the message
         */
        void top()             { /*TODO: setContentsPos(0,0); */};

        /**
         * Returns if the message html will be interepreted or displayed as
         * plain text (escaped).
         */
        bool html() const      { return m_bodyType == MessageData::HTML; };

        /**
         * Returns if the source of the message will be showed
         */
        bool source() const    { return m_bodyType == MessageData::Source; };

        /**
         * Returns if the message will be shown in fixed font, has no use
         * in html mode of course
         */
        bool fixedfont() const { return m_fixedFont; };

        /**
         * Returns if the message will load external images in html-mode
         */
        bool externalImage() const { return m_externalImage; };

        /**
         * Sets the message to be shown, remember that you probably want to
         * use TotalView::setMsg() to also get updated headers
         */
        void setMsg(MessageData* msg);

        /**
         * set @p i to true to render the html in the message
         */
        void setHTML( bool i );

        /**
         * set @p i to true to load external images in the message
         * it will update the message immediatly, unless @p updateScreen
         * is set to false.
         */
        void setExternalImage( bool i, bool updateScreen=true );

        /**
         * set this so the links will open directly, normaly it would
         * open a link in a external browser or emit the signal on middle
         * click
         */
        void setOpenLinksDirectly( bool i );

        /**
         * set @p i to true to see the message in a fixed font
         */
        void setFixedFont( bool i );

        /**
         * Call this to see the source of the message
         */
        void setViewSource();

        /**
         * this will clear the body.
         */
        void clearView();

    protected:
        virtual void urlSelected (const QString&, int, int, const QString &,
                                  KParts::URLArgs args=KParts::URLArgs());

    private slots:
        void slotAddContent(const MessageData*, const QString& filename,
                            const QString& file);
        void slotShowMsg(const MessageData*);
        void slotPopupMenu(const QString&, const QPoint&);
        void slotSetCaption(const QString& title);

    signals:
        void openComposer( const QString&, const QString&, const QString& );
        void openInTab( const QString& );
        void attachment(const QString&, const QString&);
        void pageTitle(const QString&);
        void update();

    private:
        MessageData*             m_currentMessage;
        MessageData::bodyType    m_bodyType;
        bool                     m_fixedFont;
        bool                     m_externalImage;
        bool                     m_openDirectly;
};

}

#endif
