#   MailScanner - SMTP E-Mail Virus Scanner
#   Copyright (C) 2001  Julian Field
#
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#   The author, Julian Field, can be contacted by email at
#      Jules@JulianField.net
#   or by paper mail at
#      Julian Field
#      Dept of Electronics & Computer Science
#      University of Southampton
#      Southampton
#      SO17 1BJ
#      United Kingdom
#

# Read the configuration file and set globals accordingly.

use strict;
package Config;

my $prefix = '/opt/mailscanner';

# Default values
$Config::Debugging            = 0;
$Config::MaxCleanPerRun       = 200; # Max 200 clean messages per run
$Config::MaxDirtyPerRun       = 50; # Max 50 dirty messages per run
$Config::MaxCleanBytes        = 100000*1024; # Max 100MBytes of clean messages per run
$Config::MaxDirtyBytes        = 50000*1024;  # Max 50MBytes of dirty messages per run
$Config::Sweep                = "/opt/sophos/bin/sophoswrapper";
$Config::TNEF                 = "$prefix/bin/tnef";
$Config::PidFile              = "$prefix/var/virus.pid";
$Config::DefaultConfig        = "$prefix/etc/mailscanner.conf";
$Config::Sendmail             = "/usr/lib/sendmail";
$Config::Sendmail2            = "";
$Config::QuarantineAction     = 'store';
$Config::LocalPostmaster      = 'postmaster';
$Config::InQueueDir           = '/var/spool/mqueue.in';
$Config::OutQueueDir          = '/var/spool/mqueue';
$Config::QuarantineDir        = "$prefix/var/quarantine";
$Config::SrcDir               = "$prefix/var/incoming";
$Config::RestartEvery         = 4*60*60; # Restart every 4 hours
$Config::FilenameRules        = "$prefix/etc/filename.rules";
$Config::Hostname             = 'the MailScanner';
$Config::DisinfectedReportText= "$prefix/etc/disinfected.report.txt";
$Config::TellLocalPostie      = 1;
$Config::TellSenders          = 0;
$Config::MailHeader           = 'X-MailScanner:';
$Config::CleanHeader          = 'Found to be clean';
$Config::InfectedHeader       = 'Found to be infected';
$Config::DisinfectedHeader    = 'Disinfected';
$Config::SkipAlreadyScanned   = 0;
$Config::DeliverDisinfected   = 1;
$Config::CheckSpam            = 0;
$Config::SpamHeader           = 'X-MailScanner-SpamCheck:';
@Config::AcceptSpamFrom       = ();
@Config::SpamNames            = ();
@Config::SpamDomains          = ();
#$Config::DestDir             = "$prefix/var/outgoing";
$Config::RunAsUser            = '';
$Config::RunAsGroup           = '';
$Config::MTA                  = 'sendmail';
$Config::DeliveryMethod       = 'individual';
$Config::MarkInfectedMessages = 1;
$Config::InlineTextWarning    = '';
$Config::InlineHTMLWarning    = '';
$Config::VirusScanner         = 'sophos';
$Config::LockType             = '';
$Config::MultipleHeaders      = 'append';
$Config::SpamSubjectText      = '{SPAM?}';
$Config::SpamPrependSubject   = 1;
$Config::DeliverToRecipients  = 1;
$Config::VirusScanning        = 1;
$Config::TNEFTimeout          = 120;
$Config::SweepTimeout         = 300;
$Config::WhiteListFilename    = '';
%Config::SpamWhiteList        = ();
$Config::DeliverBadTNEF       = 0;
$Config::DeliverInBackground  = 0;
$Config::SenderReportText         = "$prefix/etc/sender.report.txt";
$Config::SenderVirusReportText    = '';
$Config::SenderFilenameReportText = '';
$Config::SenderErrorReportText    = '';
$Config::StoredMessageText          = "$prefix/etc/stored.message.txt";
$Config::DeletedMessageText         = "$prefix/etc/deleted.message.txt";
$Config::StoredVirusMessageText     = '';
$Config::DeletedVirusMessageText    = '';
$Config::StoredFilenameMessageText  = '';
$Config::DeletedFilenameMessageText = '';
$Config::AttachmentWarningFilename  = 'VirusWarning.txt';
$Config::ExpandTNEF                 = 1;
$Config::SpamAssassin               = 0;
$Config::MaxAssassinSize            = 100000;
$Config::SpamAssassinTimeout        = 10;
$Config::DeliverFromLocal           = 1;
$Config::LocalDomainList            = '';
$Config::PostmasterFullHeaders      = 0;
$Config::SignCleanMessages          = 0;
$Config::InlineTextSig              = '';
$Config::InlineHTMLSig              = '';
$Config::CodeStatus                 = 'supported';
$Config::LockDir                    = '/tmp';
$Config::ScanAllMessages            = 1;
$Config::ScanningByDomain           = 0;
%Config::DomainsToScan              = ();
$Config::ScanDomainsFilename        = "$prefix/etc/domains.to.scan.conf";
$Config::SignUnscannedMessages      = 1;
$Config::UnscannedHeader            = 'not scanned';
$Config::ArchiveMail                = 0;
$Config::ArchiveMailDir             = "/var/spool/MailArchive";

sub ReadConfig {
  my($Config) = @_;
  my($key, $value);
  local(*CONF);

  $Config = $Config::DefaultConfig unless defined $Config;

  open(CONF, $Config)
    or Log::DieLog("Could not open config file $Config, %s", $!);
  while(<CONF>) {
    chomp;
    s/#.*$//;
    s/^\s*//g;
    s/\s*$//g;
    next if /^$/;

# This was
#    ($key, $value) = split(/\s*=\s*/, $_, 2);
# The following should be equivalent but with untainting
# (we trust the admin to put sensible things in the config file).
# -- nwp, 14/01/02
    /^(.*?)\s*=\s*(.*)$/;
    ($key,$value) = ($1,$2);

    $key = lc($key);
    $key =~ s/[^a-z]//g; # Delete everything except letters
    $Config::Sendmail              = $value if $key =~ /^sendmail(?!2)/;
    $Config::Sendmail2             = $value if $key =~ /^sendmail2/;
    $Config::QuarantineDir         = $value if $key =~ /^quarantine/;
    $Config::SrcDir                = $value if $key =~ /^incomingwork/;
    $Config::PidFile               = $value if $key =~ /^pid/;
    $Config::OutQueueDir           = $value if $key =~ /^outgoingq/;
    $Config::InQueueDir            = $value if $key =~ /^incomingq/;
    $Config::MaxCleanBytes         = $value if $key =~ /^maxsafebytes/;
    $Config::MaxDirtyBytes         = $value if $key =~ /^maxunsafebytes/;
    $Config::MaxCleanPerRun        = $value if $key =~ /^maxsafemess/;
    $Config::MaxDirtyPerRun        = $value if $key =~ /^maxunsafemess/;
    $Config::Sweep                 = $value if $key =~ /^sweep/;
    $Config::TNEF                  = $value if $key =~ /^tnefexpander/;
    $Config::LocalPostmaster       = $value if $key =~ /^localpostmaster/;
    $Config::Debugging             = $value if $key =~ /^debug/;
    $Config::QuarantineAction      = $value if $key =~ /^action/;
    $Config::RestartEvery          = $value if $key =~ /^restartevery/i;
    $Config::FilenameRules         = $value if $key =~ /^filenamerules/i;
    $Config::Hostname              = $value if $key =~ /^hostname/i;
    $Config::StoredMessageText     = $value if $key =~ /^storedmessagereport/i;
    $Config::DeletedMessageText    = $value if $key =~ /^deletedmessagereport/i;
    $Config::SenderReportText      = $value if $key =~ /^senderreport/i;
    $Config::DisinfectedReportText = $value if $key =~ /^disinfectedreport/i;
    $Config::TellLocalPostie       = $value if $key =~ /^notifylocalpost/i;
    $Config::TellSenders           = $value if $key =~ /^notifysender/i;
    $Config::MailHeader            = $value if $key =~ /^mailheader/i;
    $Config::CleanHeader           = $value if $key =~ /^cleanheader/i;
    $Config::InfectedHeader        = $value if $key =~ /^infectedheader/i;
    $Config::DisinfectedHeader     = $value if $key =~ /^disinfectedheader/i;
    $Config::SkipAlreadyScanned    = $value if $key =~ /^skipifalreadyscanned/i;
    $Config::DeliverDisinfected    = $value if $key =~ /^deliverdisinfected/i;
    $Config::CheckSpam             = $value if $key =~ /^spamcheck/i;
    $Config::SpamHeader            = $value if $key =~ /^spamheader/i;
    $Config::RunAsUser             = $value if $key =~ /^runasuser/i;
    $Config::RunAsGroup            = $value if $key =~ /^runasgroup/i;
    $Config::MTA                   = lc($value) if $key =~ /^mta/i;
    $Config::DeliveryMethod        = $value if $key =~ /^deliverymethod/i;
    $Config::MarkInfectedMessages  = $value if $key =~ /^markinfectedmessages/i;
    $Config::InlineTextWarning     .= "$value\n" if $key =~ /^inlinetextwarning/i;
    $Config::InlineHTMLWarning     .= "$value\n" if $key =~ /^inlinehtmlwarning/i;
    $Config::VirusScanner          = lc($value) if $key =~ /^virusscanner$/i;
    $Config::LockType              = lc($value) if $key =~ /^locktype/i;
    $Config::MultipleHeaders       = lc($value) if $key =~ /^multiplehead/i;
    $Config::SpamSubjectText       = $value if $key =~ /^spamsubjecttext/i;
    $Config::SpamPrependSubject    = $value if $key =~ /^spammodifysubject/i;
    $Config::DeliverToRecipients   = $value if $key =~ /^delivertorecipients/i;
    $Config::VirusScanning         = $value if $key =~ /^virusscanning/i;
    $Config::TNEFTimeout           = $value if $key =~ /^tneftimeout/i;
    $Config::SweepTimeout          = $value if $key =~ /^virusscannertimeout/i;
    $Config::WhiteListFilename     = $value if $key =~ /^spamwhitelist/i;
    $Config::DeliverBadTNEF        = $value if $key =~ /^deliverunpars.*tnef/i;
    $Config::DeliverInBackground   = $value if $key =~ /^deliverinbackground/i;
    $Config::SenderVirusReportText    = $value if $key =~ /^sendervirusreport/i;
    $Config::SenderFilenameReportText = $value if $key =~ /^senderbadfilenamereport/i;
    $Config::SenderErrorReportText    = $value if $key =~ /^sendererrorreport/i;
    $Config::StoredVirusMessageText   = $value if $key =~ /^storedvirusmessagereport/i;
    $Config::DeletedVirusMessageText  = $value if $key =~ /^deletedvirusmessagereport/i;
    $Config::StoredFilenameMessageText  = $value if $key =~ /^storedbadfilenamemessagereport/i;
    $Config::DeletedFilenameMessageText = $value if $key =~ /^deletedbadfilenamemessagereport/i;
    $Config::AttachmentWarningFilename = $value if $key =~ /^attachmentwarningfilename/i;
    $Config::ExpandTNEF            = $value if $key =~ /^expandtnef/i;
    $Config::SpamAssassin          = $value if $key =~ /^usespamassassin/i;
    $Config::MaxAssassinSize       = $value if $key =~ /^maxspamassassinsize/i;
    $Config::SpamAssassinTimeout   = $value if $key =~ /^spamassassintimeout/i;
    $Config::DeliverFromLocal      = $value if $key =~ /^deliverfromlocaldomains/i;
    $Config::LocalDomainList       = $value if $key =~ /^localdomains/i;
    $Config::PostmasterFullHeaders = $value if $key =~ /^postmastergetsfullheaders/i;
    $Config::SignCleanMessages     = $value if $key =~ /^signcleanmessages/i;
    $Config::InlineTextSig        .= "$value\n" if $key =~ /^inlinetextsignature/i;
    $Config::InlineHTMLSig        .= "$value\n" if $key =~ /^inlineHTMLsignature/i;
    $Config::CodeStatus            = $value if $key =~ /^minimumcodestatus/i;
    $Config::LockDir               = $value if $key =~ /^lockfiledir/i;
    $Config::ScanAllMessages       = $value if $key =~ /^scanallmessages/i;
    $Config::ScanningByDomain      = $value if $key =~ /^scanningbydomain/i;
    $Config::ScanDomainsFilename   = $value if $key =~ /^domainstoscan/i;
    $Config::SignUnscannedMessages = $value if $key =~ /^signunscanned/i;
    $Config::UnscannedHeader       = $value if $key =~ /^unscannedheader/i;
    $Config::ArchiveMail           = $value if $key =~ /^archivemail$/i;
    $Config::ArchiveMailDir        = $value if $key =~ /^archivemaildir/i;

    # Allow for unset report text files due to old configuration files
    $Config::SenderVirusReportText    = $Config::SenderReportText
      unless $Config::SenderVirusReportText;
    $Config::SenderFilenameReportText = $Config::SenderReportText
      unless $Config::SenderFilenameReportText;
    $Config::SenderErrorReportText    = $Config::SenderReportText
      unless $Config::SenderErrorReportText;

    $Config::StoredVirusMessageText     = $Config::StoredMessageText
      unless $Config::StoredVirusMessageText;
    $Config::DeletedVirusMessageText    = $Config::DeletedMessageText
      unless $Config::DeletedVirusMessageText;
    $Config::StoredFilenameMessageText  = $Config::StoredMessageText
      unless $Config::StoredFilenameMessageText;
    $Config::DeletedFilenameMessageText = $Config::DeletedMessageText
      unless $Config::DeletedFilenameMessageText;

    # Build up list of network prefixes from where I should accept spam.
    # Put your local networks in here so you still get your own mail even
    # if you end up in the ORBS list, for example.
    push @Config::AcceptSpamFrom, $value if $key =~ /^acceptspam/i;
    # Build up the list of spam RBL lists (e.g. "ORBS, relays.orbs.org.")
    # out of multiple "spamlist" configuration lines
    if ($key =~ /^spamlist/i) {
      my($spamn, $spamd);
      ($spamn, $spamd) = split(/[, ]+/, $value);
      push @Config::SpamNames,   $spamn;
      push @Config::SpamDomains, $spamd;
    }
  }
  close CONF;

  # Fix Sendmail2 if unnecessary
  $Config::Sendmail2 or $Config::Sendmail2 = $Config::Sendmail;

  # Sanitise values of QuarantineAction
  $Config::QuarantineAction = 'store'
    if $Config::QuarantineAction =~ /quar|keep|stor|save/i;
  $Config::QuarantineAction = 'delete'
    if $Config::QuarantineAction =~ /dele|wipe/i;

  $Config::TellLocalPostie      = ZeroOrOne($Config::TellLocalPostie);
  $Config::TellSenders          = ZeroOrOne($Config::TellSenders);
  $Config::CheckSpam            = ZeroOrOne($Config::CheckSpam);
  $Config::SkipAlreadyScanned   = ZeroOrOne($Config::SkipAlreadyScanned);
  $Config::DeliverDisinfected   = ZeroOrOne($Config::DeliverDisinfected);
  $Config::DeliverDisinfected   = ZeroOrOne($Config::DeliverDisinfected);
  $Config::MarkInfectedMessages = ZeroOrOne($Config::MarkInfectedMessages);
  $Config::SpamPrependSubject   = ZeroOrOne($Config::SpamPrependSubject);
  $Config::DeliverToRecipients  = ZeroOrOne($Config::DeliverToRecipients);
  $Config::VirusScanning        = ZeroOrOne($Config::VirusScanning);
  $Config::DeliverBadTNEF       = ZeroOrOne($Config::DeliverBadTNEF);
  $Config::DeliverInBackground  = ZeroOrOne($Config::DeliverInBackground);
  $Config::ExpandTNEF           = ZeroOrOne($Config::ExpandTNEF);
  $Config::SpamAssassin         = ZeroOrOne($Config::SpamAssassin);
  $Config::DeliverFromLocal     = ZeroOrOne($Config::DeliverFromLocal);
  $Config::PostmasterFullHeaders= ZeroOrOne($Config::PostmasterFullHeaders);
  $Config::SignCleanMessages    = ZeroOrOne($Config::SignCleanMessages);
  $Config::ScanAllMessages      = ZeroOrOne($Config::ScanAllMessages);
  $Config::ScanningByDomain     = ZeroOrOne($Config::ScanningByDomain);
  $Config::SignUnscannedMessages= ZeroOrOne($Config::SignUnscannedMessages);
  $Config::ArchiveMail          = ZeroOrOne($Config::ArchiveMail);

  # Remove any trailing slashes from the directory names, they will cause
  # trouble later if left in.
  $Config::InQueueDir    =~ s/\/$//;
  $Config::OutQueueDir   =~ s/\/$//;
  $Config::QuarantineDir =~ s/\/$//;
  $Config::SrcDir        =~ s/\/$//;
  $Config::LockDir       =~ s/\/$//;
  $Config::ArchiveMailDir=~ s/\/$//;

  # Check to ensure all the directories and files we need actually exist
  DirExists($Config::QuarantineDir);
  DirExists($Config::SrcDir);
  DirExists($Config::InQueueDir);
  DirExists($Config::OutQueueDir);
  DirExists($Config::LockDir);
  DirExists($Config::ArchiveMailDir) if $Config::ArchiveMail;
  FileReadable($Config::FilenameRules);
  FileReadable($Config::DisinfectedReportText);
  FileReadable($Config::WhiteListFilename);
  #FileReadable($Config::SenderReportText);
  FileReadable($Config::SenderVirusReportText);
  FileReadable($Config::SenderFilenameReportText);
  FileReadable($Config::SenderErrorReportText);
  #FileReadable($Config::StoredMessageText);
  #FileReadable($Config::DeletedMessageText);
  FileReadable($Config::StoredVirusMessageText);
  FileReadable($Config::DeletedVirusMessageText);
  FileReadable($Config::StoredFilenameMessageText);
  FileReadable($Config::DeletedFilenameMessageText);
  FileReadable($Config::ScanDomainsFilename) if $Config::ScanningByDomain;

  # Now read the filename rules database
  ReadFilenameRules() if $Config::FilenameRules;

  # Now read the spam sender white list.
  # Everyone in this list can never send us spam.
  ReadWhiteList() if $Config::WhiteListFilename;

  # Now handle the list of local domains. This may either be an address or a
  # filename containing addresses.
  if ($Config::LocalDomainList =~ /^\//) {
    FileReadable($Config::LocalDomainList);
    ReadLocalDomains();
  } else {
    $Config::LocalDomains{"$Config::LocalDomainList"} = 1;
  }

  # Now read the list of domains to scan if we are using it.
  ReadDomainsToScan() if $Config::ScanningByDomain;
}

#
# Read the list of local domains from a file. Ignore everything that doesn't
# look like an email address or domain name.
# Take the first word of each line without leading or trailing spaces, support
# # and ; comment characters.
#
sub ReadLocalDomains {
  local(*DOMAINS);

  open(DOMAINS, "$Config::LocalDomainList") or die;
  while(<DOMAINS>) {
    chomp;
    s/^#.*$//; # Delete comments
    s/^\s*//g; # and leading and
    s/\s*$//g; # trailing spaces
    next if /^$/; # and blank lines
    s/\s.*$//g; # Just use the first word
    $Config::LocalDomains{lc($_)} = 1;
  }
  close(DOMAINS);
}
    
#
# Read the list of domains whose email we are scanning.
#
sub ReadDomainsToScan {
  local(*DOMAINS);

  open(DOMAINS, "$Config::ScanDomainsFilename") or die;
  while(<DOMAINS>) {
    chomp;
    s/^#.*$//; # Delete comments
    s/^\s*//g; # and leading and
    s/\s*$//g; # trailing spaces
    next if /^$/; # and blank lines
    s/\s.*$//g; # Just use the first word
    $Config::DomainsToScan{lc($_)} = 1;
  }
  close(DOMAINS);
}

#
# Check a configuration file exists and is readable.
# Produce an error if it is not.
#
sub FileReadable {
  my($name) = @_;
  return 1 unless $name;
  return 1 if -r $name;
  Log::DieLog("Configuration file $name could not be opened for reading!");
}

# Check a directory exists and is writeable.
# Produce an error if it is not.
#
sub DirExists {
  my($name) = @_;
  return 1 unless $name;
  return 1 if -d $name;
  Log::DieLog("Directory $name does not exist!");
}

# Check two files/directories are on the same partition.
# Produce an error if it is not.
# Can't do this here as it breaks "require" by changing directory.

#
# Set a Config variable to 0 or 1 depending on its current contents
#
sub ZeroOrOne {
  my($current) = @_;
  return 1 if $current =~ /yes|true|on|1/i;
  return 0;
}

#
# Read, parse and store the attachment filename rules file
#
sub ReadFilenameRules {
  my($allow, $regexp, $logtext, $usertext, $linenum);
  local(*RULES);

  open(RULES, "$Config::FilenameRules")
    or Log::DieLog("Could not open rules data file $Config::FilenameRules, %s", $!);
  $linenum = 0;
  while(<RULES>) {
    chomp;
    s/^#.*$//;
    s/^\s*//g;
    s/\s*$//g;
    $linenum++;
    next if /^$/;
    ($allow, $regexp, $logtext, $usertext) = split(/\t+/, $_, 4);
    unless ($allow && $regexp && $logtext && $usertext) {
      Log::WarnLog("Possible syntax error on line $linenum of $Config::FilenameRules");
      Log::WarnLog("Remember to separate fields with tab characters!");
    }
    $allow = ($allow =~ /allow/i)?'allow':'deny';
    $regexp =~ s/^\/(.*)\/$/\1/;
    $logtext = "" if $logtext eq '-';
    $usertext = "" if $usertext eq '-';

    push(@Config::NameAllow, $allow);
    push(@Config::NameRE, $regexp);
    push(@Config::NameLog, $logtext);
    push(@Config::NameUser, $usertext);
  }
  close RULES;
}

#
# Read, parse and store the spam senders white list file
#
sub ReadWhiteList {
  my($allow, $regexp, $logtext, $usertext);
  local(*WHITE);

  open(WHITE, "$Config::WhiteListFilename")
    or Log::DieLog("Could not open spam whitelist file $Config::WhiteListFilename, %s",
                   $!);
  while(<WHITE>) {
    chomp;
    s/^#.*$//;
    s/^\s*//g;
    s/\s*$//g;
    next if /^$/;
    $Config::SpamWhiteList{lc($_)} = 1;
  }
  close WHITE;
}

1;
