/* This file is part of Malaga, a system for Natural Language Analysis.
 * Copyright (C) 1995-1999 Bjoern Beutel
 *
 * Bjoern Beutel
 * Universitaet Erlangen-Nuernberg
 * Abteilung fuer Computerlinguistik
 * Bismarckstrasse 12
 * D-91054 Erlangen
 * e-mail: malaga@linguistik.uni-erlangen.de 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/* description ==============================================================*/

/* This file contains data structures and functions used for grammatical 
 * analysis. */

/* includes =================================================================*/

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include "basic.h"
#include "pools.h"
#include "values.h"
#include "rule_type.h"
#include "rules.h"
#include "lexicon.h"
#include "cache.h"

#undef GLOBAL
#define GLOBAL
#include "analysis.h"

/* types ====================================================================*/

typedef struct TREE_NODE_T /* a rule application is stored in "tree_node" */
{
  struct TREE_NODE_T *mother; /* predecessor of this tree node */
  struct TREE_NODE_T *first_daughter; /* first successor of this tree node */
  struct TREE_NODE_T *sister; /* alternative tree node */
  tree_node_type_t type; /* type of this tree node */
  int_t rule; /* number of the executed rule */
  int_t index; /* index of this tree node */
  value_t right_cat; /* category of segment being added */
  value_t result_cat; /* result cat of rule application */
  int_t rule_set; /* successor rules (-1 for end state) */
  string_t input; /* the input that is not yet analysed */
} tree_node_t;

typedef struct STATE_T /* a state in morphological or syntactical analysis */
{
  struct STATE_T *next; /* link to state with same or higher <input> */
  value_t cat; /* category of input read so far */
  string_t input; /* pointer to input that is analysed next */
  int_t rule_set; /* set of rules to be applied */
  tree_node_t *tree_node; /* tree node of rule application that created
                           * this state (NULL if no tree) */
  int_t item_index; /* number of items read in so far */
} state_t;

typedef struct /* the structure for morphological and syntactical analysis */ 
{ 
  pool_t state_pool; /* all states are saved in <state_pool> */
  pool_t value_pool; /* all categories are saved in <value_pool> */

  /* states are chained by their attribute <next>. There are three chains: */
  state_t *running_states; /* states that need further analysis
                            * (in the order of their <input> indexes) */
  state_t *end_states; /* end states */
  state_t *free_states; /* states that can be reused */
} analysis_t;

/* variables ================================================================*/

/* structures used for LAG analysis, one for morphology and one for syntax. */
LOCAL analysis_t *analyses[2];

/* the data structure used to save the analysis tree */
LOCAL tree_node_t *root_tree_node; /* a pointer to the root tree node */
LOCAL pool_t tree_pool; /* pool where tree nodes are stored */
LOCAL int_t number_of_tree_nodes;

LOCAL state_t *next_result_state; /* needed for "get_next_analysis_result" */
LOCAL tree_node_t *next_tree_node; /* needed for "get_next_analysis_node" */

LOCAL string_t left_start, right_start, right_end;
/* start and end position of surface when rule is executed. READ ONLY! */

/* information needed to generate states and tree nodes */
LOCAL struct {
  analysis_t *analysis;
  bool_t build_tree; /* entering tree nodes? */
  bool_t analyse_all; /* analyse the whole input? */
  int_t rule; /* rule just executed */
  value_t right_cat; /* right category */
  tree_node_t *mother; /* predecessor tree node */
  int_t item_index; /* index of item that is added */
  string_t input; /* end of analysed input */
} state_info;

LOCAL bool_t options[NUM_ANALYSIS_OPTIONS];

/* functions for analysis options ===========================================*/

GLOBAL void set_analysis_option (analysis_option_t selected, bool_t setting)
/* Set analysis option <selected> to <setting>. */
{
  switch (selected)
  {
  case ROBUST_OPTION:
    if (rule_system[MORPHOLOGY]->robust_rule == -1)
      error ("no morphology \"robust_rule\"");
    break;
  case PRUNING_OPTION:
    if (rule_system[SYNTAX] == NULL 
	|| rule_system[SYNTAX]->pruning_rule == -1)
      error ("no morphology \"pruning_rule\"");
    break;
  case MOR_OUT_FILTER_OPTION:
    if (rule_system[MORPHOLOGY]->output_filter == -1)
      error ("no morphology \"output_filter\"");
    break;
  case SYN_OUT_FILTER_OPTION:
    if (rule_system[SYNTAX] == NULL
	|| rule_system[SYNTAX]->output_filter == -1)
      error ("no syntax \"output_filter\"");
    break;
  case SYN_IN_FILTER_OPTION:
    if (rule_system[SYNTAX] == NULL
	|| rule_system[SYNTAX]->input_filter == -1)
      error ("no syntax \"input_filter\"");
    break;
  case CACHE_OPTION:
    break;
  default:
    error ("internal (unknown option)");
  }
  options[selected] = setting;
}

/*---------------------------------------------------------------------------*/

GLOBAL bool_t get_analysis_option (analysis_option_t selected)
/* return the current setting of analysis option <selected>. */
{
  return options[selected];
}

/* functions for segmentation and preprocessing =============================*/

LOCAL bool_t is_word_part (string_t string)
/* Return whether the character *<string> may be part of a word. */
{
  /* *<string> is part of a word if it is a letter, a digit
   * or one of {".", ",", "-"} followed by a letter or a digit. */
  return (IS_ALPHA (*string) || isdigit (*string)
	  || ((*string == ',' || *string == '.' || *string == '-')
              && (IS_ALPHA (string[1]) || isdigit (string[1]))));
}

/*---------------------------------------------------------------------------*/

GLOBAL void preprocess_input (string_t input)
/* Delete heading and trailing spaces in <input>
 * and compress all whitespace sequences to a single space. */
{
  string_t input_ptr, output_ptr;

  output_ptr = input;

  /* Cut heading spaces. */
  input_ptr = next_non_space (input);
  while (*input_ptr != EOS)
  {
    if (isspace (*input_ptr))
    {
      /* Overread all whitespace and write a single space. */
      input_ptr = next_non_space (input_ptr);
      *output_ptr++ = ' ';
    }
    else
      *output_ptr++ = *input_ptr++;
  }

  /* Cut trailing spaces. */
  while (output_ptr > input && isspace (output_ptr[-1]))
    output_ptr--;
  *output_ptr = EOS;
}

/* functions for state list processing ======================================*/

LOCAL void remove_state (analysis_t *analysis, state_t **state_list_ptr)
/* Remove state <state_list_ptr> points to and enter it into the free list. */
{
  state_t *state = *state_list_ptr;

  if (state != NULL)
  {
    /* Unlink from old list. */
    *state_list_ptr = state->next;

    /* Enter in free list. */
    state->next = analysis->free_states;
    analysis->free_states = state;
  }
}

/*---------------------------------------------------------------------------*/

LOCAL state_t *insert_state (analysis_t *analysis,
                             state_t **state_list_ptr,
                             value_t cat,
                             string_t input,
                             int_t rule_set,
                             int_t item_index)
/* Insert a state, composed of <cat>, <input>, <rule_set>, and <item_index>
 * in the list *<state_list_ptr> points to, in front of all states with a
 * higher <input> index. Return this state. */
{
  state_t *new_state;

  if (analysis->free_states != NULL)
  {
    /* Get first state in the free list. */
    new_state = analysis->free_states;
    analysis->free_states = new_state->next;
  }
  else
    new_state = (state_t *) get_pool_space (analysis->state_pool, 1, NULL);

  /* Set values. */
  new_state->cat = cat;
  new_state->input = input;
  new_state->rule_set = rule_set;
  new_state->item_index = item_index;
  new_state->tree_node = NULL;

  /* Insert new state in list. */
  while (*state_list_ptr != NULL && (*state_list_ptr)->input <= input)
    state_list_ptr = &(*state_list_ptr)->next;
  new_state->next = *state_list_ptr;
  *state_list_ptr = new_state;
  
  return new_state;
}

/*---------------------------------------------------------------------------*/

LOCAL tree_node_t *add_tree_node (value_t result_cat, 
                                  string_t input, 
                                  int_t rule_set,
				  tree_node_type_t type)
/* Add a tree node for a rule that fired with <result_cat> and <rule_set>,
 * where <input> is yet to be analysed. */
{
  tree_node_t **tree_node_ptr;
  tree_node_t *tree_node;
  
  /* Get a new tree node. */
  tree_node = (tree_node_t *) get_pool_space (tree_pool, 1, NULL); 
  
  tree_node->mother = state_info.mother;
  tree_node->first_daughter = NULL;
  tree_node->sister = NULL;
  tree_node->type = type;
  tree_node->rule = state_info.rule;
  tree_node->index = number_of_tree_nodes;
  tree_node->right_cat = state_info.right_cat;
  tree_node->result_cat = result_cat;
  tree_node->rule_set = rule_set;
  tree_node->input = input;
  
  /* Link the tree node into the tree structure. */
  tree_node_ptr = &state_info.mother->first_daughter;
  while (*tree_node_ptr != NULL)
    tree_node_ptr = &(*tree_node_ptr)->sister;
  *tree_node_ptr = tree_node;
  
  /* Increment the number of tree nodes. */
  number_of_tree_nodes++;

  return tree_node;
}

/*---------------------------------------------------------------------------*/

LOCAL void add_state (state_t **list, 
                      value_t cat, 
                      int_t rule_set,
		      tree_node_type_t type)
/* Add state, consisting of <cat> and <rule_set> in list **<list>.
 * When <state_info.build_tree> == TRUE, also generate a tree node. */
{ 
  value_t new_value;
  state_t *state;
  
  /* Preserve the category. */
  new_value = copy_value_to_pool (state_info.analysis->value_pool, cat, NULL);

  state = insert_state (state_info.analysis, list, new_value, state_info.input,
                        rule_set, state_info.item_index);
  if (state_info.build_tree)
    state->tree_node = add_tree_node (new_value, state_info.input, rule_set,
				      type);
}

/* callback functions needed by rules =======================================*/

LOCAL void local_add_end_state (value_t cat)
/* Add a state, consisting of <cat>, as an end state. */
{ 
  string_t input = state_info.input;

  /* Only add an end state if at the end of input
   * or at the end of a word in a subordinate morphological analysis. */
  if (*input == EOS || ! (state_info.analyse_all 
                          || (is_word_part (input-1) && is_word_part (input))))
    add_state (&state_info.analysis->end_states, cat, -1, FINAL_NODE);
}

/*---------------------------------------------------------------------------*/

LOCAL void local_add_running_state (value_t cat, int_t rule_set)
/* Add a running state, consisting of <cat> and <rule_set>. */
{ 
  add_state (&state_info.analysis->running_states, cat, rule_set, INTER_NODE);
}

/*---------------------------------------------------------------------------*/

LOCAL string_t local_get_surface (surface_t surface_type)
/* Return surface <surface_type> for currently executed rule.
 * The result must be freed after use. */
{
  string_t left_end;

  if (right_start > left_start && right_start[-1] == ' ')
    left_end = right_start - 1;
  else
    left_end = right_start;
  
  switch (surface_type)
  {
  case LEFT_SURFACE:
    return new_string_readable (left_start, left_end);
  case RIGHT_SURFACE:
    return new_string_readable (right_start, right_end);
  case RESULT_SURFACE:
    return new_string_readable (left_start, right_end);
  default:
    error ("internal (unknown surface type)");
  }
}

/* analysis functions =======================================================*/

LOCAL analysis_t *new_analysis (void)
/* Create a new analysis structure. */
{
  analysis_t *analysis = new_mem (sizeof (analysis_t));

  analysis->state_pool = new_pool (sizeof (state_t));
  analysis->value_pool = new_pool (sizeof (cell_t));
  analysis->running_states = NULL;
  analysis->end_states = NULL;
  analysis->free_states = NULL;

  return analysis;
}

/*---------------------------------------------------------------------------*/

LOCAL void free_analysis (analysis_t **analysis)
/* Destroy an analysis structure. */
{
  if (*analysis != NULL)
  {
    free_pool (&(*analysis)->state_pool);
    free_pool (&(*analysis)->value_pool);
    free_mem (analysis);
  }
}

/*---------------------------------------------------------------------------*/

GLOBAL void init_analysis (string_t morphology_file, string_t syntax_file)
/* Initialise the analysis module.
 * <morphology_file> and <syntax_file> are the rule files to load. 
 * <syntax_file> may be NULL. */
{
  int_t i;

  /* Read rule files. */
  rule_system[MORPHOLOGY] = read_rule_sys (morphology_file);
  if (syntax_file != NULL)
    rule_system[SYNTAX] = read_rule_sys (syntax_file);

  /* Init analysis structure. */
  analyses[MORPHOLOGY] = new_analysis ();
  analyses[SYNTAX] = new_analysis ();
  tree_pool = new_pool (sizeof (tree_node_t));

  /* Set analysis options to start values. */
  for (i = 0; i < NUM_ANALYSIS_OPTIONS; i++)
    options[i] = FALSE;
  options[MOR_OUT_FILTER_OPTION] = 
    (rule_system[MORPHOLOGY]->output_filter != -1);
  options[SYN_IN_FILTER_OPTION] = 
    (rule_system[SYNTAX] != NULL && rule_system[SYNTAX]->input_filter != -1);
  options[SYN_OUT_FILTER_OPTION] = 
    (rule_system[SYNTAX] != NULL && rule_system[SYNTAX]->output_filter != -1);
}

/*---------------------------------------------------------------------------*/

GLOBAL void terminate_analysis (void)
/* Terminate the analysis module. */
{
  free_rule_sys (&rule_system[SYNTAX]);
  free_rule_sys (&rule_system[MORPHOLOGY]);
  free_analysis (&analyses[MORPHOLOGY]);
  free_analysis (&analyses[SYNTAX]);
  free_pool (&tree_pool);
  clear_cache ();
  free_switches ();
}

/*---------------------------------------------------------------------------*/

GLOBAL bool_t reset_analysis_results (void)
/* Restart to read analysis results. 
 * Return TRUE iff there are any analysis results. */
{
  next_result_state = analyses[top_grammar]->end_states;
  return next_result_state != NULL;
}

/*---------------------------------------------------------------------------*/

GLOBAL value_t get_next_analysis_result (void)
/* Return the category of the next analysis result. */
{
  if (next_result_state != NULL)
  {
    value_t result = next_result_state->cat;

    next_result_state = next_result_state->next;
    return result;
  }
  else
    return NULL;
}

/*---------------------------------------------------------------------------*/

GLOBAL bool_t reset_analysis_nodes (void)
/* Restart to read analysis nodes. 
 * Return TRUE iff there are any analysis nodes. */
{
  next_tree_node = root_tree_node;
  return next_tree_node != NULL;
}

/*---------------------------------------------------------------------------*/

GLOBAL void free_analysis_node (analysis_node_t **node)
/* Free the memory occupied by <node>. */
{
  if (*node != NULL)
  {
    free_mem (&(*node)->right_surf);
    free_mem (&(*node)->result_surf);
    free_mem (&(*node)->rule_set);
    free_mem (node);
  }
}

/*---------------------------------------------------------------------------*/

GLOBAL analysis_node_t *get_next_analysis_node (void)
/* Return the next analysis tree node of the last call of "analyse_item".
 * Return NULL if there is no more node. 
 * The node must be freed with "free_analysis_node" after use. */
{
  analysis_node_t *node;
  string_t right_start;
  rule_sys_t *rule_sys = rule_system[top_grammar];

  if (next_tree_node == NULL)
    return NULL;

  node = new_mem (sizeof (analysis_node_t));

  /* Set node index. */
  node->index = next_tree_node->index;
  
  /* Set node type. */
  node->type = next_tree_node->type;
  
  /* Set mother index. */
  if (next_tree_node->mother == NULL)
    node->mother_index = -1;
  else
    node->mother_index = next_tree_node->mother->index;

  /* Set rule name. */
  if (next_tree_node->rule != -1)
    node->rule_name = (rule_sys->strings 
		      + rule_sys->rules[next_tree_node->rule].name);
  else
    node->rule_name = NULL;

  /* Set right surface and category. */
  if (next_tree_node->mother == NULL) /* no predecessor */
    right_start = last_analysis_input;
  else
    right_start = next_non_space (next_tree_node->mother->input);
  if (right_start != next_tree_node->input)
    node->right_surf = new_string (right_start, next_tree_node->input);
  node->right_cat = next_tree_node->right_cat;

  /* Set result surface. */
  node->result_surf = new_string (last_analysis_input, next_tree_node->input);
  node->result_cat = next_tree_node->result_cat;

  /* Set rule set. */
  if (next_tree_node->result_cat != NULL)
    node->rule_set = rule_set_readable (rule_sys, next_tree_node->rule_set);
  
  /* Update <next_tree_node>. */
  if (next_tree_node->first_daughter != NULL)
    next_tree_node = next_tree_node->first_daughter;
  else if (next_tree_node->sister != NULL)
    next_tree_node = next_tree_node->sister;
  else /* Go back to the next node not yet visited. */
  {
    while (TRUE)
    {
      next_tree_node = next_tree_node->mother;
      if (next_tree_node == NULL)
	break;
      
      if (next_tree_node->sister != NULL)
      {
	next_tree_node = next_tree_node->sister;
	break;
      }
    }
  }

  return node;
}

/*---------------------------------------------------------------------------*/

LOCAL string_t get_word_end (string_t input, bool_t analyse_all)
/* Return the end of the word that starts at <input>. */
{
  string_t input_end;

  if (analyse_all)
    input_end = input + strlen (input);
  else if (! is_word_part (input))
    input_end = input + 1;
  else
  {
    input_end = input + 1;
    while (is_word_part (input_end))
      input_end++;
  }

  return input_end;
}

/*---------------------------------------------------------------------------*/

LOCAL bool_t get_from_cache (analysis_t *analysis, string_t input, 
			     bool_t analyse_all)
/* If next word at <input> is in analysis cache, 
 * enter its results in <analysis>, and return TRUE. Otherwise return FALSE. */
{
  string_t input_end = get_word_end (input, analyse_all);
  
  if (word_in_cache (input, input_end))
  {
    value_t result;
      
    while (TRUE)
    {
      result = next_result_in_cache ();
      if (result == NULL)
	break;
	  
      insert_state (analysis, &analysis->end_states, result, input_end, -1, 0);
    }
    return TRUE;
  }
  else
    return FALSE;
}

/*---------------------------------------------------------------------------*/

LOCAL void put_into_cache (analysis_t *analysis, string_t input,
			   bool_t analyse_all)
/* Store the results in <analysis> for the word form that starts at <input>
 * in the cache. */
{
  value_t *cats;
  int_t num_cats, i;
  state_t *state;
  string_t input_end = get_word_end (input, analyse_all);

  /* Count categories. */
  num_cats = 0;
  for (state = analysis->end_states; state != NULL; state = state->next)
  {
    if (state->input != input_end) 
      /* Only put into cache if all entries will be found in cache. */
      return;
    num_cats++;
  }
  
  /* Allocate a new vector which takes the categories. */
  cats = new_vector (sizeof (value_t), num_cats);
  for (i = 0, state = analysis->end_states; 
       state != NULL && state->input == input_end; 
       i++, state = state->next)
    cats[i] = new_value (state->cat);
  
  enter_in_cache (input, input_end, num_cats, cats);
}

/*---------------------------------------------------------------------------*/

LOCAL void execute_robust_rule (analysis_t *analysis,
				rule_sys_t *rule_sys,
				string_t input,
				bool_t analyse_all)
/* Execute robust_rule in <rule_sys> for the next word in <analysis>.
 * Word starts at <input>.
 * Word must extend until end of string iff <analyse_all> == TRUE. */
{
  string_t input_end;
  
  input_end = get_word_end (input, analyse_all);

  /* Set debugging information. */
  left_start = input;
  right_start = input;
  right_end = input_end;

  /* Setup <state_info>. */
  state_info.analysis = analysis;
  state_info.build_tree = FALSE;
  state_info.analyse_all = analyse_all;
  state_info.item_index = 1;
  state_info.input = input_end;

  top = 0;
  push_string_value (input, input_end);
  execute_rule (rule_sys, rule_sys->robust_rule);
  if (analysis->end_states != NULL && analyse_all)
    recognised_by_robust_rule = TRUE;
}

/*---------------------------------------------------------------------------*/

LOCAL void execute_filter_rule (analysis_t *analysis, 
				rule_sys_t *rule_sys,
				int_t filter_rule,
				bool_t analyse_all)
/* Execute <filter_rule> in <rule_sys> for <analysis>. */
{
  state_t *old_end_states = analysis->end_states;

  /* Go through all results with the same length. */
  old_end_states = analysis->end_states;
  analysis->end_states = NULL;
  while (old_end_states != NULL) 
  {
    int_t results, i;
    state_t *state;
    string_t input = old_end_states->input;

    /* Count number of results. */
    results = 0;
    for (state = old_end_states; 
	 state != NULL && state->input == input;
	 state = state->next)
      results++;

    /* Create a list with the results of all states and remove states. */
    top = 0;
    for (i = 0; i < results; i++)
    {
      push_value (old_end_states->cat);
      remove_state (analysis, &old_end_states);
    }
    build_list (results);

    /* Set debugging information. */
    right_start = right_end = input;

    state_info.analysis = analysis;
    state_info.build_tree = FALSE;
    state_info.analyse_all = analyse_all;
    state_info.item_index = 0;
    state_info.input = input;
    execute_rule (rule_sys, filter_rule);
  }

}

/*---------------------------------------------------------------------------*/

LOCAL void execute_pruning_rule (analysis_t *analysis, grammar_t grammar)
/* Execute pruning_rule in <grammar> for the running states in <analysis>. */
{
  int_t results, i;
  state_t *state;
  state_t **state_ptr;
  string_t input = analysis->running_states->input;
  rule_sys_t *rule_sys = rule_system[grammar];
  value_t list;
  symbol_t symbol;

  /* Create a list that contains the results. */
  top = 0;
  results = 0;
  for (state = analysis->running_states; 
       state != NULL && state->input == input;
       state = state->next)
  {
    results++;
    push_value (state->cat);
  }
  build_list (results);

  /* Set debugging information. */
  right_start = right_end = input;
  
  execute_rule (rule_sys, rule_sys->pruning_rule);

  list = value_stack[top-1];
  if (get_value_type (list) != LIST_SYMBOL)
    error ("pruning rule result must be a list");
  if (get_list_length (list) != results)
    error ("pruning rule result must have as much elements as rule argument");

  state_ptr = &analysis->running_states;
  for (i = 0; i < results; i++)
  {
    symbol = value_to_symbol (get_element (list, i + 1));
    if (symbol == NO_SYMBOL)
    {
      if ((*state_ptr)->tree_node != NULL)
	(*state_ptr)->tree_node->type = PRUNED_NODE;
      remove_state (analysis, state_ptr);
    }
    else if (symbol == YES_SYMBOL)
      state_ptr = &(*state_ptr)->next;
    else
      error ("pruning rule result list may only contain yes/no symbols");
  }
}

/*---------------------------------------------------------------------------*/

LOCAL void execute_rules (analysis_t *analysis, 
                          rule_sys_t *rule_sys, 
                          state_t *state, 
                          value_t right_cat,
                          string_t right_surf_start, 
                          string_t right_surf_end, 
                          bool_t build_tree, 
			  rule_type_t rule_type,
                          bool_t analyse_all)
/* Execute the successor rules in <rule_sys> for <state> in <analysis>.
 * Consume the segment from <right_surf_start> to <right_surf_end>
 * with category <right_cat>. Enter a tree node if <build_tree> == TRUE. */
{
  int_t *rule_ptr;
  bool_t rules_successful;

  /* Setup <state_info>. */
  state_info.analysis = analysis;
  state_info.build_tree = build_tree;
  state_info.analyse_all = analyse_all;
  state_info.right_cat = right_cat;
  state_info.mother = state->tree_node;
  state_info.item_index = state->item_index + 1;
  state_info.input = right_surf_end;

  /* Set debugging information. */
  right_start = right_surf_start;
  right_end = right_surf_end;

  /* Check if we are now executing the rules for a state to be debugged */
  if (debug_state != NULL && state->tree_node != NULL)
    debug_state (state->tree_node->index);

  rules_successful = FALSE;
  for (rule_ptr = rule_sys->rule_sets + state->rule_set; 
       *rule_ptr != -1; 
       rule_ptr++) 
  {
    if (*rule_ptr == -2)
    {
      if (rule_type == END_RULE || rules_successful)
	break;
    }
    else if (rule_sys->rules[*rule_ptr].type == rule_type)
    {
      state_info.rule = *rule_ptr;
      top = 0;
      push_value (state->cat);
      push_value (right_cat);
      push_string_value (right_surf_start, right_surf_end);
      push_number_value (state_info.item_index);
      execute_rule (rule_sys, *rule_ptr);
      rules_successful |= rule_successful;
    }
  }

  if (build_tree && rule_type != END_RULE) 
  {
    /* Enter a tree node for a rule set that did not fire. */
    state_info.rule = -1;
    add_tree_node (NULL, right_surf_end, -1, BREAK_NODE);
  }
}

/*---------------------------------------------------------------------------*/

GLOBAL void analyse (grammar_t grammar, 
                     string_t input, 
                     bool_t build_tree,
                     bool_t analyse_all)
/* Perform a LAG analysis of <input> using <grammar> (MORPHOLOGY or SYNTAX).
 * An analysis tree will be built if <build_tree> == TRUE.
 * The whole input will be analysed if <analyse_all> == TRUE. */
{
  rule_sys_t *rule_sys;
  state_t *initial_state;
  analysis_t *analysis;

  if (analyse_all)
  {
    top_grammar = grammar;
    root_tree_node = NULL;
    last_analysis_input = input;
    recognised_by_robust_rule = recognised_by_combi_rules = FALSE;
  }

  analysis = analyses[grammar];
  rule_sys = rule_system[grammar];
  if (rule_sys == NULL)
    error ("missing rule system");
  
  /* Set callback functions for <execute_rules>. */
  add_running_state = local_add_running_state;
  add_end_state = local_add_end_state;

  /* Reset the analysis, we start anew. */
  analysis->running_states = NULL;
  analysis->end_states = NULL;
  analysis->free_states = NULL;
  clear_pool (analysis->state_pool);
  clear_pool (analysis->value_pool);

  /* Set debug information. */
  get_surface = local_get_surface;
  left_start = input;

  if (grammar == MORPHOLOGY && options[CACHE_OPTION] && ! build_tree)
  {
    if (get_from_cache (analysis, input, analyse_all))
      return;
  }

  /* Enter the initial state. */
  DB_ASSERT (rule_sys->initial_cat < rule_sys->values_size);
  initial_state = insert_state (analysis, &analysis->running_states,
                                rule_sys->values 
                                + rule_sys->initial_cat,
                                input, rule_sys->initial_rule_set, 0);

  if (build_tree)
  {
    /* Clear all tree nodes and setup <root_tree_node>. */
    clear_pool (tree_pool);
    root_tree_node = (tree_node_t *) get_pool_space (tree_pool, 1, NULL);
    root_tree_node->mother = NULL;
    root_tree_node->first_daughter = NULL;
    root_tree_node->sister = NULL;
    root_tree_node->type = INTER_NODE;
    root_tree_node->rule = -1;
    root_tree_node->index = 0;
    root_tree_node->right_cat = NULL;
    root_tree_node->result_cat = rule_sys->values + rule_sys->initial_cat;
    root_tree_node->rule_set = rule_sys->initial_rule_set;
    root_tree_node->input = input;
    initial_state->tree_node = root_tree_node;
    number_of_tree_nodes = 1;
  }

  /* Analyse while there are running states. */
  while (analysis->running_states != NULL) 
  {
    state_t *state;
    string_t current_input = analysis->running_states->input;

    if (options[PRUNING_OPTION] && current_input > input 
	&& rule_sys->pruning_rule != -1)
      execute_pruning_rule (analysis, grammar);

    /* Apply end_rules only if all input has been parsed
     * or if in subordinate analysis. */
    if (current_input > input 
	&& (*current_input == EOS 
	    || ! (analyse_all || (is_word_part (current_input-1) 
 				  && is_word_part (current_input)))))
    {
      /* Apply all end_rules to states at <current_input>. */
      for (state = analysis->running_states; 
 	   state != NULL && state->input == current_input; 
 	   state = state->next)
 	execute_rules (analysis, rule_sys, state, NULL, current_input, 
 		       current_input, build_tree, END_RULE, analyse_all);
    }
    
    /* If analysis has consumed all input, leave. */
    if (*current_input == EOS)
      break;

    if (grammar == MORPHOLOGY) 
    {
      string_t right_surf_end; /* end of surface of the next allomorph */
      value_t cat;
      
      /* Look for prefixes of increasing length
       * that match the string at <current_input>. */
      search_for_prefix (current_input);
      while (get_next_prefix (&right_surf_end, &cat))
      {
	/* Apply this next-variable to all morphological states. */
	for (state = analysis->running_states; 
	     state != NULL && state->input == current_input; 
	     state = state->next) 
	  execute_rules (analysis, rule_sys, state, cat, current_input,
			 right_surf_end, build_tree, COMBI_RULE, analyse_all);
      }
    } 
    else /* <grammar> == SYNTAX */ 
    { 
      state_t *morph_result;
      string_t input_behind_space = next_non_space (current_input);

      /* Call morphological analysis to get right-categories. */
      analyse (MORPHOLOGY, input_behind_space, FALSE, FALSE);

      /* Execution of morphology rules has changed <left_start>. */
      left_start = input;

      /* Step through all morphological results. */
      for (morph_result = analyses[MORPHOLOGY]->end_states;
	   morph_result != NULL;
	   morph_result = morph_result->next) 
      {
	/* The morphology pool may be cleared,
	 * so copy <cat> to the syntax pool. */ 
	value_t right_cat = copy_value_to_pool (analysis->value_pool, 
						morph_result->cat, NULL);
              
	/* Apply this right category to all syntactic states. */
	for (state = analysis->running_states; 
	     state != NULL && state->input == current_input; 
	     state = state->next)
	  execute_rules (analysis, rule_sys, state, right_cat,
			 input_behind_space, morph_result->input, 
			 build_tree, COMBI_RULE, TRUE);
      }
    }

  /* We have combined all analyses at <current_input> with all states
   * that were at <current_input>, so we can kill these states. */
  while (analysis->running_states != NULL 
	 && analysis->running_states->input == current_input)
    remove_state (analysis, &analysis->running_states);
  } /* end of loop that consumes all running states */

  if (analysis->end_states != NULL)
    recognised_by_combi_rules = TRUE;

  if (grammar == MORPHOLOGY)
  {
    if (analysis->end_states == NULL && options[ROBUST_OPTION])
      execute_robust_rule (analysis, rule_sys, input, analyse_all);
     
    if (options[MOR_OUT_FILTER_OPTION])
      execute_filter_rule (analysis, rule_system[MORPHOLOGY],
			   rule_system[MORPHOLOGY]->output_filter, 
			   analyse_all);

    if (options[SYN_IN_FILTER_OPTION])
      execute_filter_rule (analysis, rule_system[SYNTAX],
			   rule_system[SYNTAX]->input_filter, analyse_all);

    if (options[CACHE_OPTION] && ! build_tree)
      put_into_cache (analysis, input, analyse_all);
  }
  else /* grammar == SYNTAX */
  {
    if (options[SYN_OUT_FILTER_OPTION])
      execute_filter_rule (analysis, rule_system[SYNTAX],
			   rule_system[SYNTAX]->output_filter, analyse_all);
  }
}

/* end of file ==============================================================*/
