/* minitime - a mini dockable clock

   Copyright 2002 Matthew Allum

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
*/


#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xatom.h>
#include <X11/extensions/shape.h>
 
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <sys/time.h>
#include <unistd.h>

#define TIME_DISP 1
#define DATE_DISP 0
#define PADDING   2

int WIN_WIDTH, WIN_HEIGHT; 			      /* stupid globals */
XFontStruct *font;                               /* need to move to struct */

int
ColorFromStr(Display *display, XColor *col, const char *defstr)
{
  char *str;
  const char delim[] = ",:";
  char *token;
  XColor exact;

  if ((strchr(defstr, delim[0]) != NULL)
      || (strchr(defstr, delim[1]) != NULL) )
  {
     str = strdup(defstr);

     token = strsep (&str, delim); 
     col->red = ( atoi(token) * 65535 ) / 255; 
     token = strsep (&str, delim); 
     col->green = ( atoi(token) * 65535 ) / 255;
     token = strsep (&str, delim); 
     col->blue = ( atoi(token) * 65535 ) / 255;

     return XAllocColor(display,
			DefaultColormap(display, DefaultScreen(display)),
			col);
  } else {
     return XAllocNamedColor(display,
			     DefaultColormap(display, DefaultScreen(display)),
			     defstr, col, &exact);
  }
}


int update(Display* display, Window win, GC gc, 
	   int mode, Pixmap mask, GC mask_gc)
{
  struct tm *tm;
  time_t curtime;
  char date[10];


  
  XSetForeground(display, mask_gc, XWhitePixel (display, DefaultScreen(display)));
  XSetForeground(display, mask_gc, XBlackPixel (display, DefaultScreen(display)));

  XFillRectangle(display, mask, mask_gc, 0, 0, WIN_WIDTH, WIN_HEIGHT);

  XSetForeground(display, mask_gc, XBlackPixel (display, DefaultScreen(display)));
  XSetForeground(display, mask_gc, XWhitePixel (display, DefaultScreen(display)));
  

      if (mode == DATE_DISP)
	{
	   curtime=time(NULL);
	   tm=localtime(&curtime);
	   strftime(date,9,"%D",tm);
	} else {
	   curtime=time(NULL);
	   tm=localtime(&curtime);
	   strftime(date,9,"%H:%M",tm);
	}
      XDrawString(display, win, gc,
		  PADDING, PADDING + font->ascent, date, strlen(date));
  
      XDrawString(display, mask, mask_gc, PADDING   , PADDING + font->ascent,
		  date, strlen(date));

      XShapeCombineMask (display, win, ShapeBounding, 0, 0, mask, ShapeSet);

      XFlush(display);

      return 0;
}


void printversion(void)
{
  printf("Version: 0.1 \n");
}

void usage(void)
{
  printf("Usage: miniapm <options>\n");
  printf("Options:\n");
  printf("  -display  <display>\n");
  printf("  -color    <color>\n");
  printf("  -fn       <font>\n");
  printf("  -m        show date rather than time\n");
  printf("  -v        version\n");
}



int main(int argc, char* argv[])
{
  Display* display;	
  int screen_num;	
  Window win;
  GC gc, mask_gc;
  Pixmap mask;
  XGCValues gv;
  XColor fg_col;
  
  char *display_name = (char *)getenv("DISPLAY");
  char *font_name = "fixed";
  char *color_name = "black";
  int depth, i;
  int mode = TIME_DISP;

  time_t last_time = 0;
  time_t curTime;

  int update_req = 0;

  /* pass command line */
  for (i=1; argv[i]; i++) {
    char *arg = argv[i];
    if (*arg=='-') {
      switch (arg[1]) {
	case 'd' : /* display */
	  display_name = argv[i+1];
	  i++;
	  break;
	case 'f' :
	  font_name = argv[i+1];
	  i++;
	  break;
	case 'c' :
	   color_name = argv[i+1];
	   i++;
	  break;
	case 'm' :
	   mode = DATE_DISP;
	  break;
	 case 'v' :
	  printversion();
	  exit(0);
	  break;
	default:
	  usage();
	  exit(0);
	  break;
      }
    }
  }

  display = XOpenDisplay(display_name);
  if (display != NULL) {

    Atom window_type_atom = XInternAtom(display, "_NET_WM_WINDOW_TYPE" , False);
    Atom window_type_dock_atom = XInternAtom(display, "_NET_WM_WINDOW_TYPE_DOCK",False);

  screen_num = DefaultScreen(display);

  if (!(font = XLoadQueryFont(display, font_name)))
  {
     printf("font %s not found", font_name);
     exit(1);
  }

  if (mode == TIME_DISP)
     WIN_WIDTH = XTextWidth(font, "12:34", 5) + 2*PADDING;
  else 
     WIN_WIDTH = XTextWidth(font, "12/34/99", 8) + 2*PADDING;

  WIN_HEIGHT = font->ascent + font->descent + 2*PADDING;

  win = XCreateSimpleWindow(display,
			    RootWindow(display, screen_num),
			    0, 0,
			    WIN_WIDTH, WIN_HEIGHT,
			    0, BlackPixel(display, screen_num),
			    WhitePixel(display, screen_num));

  
  ColorFromStr(display, &fg_col, color_name);

  
  gv.graphics_exposures = 0;
  gv.function   = GXcopy;
  gv.foreground = fg_col.pixel;
  gv.font       = font->fid;
  gc = XCreateGC(display, RootWindow(display, screen_num),
		 GCGraphicsExposures|GCFunction|GCForeground|GCFont, &gv);
  
  depth = DefaultDepth(display, DefaultScreen(display));

  mask = XCreatePixmap(display, win, WIN_WIDTH, WIN_HEIGHT, 1);
  mask_gc = XCreateGC(display, mask,
		      GCGraphicsExposures|GCFunction|GCForeground|GCFont, &gv);

  XFillRectangle(display, win, gc, 0, 0, WIN_WIDTH, WIN_HEIGHT );

  XFillRectangle(display, mask, mask_gc, 0, 0, WIN_WIDTH, WIN_HEIGHT );

  XShapeCombineMask (display, win, ShapeBounding, 0, 0, mask, ShapeSet);

  XChangeProperty(display, win, window_type_atom, XA_ATOM, 32, 
		 PropModeReplace, (unsigned char *) &window_type_dock_atom, 1);

  XMapWindow(display, win);

  update(display,win,gc,mode,mask,mask_gc);

  XSync(display, False);
  XFlush(display);


  XSelectInput(display, win, ExposureMask | StructureNotifyMask |
	       SubstructureNotifyMask | ButtonPressMask |  
	       ButtonReleaseMask ); 
  {
    int done = 0;
    int isButtonPressed = 0;
    XEvent an_event;

    while (!done)
      {
	curTime = time(0);
	if ( curTime != last_time )
	  {
	    last_time = curTime;
	    update_req = 1;
	  }
	while (XPending(display)) {
	  XNextEvent(display, &an_event);
	  switch (an_event.type) {
	    case Expose:
	      update_req = 1;
	      break;
	    case DestroyNotify:	      
	      done = 1;
	      break;
	    case ButtonPress:
	      isButtonPressed = 1;
	      break;
	    default:
	      break;
	  } 
	}
	if (update_req)
	  {
	     update(display,win,gc,mode,mask, mask_gc);
	     update_req = 0;
	  } 
	    usleep(300000L);
      } 
    
  }

  XFreePixmap(display,mask);

  XCloseDisplay(display);
  } else {
    fprintf(stderr, "%s: cannot connect to X server '%s'\n",
            argv[0], display_name);
    exit(1);
  }

  
  return 0;
}








