/*!
  @file           Log_ActionUnloadFile.hpp
  @author         UweH
  @ingroup        ActionHandling
  @brief          

\if EMIT_LICENCE

    ========== licence begin  GPL
    Copyright (c) 2000-2005 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end



\endif
*/



#ifndef LOG_ACTION_UNLOAD_FILE_H
#define LOG_ACTION_UNLOAD_FILE_H

/*===========================================================================*
 *  INCLUDES                                                                 *
 *===========================================================================*/

#include "SAPDBCommon/SAPDB_Types.hpp"
#include "Logging/Log_Types.hpp"
#include "Logging/Log_ReadWriteActionImage.hpp"

#include "ggg00.h"

/*===========================================================================*
 *  CLASSES, STRUCTURES, TYPES, UNIONS ...                                   *
 *===========================================================================*/

//===================================================================================
/*!
   @brief          defines a class to handle the after image of unload file
   @copydoc        Log_ReadWriteActionImage
 */

class Log_ActionUnloadFile : public Log_ReadWriteActionImage
{

private:
    
    /// pysical representation
    struct Head
    {
        tgg00_Filename filename;      ///< the filename (16 bytes)
        tgg00_Filename auxfilename;   ///< the temporary filename (16 bytes)

        /// constructor
        Head()
        {
            memset (filename,    0, sizeof(filename));
            memset (auxfilename, 0, sizeof(auxfilename));
            filename[0] = tfnNil_egg00;
            auxfilename[0] = tfnNil_egg00;
        }
        
        /// constructor
        Head( const tgg00_Filename &newfilename )
        {
            filename = newfilename;
            memset (auxfilename, 0, sizeof(auxfilename));
            auxfilename[0] = tfnNil_egg00;
        }

        /// constructor
        Head( const tgg00_Filename &newfilename,
              const tgg00_Filename &newauxfilename )
        {
            filename    = newfilename;
            auxfilename = newauxfilename;
        }
    };
    
    /// the in memory representation
    Head m_head;

public:
    
    /// This is used for reading persistent images.
    Log_ActionUnloadFile()
    : Log_ReadWriteActionImage (Log_UnloadFile)
    {}

    /// This is used for rollback of begin table load
    Log_ActionUnloadFile( const tgg00_Filename &filename )
    : Log_ReadWriteActionImage (Log_UnloadFile),
      m_head (filename)
    {}

    /// This is used for undo file
    Log_ActionUnloadFile( const tgg00_Filename &filename,
                          const tgg00_Filename &auxfilename )
    : Log_ReadWriteActionImage (Log_UnloadFile),
      m_head (filename, auxfilename)
    {}

    /// called by command
    void ExecuteAction (tgg00_TransContext &TransContext,
                        tgg00_FileId       *pCommandFileId    = NULL,
                        tgg00_FileId       *pCommandAuxFileId = NULL) const;

    virtual void Execute (tgg00_TransContext &TransContext) const
    {
        ExecuteAction (TransContext);
    }

    virtual SAPDB_UInt GetPersistentLength () const
    {
        return Log_AlignedImageLen(sizeof(m_head));
    }

    /// reads the in memory representation from the reader
    void ReadImagePersistent (Log_ActionImageReader &reader,
                              bool                  &isOk)
    {
        reader.CopyImage( reinterpret_cast<SAPDB_Byte*>(&m_head),
                          sizeof(m_head),
                          isOk );
    }

    /// writes the in memory representation via writer to disk 
    void WriteImagePersistent (Log_ActionImageWriter &writer,
                               bool                  &isOk) const
    {
        writer.PersistentWrite (&m_head, sizeof (m_head), isOk);
    }
    
};


/*==========================================================================*
*  DEFINITION OF INLINE METHODS                                             *
*==========================================================================*/

#endif  /* LOG_ACTION_UNLOAD_FILE_H */
