""" Creates plugin implementations from plugin definitions. """


# Standard library imports.
import logging

# Enthought library imports.
from enthought.traits.api import HasTraits, Instance

# Local imports.
from plugin import Plugin


# Setup a logger for this module.
logger=logging.getLogger(__name__)


class PluginFactory(HasTraits):
    """ Creates plugin implementations from plugin definitions. """

    #### 'PluginFactory' interface ############################################

    # The application that the factory is part of.
    application = Instance('enthought.envisage.core.application.Application')

    ###########################################################################
    # 'PluginFactory' interface.
    ###########################################################################

    def create_plugin(self, definition):
        """ Creates a plugin implementation from a plugin definition. """

        # Determine the plugin implementation class. If no specific class is
        # required then we use the default 'Plugin' class.
        if len(definition.class_name) == 0:
            klass = Plugin

        else:
            klass = self.application.import_symbol(definition.class_name)

        # Create an instance of the plugin implementation class.
        plugin = klass(application=self.application, id=definition.id)

        # Register the plugin with the extension registry so that it will get
        # notified when any additional contributions are made to any of the
        # extension points it offers.
        extension_registry = self.application.extension_registry

        for extension_point in definition.extension_points:
            extension_registry.add_extension_listener(
                extension_point, plugin.extension_added
            )

        logger.debug('plugin %s created' % plugin.id)

        return plugin

#### EOF ######################################################################
