#include <libgnome/libgnome.h>
#include <math.h>
#include <sys/wait.h>
#include "mplayer_engine.hh"
#include "exception.hh"
#include "log.hh"
#include "application.hh"
#include "exception_handler.hh"
#include "gdk_lock.hh"
#include "mutex_lock.hh"

#include <X11/Xlib.h>
#include <X11/extensions/XTest.h>

class Parameters
{
private:
	GSList* parameters;
	guint length;

public:
	Parameters()
	{
		length = 0;
		parameters = NULL;
	}
	
	~Parameters()
	{
		clear();
	}
	
	void clear()
	{
		GSList* iterator = parameters;
		while (iterator != NULL)
		{
			g_free(iterator->data);
			iterator = g_slist_next(iterator);
		}
		g_slist_free(parameters);
		
		length = 0;
		parameters = NULL;
	}
	
	void add(const String& value)
	{
		parameters = g_slist_append(parameters, g_strdup(value.c_str()));
		length++;
	}
	
	void add(gint value)
	{
		parameters = g_slist_append(parameters, g_strdup_printf("%d", value));
		length++;
	}
	
	String join(const gchar* separator = " ")
	{
		gchar** arguments = get_arguments();
		gchar* joined = g_strjoinv(separator, arguments);
		String result = joined;
		g_free(joined);
		delete [] arguments;
		
		return result;
	}
	
	gchar** get_arguments()
	{
		gchar** arguments = new gchar*[length + 1];
		guint index = 0;

		GSList* iterator = parameters;
		while (iterator != NULL)
		{
			arguments[index++] = (gchar*)iterator->data;
			iterator = g_slist_next(iterator);
		}
		arguments[index++] = NULL;
				
		return arguments;
	}
};

MPlayerEngine::MPlayerEngine ()
{		
	Display*		display;
	int				screen;
	double			res_h, res_v;

	mute_state			= false;
	fifo_output_stream	= NULL;
	pid					= -1;
	standard_input		= -1;
	dual_language_state = ENGINE_DUAL_LANGUAGE_STATE_DISABLED;
	audio_channel		= 0;
	subtitle_channel	= 0;

	Application& application = Application::get_current();
	Configuration& configuration = application.get_configuration();

	configuration.set_default_string_value("mplayer.video_driver", "xv");
	configuration.set_default_string_value("mplayer.audio_driver", "alsa");
	configuration.set_default_string_value("mplayer.fifo_path", application.get_directory() + "/video.fifo");

	fifo_path = configuration.get_string_value("mplayer.fifo_path");
	if (!IO::FIFO::exists(fifo_path))
	{
		IO::FIFO::create(fifo_path);
	}
	
	GtkWidget* event_box_video = application.get_glade().get_widget("event_box_video");
	GtkWidget* aspect_frame = gtk_aspect_frame_new(NULL, 0.5, 0.5, 16/9.0, false);
	widget = gtk_drawing_area_new();
	GdkColor black = {0, 0, 0,  0};
	gtk_widget_modify_bg(widget, GTK_STATE_NORMAL, &black);
	gtk_widget_modify_bg(aspect_frame, GTK_STATE_NORMAL, &black);
	gtk_widget_modify_bg(event_box_video, GTK_STATE_NORMAL, &black);
	gtk_container_add(GTK_CONTAINER(event_box_video), aspect_frame);
	gtk_container_add(GTK_CONTAINER(aspect_frame), widget);
	
	gtk_widget_set_double_buffered(widget, false);
	gtk_widget_show_all(event_box_video);
	gtk_widget_realize(widget);
}

MPlayerEngine::~MPlayerEngine()
{
	close();
}

Window MPlayerEngine::get_window_id()
{
	return GDK_WINDOW_XID(widget->window);
}

void MPlayerEngine::mute(gboolean state)
{
	if (standard_input != -1 && mute_state != state)
	{
		mute_state = state;
		Log::write(mute_state ? "Muting" : "Unmuting");
		::write(standard_input, "m\n", 2);	
	}
}

void MPlayerEngine::write(const gchar* buffer, gsize length)
{
	if (fifo_output_stream != NULL)
	{
		fifo_output_stream->write(buffer, length);
	}
}

void MPlayerEngine::open()
{
	if (pid == -1)
	{
		GError* error = NULL;
		
		Application& application = Application::get_current();
		Configuration& configuration = application.get_configuration();

		String video_driver = configuration.get_string_value("mplayer.video_driver");
		String audio_driver = configuration.get_string_value("mplayer.audio_driver");

		Parameters parameters;
		
		parameters.add("mplayer");
		switch (dual_language_state)
		{
		case ENGINE_DUAL_LANGUAGE_STATE_LEFT:
			parameters.add("-af");
			parameters.add("channels=2:2:0:0:0:1");
			break;
		case ENGINE_DUAL_LANGUAGE_STATE_RIGHT:
			parameters.add("-af");
			parameters.add("channels=2:2:1:1:1:0");
			break;
		}
		if (subtitle_channel != 0)
		{
			parameters.add("-sid");
			parameters.add(subtitle_channel);
		}
		if (audio_channel != 0)
		{
			parameters.add("-aid");
			parameters.add(audio_channel);
		}
		parameters.add("-really-quiet");
		parameters.add("-slave");
		parameters.add("-softvol");
		parameters.add("-stop-xscreensaver");
		parameters.add("-vf");
		parameters.add("pp=fd");
		parameters.add("-vo");
		parameters.add(video_driver);
		parameters.add("-ao");
		parameters.add(audio_driver);
		parameters.add("-wid");
		parameters.add(get_window_id());
		parameters.add(fifo_path);
		
		String command_line = parameters.join();
		Log::write("Command: %s", command_line.c_str());
		
		gchar** arguments = parameters.get_arguments();
		
		g_spawn_async_with_pipes(
								 NULL,
								 arguments,
								 NULL,
								 G_SPAWN_SEARCH_PATH,
								 NULL,
								 NULL,
								 &pid,
								 &standard_input,
								 NULL,
								 NULL,
								 &error
								 );
		
		delete [] arguments;
		
		if (error != NULL)
		{
			throw Exception(error->message);
		}

		if (fifo_output_stream == NULL)
		{
			SCOPE_LOG(N_("Opening FIFO for writing"));
			fifo_output_stream = new IO::Channel(fifo_path, O_WRONLY);
			fifo_output_stream->set_encoding(NULL);
		}
	}
}

void MPlayerEngine::close()
{
	if (fifo_output_stream != NULL)
	{
		Log::write(N_("Closing FIFO stream"));
		delete fifo_output_stream;
		fifo_output_stream = NULL;
	}
	
	if (pid != -1)
	{
		Log::write("Quitting MPlayer");
		::write(standard_input, "q\n", 2);
		
		waitpid(pid, NULL, 0);
		pid = -1;
		Log::write("MPlayer has quit");
	}
}

void MPlayerEngine::set_subtitle_channel(gint channel)
{
	subtitle_channel = channel;
	Application::get_current().restart_stream();
}

void MPlayerEngine::set_audio_channel(gint channel)
{
	if (audio_channel != channel)
	{
		audio_channel = channel;
		Application::get_current().restart_stream();
	}
}

void MPlayerEngine::set_dual_language_state(gint state)
{
	if (dual_language_state != state)
	{
		dual_language_state = state;
		Application::get_current().restart_stream();
	}
}

