/********************************************************
*  mgaXconfig.c
*
*  Source file for Matrox video parameters tuning utility.
*  Functions accessing/modifying XF86ConfigRec struct.
*
*  Copyright (C) 2001, Matrox Graphics Inc.
*
*  Author : Luugi Marsan and Stephane Duguay
*  Last modified : April 2001.
*********************************************************/


#include "io.h"
#include "mgaXconfig.h"

typedef enum {
    ediDisplay1 = 0,
    ediDisplay2 = 1,
    ediDisplayMerged = 2
} eDisplayIdentifier;

char* edi2str[] = {
    "Display 1",
    "Display 2",
    "Display Merged"
};

/*obsolete, must remove */
typedef enum {
    ediDevice1 = 0,
    ediDevice2 = 1,
    ediDeviceMerged = 2
} eDeviceIdentifier;

char* edevi2str[] = {
    "MATROX CARD 1",
    "MATROX CARD 2",
    "MATROX CARD 1"
};


static XF86ConfScreenPtr getScreenFromIndex(XF86ConfigPtr conf, int screenIndex);
static int getIndexFromScreen(XF86ConfigPtr conf, char* screen);
static BOOL replaceString(char** dest, char* src);
static XF86OptionPtr removeOption(XF86OptionPtr pOptionsList, char* OptionType);
static XF86OptionPtr addOption(XF86OptionPtr pConfOption, char* OptionType, char* Value);
void MsgBox(char* title, char* msg, int exit);

XF86ConfMonitorPtr MakeMonitor( XF86ConfigPtr conf, eDisplayIdentifier did );
XF86ConfDevicePtr MakeDevice( XF86ConfigPtr conf, eDeviceIdentifier edi );
XF86ConfScreenPtr MakeScreen( XF86ConfigPtr conf, eDisplayIdentifier edi, XF86ConfDevicePtr pNewDevice, XF86ConfMonitorPtr pNewMonitor );
XF86ConfScreenPtr FindScreen(XF86ConfigPtr conf, eDisplayIdentifier edi);
XF86ConfMonitorPtr FindMonitor(XF86ConfigPtr conf, eDisplayIdentifier edi);


static BOOL replaceString(char** dest, char* src)

{


	free(*dest); /* even if NULL, because free don't do nothing if parameter is NULL */


	if(!src)
	{
		*dest = NULL;
		return TRUE;
	}
	else
	{


		*dest = (char*) calloc(strlen(src) + 1, sizeof(char));


		if(!*dest) return FALSE;


		strcpy(*dest, src);


		return TRUE;
	}
}


static XF86OptionPtr removeOption(XF86OptionPtr pOptionsList, char* OptionType)
{
	XF86OptionPtr pPrevOption;
	XF86OptionPtr pOption;
	XF86OptionPtr pRetour;

	if (!pOptionsList) return NULL;

	pOption = pOptionsList;
	pPrevOption = pOption;


	while (pOption)
	{
		if (!(strcmp(pOption->opt_name, OptionType)))
		{
			/* Found option to be deleted */
			/* Re-link the list around it */
			if ( pOption == pOptionsList)
				pRetour = pOption->list.next; /* We're deleting the head */
			else
			{
				pPrevOption->list.next = pOption->list.next;
				pRetour = pOptionsList;
			}

			free(pOption);
			return pRetour;
		}

		pPrevOption = pOption;
		pOption = pOption->list.next;


	}


	return pOptionsList;


}

static XF86OptionPtr addOption(XF86OptionPtr pConfOption, char* OptionType, char* Value)
{
    XF86OptionPtr pTmpConfOption;
    
    /* If the option exists */           
    if((pTmpConfOption = xf86findOption(pConfOption, OptionType)))
    {
            if(strcmp(xf86findOptionValue(pConfOption, OptionType), Value))
            {
                    pTmpConfOption->opt_val = calloc(strlen(Value)+1,sizeof(char));
                    replaceString(&pTmpConfOption->opt_val,Value);
            }                    
    }
    else /* if not then add it */
    {
            char* optType = calloc(strlen(OptionType) + 1, sizeof(char));
            char* optValue = calloc(strlen(Value) + 1, sizeof(char));

            strcpy(optType, OptionType);
            strcpy(optValue, Value);
            pConfOption = xf86addNewOption(pConfOption, optType, optValue);
    }
    return pConfOption;
}

/* This routine outputs the index of the screen string supplied by the
 * calling routine. It traveses the Screen structure and searches for
 * the string*/


static int getIndexFromScreen(XF86ConfigPtr conf, char* screen){

        if(strcmp(screen,"Display 1") == 0) 
            return 0;
        else if(strcmp(screen,"Display 2") == 0) 
            return 1;
        else
        
	return -1;
}


static XF86ConfScreenPtr getScreenFromIndex(XF86ConfigPtr conf, int screenIndex)
{
#ifdef DEBUG
	printf("getScreenFromIndex(%d)",screenIndex); 
#endif

	if((screenIndex < 0) || !(conf)) return NULL;


        switch(screenIndex) 
        {
            case 0 : return FindScreen(conf,ediDisplay1);
            case 1 : return FindScreen(conf,ediDisplay2);
            default : return NULL;
        }

}



/* Opens the XF86Config file and returns the pointer to the structure
 * of the config file
 */ 
XF86ConfigPtr getConfigPtr(const char* filename)
{


	XF86ConfigPtr conf;

	if (!filename) return NULL;


	filename = xf86openConfigFile ("%A", filename, NULL);


	if (filename)
	{
		printf("Matrox PowerDesk: Configuration file (%s) opened.\n", filename);
	}
	else
	{
		printf ("Matrox PowerDesk: Couldn't open configuration file.\n");
		return NULL;
	}


	if ((conf = xf86readConfigFile ()) == NULL)
	{
		fprintf (stderr, "Matrox PowerDesk: Couldn't parse the configuration file.\n");
		xf86closeConfigFile();
		return NULL;
	}
	else
	{
		fprintf (stderr, "Matrox PowerDesk: Configuration file parsed.\n");
	}


	/* Close Config file */
	xf86closeConfigFile();

	replaceString(&(conf->conf_layout_lst->lay_identifier), "Matrox PowerDesk configured.");


	return conf;
}


/* Write Conf file */
BOOL confWriteConfigFile( char* filename,XF86ConfigPtr conf)
{
	if (!conf) return FALSE;

        if(confMergedFBIsActive(conf)) {
            confGenerateMetaModes(conf);
        }
        
	return xf86writeConfigFile (filename, conf);
}


/* Inverts the Screens Layout. It assumes only 2 monitors */
XF86ConfigPtr confSetLayout (XF86ConfigPtr conf, int where)
{
	XF86ConfAdjacencyPtr aptr;
        XF86ConfScreenPtr pScrn;

	if (!conf) return NULL;


	aptr = conf->conf_layout_lst->lay_adjacency_lst;

        /* Set the Monitor2Position even if not in merge mode to prevent inconsistencys in switches */
        if((pScrn = FindScreen(conf,ediDisplayMerged)) != NULL) 
        {
            switch(where) 
            {
                case CONF_ADJ_LEFTOF :
                    pScrn->scrn_option_lst = addOption(pScrn->scrn_option_lst,"Monitor2Position","RightOf");
                    break;
                    
                case CONF_ADJ_RIGHTOF :
                    pScrn->scrn_option_lst = addOption(pScrn->scrn_option_lst,"Monitor2Position","LeftOf");
                    break;

                case CONF_ADJ_ABOVE :
                    pScrn->scrn_option_lst = addOption(pScrn->scrn_option_lst,"Monitor2Position","Below");
                    break;

                case CONF_ADJ_BELOW :
                    pScrn->scrn_option_lst = addOption(pScrn->scrn_option_lst,"Monitor2Position","Above");
                    break;
                default :
                    pScrn->scrn_option_lst = addOption(pScrn->scrn_option_lst,"Monitor2Position","RightOf");
                    
            }
            aptr->adj_where = 0;
        }   

        if(strcmp(aptr->adj_screen_str, "Display Merged") != 0)
            aptr->adj_where = where;

	return conf;
}


/* MakeMonitor, MakeScreen, MakeDevice
 * Finds or Makes and fills the requested items with proper values */

XF86ConfMonitorPtr MakeMonitor( XF86ConfigPtr conf, eDisplayIdentifier did )
{
    XF86ConfMonitorPtr pMonitor, pNewMonitor = NULL;
    int i;

    if (!conf)
    {
            printf("Matrox PowerDesk: Configuration problem encountered.\n");
            return NULL;
    }
    /* Look for existing monitor */
    for(pMonitor = conf->conf_monitor_lst; pMonitor != NULL; pMonitor = pMonitor->list.next) 
    {
        if(strcmp(pMonitor->mon_identifier, edi2str[did]) == 0) {
            pNewMonitor = pMonitor;
            break;
        }
        if( (pNewMonitor == NULL) &&
            strcmp(pMonitor->mon_identifier, edi2str[0]) &&
            strcmp(pMonitor->mon_identifier, edi2str[1]) &&
            strcmp(pMonitor->mon_identifier, edi2str[2]) ) 
        {
            /* none of the above, so pMonitor is like free monitor. */
            pNewMonitor = pMonitor;
        }
    }
    /*set pMonitor to any monitor, to copy unknown values */
    pMonitor = conf->conf_monitor_lst; 
    while(pMonitor->list.next != NULL) pMonitor = pMonitor->list.next; /* we'll chose last one */
            
    
    if (!pNewMonitor) /* Need to create new */
    {
        /* add a new monitor */
        pNewMonitor = calloc(1, sizeof(XF86ConfMonitorRec));
        pNewMonitor->list.next = NULL;


        replaceString(&pNewMonitor->mon_vendor, pMonitor->mon_vendor);
        replaceString(&pNewMonitor->mon_modelname, pMonitor->mon_modelname);
        pNewMonitor->mon_width = pMonitor->mon_width;
        pNewMonitor->mon_height = pMonitor->mon_height;
        pNewMonitor->mon_modeline_lst = NULL;


        pNewMonitor->mon_n_hsync = pMonitor->mon_n_hsync;
        for(i = 0; i < pMonitor->mon_n_hsync; i++) pNewMonitor->mon_hsync[i] = pMonitor->mon_hsync[i];

        pNewMonitor->mon_n_vrefresh = pMonitor->mon_n_vrefresh;
        for(i = 0; i < pMonitor->mon_n_vrefresh; i++) pNewMonitor->mon_vrefresh[i] = pMonitor->mon_vrefresh[i];

        pNewMonitor->mon_gamma_red = pMonitor->mon_gamma_red;
        pNewMonitor->mon_gamma_green = pMonitor->mon_gamma_green;
        pNewMonitor->mon_gamma_blue = pMonitor->mon_gamma_blue;
        pNewMonitor->mon_option_lst = NULL;
        pNewMonitor->mon_modes_sect_lst = NULL;
        pNewMonitor->mon_comment = NULL;

        /* Add it the list */
        xf86addListItem ( (GenericListPtr) pMonitor, (GenericListPtr) pNewMonitor);
    }
   
    replaceString(&pNewMonitor->mon_identifier ,edi2str[did]);
    return pNewMonitor;
}


XF86ConfDevicePtr MakeDevice( XF86ConfigPtr conf, eDeviceIdentifier edi )
{
    XF86ConfDevicePtr pDevice, pNewDevice = NULL;
    int i;
	
    if (!conf)
    {
            printf("Matrox PowerDesk: Configuration problem encountered.\n");
            return NULL;
    }
    /* Look for existing device */
    for(pDevice = conf->conf_device_lst; pDevice != NULL; pDevice = pDevice->list.next) 
    {
        if(strcmp(pDevice->dev_identifier, edevi2str[edi]) == 0) {
            pNewDevice = pDevice;
            break;
        }
        if( (pNewDevice == NULL) &&
            strcmp(pDevice->dev_identifier, edevi2str[0]) &&
            strcmp(pDevice->dev_identifier, edevi2str[1]))
        {
            /* none of the above, so pDevice is like free dev. */
            pNewDevice = pDevice;
        }
    }
    /*set pDevice to any dev, to copy unknown values */
    pDevice = conf->conf_device_lst; 
    while(pDevice->list.next != NULL) pDevice = pDevice->list.next; /* we'll chose last one */
    
    if (!pNewDevice) /* Need to create one */
    {
        /* add a new device */
        pNewDevice = calloc(1, sizeof(XF86ConfDeviceRec));

        pNewDevice->list.next = NULL;
        replaceString(&pNewDevice->dev_vendor, pDevice->dev_vendor);
        replaceString(&pNewDevice->dev_board, pDevice->dev_board);
        replaceString(&pNewDevice->dev_chipset, pDevice->dev_chipset);
        replaceString(&pNewDevice->dev_busid, pDevice->dev_busid);
        replaceString(&pNewDevice->dev_card, pDevice->dev_card);
        replaceString(&pNewDevice->dev_driver, pDevice->dev_driver);
        replaceString(&pNewDevice->dev_ramdac, pDevice->dev_ramdac);
        replaceString(&pNewDevice->dev_clockchip, pDevice->dev_clockchip);
        for(i = 0; i < 4; i++) pNewDevice->dev_dacSpeeds[i] = pDevice->dev_dacSpeeds[i];
        pNewDevice->dev_videoram = pDevice->dev_videoram;
        pNewDevice->dev_textclockfreq = pDevice->dev_textclockfreq;
        pNewDevice->dev_videoram = pDevice->dev_videoram;
        pNewDevice->dev_bios_base = pDevice->dev_bios_base;
        pNewDevice->dev_mem_base = pDevice->dev_mem_base;
        pNewDevice->dev_io_base = pDevice->dev_io_base;
        pNewDevice->dev_clocks = pDevice->dev_clocks;
        for(i = 0; i < pDevice->dev_clocks; i++) pNewDevice->dev_clock[i] = pDevice->dev_clock[i];
        pNewDevice->dev_chipid = pDevice->dev_chipid;
        pNewDevice->dev_chiprev = pDevice->dev_chiprev;
        pNewDevice->dev_irq = pDevice->dev_irq;


        pNewDevice->dev_comment = NULL;
        pNewDevice->dev_option_lst = NULL;

        /* Add it the list */
        xf86addListItem ( (GenericListPtr) pDevice, (GenericListPtr) pNewDevice);
    }
    
    pNewDevice->dev_screen = (int)edi;
    replaceString(&pNewDevice->dev_identifier ,edevi2str[edi]);
    replaceString(&pNewDevice->dev_driver, "mga");
    
    return pNewDevice;
}

XF86ConfScreenPtr MakeScreen( XF86ConfigPtr conf, eDisplayIdentifier edi, XF86ConfDevicePtr pNewDevice, XF86ConfMonitorPtr pNewMonitor ) 
{
    XF86ConfScreenPtr pScrn, pNewScrn = NULL;

    if (!conf)
    {
            printf("Matrox PowerDesk: Configuration problem encountered.\n");
            return NULL;
    }
    /* Look for existing screen */
    for(pScrn = conf->conf_screen_lst; pScrn != NULL; pScrn = pScrn->list.next) 
    {
        if(strcmp(pScrn->scrn_identifier, edi2str[edi]) == 0) {
            pNewScrn = pScrn;
            break;
        }
        if( (pNewScrn == NULL) &&
            strcmp(pScrn->scrn_identifier, edi2str[0]) &&
            strcmp(pScrn->scrn_identifier, edi2str[1]) &&
            strcmp(pScrn->scrn_identifier, edi2str[2]))
        {
            /* none of the above, so pScreen is like free scrn. */
            pNewScrn = pScrn;
        }
    }
    
    /*set pScreen to any scrn, to copy unknown values */
    pScrn = conf->conf_screen_lst; 
    while(pScrn->list.next != NULL) pScrn = pScrn->list.next; /* we'll chose last one */
    
    
    if (!pNewScrn) /* need to create */
    {
        XF86ConfDisplayPtr pDisplay;
        XF86ModePtr pConfMode;

        /* Add a new screen */
        pNewScrn = calloc(1, sizeof(XF86ConfScreenRec));
        pNewScrn->list.next = NULL;

        replaceString(&pNewScrn->scrn_obso_driver, pScrn->scrn_obso_driver);

        pNewScrn->scrn_defaultdepth = pScrn->scrn_defaultdepth;
        pNewScrn->scrn_defaultbpp = pScrn->scrn_defaultbpp;
        pNewScrn->scrn_defaultfbbpp = pScrn->scrn_defaultfbbpp;

        pNewScrn->scrn_adaptor_lst = NULL;
        pNewScrn->scrn_option_lst = NULL;
        pNewScrn->scrn_comment = NULL;

        pDisplay = calloc(1, sizeof(XF86ConfDisplayRec));
        pDisplay->disp_option_lst = NULL;
        pDisplay->disp_mode_lst = NULL;
        pDisplay->disp_comment = NULL;
        pDisplay->disp_weight.red = 0;
        pDisplay->disp_weight.blue = 0;
        pDisplay->disp_weight.green = 0;
        pDisplay->disp_white.red = -1;
        pDisplay->disp_white.blue = -1;
        pDisplay->disp_white.green = -1;
        pDisplay->disp_black.red = -1;
        pDisplay->disp_black.blue = -1;
        pDisplay->disp_black.green = -1;
        pDisplay->list.next = NULL;


        /* add new screen at 640x480 */
        pDisplay->disp_depth = pNewScrn->scrn_defaultdepth;
        pConfMode = calloc(1, sizeof(XF86ModeRec));
        pConfMode->mode_name = calloc(strlen("640x480") + 1, sizeof(char));
        strcpy(pConfMode->mode_name, "640x480");
        pConfMode->list.next = NULL;
        pDisplay->disp_mode_lst = (XF86ModePtr)xf86addListItem(     (GenericListPtr) pDisplay->disp_mode_lst, \
                        (GenericListPtr) pConfMode);

        pNewScrn->scrn_display_lst = (XF86ConfDisplayPtr) xf86addListItem(      (GenericListPtr) pNewScrn->scrn_display_lst, \
                        (GenericListPtr) pDisplay);


        xf86addListItem ( (GenericListPtr) pScrn, (GenericListPtr) pNewScrn);


        if( edi == ediDisplay2 )
            MsgBox( "Matrox PowerDesk",
                        "Your secondary display has been enabled. Its resolution has been set to 640 x 480.\n\nTo change its resolution, click the button for the secondary display (display 2).",
                        0);
    }

    /*make sure the option is not lying around */
    pNewScrn->scrn_option_lst = removeOption(pNewScrn->scrn_option_lst,"MergedFB");
    
    replaceString(&pNewScrn->scrn_identifier , edi2str[edi]);

    /* link */
    replaceString(&pNewScrn->scrn_device_str, pNewDevice->dev_identifier);
    replaceString(&pNewScrn->scrn_monitor_str, pNewMonitor->mon_identifier);
    pNewScrn->scrn_monitor = pNewMonitor;
    pNewScrn->scrn_device = pNewDevice;

    return pNewScrn;
}

XF86ConfScreenPtr FindScreen(XF86ConfigPtr conf, eDisplayIdentifier edi) 
{
    XF86ConfScreenPtr pScrn = conf->conf_screen_lst;
    if (!conf)
    {
            printf("Matrox PowerDesk: Configuration problem encountered.\n");
            return NULL;
    }
    while(pScrn != NULL && (strcmp(pScrn->scrn_identifier, edi2str[edi]) != 0)) {
        pScrn = pScrn->list.next;
    }
    return pScrn;
}

XF86ConfMonitorPtr FindMonitor(XF86ConfigPtr conf, eDisplayIdentifier edi) 
{
    XF86ConfMonitorPtr pMonitor = conf->conf_monitor_lst;
    if (!conf)
    {
            printf("Matrox PowerDesk: Configuration problem encountered.\n");
            return NULL;
    }
    while(pMonitor != NULL && (strcmp(pMonitor->mon_identifier, edi2str[edi]) != 0)) {
        pMonitor = pMonitor->list.next;
    }
    return pMonitor;
}


/* Enable Dual Head
 * Adds a second Screen and Device section if it's not done already
 *  
 * 1. it will check to see how many devices that you have
 * 2. add another device Screen section and a Second device section
 * 3. Create a Screen struct and link it the rest
 */


XF86ConfigPtr confEnableDualHead ( XF86ConfigPtr conf, int where)
{
	XF86ConfScreenPtr pScrn, pNewScrn;
	XF86ConfDevicePtr pDevice, pNewDevice;
	XF86ConfMonitorPtr pMonitor, pNewMonitor;
	XF86ConfLayoutPtr pLayout;
	XF86ConfAdjacencyPtr pAdj, pNewAdj;


	if (!conf)
	{
		printf("Matrox PowerDesk: Configuration problem encountered.\n");
		return NULL;
	}


        /* setting twice does no harm, and we need to bypass this check on startup */
/*	if(confDualHeadIsActive(conf)) return conf;*/
        
        pDevice = MakeDevice(conf,ediDevice1);
	pMonitor = MakeMonitor(conf,ediDisplay1);
	pScrn = MakeScreen(conf,ediDisplay1,pDevice,pMonitor);


	pLayout = conf->conf_layout_lst;
	pAdj = pLayout->lay_adjacency_lst;


	pNewDevice = MakeDevice(conf,ediDevice2);
	pNewMonitor = MakeMonitor(conf,ediDisplay2);
	pNewScrn = MakeScreen(conf,ediDisplay2,pNewDevice,pNewMonitor);


	pDevice->dev_option_lst = removeOption(pDevice->dev_option_lst, "NoHal");
	pNewDevice->dev_option_lst = removeOption(pNewDevice->dev_option_lst, "NoHal");

	/* Server Layout Section */
	replaceString(&pAdj->adj_refscreen, "Display 2");


	replaceString(&pAdj->adj_screen_str, pScrn->scrn_identifier);


	pAdj->adj_where= where;


	if (pAdj->list.next == NULL)
	{
		pNewAdj = calloc(1, sizeof(XF86ConfAdjacencyRec));
		replaceString(&pNewAdj->adj_screen_str ,"Display 2");
		pNewAdj->adj_screen = pNewScrn;
		pNewAdj->adj_scrnum = -1;
		pLayout->lay_adjacency_lst= (XF86ConfAdjacencyPtr) xf86addListItem ( (GenericListPtr) pLayout->lay_adjacency_lst, (GenericListPtr) pNewAdj);
	}
	else
	{
		pNewAdj = pAdj->list.next;
		pNewAdj->adj_screen = pNewScrn;
		pNewAdj->adj_scrnum = -1;
		replaceString(&pNewAdj->adj_screen_str, "Display 2");
	}


	return conf;


}


/* Disables dual head */
XF86ConfigPtr confDisableDualHead ( XF86ConfigPtr conf)
{
	XF86ConfScreenPtr pScrn;
	XF86ConfDevicePtr pDevice;
	XF86ConfLayoutPtr pLayout;
	XF86ConfAdjacencyPtr pAdj;


	if (!conf)
	{
		printf("Matrox PowerDesk: Configuration problem encountered.\n");
		return NULL;
	}


	if(!(confDualHeadIsActive(conf))) return conf;


	pScrn = conf->conf_screen_lst;
	pDevice = conf->conf_device_lst;
	pLayout = conf->conf_layout_lst;
	pAdj = pLayout->lay_adjacency_lst;


	free(pAdj->adj_refscreen);
	pAdj->adj_refscreen = NULL;
	pAdj->adj_where = 0;
	free((XF86ConfAdjacencyPtr)pAdj->list.next);
	pAdj->list.next = NULL;

	return conf;


}


/* I will check how many screns there is and how many devices */
BOOL confDualHeadIsActive(XF86ConfigPtr conf)
{
	XF86ConfScreenPtr pScrn, pTempScrn;
	XF86ConfDevicePtr pDevice, pTempDevice;
	XF86ConfLayoutPtr pLayout;
	XF86ConfAdjacencyPtr pAdj,pNextAdj;
	int numScreens;
	int numDevices;


	if (!conf)
	{
		printf("Matrox PowerDesk: Configuration problem encountered.\n");
		return FALSE;
	}   

	pScrn = conf->conf_screen_lst;
	pDevice = conf->conf_device_lst;

	pLayout = conf->conf_layout_lst;
	pAdj = pLayout->lay_adjacency_lst;


	if (pAdj->list.next)
		pNextAdj= pAdj->list.next;
	else
		pNextAdj = NULL;


	/* Count how many Screen Sections */
	numScreens = 1 ;
	for (pTempScrn = pScrn; pTempScrn->list.next != NULL; pTempScrn=pTempScrn->list.next)
		numScreens++;


	/* Count how many Device Sections */
	numDevices = 1;
	for (pTempDevice = pDevice; pTempDevice->list.next != NULL; pTempDevice=pTempDevice->list.next)
		if (!(strcmp(pTempDevice->dev_driver, "mga")))
			numDevices++;


	if(!confValidateLayout(conf)) return FALSE;


	/*Check if one of the AdjencyRec has a screen that it references
	 * to. If so Dual is active*/


	if  (pAdj->adj_refscreen)
		return TRUE;


	if (pNextAdj && pNextAdj->adj_refscreen)
		return TRUE;


	/* Dual head is not active */
	return FALSE;


}



/* Clone */
XF86ConfigPtr confEnableClone ( XF86ConfigPtr conf)
{
	XF86ConfScreenPtr pScrn, pNewScrn;
	XF86ConfDevicePtr pDevice, pNewDevice;
	XF86ConfMonitorPtr pMonitor, pNewMonitor;
	XF86ConfLayoutPtr pLayout;
	XF86ConfAdjacencyPtr pAdj, pNewAdj;

	if (!conf)
	{
		printf("Matrox PowerDesk: Configuration problem encountered.\n");
		return NULL;
	}


        /* setting twice does no harm, and we need to bypass this check on startup */
/*	if(confCloneIsActive(conf)) return conf;*/


	pDevice = MakeDevice(conf,ediDevice1);
	pMonitor = MakeMonitor(conf,ediDisplay1);
	pScrn = MakeScreen(conf,ediDisplay1,pDevice,pMonitor);


	pLayout = conf->conf_layout_lst;
	pAdj = pLayout->lay_adjacency_lst;


	pNewDevice = MakeDevice(conf,ediDevice2);
	pNewMonitor = MakeMonitor(conf,ediDisplay2);
	pNewScrn = MakeScreen(conf,ediDisplay2,pNewDevice,pNewMonitor);


        pDevice->dev_option_lst = removeOption(pDevice->dev_option_lst, "NoHal");
	pNewDevice->dev_option_lst = removeOption(pNewDevice->dev_option_lst, "NoHal");



	/*****************************************
	  Server Layout Section - Enable clone
	  Check if the there's a second adjancy list
	  if so: make sure there's not refscreen
	  if not: create a new adjancy list with the Screen 2
	 ********************************************/

	replaceString(&pAdj->adj_screen_str, "Display 1");


	pAdj->adj_screen = pScrn;
	pAdj->adj_scrnum = -1;


	if (pAdj->list.next == NULL)
	{
		pNewAdj = calloc (1, sizeof(XF86ConfAdjacencyRec));

		replaceString(&pNewAdj->adj_screen_str, "Display 2");

		pNewAdj->adj_where = 0;
		pNewAdj->adj_screen = pNewScrn;
		pNewAdj->adj_scrnum = -1;
		pLayout->lay_adjacency_lst = (XF86ConfAdjacencyPtr) xf86addListItem ( (GenericListPtr)  \
				pAdj, (GenericListPtr) pNewAdj);
	}
	else
	{
		pNewAdj = pAdj->list.next;

		replaceString(&pNewAdj->adj_screen_str, "Display 2");

		pNewAdj->adj_where = 0;
		pNewAdj->adj_screen = pNewScrn;
		pNewAdj->adj_scrnum = -1;
	}


	pAdj->adj_where = 0;


	return conf;


}


XF86ConfigPtr confDisableClone ( XF86ConfigPtr conf)
{
	XF86ConfScreenPtr pScrn;
	XF86ConfDevicePtr pDevice;
	XF86ConfLayoutPtr pLayout;
	XF86ConfAdjacencyPtr pAdj;


	if (!conf)
	{
		printf("Matrox PowerDesk: Configuration problem encountered.\n");
		return NULL;
	}


	pScrn = conf->conf_screen_lst;
	pDevice = conf->conf_device_lst;
	pLayout = conf->conf_layout_lst;
	pAdj = pLayout->lay_adjacency_lst;


	pAdj->adj_refscreen = NULL;
	pAdj->adj_where = 0;
	/* Make the second monitor NULL */
	free(pAdj->list.next);
	pAdj->list.next = NULL;

	return conf;


}


BOOL confCloneIsActive(XF86ConfigPtr conf)
{
	XF86ConfScreenPtr pScrn, pTempScrn;
	XF86ConfDevicePtr pDevice, pTempDevice;
	XF86ConfLayoutPtr pLayout;
	XF86ConfAdjacencyPtr pAdj,pNextAdj;
	int numScreens;
	int numDevices;


	if (!conf)
	{
		printf("Matrox PowerDesk: Configuration problem encountered.\n");
		return FALSE;
	}   


	pScrn = conf->conf_screen_lst;
	pDevice = conf->conf_device_lst;


	pLayout = conf->conf_layout_lst;
	pAdj = pLayout->lay_adjacency_lst;


	if (pAdj->list.next)
		pNextAdj= pAdj->list.next;
	else
		pNextAdj = NULL;


	/* Count how many Screen Sections */
	numScreens = 1 ;
	for (pTempScrn = pScrn; pTempScrn->list.next != NULL; pTempScrn=pTempScrn->list.next)
		numScreens++;


	/* Count how many Device Sections */
	numDevices = 1;
	for (pTempDevice = pDevice; pTempDevice->list.next != NULL; pTempDevice=pTempDevice->list.next)
		if (!(strcmp(pTempDevice->dev_driver, "mga")))
			numDevices++;


	if(!confValidateLayout(conf)) return FALSE;


	/* Checks to see that both both adj structs have to ref_screen */
	if  (!pAdj->adj_refscreen )
	{
		if (pNextAdj && !(pNextAdj->adj_refscreen))
			return TRUE;
	}


	return FALSE;


}


/* Direct Rendering Layer (DRI) */
/* Adds module at end of list */
XF86ConfigPtr confLoadModule (XF86ConfigPtr conf, char* load_name)
{
	XF86ConfModulePtr pModule;
	XF86LoadPtr pLoad, pTempLoad;
	XF86LoadPtr pNewLoad;

	pModule = conf->conf_modules;
	pLoad = pModule-> mod_load_lst;


	/* Search for DRI in the Module List*/
	for (pTempLoad = pModule-> mod_load_lst; pTempLoad; pTempLoad= pTempLoad->list.next)
		if (!strcmp(pTempLoad->load_name, load_name)) return conf;


#ifdef DEBUG
	printf("Did not find the module %s",load_name);
	printf(" Adding new module\n");
#endif

	/* If it doesn't find DRI */
	/* Add DRI section */
	pNewLoad = calloc (1, sizeof(XF86LoadRec));


	replaceString(&pNewLoad->load_name, load_name);


#ifdef DEBUG
	printf("pNewLoad->load_name = %s\n",pNewLoad->load_name);
#endif

	pNewLoad->load_type = 0; /* LOAD_MODULE */
	pNewLoad->load_opt = NULL;
	pNewLoad->list.next = NULL;


	pModule-> mod_load_lst = (XF86LoadPtr) xf86addListItem ( (GenericListPtr) \
			pLoad, (GenericListPtr) pNewLoad);

	return conf;


}


XF86ConfigPtr confUnloadModule (XF86ConfigPtr conf, char* load_name)
{
	XF86LoadPtr pPrevLoad;
	XF86LoadPtr pLoad;
	XF86ConfModulePtr pModule;


	if (!conf)
	{
		printf("Matrox PowerDesk: Configuration problem encountered.\n");
		return NULL;
	}

	pModule = conf->conf_modules;
	pLoad = pModule->mod_load_lst;
	pPrevLoad = pLoad; 


	while (pLoad)
	{
		if (!(strcmp(pLoad->load_name, load_name)))
		{
			/* Found module to be deleted */
			/* Re-link the list around it */
			if ( pLoad == pModule->mod_load_lst)
				pModule->mod_load_lst = pLoad->list.next; /* We're deleting the head */
			else
				pPrevLoad->list.next = pLoad->list.next;

			free(pLoad);
			break;
		}

		pPrevLoad = pLoad;
		pLoad = pLoad->list.next;


	}


	return conf;


}



BOOL confIsModuleActive(XF86ConfigPtr conf, char* load_name)
{
	XF86LoadPtr pLoad;
	XF86ConfModulePtr pModule;


	pModule = conf->conf_modules;
	pLoad = pModule->mod_load_lst;


	while (pLoad != NULL)
	{
		if (!(strcmp(pLoad->load_name,load_name)))
			return TRUE;

		pLoad= pLoad->list.next;
	}


	return FALSE;


}

/* Set the default Depth */
XF86ConfigPtr confSetDefaultDepthFromScreenPtr(XF86ConfigPtr conf, XF86ConfScreenPtr pScrn, int depth)
{
	if (!pScrn)
	{
		return NULL;
	}


    /* For 32 bit */
	if (depth ==32) {
	pScrn->scrn_defaultdepth = 24;
	pScrn->scrn_defaultfbbpp = 32;
	}
	else 
	{
	pScrn->scrn_defaultdepth = depth;
	pScrn->scrn_defaultfbbpp = 0;
    }

	return conf;
}

/* Set the default Depth */
XF86ConfigPtr confSetDefaultDepth(XF86ConfigPtr conf, int screen, int depth)
{
	XF86ConfScreenPtr pScrn ;

	pScrn= getScreenFromIndex(conf, screen);


	if (!pScrn)
	{
		printf("getScreenFromIndex failed\n");
		return NULL;
	}

	return confSetDefaultDepthFromScreenPtr(conf, pScrn, depth);
}

/* Get the default Depth */
int confGetDefaultDepthFromScreenPtr(XF86ConfigPtr conf, XF86ConfScreenPtr pScrn )
{

        if (!pScrn)
        {
                return 0;
        }

    /* For 32 bit */
        if(	pScrn->scrn_defaultdepth == 24 &&
	        pScrn->scrn_defaultfbbpp == 32 ) 
	{
		return 32;
        }
        else
        {
        	return pScrn->scrn_defaultdepth;
	}

}

/* Get the default Depth */
int confGetDefaultDepth(XF86ConfigPtr conf, int screen)
{
        XF86ConfScreenPtr pScrn ;

        pScrn= getScreenFromIndex(conf, screen);


        if (!pScrn)
        {
                printf("getScreenFromIndex failed\n");
                return 0;
        }


        return confGetDefaultDepthFromScreenPtr(conf, pScrn);
}


/* This function sets the different resolutions and depth */
XF86ConfigPtr confSetResolutionsFromScreenPtr(XF86ConfigPtr conf, XF86ConfScreenPtr pScrn, int nbRes, char** Resolutions, int colorDepth)
{
	XF86ConfDisplayPtr pDisplay;
	XF86ModePtr pConfMode;
	int i;
	BOOL bFoundDisplay = FALSE;


	if (!conf) return NULL;


	if (!pScrn) return NULL; 


	pDisplay = pScrn->scrn_display_lst;

	/* Search for a display subsection with same bpp */
	while(pDisplay)
	{
		if((pDisplay->disp_depth == colorDepth) || ((pDisplay->disp_depth == 24) &&  (colorDepth == 32)))
		{       /* if found, delete previous resolution */
			bFoundDisplay = TRUE;
			free(pDisplay->disp_mode_lst);
			pDisplay->disp_mode_lst = NULL;
			break;
		}


		pDisplay = pDisplay->list.next;
	}

	if(!bFoundDisplay)
	{       /* if not found same bpp, add a new display subsection */
		pDisplay = calloc(1, sizeof(XF86ConfDisplayRec));
		pDisplay->disp_option_lst = NULL;
		pDisplay->disp_mode_lst = NULL;
		pDisplay->disp_comment = NULL;
		pDisplay->disp_weight.red = 0;
		pDisplay->disp_weight.blue = 0;
		pDisplay->disp_weight.green = 0;
		pDisplay->disp_white.red = -1;
		pDisplay->disp_white.blue = -1;
		pDisplay->disp_white.green = -1;
		pDisplay->disp_black.red = -1;
		pDisplay->disp_black.blue = -1;
		pDisplay->disp_black.green = -1;
		pDisplay->list.next = NULL;
		pScrn->scrn_display_lst = (XF86ConfDisplayPtr) xf86addListItem( (GenericListPtr) pScrn->scrn_display_lst, \
				(GenericListPtr) pDisplay);
	}

	/* set value for the pointer (found or allocated) */
	if ((colorDepth == 8 ) || (colorDepth == 16) || (colorDepth == 24)
	|| (colorDepth == 32)) {
		if (colorDepth == 32) 
		{
			pDisplay->disp_depth = 24;
		}
		else
			pDisplay->disp_depth = colorDepth;
		}
	else
		return NULL;


	/* Write the resolutions in Display section */
	for (i= 0; i < nbRes;i ++)
	{
		pConfMode = calloc(1, sizeof(XF86ModeRec));
		pConfMode->mode_name = calloc(strlen(Resolutions[i]) + 1, sizeof(char));
		strcpy(pConfMode->mode_name, Resolutions[i]);
		pConfMode->list.next = NULL;
		pDisplay->disp_mode_lst = (XF86ModePtr)xf86addListItem(     (GenericListPtr) pDisplay->disp_mode_lst, \
				(GenericListPtr) pConfMode);
	}

	return conf;
}


/* This function sets the different resolutions and depth */
XF86ConfigPtr confSetResolutions(XF86ConfigPtr conf, int screenIndex, int nbRes, char** Resolutions, int colorDepth)
{
	XF86ConfScreenPtr pScrn;

	if (!conf) return NULL;


	pScrn = getScreenFromIndex(conf, screenIndex);


	if (!pScrn)
	{
		printf("getScreenFromIndex failed\n");
		return NULL;
	}

        return confSetResolutionsFromScreenPtr(conf,pScrn,nbRes, Resolutions, colorDepth);
}

/* This function gets the different resolutions at a specified depth */
int confGetResolutionsFromScreenPtr(XF86ConfigPtr conf, XF86ConfScreenPtr pScrn, int colorDepth, char** Resolutions, int maxnbRes)
{
	XF86ConfDisplayPtr pDisplay;
	XF86ModePtr pConfMode;
	int ResIndex;
	
        
	if (!conf) return 0;

        if (!pScrn) return 0;

	pDisplay = pScrn->scrn_display_lst;

	/* Search for a display subsection with same bpp */
	while(pDisplay)
	{
		if((pDisplay->disp_depth == colorDepth) || ((pDisplay->disp_depth == 24) &&  (colorDepth == 32)))
		{      /* we've found our pDisplay, break.*/ 
			break;
		}


		pDisplay = pDisplay->list.next;
	}

	if(!pDisplay) return 0;

	/* Read the resolutions in Display section */
	pConfMode = pDisplay->disp_mode_lst;
	ResIndex = 0;
	while((pConfMode != NULL) && (ResIndex < maxnbRes)) {
		Resolutions[ResIndex] = pConfMode->mode_name;
		ResIndex++;
		pConfMode = pConfMode->list.next;
		}


	return ResIndex;
}

/* This function gets the different resolutions at a specified depth */
int confGetResolutions(XF86ConfigPtr conf, int screenIndex, int colorDepth, char** Resolutions, int maxnbRes)
{
	XF86ConfScreenPtr pScrn;


	if (!conf) return 0;


	pScrn = getScreenFromIndex(conf, screenIndex);

	if (!pScrn)
	{
		printf("getScreenFromIndex failed\n");
		return 0;
	}

        return confGetResolutionsFromScreenPtr(conf, pScrn, colorDepth, Resolutions, maxnbRes);
}
        

/* Xinerama */
XF86ConfigPtr confDisableXinerama(XF86ConfigPtr conf)
{
	XF86OptionPtr pConfOptions, pConfOptionPrev, pConfXineramaOption;

	if(!conf) return NULL;

	if((!(confXineramaIsActive(conf))) || (!(conf->conf_flags)) || (!(conf->conf_flags->flg_option_lst)))
		return conf;

	pConfOptions = conf->conf_flags->flg_option_lst;

	if(!(pConfXineramaOption = xf86findOption(pConfOptions,
					"Xinerama"))) return conf;

	pConfOptionPrev = pConfOptions;

	while (pConfOptions)
	{
		if (pConfOptions  == pConfXineramaOption)
		{
			/* Found module to be deleted */
			/* Re-link the list around it */
			if (conf->conf_flags->flg_option_lst ==
					pConfXineramaOption)
				/* We're deleting the head */
				conf->conf_flags->flg_option_lst =
					pConfXineramaOption->list.next;
			else
				pConfOptionPrev->list.next =
					pConfOptions->list.next;

			free(pConfXineramaOption);
			break;
		}

		pConfOptionPrev = pConfOptions;


		pConfOptions = pConfOptions->list.next;
	}



	/* Load dri as Xinerama got off */
	confLoadModule(conf, "dri");

	return conf;


}


XF86ConfigPtr confEnableXinerama(XF86ConfigPtr conf)
{
	char* optXinerama;

	if(!conf) return NULL;

	if(confXineramaIsActive(conf)) return conf;

	if(!(conf->conf_flags))
		conf->conf_flags = calloc(1, sizeof(XF86ConfFlagsRec));

	optXinerama = calloc(strlen("Xinerama") + 1, sizeof(char));
	strcpy(optXinerama, "Xinerama");

	conf->conf_flags->flg_option_lst =
		xf86addNewOption(conf->conf_flags->flg_option_lst, optXinerama, NULL);


	/* Unload dri because Xinerama just got ON. */
	confUnloadModule(conf, "dri");

	return conf;
}


BOOL confXineramaIsActive(XF86ConfigPtr conf)
{
	XF86ConfFlagsPtr pConfFlags = conf->conf_flags;

	if(!conf) return FALSE;

	if((!(pConfFlags)) || (!(pConfFlags->flg_option_lst)))
		return FALSE;


	else if(xf86findOption(pConfFlags->flg_option_lst, "Xinerama") !=
			NULL)
		return TRUE;
	else
		return FALSE;


}


/* add a Modeline */
XF86ConfigPtr confAddModeline(XF86ConfigPtr conf, int screenIndex, XF86VidModeModeLine* pModeLine, int dotClock)
{
	XF86ConfScreenPtr pConfScreen = getScreenFromIndex(conf, screenIndex);
	XF86ConfModeLinePtr pModeLineList;
	XF86ConfModeLinePtr pNewConfModeLine;
	int refreshRate, hits = 0;
	char newRes[255];

	if (!pConfScreen)
	{
		printf("getScreenFromIndex failed\n");
		return NULL;
	}
	
        pModeLineList = pConfScreen->scrn_monitor->mon_modeline_lst;
	
        refreshRate = (int)(((((dotClock * 1000) / (float)(pModeLine->vtotal * pModeLine->htotal)) * 100.0) + 50) / 100);
	
        snprintf(newRes, 255, "%dx%d@%dHz:%d", pModeLine->hdisplay, pModeLine->vdisplay, refreshRate,screenIndex);
	
	/* find a unique name if this one is already used by another mode */
	
	while (xf86findModeLine(newRes, pModeLineList))
	    snprintf(newRes, 255, "%dx%d@%dHz:%d(%d)", pModeLine->hdisplay, pModeLine->vdisplay, refreshRate, screenIndex, ++hits);

	pNewConfModeLine = calloc(1, sizeof(XF86ConfModeLineRec));

	pNewConfModeLine->ml_identifier = calloc(strlen(newRes) + 1, sizeof(char));
	strcpy(pNewConfModeLine->ml_identifier, newRes);

	pNewConfModeLine->ml_clock = dotClock;
	pNewConfModeLine->ml_hdisplay = pModeLine->hdisplay;
	pNewConfModeLine->ml_hsyncstart = pModeLine->hsyncstart;
	pNewConfModeLine->ml_hsyncend = pModeLine->hsyncend;
	pNewConfModeLine->ml_htotal = pModeLine->htotal;
	pNewConfModeLine->ml_vdisplay = pModeLine->vdisplay;
	pNewConfModeLine->ml_vsyncstart = pModeLine->vsyncstart;
	pNewConfModeLine->ml_vsyncend = pModeLine->vsyncend;
	pNewConfModeLine->ml_vtotal = pModeLine->vtotal;
	pNewConfModeLine->ml_vscan = 0;
	pNewConfModeLine->ml_flags = pModeLine->flags;
	pNewConfModeLine->ml_hskew = pModeLine->hskew;


	pConfScreen->scrn_monitor->mon_modeline_lst=
	    (XF86ConfModeLinePtr)xf86addListItem((GenericListPtr) pModeLineList, (GenericListPtr) pNewConfModeLine);
	return conf;
}


/* retrieve user-defined Modelines */
XF86ConfModeLinePtr confGetUserModeLines(XF86ConfigPtr conf, int screenIndex)
{
	XF86ConfScreenPtr pConfScreen = getScreenFromIndex(conf, screenIndex);

	if (!pConfScreen)
	{
		printf("getScreenFromIndex failed\n");
		return NULL;
	}

	if(!(pConfScreen->scrn_monitor)) return NULL;

	return pConfScreen->scrn_monitor->mon_modeline_lst;
}


/* HW Cursor */
XF86ConfigPtr confEnableHWCursor (XF86ConfigPtr conf)
{

	XF86ConfDevicePtr pConfDevice, pTmpConfDevice;
	XF86OptionPtr pConfOption, pTmpConfOption;

	if(!conf) return NULL;

	pConfDevice = conf->conf_device_lst;

	/* for each device */
	for (pTmpConfDevice = pConfDevice; pTmpConfDevice; pTmpConfDevice = pTmpConfDevice->list.next)
	{
		/* if not mga, skip it. */
		if(strcmp(pTmpConfDevice->dev_driver, "mga")) continue;


		pConfOption = pTmpConfDevice->dev_option_lst;

		/* If the "hw cursor" option exist */           
		if((pTmpConfOption = xf86findOption(pTmpConfDevice->dev_option_lst, "hw cursor")))
			if(strcmp(xf86findOptionValue(pTmpConfDevice->dev_option_lst, "hw cursor"), "on"))
			{
				free(pTmpConfOption->opt_val);
				pTmpConfOption->opt_val = calloc(3, sizeof(char));
				strcpy(pTmpConfOption->opt_val,"on");
			}


		/* If the "hw cursor" do not exist, let it like that because it's ON by default */
	}


	return conf;


}


XF86ConfigPtr confDisableHWCursor (XF86ConfigPtr conf)
{


	XF86ConfDevicePtr pConfDevice, pTmpConfDevice;
	XF86OptionPtr pConfOption, pTmpConfOption;


	if(!conf) return NULL;

	pConfDevice = conf->conf_device_lst;

	for (pTmpConfDevice = pConfDevice; pTmpConfDevice; pTmpConfDevice = pTmpConfDevice->list.next)
	{
		/* if not mga, skip it. */
		if(strcmp(pTmpConfDevice->dev_driver, "mga")) continue;


		pConfOption = pTmpConfDevice->dev_option_lst;

		/* If the "hw cursor" option exist */           
		if((pTmpConfOption = xf86findOption(pTmpConfDevice->dev_option_lst, "hw cursor")))
		{
			if(strcmp(xf86findOptionValue(pTmpConfDevice->dev_option_lst, "hw cursor"), "off"))
			{
				/*free(pTmpConfOption->opt_val);*/
				pTmpConfOption->opt_val = calloc(4, sizeof(char));
				strcpy(pTmpConfOption->opt_val,"off");
			}
		}
		else /* if not then add it */
		{
			/* add the option "hw cursor" "off" */
			char* optHwCursor = calloc(strlen("hw cursor") + 1, sizeof(char));
			char* optOff = calloc(strlen("off") + 1, sizeof(char));

			strcpy(optHwCursor, "hw cursor");
			strcpy(optOff, "off");
			pTmpConfDevice->dev_option_lst = xf86addNewOption(pConfOption, optHwCursor, optOff);
		}


	}
	return NULL;
}


BOOL confHWCursorIsActive(XF86ConfigPtr conf)
{


	XF86ConfDevicePtr pConfDevice, pTmpConfDevice;
	XF86OptionPtr pConfOption, pTmpConfOption;

	if(!conf) return FALSE;

	pConfDevice = conf->conf_device_lst;

	for (pTmpConfDevice = pConfDevice; pTmpConfDevice; pTmpConfDevice = pTmpConfDevice->list.next)
	{


		if(strcmp(pTmpConfDevice->dev_driver, "mga")) continue;


		pConfOption = pTmpConfDevice->dev_option_lst;

		for(pTmpConfOption = pConfOption; pTmpConfOption; pTmpConfOption = pTmpConfOption->list.next)
		{
			if(!strcmp(pTmpConfOption->opt_name, "hw cursor"))
			{
				if(!strcmp(pTmpConfOption->opt_val, "off"))
					return FALSE;
				else if(!strcmp(pTmpConfOption->opt_val, "on"))
					return TRUE;
			}
		}
	}


	/* By default : it's ON */
	return TRUE;

}


/* TV */
XF86ConfigPtr confEnableTVout( XF86ConfigPtr conf, int screenIndex, char* tvstandard , char* cabletype)
{       

	XF86ConfScreenPtr pConfScreen;
	XF86ConfDevicePtr pConfDevice;
	XF86OptionPtr pConfOption;

	if(!conf) return NULL;

	pConfScreen = getScreenFromIndex(conf, screenIndex);
	
        if (!pConfScreen)
	{
		printf("getScreenFromIndex failed\n");
		return NULL;
	}


	pConfDevice = pConfScreen->scrn_device;

	/* If the "TV" option exist */          
	if((pConfOption = xf86findOption(pConfDevice->dev_option_lst, "TV")))
	{
		if(strcmp(xf86findOptionValue(pConfDevice->dev_option_lst, "TV"), "yes"))
		{
			free(pConfOption->opt_val);
			pConfOption->opt_val = calloc(4, sizeof(char));
			strcpy(pConfOption->opt_val,"yes");
		}
	}
	else /* if not then add it */
	{
		char* optTV = calloc(strlen("TV") + 1, sizeof(char));
		char* optValYes = calloc(strlen("yes") + 1, sizeof(char));


		strcpy(optTV, "TV");
		strcpy(optValYes, "yes");
		pConfDevice->dev_option_lst = xf86addNewOption(pConfDevice->dev_option_lst, optTV, optValYes);
	}

	/* If the "CableType" option exist */           
	if((pConfOption = xf86findOption(pConfDevice->dev_option_lst, "CableType")))
	{
		free(pConfOption->opt_val);
		pConfOption->opt_val = calloc(strlen(cabletype)+1, sizeof(char));
		strcpy(pConfOption->opt_val,cabletype);
	}
	else /* if not then add it */
	{
		char* optValCable = calloc(strlen(cabletype) + 1, sizeof(char));
		char* optCableType = calloc(strlen("CableType") + 1, sizeof(char));


		strcpy(optCableType, "CableType");
		strcpy(optValCable, cabletype);
		pConfDevice->dev_option_lst = xf86addNewOption(pConfDevice->dev_option_lst, optCableType, optValCable);
	}


	/* If the "TVStandard" option exist */
	if((pConfOption = xf86findOption(pConfDevice->dev_option_lst, "TVStandard")))
	{
		free(pConfOption->opt_val);
		pConfOption->opt_val = calloc(strlen(tvstandard) + 1, sizeof(char));
		strcpy(pConfOption->opt_val,tvstandard);
	}
	else /* if not then add it */
	{
		char* optValStandard = calloc(strlen(tvstandard) + 1, sizeof(char));
		char* optStandard = calloc(strlen("TVStandard") + 1, sizeof(char));


		strcpy(optStandard, "TVStandard");
		strcpy(optValStandard, tvstandard);
		pConfDevice->dev_option_lst = xf86addNewOption(pConfDevice->dev_option_lst, optStandard, optValStandard);
	}


	return conf;


}


XF86ConfigPtr confDisableTVout (XF86ConfigPtr conf, int screenIndex)
{

	XF86ConfDevicePtr pConfDevice;
	XF86ConfScreenPtr pConfScreen;
	XF86OptionPtr pConfOption;

	if(!conf) return NULL;

	if(!confTVIsActive(conf, screenIndex)) return conf;

	pConfScreen = getScreenFromIndex(conf, screenIndex);

	if (!pConfScreen)
	{
		printf("getScreenFromIndex failed\n");
		return NULL;
	}
	pConfDevice = pConfScreen->scrn_device;

	/* If the "TV" option exist */          
	if((pConfOption = xf86findOption(pConfDevice->dev_option_lst, "TV")))
	{
		if(strcmp(xf86findOptionValue(pConfDevice->dev_option_lst, "TV"), "no"))
		{
			free(pConfOption->opt_val);
			pConfOption->opt_val = calloc(3, sizeof(char));
			strcpy(pConfOption->opt_val,"no");
		}
	}


	return conf;


}


BOOL confTVIsActive(XF86ConfigPtr conf, int screenIndex)
{

	XF86ConfDevicePtr pConfDevice;
	XF86ConfScreenPtr pConfScreen;
	XF86OptionPtr pConfOption;

	if(!conf) return FALSE;

	pConfScreen = getScreenFromIndex(conf, screenIndex);
	if (!pConfScreen)
	{
		printf("getScreenFromIndex failed\n");
		return FALSE;
	}
	pConfDevice = pConfScreen->scrn_device;

	/* If the "TV" option exist */      
	if((pConfOption = xf86findOption(pConfDevice->dev_option_lst, "TV")))
	{
		if(!strcmp(xf86findOptionValue(pConfDevice->dev_option_lst, "TV"), "yes"))
		{
			return TRUE;
		}
	}


	return FALSE;


}


char* confGetTVStandard(XF86ConfigPtr conf, int screenIndex)
{
	XF86ConfScreenPtr pConfScreen;
	XF86ConfDevicePtr pConfDevice;
	XF86OptionPtr pConfOption;


	if(!conf) return (char*)NULL;


	pConfScreen = getScreenFromIndex(conf, screenIndex);
	if (!pConfScreen)
	{
		printf("getScreenFromIndex failed\n");
		return NULL;
	}
	pConfDevice = pConfScreen->scrn_device;


	if((pConfOption = xf86findOption(pConfDevice->dev_option_lst, "TVStandard")))
	{
		return xf86findOptionValue(pConfDevice->dev_option_lst, "TVStandard");
	}
	else
		return (char*)NULL;


}


char* confGetCableType(XF86ConfigPtr conf, int screenIndex)
{
	XF86ConfScreenPtr pConfScreen;
	XF86ConfDevicePtr pConfDevice;
	XF86OptionPtr pConfOption;


	if(!conf) return (char*)NULL;


	pConfScreen = getScreenFromIndex(conf, screenIndex);
	if (!pConfScreen)
	{
		printf("getScreenFromIndex failed\n");
		return NULL;
	}
	pConfDevice = pConfScreen->scrn_device;


	if((pConfOption = xf86findOption(pConfDevice->dev_option_lst, "CableType")))
	{
		return xf86findOptionValue(pConfDevice->dev_option_lst, "CableType");
	}
	else
		return (char*)NULL;


}


/* DigitalScreen */
XF86ConfigPtr confEnableDigitalScreen( XF86ConfigPtr conf, int screenIndex)
{       
	XF86ConfScreenPtr pConfScreen;
	XF86ConfDevicePtr pConfDevice;
	XF86OptionPtr pConfOption;

	if(!conf) return NULL;

	pConfScreen = getScreenFromIndex(conf, screenIndex);

	if (!pConfScreen)
	{
		printf("getScreenFromIndex failed\n");
		return NULL;
	}

	pConfDevice = pConfScreen->scrn_device;

	/* If the "DigitalScreen" option exist */       
	if((pConfOption = xf86findOption(pConfDevice->dev_option_lst, "DigitalScreen")))
	{
		if(strcmp(xf86findOptionValue(pConfDevice->dev_option_lst, "DigitalScreen"), "yes"))
		{
			free(pConfOption->opt_val);
			pConfOption->opt_val = calloc(4, sizeof(char));
			strcpy(pConfOption->opt_val,"yes");
		}
	}
	else /* if not then add it */
	{
		char* optValYes = calloc(strlen("yes") + 1, sizeof(char));
		char* optDigital = calloc(strlen("DigitalScreen") + 1, sizeof(char));


		strcpy(optDigital, "DigitalScreen");
		strcpy(optValYes, "yes");
		pConfDevice->dev_option_lst = xf86addNewOption(pConfDevice->dev_option_lst, optDigital, optValYes);
	}


	return conf;
}


XF86ConfigPtr confDisableDigitalScreen(XF86ConfigPtr conf, int screenIndex)
{
	XF86ConfDevicePtr pConfDevice;
	XF86ConfScreenPtr pConfScreen;
	XF86OptionPtr pConfOption;

	if(!conf) return NULL;

	if(!confDigitalScreenIsActive(conf, screenIndex)) return conf;

	pConfScreen = getScreenFromIndex(conf, screenIndex);
	
        if (!pConfScreen)
	{
		printf("getScreenFromIndex failed\n");
		return NULL;
	}

	pConfDevice = pConfScreen->scrn_device;

	/* If the "DigitalScreen" option exist */       
	if((pConfOption = xf86findOption(pConfDevice->dev_option_lst, "DigitalScreen")))
	{
		if(strcmp(xf86findOptionValue(pConfDevice->dev_option_lst, "DigitalScreen"), "no"))
		{
			/*free(pConfOption->opt_val); */
			pConfOption->opt_val = calloc(3, sizeof(char));
			strcpy(pConfOption->opt_val,"no");
		}
	}


	return conf;
}


BOOL confDigitalScreenIsActive(XF86ConfigPtr conf, int screenIndex)
{
	XF86ConfDevicePtr pConfDevice;
	XF86ConfScreenPtr pConfScreen;
	XF86OptionPtr pConfOption;

	if(!conf) return FALSE;

	pConfScreen = getScreenFromIndex(conf, screenIndex);
	
        if (!pConfScreen)
	{
		printf("getScreenFromIndex failed\n");
		return FALSE;
	}
	pConfDevice = pConfScreen->scrn_device;

	/* If the "DigitalScreen" option exist */
	if((pConfOption = xf86findOption(pConfDevice->dev_option_lst, "DigitalScreen")))
	{
		if(!strcmp(xf86findOptionValue(pConfDevice->dev_option_lst, "DigitalScreen"), "yes"))
		{
			return TRUE;
		}
	}


	return FALSE;
}


BOOL confValidateLayout (XF86ConfigPtr p)
{
	XF86ConfLayoutPtr layout = p->conf_layout_lst;
	XF86ConfAdjacencyPtr adj;
	XF86ConfInactivePtr iptr;
	XF86ConfScreenPtr screen;
	XF86ConfDevicePtr device;


	while (layout)
	{
		adj = layout->lay_adjacency_lst;
		while (adj)
		{
			/* the first one can't be "" but all others can */
			screen = xf86findScreen (adj->adj_screen_str, p->conf_screen_lst);
			if (!screen)
			{
				printf("Invalid Layout\n");
				return (FALSE);
			}
			else
				adj->adj_screen = screen;


			adj = adj->list.next;
		}

		iptr = layout->lay_inactive_lst;

		while (iptr)
		{
			device = xf86findDevice (iptr->inactive_device_str, p->conf_device_lst);
			if (!device)
			{
				printf("Bad Layout. Something wrong with device\n");
				return (FALSE);
			}
			else
				iptr->inactive_device = device;


			iptr = iptr->list.next;

		}

		layout = layout->list.next;

	}


#ifdef DEBUG
	printf("Layout is valid\n");
#endif


	return (TRUE);
}


/* Gives the ServerLayout */
/* This routine sets the where variable easily
 * But uses the getIndexFromScreen function to return the proper index
 * */
XF86ConfigPtr confGetServerLayout(XF86ConfigPtr conf, int* scrn1, int* where, int* scrn2 ){


	XF86ConfLayoutPtr pLayout; 
	XF86ConfAdjacencyPtr pAdj;
	XF86ConfScreenPtr pScreen;

	pLayout = conf->conf_layout_lst;
	pAdj = pLayout->lay_adjacency_lst;
	pScreen = pAdj->adj_screen;


#ifdef DEBUG
	printf("Entering: confGetServerLayout\n");
#endif


	/* Loops trought the Adjacency list. If there's a ref screen
	 * return that line*/
	while(pAdj)
	{

		/* Checks to see if theire is a refscreen */
		if (pAdj->adj_refscreen != NULL)
		{
			*where = pAdj->adj_where;
			if(scrn1) *scrn1 = getIndexFromScreen(conf, pAdj->adj_screen_str);
			if(scrn2) *scrn2 = getIndexFromScreen(conf, pAdj->adj_refscreen);
			break;
		}
                if (strcmp(pAdj->adj_screen_str, "Display Merged") == 0) 
                {
                    char* sval = xf86findOptionValue(pScreen->scrn_option_lst, "Monitor2Position");
		    if(scrn1) *scrn1 = 0;
		    if(scrn2) *scrn2 = 1;
                    
                    if(sval == NULL) {
                        *where = CONF_ADJ_RIGHTOF;
                        break;
                    }

                    switch(sval[0]) 
                    {
                        case 'L': case 'l': case 'G': case 'g':
                            *where = CONF_ADJ_RIGHTOF;
                            break;
                        case 'R': case 'r': case 'D': case 'd':
                            *where = CONF_ADJ_LEFTOF;
                            break;

                        case 'A': case 'a': case 'H': case 'h':
                            *where = CONF_ADJ_BELOW;
                            break;

                        case 'B': case 'b':
                            *where = CONF_ADJ_ABOVE;
                            break;

                        case 'C': case 'c':
                        default:
                            *where = CONF_ADJ_LEFTOF;
                            break;
                    }
                    break;
                    
                }

		pAdj = pAdj->list.next;

	}

	return conf;


}


/* This routine renames all the screens in the config file */


BOOL confRenameScreen(XF86ConfigPtr conf, XF86ConfScreenPtr pScrn, char *name)
{
	XF86ConfLayoutPtr pLayout;
	XF86ConfAdjacencyPtr pAdj; 
	pLayout= conf->conf_layout_lst;


	if (conf == NULL || pScrn == NULL) 
		return (False);


	while (pLayout != NULL) {
		pAdj = pLayout->lay_adjacency_lst;


		while (pAdj != NULL) {
			if (pAdj->adj_screen == pScrn) {
				replaceString(&pAdj->adj_screen_str, name);
			}
			else if (pAdj->adj_top == pScrn) {
				replaceString(&pAdj->adj_top_str, name);
			}
			else if (pAdj->adj_bottom == pScrn) {
				replaceString(&pAdj->adj_bottom_str, name);
			}
			else if (pAdj->adj_left == pScrn) {
				replaceString(&pAdj->adj_left_str, name);
			}
			else if (pAdj->adj_right == pScrn) {
				replaceString(&pAdj->adj_right_str, name);
			}
			else if (pAdj->adj_refscreen != NULL &&
					strcasecmp(pAdj->adj_refscreen, name) == 0) {
				replaceString(&pAdj->adj_refscreen, name);
			}


			pAdj = (XF86ConfAdjacencyPtr)(pAdj->list.next);
		}
		pLayout = (XF86ConfLayoutPtr)(pLayout->list.next);
	}


	replaceString(&pScrn->scrn_identifier, name);


	return (True);
}


/* This routine renames all the devices in the config file */
BOOL confRenameDevice(XF86ConfigPtr conf, XF86ConfDevicePtr pDevice, char *name)
{
	XF86ConfScreenPtr pScrn;
	pScrn= conf->conf_screen_lst;


	if (conf == NULL || pDevice == NULL) 
		return (False);


	while (pScrn != NULL) {
		if (pScrn->scrn_device == pDevice) {
			replaceString(&pScrn->scrn_device_str , name);
		}


		pScrn = (XF86ConfScreenPtr)(pScrn->list.next);
	}


	replaceString(&pDevice->dev_identifier , name);


	return (True);
}

/* MergedFB */

XF86ConfigPtr confDisableMergedFB(XF86ConfigPtr conf)
{
	XF86ConfScreenPtr pScrn;
	XF86ConfMonitorPtr pMonitor;
	XF86ConfDevicePtr pDevice;
	XF86ConfLayoutPtr pLayout;
	XF86ConfAdjacencyPtr pAdj;

	XF86OptionPtr pConfOptions, pConfOptionPrev, pConfMergedFBOption;

	if(!conf) return NULL;

	if((!(confMergedFBIsActive(conf))) || (!(conf->conf_screen_lst)) )
		return conf;

	pDevice = MakeDevice(conf,ediDevice1);
	pMonitor = MakeMonitor(conf,ediDisplay1);
	pScrn = MakeScreen(conf,ediDisplay1,pDevice,pMonitor);
        
        if(!pScrn) return conf;

        /* remove MergedFB option from Display 1 if one exists */
        pConfOptions = pScrn->scrn_option_lst;

	if((pConfMergedFBOption = xf86findOption(pConfOptions,
					"MergedFB"))) 
        {
            pConfOptionPrev = pConfOptions;
            while (pConfOptions)
            {
                    if (pConfOptions  == pConfMergedFBOption)
                    {
                            /* Found module to be deleted */
                            /* Re-link the list around it */
                            if (pScrn->scrn_option_lst ==
                                            pConfMergedFBOption)
                                    /* We're deleting the head */
                                    pScrn->scrn_option_lst =
                                            pConfMergedFBOption->list.next;
                            else
                                    pConfOptionPrev->list.next =
                                            pConfOptions->list.next;
                            free(pConfMergedFBOption);
                            break;
                    }
                    pConfOptionPrev = pConfOptions;
                    pConfOptions = pConfOptions->list.next;
            }
        }

        /* Reset layout */	
	pLayout = conf->conf_layout_lst;
	pAdj = pLayout->lay_adjacency_lst;

        pAdj->adj_screen = pScrn;
	replaceString(&pAdj->adj_screen_str, pScrn->scrn_identifier);

	free(pAdj->adj_refscreen);
	pAdj->adj_refscreen = NULL;
	pAdj->adj_where = 0;
	free((XF86ConfAdjacencyPtr)pAdj->list.next);
	pAdj->list.next = NULL;

	return conf;


}


XF86ConfigPtr confEnableMergedFB(XF86ConfigPtr conf, int where)
{
	XF86ConfScreenPtr pScrn, pNewScrn, pMergedScrn;
	XF86ConfDevicePtr pDevice, pNewDevice;
	XF86ConfMonitorPtr pMonitor, pNewMonitor, pMergedMonitor;
	XF86ConfLayoutPtr pLayout;
	XF86ConfAdjacencyPtr pAdj;
	char* optMergedFB;

        if(!conf) return NULL;

        /* setting twice does no harm, and we need to bypass this check on startup */
	/*if(confMergedFBIsActive(conf)) return conf;*/

	if(!(conf->conf_screen_lst))
                return NULL;
                
        /* Make sections */
        pDevice = MakeDevice(conf,ediDevice1);
	pMonitor = MakeMonitor(conf,ediDisplay1);
	pScrn = MakeScreen(conf,ediDisplay1,pDevice,pMonitor);

	pNewDevice = MakeDevice(conf,ediDevice2);
	pNewMonitor = MakeMonitor(conf,ediDisplay2);
	pNewScrn = MakeScreen(conf,ediDisplay2,pNewDevice,pNewMonitor);

	pMergedMonitor = MakeMonitor(conf, ediDisplayMerged);
	pMergedScrn = MakeScreen(conf, ediDisplayMerged, pDevice, pMergedMonitor);
	
        pDevice->dev_option_lst = removeOption(pDevice->dev_option_lst, "NoHal");
	pNewDevice->dev_option_lst = removeOption(pNewDevice->dev_option_lst, "NoHal");

        /* Set layout */
	pLayout = conf->conf_layout_lst;
	pAdj = pLayout->lay_adjacency_lst;

        pAdj->adj_screen = pMergedScrn;
	replaceString(&pAdj->adj_screen_str, pMergedScrn->scrn_identifier);

        if(pAdj->adj_refscreen)
            free(pAdj->adj_refscreen);
	pAdj->adj_refscreen = NULL;
	pAdj->adj_where = 0;
	if(pAdj->list.next)
            free((XF86ConfAdjacencyPtr)pAdj->list.next);
	pAdj->list.next = NULL;
	

        /* set layout */
        conf = confSetLayout(conf,where);

        /*Merged FB option ON */
        
        optMergedFB = calloc(strlen("MergedFB") + 1, sizeof(char));
	strcpy(optMergedFB, "MergedFB");

	pMergedScrn->scrn_option_lst =
		xf86addNewOption(pMergedScrn->scrn_option_lst, optMergedFB, NULL);

	return conf;
}


BOOL confMergedFBIsActive(XF86ConfigPtr conf)
{
	XF86ConfScreenPtr pConfScreen = FindScreen(conf,ediDisplayMerged);
	XF86ConfLayoutPtr pLayout;
	XF86ConfAdjacencyPtr pAdj;

	if(!conf) return FALSE;

	if((!(pConfScreen)) || (!(pConfScreen->scrn_option_lst)))
		return FALSE;
	
        pLayout = conf->conf_layout_lst;
	pAdj = pLayout->lay_adjacency_lst;

	return (strcmp( pAdj->adj_screen_str, "Display Merged") == 0 ) &&
	        (xf86findOption(pConfScreen->scrn_option_lst, "MergedFB") != NULL);
                 
}

#define MIN(x,y) (x > y ? y : x)
#define MAX(x,y) (x < y ? y : x)

void FitDesk(int* rx, int* ry, int where, int x1, int y1, int x2, int y2) 
{
    switch(where) 
    {
        case CONF_ADJ_LEFTOF :
        case CONF_ADJ_RIGHTOF :
            *rx = MAX(*rx,x1+x2);
            *ry = MAX(*ry,MAX(y1,y2));
            break;

        case CONF_ADJ_ABOVE :
        case CONF_ADJ_BELOW :
            *rx = MAX(*rx,MAX(x1,x2));
            *ry = MAX(*ry,y1+y2);
            break;
        default :
            *rx = MAX(*rx,MAX(x1,x2));
            *ry = MAX(*ry,MAX(y1,y2));
    }
}


/* 
 * confGenerateMetaModes
 * 
 * Alright, this does a little more than generates meta-modes.
 * it takes the 2 displays, and "merges" them in a "Merged Display"
 *
 * to be called at least right before writing the config file.
 */
typedef struct pair {
    int x;
    int y;
} pairrec;

XF86ConfigPtr confGenerateMetaModes(XF86ConfigPtr conf)
{
#define MAXRESOLUTIONS 200
	char* optMetaModes;
        char* sMetaModes;
	char*	asRes1[MAXRESOLUTIONS];
	pairrec	apRes1[MAXRESOLUTIONS];
        int iNumRes1;
	char*	asRes2[MAXRESOLUTIONS];
	pairrec	apRes2[MAXRESOLUTIONS];
        int iNumRes2;
        int i,j,i1,i2;
        BOOL NewRes;
        int xvport = 0;
        int yvport = 0;
        int where;
        
        XF86ConfScreenPtr pScrn,pScrn1,pScrn2;
	XF86ConfMonitorPtr pMonitor,pMonitor1,pMonitor2;
	XF86ConfDisplayPtr pDisplay;
        
        
	if(!conf) return NULL;

	if(!confMergedFBIsActive(conf)) return conf;

	if(!(conf->conf_screen_lst))
            return NULL;
        
	pMonitor = FindMonitor(conf, ediDisplayMerged);
	pMonitor1 = FindMonitor(conf, ediDisplay1);
	pMonitor2 = FindMonitor(conf, ediDisplay2);
	pScrn = FindScreen(conf, ediDisplayMerged);
        pScrn1 = FindScreen(conf, ediDisplay1);
        pScrn2 = FindScreen(conf, ediDisplay2);
       
        if(!pMonitor || !pMonitor1 || !pMonitor2 || !pScrn || !pScrn1 || !pScrn2)
            return NULL;

        confGetServerLayout(conf,NULL,&where,NULL);

        iNumRes1 = confGetResolutionsFromScreenPtr(conf, pScrn1, confGetDefaultDepthFromScreenPtr(conf, pScrn1), asRes1, MAXRESOLUTIONS-1);
        iNumRes2 = confGetResolutionsFromScreenPtr(conf, pScrn2, confGetDefaultDepthFromScreenPtr(conf, pScrn2), asRes2, MAXRESOLUTIONS-1);

        /* duplicate last element to simplify "Make dual modes" loop*/
        asRes1[iNumRes1] = asRes1[iNumRes1-1];
        asRes2[iNumRes2] = asRes2[iNumRes2-1];
        
        for(i = 0; i <= iNumRes1; i++) 
            sscanf(asRes1[i], " %dx%d", &apRes1[i].x, &apRes1[i].y);
        for(i = 0; i <= iNumRes2; i++) 
            sscanf(asRes2[i], " %dx%d", &apRes2[i].x, &apRes2[i].y);
	
        
        optMetaModes = calloc(strlen("MetaModes") + 1, sizeof(char));
	strcpy(optMetaModes, "MetaModes");

        sMetaModes = calloc(1024+1, sizeof(char));
        
        /* Make dual modes */
        i1 = i2 = 0;
        while(!(i1 >= iNumRes1  &&  i2 >= iNumRes2))
        {
            /* add mode to list */
            strcat(sMetaModes, asRes1[i1]);
            strcat(sMetaModes, "-");
            strcat(sMetaModes, asRes2[i2]);
            strcat(sMetaModes, " ");

            /* find virtual desktop size */
            FitDesk(&xvport,&yvport,where, apRes1[i1].x,
                                           apRes1[i1].y,
                                           apRes2[i2].x,
                                           apRes2[i2].y);
            /* figure out next mode */
            if( i1 == iNumRes1 )
                i2++;
            else if( i2 == iNumRes2 )
                i1++;
            else if( apRes1[i1].x * apRes1[i1].y == apRes2[i2].x * apRes2[i2].y )
                i1++, i2++;
            else if( abs(apRes1[i1].x * apRes1[i1].y - apRes2[i2].x * apRes2[i2].y) >
                     abs(apRes1[i1+1].x * apRes1[i1+1].y - apRes2[i2].x * apRes2[i2].y) ) /* i1+1 closer to i2 */
                i1++;
            else if( abs(apRes1[i1].x * apRes1[i1].y - apRes2[i2].x * apRes2[i2].y) >
                     abs(apRes1[i1].x * apRes1[i1].y - apRes2[i2+1].x * apRes2[i2+1].y) ) /* i2+1 closer to i1 */
                i2++;
            else { /*nothing beter close, pass to next*/
                if(i1 != iNumRes1) i1++;
                if(i2 != iNumRes2) i2++;
            }
        }
        
        /* Make single cloned modes */
        for(i = 0; i < iNumRes1; i++) 
            for(j = 0; j < iNumRes2; j++) 
                if( strcmp(asRes1[i], asRes2[j]) == 0 ) {
                    char *r1;
                    int x1,y1;
                    
                    strcat(sMetaModes, r1 = asRes1[i]);
                    strcat(sMetaModes, " ");
                    
                    /* find virtual desktop size */
                    sscanf(r1," %dx%d",&x1,&y1);
                    FitDesk(&xvport,&yvport,0,x1,y1,x1,y1);
                }

	printf("Metamodes: %s \n",sMetaModes);
        pScrn->scrn_option_lst =
		xf86addNewOption(pScrn->scrn_option_lst, optMetaModes, sMetaModes);

        /* set viewports */
        for(pDisplay = pScrn->scrn_display_lst; pDisplay != NULL; pDisplay = pDisplay->list.next) {
            pDisplay->disp_virtualX = xvport;
            pDisplay->disp_virtualY = yvport;
        }
        
        /* Merge Resolutions */
        for(i = 0; i < iNumRes2; i++) 
        {
            NewRes = TRUE;
            for(j = 0; j < iNumRes1; j++) 
                NewRes &= (strcmp(asRes1[j], asRes2[i]) != 0);
            if(NewRes) 
            {
                asRes1[iNumRes1] = asRes2[i];
                iNumRes1++;
            }
        }
            
        confSetResolutionsFromScreenPtr(conf, pScrn, iNumRes1, asRes1, confGetDefaultDepthFromScreenPtr(conf, pScrn));


        /* Merge modelines */
        {
        XF86ConfModeLinePtr ml,newml;
        /* free list */            
        while(pMonitor->mon_modeline_lst != NULL) 
        {
            ml = pMonitor->mon_modeline_lst->list.next;
            free(pMonitor->mon_modeline_lst);
            pMonitor->mon_modeline_lst = ml;
        }

        for(ml = pMonitor1->mon_modeline_lst; ml != NULL; ml = ml->list.next) 
        {
            newml = malloc(sizeof(*newml));
            memcpy(newml,ml,sizeof(*ml));
            newml->list.next = NULL;
            newml->ml_identifier = NULL;
	    replaceString(&newml->ml_identifier, ml->ml_identifier);
            
            /* Add it the list */
            pMonitor->mon_modeline_lst = (XF86ConfModeLinePtr) xf86addListItem ( (GenericListPtr) pMonitor->mon_modeline_lst, (GenericListPtr) newml );
        }
        
        for(ml = pMonitor2->mon_modeline_lst; ml != NULL; ml = ml->list.next) 
        {
            newml = malloc(sizeof(*newml));
            memcpy(newml,ml,sizeof(*ml));
            newml->list.next = NULL;
            newml->ml_identifier = NULL;
	    replaceString(&newml->ml_identifier, ml->ml_identifier);

            
            /* Add it the list */
            pMonitor->mon_modeline_lst = (XF86ConfModeLinePtr) xf86addListItem ( (GenericListPtr) pMonitor->mon_modeline_lst, (GenericListPtr) newml );
        }
        
        } /* end Merge modelines */

    
	/* Set up monitor 2's Hsync and Vrefresh */
        {
            char sacc[256] = "";

            for(i = 0; i < pMonitor2->mon_n_hsync; i++) {
                sprintf(sacc + strlen(sacc), "%.1f-%.1f ", pMonitor2->mon_hsync[i].lo, pMonitor2->mon_hsync[i].hi);
            }
            
            pScrn->scrn_option_lst = addOption(pScrn->scrn_option_lst,"Monitor2HSync",sacc);
            
            sacc[0] = 0;
            for(i = 0; i < pMonitor2->mon_n_vrefresh; i++) {
                sprintf(sacc + strlen(sacc), "%.1f-%.1f ", pMonitor2->mon_vrefresh[i].lo, pMonitor2->mon_vrefresh[i].hi);
            }
            
            pScrn->scrn_option_lst = addOption(pScrn->scrn_option_lst,"Monitor2VRefresh",sacc);
        }
            
                
        return conf;
}

/* called at startup to put config file in a sane state */
XF86ConfigPtr confMakeSections ( XF86ConfigPtr conf)
{
    XF86ConfScreenPtr pScrn;
    XF86ConfDevicePtr pDevice;
    XF86ConfMonitorPtr pMonitor;
    XF86ConfLayoutPtr pLayout;
    XF86ConfAdjacencyPtr pAdj;
    
    pDevice = MakeDevice(conf,ediDevice1);
    pMonitor = MakeMonitor(conf,ediDisplay1);
    pScrn = MakeScreen(conf,ediDisplay1,pDevice,pMonitor);
    pLayout = conf->conf_layout_lst;
    pAdj = pLayout->lay_adjacency_lst;
    
    replaceString(&pAdj->adj_screen_str, pScrn->scrn_identifier);
    pAdj->adj_screen =  pScrn;
    
    return conf;
}
