/*
 * Written by Bastien Chevreux (BaCh)
 *
 * Copyright (C) 1997-2000 by the German Cancer Research Center (Deutsches
 *   Krebsforschungszentrum, DKFZ Heidelberg) and Bastien Chevreux
 * Copyright (C) 2000 and later by Bastien Chevreux
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the 
 * Free Software Foundation, Inc., 
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 * 
 */

// 	$Id$

#ifndef _mira_multitag_h_
#define _mira_multitag_h_


#include <string>

#include "stdinc/stlincludes.H"

#include "stdinc/defines.H"
#include "errorhandling/errorhandling.H"

#include "io/exp.H"
#include "mira/stringcontainer.H"



class multitag_t {
public:
  typedef StringContainer<uint8>::Entry mte_src_t;
  typedef StringContainer<uint16>::Entry mte_id_t;
  typedef StringContainer<uint32>::Entry mte_co_t;

private:
  // string container for multitag tagsource, identifier and comments
  // tagsource = either MIRA, GenBank (when read from gbk) or read from GFF
  // identifier = id of the tag (gap4-IDs with MIRA additions)
  // comment = free string
  static StringContainer<uint8> MT_sc_mttagsource;
  static StringContainer<uint16> MT_sc_mtidentifier;
  static StringContainer<uint32> MT_sc_mtcomment;

  //uint32 MT_rtid;
  //uint16 MT_order;



public:
  static const mte_src_t MT_tagsrcentry_idEmpty;
  static const mte_src_t MT_tagsrcentry_idMIRA;
  static const mte_src_t MT_tagsrcentry_idGenBank;
  static const mte_src_t MT_tagsrcentry_idGFF3;

public:
  uint32    from;                  // including 
  uint32    to;                    // including
  mte_co_t  comment;               // comment to that tag
  mte_id_t  identifier;            // gap4 identifier (extended by mira)
  mte_src_t source;                // where does the tag come from?
  char      strand;                // +, -, = means both

public:

  multitag_t() {
    from=0;
    to=0;
    strand='=';
    comment=MT_sc_mtcomment.addEntry("");
    identifier=MT_sc_mtidentifier.addEntry("");
    source=MT_tagsrcentry_idEmpty;
    //cout << "mt init" << endl;
  };

  //multitag_t const & operator=(tag_t const & other);
  multitag_t const & operator=(tag_t const & simpletag) {
    //MT_rtid=0;
    //MT_order=0;
    from=simpletag.from;
    to=simpletag.to;
    strand=simpletag.strand;
    comment=multitag_t::newComment(simpletag.comment);
    identifier=multitag_t::newIdentifier(simpletag.identifier);
    // tag_t has no source, set to empty
    source=MT_tagsrcentry_idEmpty;
    return *this;
  };

  friend ostream & operator<<(ostream &ostr, const multitag_t &t){
    ostr << "From: " << t.from <<  endl;
    ostr << "To: " << t.to << endl;
    ostr << "Strand: " << t.strand << endl;
    //ostr << "Identifier: " << t.identifier << endl;
    //ostr << "Comment: " << t.comment << endl;
    //ostr << "Source: " << t.source << endl;
    return ostr;
  }

  inline const string & getIdentifierStr() const {return MT_sc_mtidentifier.getEntry(identifier);}
  inline static const string & getIdentifierStr(mte_id_t e) {return MT_sc_mtidentifier.getEntry(e);}
  inline const string & getCommentStr() const {return MT_sc_mtcomment.getEntry(comment);}
  inline static const string & getCommentStr(mte_co_t e) {return MT_sc_mtcomment.getEntry(e);}
  inline const string & getSourceStr() const {return MT_sc_mttagsource.getEntry(source);}
  inline static const string & getSourceStr(mte_src_t e) {return MT_sc_mttagsource.getEntry(e);}

  // get that from tag_t
  inline void serialiseComment(string & result) const {
    tag_t::serialiseComment(getCommentStr(),result);
  }

  inline bool extractGenBankKeyValueFromComment(const string & whatkey,
					 string & result) const {
    return tag_t::extractGenBankKeyValueFromComment(getCommentStr(), whatkey, result);
  };

  inline static mte_id_t newIdentifier(const char * s) {
    //cout << "nI*: " << s << endl;
    return MT_sc_mtidentifier.addEntry(s);
  };
  inline static mte_id_t newIdentifier(const string & s) {
    //cout << "nIs: " << s << endl;
    return newIdentifier(s.c_str());
  };
  inline static mte_id_t hasIdentifier(const char * s) {
    //cout << "nI*: " << s << endl;
    return MT_sc_mtidentifier.hasEntry(s);
  };
  inline static mte_id_t hasIdentifier(const string & s) {
    //cout << "nIs: " << s << endl;
    return hasIdentifier(s.c_str());
  };
  inline static mte_co_t newComment(const char * s) {
    //cout << "nC*: " << s << endl;
    return MT_sc_mtcomment.addEntry(s);
  };
  inline static mte_co_t newComment(const string & s) {
    //cout << "nCs: " << s << endl;
    return newComment(s.c_str());
  };
  inline static mte_src_t newSource(const char * s) {
    //cout << "nC*: " << s << endl;
    return MT_sc_mttagsource.addEntry(s);
  };
  inline static mte_src_t newSource(const string & s) {
    //cout << "nCs: " << s << endl;
    return newSource(s.c_str());
  };
  
};

#endif
