// header_config.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


// methods for protected Header::ConfigRequester class and its subclasses

#ifdef __GNUG__
#pragma implementation
#endif

#include "localdefs.h"
#include "header.h"
#include "pvocdata.h"
#include "pvocheader.h"
#include "query_templates.h"
#include "range.h"
#include "request.h"
#include "formatrequester.h"
#include "sound.h"
#include "soundheader.h"

Header::ConfigRequester::ConfigRequester(const char* title, Header* header)
		: TitledRequester(title),
		  myHeader(header), byteOrder(NotSwapped) {
	myHeader->setRaw();	// set to raw mode to skip all disk reads
	myHeader->setDataOffset(0);	// assume no header to start with
}

void
Header::ConfigRequester::configureRequest(Request* request) {
	const char* label = "Bytes to skip before reading samples:";
	request->appendValue(
#ifndef __GNUG__
		createQuery(
			label,
			myHeader,
			&Header::setDataOffset,
			&Header::dataOffset,
			NonNegativeIntegers,
			false
		)
#else
		new Query<int>(
			label,
			new MethodSetterGetter<Header, boolean, int>(
				myHeader,
				&Header::setDataOffset,
				&Header::dataOffset
			),
			NonNegativeIntegers,
			false
		)
#endif
	);
	request->appendChoice("Swap bytes after reading?", "|Yes|No|",
			     &byteOrder, true);
}

boolean
Header::ConfigRequester::confirmValues() {
	myHeader->forceByteSwapping(byteOrder == Swapped);
	return true;
}

//********

SoundHeader::SoundConfigRequester::SoundConfigRequester(Header* header)
	: ConfigRequester(nil, header),
	  FormatRequester(nil, "Sample format:",
			  ((SoundHeader *)header)->data_type),
	  TitledRequester("Read Raw Soundfile:") {}

void
SoundHeader::SoundConfigRequester::configureRequest(Request* request) {
	SoundHeader* sh = (SoundHeader *) myHeader;
	request->appendValue("Sample rate:", &sh->samprate, PositiveIntegers);
	request->appendValue("Channels:", &sh->nchans, PositiveIntegers);
	FormatRequester::configureRequest(request);
	ConfigRequester::configureRequest(request);
}

//********

void
FrameDataHeader::FrameConfigRequester::configureRequest(Request* request) {
	FrameDataHeader* fh = (FrameDataHeader *) myHeader;
	request->appendValue("Frame rate (frames/sec):", &fh->framerate,
			    PositiveNumbers);
	request->appendValue("Source sample rate:", &fh->samprate,
	                    PositiveNumbers);
}

//********

void
LPCHeader::LPCConfigRequester::configureRequest(Request* request) {
	LPCHeader* lh = (LPCHeader *) myHeader;
	request->appendValue("Number of filter poles:", &lh->npoles,
			    Range(12, 64), true);
	Super::configureRequest(request);
}

boolean
LPCHeader::LPCConfigRequester::confirmValues() {
	if(Super::confirmValues()) {
		LPCHeader* lh = (LPCHeader *) myHeader;
		lh->nchans = lh->npoles + 4;
		return true;
	}
	return false;
}

//********

void
FFTHeader::FFTConfigRequester::configureRequest(Request* request) {
	FFTHeader* fh = (FFTHeader *) myHeader;
	request->appendValue("Number of frequency points:", &fh->nchans,
			    PositiveIntegers);
	Super::configureRequest(request);
}

//********

PvocHeader::PvocConfigRequester::PvocConfigRequester(Header* header)
	: Header::ConfigRequester("Read Raw Pvoc Datafile:", header),
	  freqBands(PvocData::defaultNumberOfBands() >> 6) {}

void
PvocHeader::PvocConfigRequester::configureRequest(Request* request) {
	PvocHeader* pvh = (PvocHeader *) myHeader;
	request->appendValue("Frame rate (frames/sec):", &pvh->samprate,
			    PositiveNumbers);
	request->appendChoice("Number of frequency points (N/2):",
		"|64|128|256|512|1024|2048|4096|16384|", &freqBands, true);
	Super::configureRequest(request);
}

boolean
PvocHeader::PvocConfigRequester::confirmValues() {
	if(Super::confirmValues()) {
		PvocHeader* pvh = (PvocHeader *) myHeader;
		pvh->nchans = ((freqBands << 6) + 1) * 2;
		return true;
	}
	return false;
}
