# -------------------------------------------------------------------------
#     This file is part of mMass - the spectrum analysis tool for MS.
#     Copyright (C) 2005-07 Martin Strohalm <mmass@biographics.cz>

#     This program is free software; you can redistribute it and/or modify
#     it under the terms of the GNU General Public License as published by
#     the Free Software Foundation; either version 2 of the License, or
#     (at your option) any later version.

#     This program is distributed in the hope that it will be useful,
#     but WITHOUT ANY WARRANTY; without even the implied warranty of
#     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#     GNU General Public License for more details.

#     Complete text of GNU GPL can be found in the file LICENSE in the
#     main directory of the program
# -------------------------------------------------------------------------

# Function: Mascot search tools.

# load libs
import wx
import urllib
import re
import string
import os.path

# load modules
from dlg_search import dlgMascotSearch
from params import parameters


class mMascotSearch:
    """ Mascot search tools. """

    # ----
    def __init__(self, parent, document):

        self.parent = parent
        self.config = parent.config
        self.docMonitor = parent.docMonitor
        self.docData = document
    # ----


    # ----
    def search(self, searchType):
        """ Raise protein mass fingerprint dialog. """

        # get title
        title = os.path.basename(self.docData.getPath())
        scanID = self.docData.getScanID()
        if scanID:
            title = '%s (scan #%s)' % (title, scanID)

        # get peaklist data
        peaklist = self.getConvertedPeaklist()

        # get parameters from mascot server
        parameters = self.getMascotParams()

        # raise dialog
        dlg = dlgMascotSearch(self.parent, searchType, title, peaklist, parameters, self.config.cfg['mascot'])
        dlg.ShowModal()
        dlg.Destroy()
    # ----


    # ----
    def getConvertedPeaklist(self):
        """ Get peaklist from document and convert to text. """

        peaklist = ''
        digitsFormat = '%0.' + `self.config.cfg['common']['digits']` + 'f'
        for peak in self.docData.getPeaks():
            mass = digitsFormat % peak[0]
            intensity = digitsFormat % peak[1]
            peaklist += '%s %s\n' % (mass, intensity)

        return peaklist
    # ----


    # ----
    def getMascotParams(self):
        """ Get parameters from MASCOT server. """

        # get parameter file
        self.docMonitor('setAppStatus', 'Connecting to Mascot server...')
        try:
            paramFile = urllib.urlopen(self.config.cfg['mascot']['urlparams'])

        except IOError:
            message = "Unable to get search parameters from Mascot server.\nDefault parameters will be used."
            dlg = wx.MessageDialog(self.parent, message, "Connection Failed", wx.OK|wx.ICON_EXCLAMATION)
            dlg.ShowModal()
            dlg.Destroy()

            self.docMonitor('setAppStatus', 0)
            return parameters

        self.docMonitor('setAppStatus', 0)

        # clear params
        parameters['database'] = []
        parameters['mods'] = ['none']
        parameters['enzyme'] = []
        parameters['taxonomy'] = []
        parameters['instrument'] = []

        # parse parameter file
        secPatt = re.compile('^\[([a-zA-Z_]*)\]$')
        section = ''
        for line in paramFile.readlines():
            line = string.strip(line)
            if line:

                # get section header
                match = secPatt.match(line)
                if match:
                    section = match.group(1)

                # get data
                elif section == 'DB':
                    parameters['database'].append(line)
                elif section == 'MODS':
                    parameters['mods'].append(line)
                elif section == 'CLE':
                    parameters['enzyme'].append(line)
                elif section == 'TAXONOMY':
                    parameters['taxonomy'].append(line)
                elif section == 'INSTRUMENT':
                    parameters['instrument'].append(line)

        return parameters
    # ----
