# -------------------------------------------------------------------------
#     This file is part of mMass - the spectrum analysis tool for MS.
#     Copyright (C) 2005-07 Martin Strohalm <mmass@biographics.cz>

#     This program is free software; you can redistribute it and/or modify
#     it under the terms of the GNU General Public License as published by
#     the Free Software Foundation; either version 2 of the License, or
#     (at your option) any later version.

#     This program is distributed in the hope that it will be useful,
#     but WITHOUT ANY WARRANTY; without even the implied warranty of
#     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#     GNU General Public License for more details.

#     Complete text of GNU GPL can be found in the file LICENSE in the
#     main directory of the program
# -------------------------------------------------------------------------

# Function: Notebook pages for application preferences.

# load libs
import wx
import string

# load modules
from nucleus import mwx


class mMassPage(wx.Panel):
    """General controls"""

    # ----
    def __init__(self, parent, data):
        wx.Panel.__init__(self, parent, -1)

        self.data = data

        # pack main frame
        if wx.Platform == '__WXMAC__':
            margin = 10
        else:
            margin = 5

        col2Sizer = wx.BoxSizer(wx.VERTICAL)
        col2Sizer.Add(self.makeMatchingBox(), 0, wx.EXPAND|wx.BOTTOM, margin)
        col2Sizer.Add(self.makePrintBox(), 1, wx.EXPAND, 0)

        row1Sizer = wx.BoxSizer(wx.HORIZONTAL)
        row1Sizer.Add(self.makeMassBox(), 1, wx.EXPAND|wx.ALL, margin)
        row1Sizer.Add(col2Sizer, 1, wx.EXPAND|wx.ALL, margin)

        row2Sizer = wx.BoxSizer(wx.HORIZONTAL)
        row2Sizer.Add(self.makeDescriptionBox(), 1, wx.EXPAND|wx.LEFT|wx.RIGHT|wx.BOTTOM, margin)

        label = wx.StaticText(self, -1, "General application parameters")
        mainSizer = wx.BoxSizer(wx.VERTICAL)
        mainSizer.Add(label, 0, wx.ALL|wx.ALIGN_CENTER, 10)
        mainSizer.Add(row1Sizer, 0, wx.EXPAND, 0)
        mainSizer.Add(row2Sizer, 0, wx.EXPAND, 0)
        self.SetSizer(mainSizer)
    # ----


    # ----
    def getData(self):
        """ Check and get data from page. """

        self.data['nomatchalert'] = 0
        self.data['showmatchinfo'] = 0
        self.data['masstype'] = 'mmass'
        self.data['updatedescr'] = 0

        # get mass type, charge, and error type
        if self.massAv_radio.GetValue():
            self.data['masstype'] = 'amass'
        self.data['charge'] = self.massCharge_combo.GetValue()
        self.data['errortype'] = self.massErrorType_combo.GetValue()

        # get digits
        digits = self.massDigits_value.GetValue()
        try:
            self.data['digits'] = int(digits)
        except ValueError:
            pass

        # get tolerance
        tolerance = self.massTolerance_value.GetValue()
        tolerance = tolerance.replace(',', '.')
        try:
            self.data['tolerance'] = float(tolerance)
        except ValueError:
            pass

        # get data matching
        if self.matchNoMatchAlert_check.IsChecked():
            self.data['nomatchalert'] = 1
        if self.matchShowInfo_check.IsChecked():
            self.data['showmatchinfo'] = 1

        # get digits
        printFilter = self.printFilter_value.GetValue()
        try:
            self.data['printfilter'] = int(printFilter)
        except ValueError:
            pass

        # get description
        self.data['operator'] = self.descrOperator_value.GetValue()
        self.data['contact'] = self.descrContact_value.GetValue()
        self.data['institution'] = self.descrInstitution_value.GetValue()
        self.data['instrument'] = self.descrInstrument_value.GetValue()
        if self.descrAutoUpdate_check.IsChecked():
            self.data['updatedescr'] = 1

        return self.data
    # ----


    # ----
    def makeMassBox(self):
        """ Box for mass and error. """

        # make items
        mainBox = wx.StaticBoxSizer(wx.StaticBox(self, -1, "Startup Mass and Tolerance"), wx.VERTICAL)
        grid = mwx.GridBagSizer()

        self.massMo_radio = wx.RadioButton(self, -1, "Monoisotopic mass", style=wx.RB_GROUP)
        self.massAv_radio = wx.RadioButton(self, -1, "Average mass")

        massCharge_choices = ['M', '1+', '2+', '1-', '[2-']
        massCharge_label = wx.StaticText(self, -1, "Charge: ")
        self.massCharge_combo = wx.ComboBox(self, -1, size=(85, -1), choices=massCharge_choices, style=wx.CB_READONLY)

        massTolerance_label = wx.StaticText(self, -1, "Tolerance: ")
        self.massTolerance_value = wx.TextCtrl(self, -1, str(self.data['tolerance']), size=(85, -1), validator=mwx.txtValidator('float'))

        massErrorType_choices = ['Da', 'ppm', '%']
        massErrorType_label = wx.StaticText(self, -1, "Tol. units: ")
        self.massErrorType_combo = wx.ComboBox(self, -1, size=(85, -1), choices=massErrorType_choices, style=wx.CB_READONLY)

        massDigits_label = wx.StaticText(self, -1, "Digits: ")
        self.massDigits_value = wx.SpinCtrl(self, -1, str(self.data['digits']), size=(83, -1), min=0, max=6, style=wx.SP_ARROW_KEYS)

        # pack items
        grid.Add(massCharge_label, (0, 0), flag=wx.ALIGN_CENTER_VERTICAL|wx.ALIGN_RIGHT)
        grid.Add(self.massCharge_combo, (0, 1))
        grid.Add(massTolerance_label, (1, 0), flag=wx.ALIGN_CENTER_VERTICAL|wx.ALIGN_RIGHT)
        grid.Add(self.massTolerance_value, (1, 1))
        grid.Add(massErrorType_label, (2, 0), flag=wx.ALIGN_CENTER_VERTICAL|wx.ALIGN_RIGHT)
        grid.Add(self.massErrorType_combo, (2, 1))
        grid.Add(massDigits_label, (3, 0), flag=wx.ALIGN_CENTER_VERTICAL|wx.ALIGN_RIGHT)
        grid.Add(self.massDigits_value, (3, 1))
        
        mainBox.Add(self.massMo_radio, 0, wx.TOP|wx.LEFT|wx.RIGHT, 5)
        mainBox.Add(self.massAv_radio, 0, wx.ALL, 5)
        mainBox.Add(grid, 0, wx.ALL, 5)

        # set defaults
        if self.data['masstype'] == 'mmass':
            self.massMo_radio.SetValue(True)
        else:
            self.massAv_radio.SetValue(True)
        self.massCharge_combo.Select(massCharge_choices.index(self.data['charge']))
        self.massErrorType_combo.Select(massErrorType_choices.index(self.data['errortype']))

        return mainBox
    # ----


    # ----
    def makeMatchingBox(self):
        """ Make box for data matching options. """

        # make items
        mainBox = wx.StaticBoxSizer(wx.StaticBox(self, -1, "Data Matching"), wx.VERTICAL)

        self.matchNoMatchAlert_check = wx.CheckBox(self, -1, "Warn if no match")
        self.matchShowInfo_check = wx.CheckBox(self, -1, "Show match info")

        # pack items
        mainBox.Add(self.matchNoMatchAlert_check, 0, wx.TOP|wx.LEFT|wx.RIGHT, 5)
        mainBox.Add(self.matchShowInfo_check, 0, wx.ALL, 5)

        # set defaults
        self.matchNoMatchAlert_check.SetValue(self.data['nomatchalert'])
        self.matchShowInfo_check.SetValue(self.data['showmatchinfo'])

        return mainBox
    # ----


    # ----
    def makePrintBox(self):
        """ Box for print quality. """

        # make items
        mainBox = wx.StaticBoxSizer(wx.StaticBox(self, -1, "Printing"), wx.VERTICAL)
        grid = mwx.GridBagSizer()

        printFilter_label = wx.StaticText(self, -1, "Spectrum filter: ")
        self.printFilter_value = wx.SpinCtrl(self, -1, str(self.data['printfilter']), size=(70, -1), min=0, max=6, style=wx.SP_ARROW_KEYS)
        self.printFilter_value.SetToolTip(wx.ToolTip("Lower value means better quality"))

        # pack items
        grid.Add(printFilter_label, (0, 0), flag=wx.ALIGN_CENTER_VERTICAL)
        grid.Add(self.printFilter_value, (0, 1))
        mainBox.Add(grid, 0, wx.TOP|wx.LEFT|wx.RIGHT, 5)

        return mainBox
    # ----


    # ----
    def makeDescriptionBox(self):
        """ Box for document description. """

        # make items
        mainBox = wx.StaticBoxSizer(wx.StaticBox(self, -1, "Default Document Description"), wx.VERTICAL)
        grid = mwx.GridBagSizer()

        operator_label = wx.StaticText(self, -1, "Operator: ")
        self.descrOperator_value = wx.TextCtrl(self, -1, unicode(self.data['operator']), size=(250, -1))

        contact_label = wx.StaticText(self, -1, "Contact: ")
        self.descrContact_value = wx.TextCtrl(self, -1, unicode(self.data['contact']), size=(250, -1))

        institution_label = wx.StaticText(self, -1, "Institution: ")
        self.descrInstitution_value = wx.TextCtrl(self, -1, unicode(self.data['institution']), size=(250, -1))

        instrument_label = wx.StaticText(self, -1, "Instrument: ")
        self.descrInstrument_value = wx.TextCtrl(self, -1, unicode(self.data['instrument']), size=(250, -1))

        self.descrAutoUpdate_check = wx.CheckBox(self, -1, "Paste description automatically to imported file")

        # pack items
        grid.Add(operator_label, (0, 0), flag=wx.ALIGN_CENTER_VERTICAL|wx.ALIGN_RIGHT)
        grid.Add(self.descrOperator_value, (0, 1), flag=wx.EXPAND)
        grid.Add(contact_label, (1, 0), flag=wx.ALIGN_CENTER_VERTICAL|wx.ALIGN_RIGHT)
        grid.Add(self.descrContact_value, (1, 1), flag=wx.EXPAND)
        grid.Add(institution_label, (2, 0), flag=wx.ALIGN_CENTER_VERTICAL|wx.ALIGN_RIGHT)
        grid.Add(self.descrInstitution_value, (2, 1), flag=wx.EXPAND)
        grid.Add(instrument_label, (3, 0), flag=wx.ALIGN_CENTER_VERTICAL|wx.ALIGN_RIGHT)
        grid.Add(self.descrInstrument_value, (3, 1), flag=wx.EXPAND)
        grid.AddGrowableCol(1)

        mainBox.Add(grid, 0, wx.EXPAND|wx.ALL, 5)
        mainBox.Add(self.descrAutoUpdate_check, 0, wx.ALL, 5)

        # set defaults
        self.descrAutoUpdate_check.SetValue(self.data['updatedescr'])

        return mainBox
    # ----


class mPeakPage(wx.Panel):
    """Controls for mPeak module"""

    # ----
    def __init__(self, parent, data):
        wx.Panel.__init__(self, parent, -1)

        self.data = data

        # pack main frame
        if wx.Platform == '__WXMAC__':
            margin = 10
        else:
            margin = 5

        row1Sizer = wx.BoxSizer(wx.HORIZONTAL)
        row1Sizer.Add(self.makeLayoutBox(), 1, wx.EXPAND|wx.ALL, margin)
        row1Sizer.Add(self.makeTresholdBox(), 1, wx.EXPAND|wx.ALL, margin)

        label = wx.StaticText(self, -1, "Peaklist panel settings")
        mainSizer = wx.BoxSizer(wx.VERTICAL)
        mainSizer.Add(label, 0, wx.TOP|wx.ALL|wx.ALIGN_CENTER, 10)
        mainSizer.Add(row1Sizer, 0, wx.EXPAND)
        mainSizer.Add(self.makeCopyBox(), 0, wx.EXPAND|wx.LEFT|wx.RIGHT|wx.BOTTOM, margin)
        self.SetSizer(mainSizer)
    # ----


    # ----
    def getData(self):
        """ Check and get data from page. """

        self.data['showpeaklist'] = 0
        self.data['splitting'] = 'v'
        self.data['copymass'] = 0
        self.data['copyintens'] = 0
        self.data['copyannot'] = 0

        # peaklist
        if self.layoutShowPeaklist_check.GetValue():
            self.data['showpeaklist'] = 1
        if self.layoutSplitHorizontally_radio.GetValue():
            self.data['splitting'] = 'h'

        # get copy params
        if self.copyMass_check.IsChecked():
            self.data['copymass'] = 1
        if self.copyIntensity_check.IsChecked():
            self.data['copyintens'] = 1
        if self.copyAnnotations_check.IsChecked():
            self.data['copyannot'] = 1

        # get delimiter
        if self.delimiterTab_radio.GetValue():
            self.data['delimiter'] = 'tab'
        else:
            self.data['delimiter'] = self.delimiter_value.GetValue()

        # get treshold
        treshold = self.tresholdValue_value.GetValue()
        self.data['thdirection'] = self.tresholdDirection_combo.GetValue()
        try:
            self.data['treshold'] = int(treshold)
        except ValueError:
            pass

        return self.data
    # ----


    # ----
    def makeLayoutBox(self):
        """ Make box for application layout options. """

        # make items
        mainBox = wx.StaticBoxSizer(wx.StaticBox(self, -1, "Peaklist Layout"), wx.VERTICAL)

        self.layoutShowPeaklist_check = wx.CheckBox(self, -1, "Show peaklist window at startup")
        self.layoutSplitVertically_radio = wx.RadioButton(self, -1, "Peaklist left", style=wx.RB_GROUP)
        self.layoutSplitHorizontally_radio = wx.RadioButton(self, -1, "Peaklist bottom")

        # pack items
        mainBox.Add(self.layoutShowPeaklist_check, 0, wx.TOP|wx.LEFT|wx.RIGHT, 5)
        mainBox.Add(self.layoutSplitVertically_radio, 0, wx.TOP|wx.LEFT|wx.RIGHT, 5)
        mainBox.Add(self.layoutSplitHorizontally_radio, 0, wx.ALL, 5)

        # set defaults
        self.layoutShowPeaklist_check.SetValue(self.data['showpeaklist'])
        if self.data['splitting'] == 'h':
            self.layoutSplitHorizontally_radio.SetValue(True)
        else:
            self.layoutSplitVertically_radio.SetValue(True)

        return mainBox
    # ----


    # ----
    def makeCopyBox(self):
        """ Make box for clipboard options. """

        # make items
        mainBox = wx.StaticBoxSizer(wx.StaticBox(self, -1, "Clipboard/Export Options"), wx.VERTICAL)
        delimBox = wx.BoxSizer(wx.HORIZONTAL)

        self.copyMass_check = wx.CheckBox(self, -1, "Copy mass")
        self.copyIntensity_check = wx.CheckBox(self, -1, "Copy intensity")
        self.copyAnnotations_check = wx.CheckBox(self, -1, "Copy annotations")

        self.delimiterTab_radio = wx.RadioButton(self, -1, "Delimit by \"Tab\"", style=wx.RB_GROUP)
        self.delimiterValue_radio = wx.RadioButton(self, -1, "Delimit by value: ")
        self.delimiter_value = wx.TextCtrl(self, -1, '', size=(50, -1))

        # pack items
        delimBox.Add(self.delimiterValue_radio, 0, wx.TOP|wx.RIGHT, 5)
        delimBox.Add(self.delimiter_value, 0, wx.RIGHT, 5)

        mainBox.Add(self.copyMass_check, 0, wx.TOP|wx.LEFT|wx.RIGHT, 5)
        mainBox.Add(self.copyIntensity_check, 0, wx.TOP|wx.LEFT|wx.RIGHT, 5)
        mainBox.Add(self.copyAnnotations_check, 0, wx.TOP|wx.LEFT|wx.RIGHT|wx.BOTTOM, 5)
        mainBox.Add(self.delimiterTab_radio, 0, wx.TOP|wx.LEFT|wx.RIGHT, 5)
        mainBox.Add(delimBox, 0, wx.LEFT|wx.RIGHT|wx.BOTTOM, 5)

        # set defaults
        self.copyMass_check.SetValue(self.data['copymass'])
        self.copyIntensity_check.SetValue(self.data['copyintens'])
        self.copyAnnotations_check.SetValue(self.data['copyannot'])
        if not self.data['delimiter'] or self.data['delimiter'] == 'tab':
            self.delimiterTab_radio.SetValue(True)
        else:
            self.delimiterValue_radio.SetValue(True)
            self.delimiter_value.SetValue(str(self.data['delimiter']))

        return mainBox
    # ----


    # ----
    def makeTresholdBox(self):
        """ Make box for treshold params. """

        # make items
        mainBox = wx.StaticBoxSizer(wx.StaticBox(self, -1, "Default Treshold Values"), wx.VERTICAL)
        grid = mwx.GridBagSizer()

        tresholdValue_label = wx.StaticText(self, -1, "Intensity: ")
        self.tresholdValue_value = wx.TextCtrl(self, -1, str(self.data['treshold']), size=(85, -1), validator=mwx.txtValidator('int'))
        tresholdDirection_label = wx.StaticText(self, -1, "Direction: ")
        self.tresholdDirection_combo = wx.ComboBox(self, -1, size=(85, -1), choices=('above', 'below'), style=wx.CB_READONLY)

        # pack items
        grid.Add(tresholdValue_label, (0, 0), flag=wx.ALIGN_CENTER_VERTICAL|wx.ALIGN_RIGHT)
        grid.Add(self.tresholdValue_value, (0, 1))
        grid.Add(tresholdDirection_label, (1, 0), flag=wx.ALIGN_CENTER_VERTICAL|wx.ALIGN_RIGHT)
        grid.Add(self.tresholdDirection_combo, (1, 1))

        mainBox.Add(grid, 0, wx.LEFT|wx.RIGHT|wx.BOTTOM, 5)

        # set defaults
        self.tresholdDirection_combo.Select(0)
        if self.data['thdirection'] == 'below':
            self.tresholdDirection_combo.Select(1)

        return mainBox
    # ----


class mSpecPage(wx.Panel):
    """Controls for mSpec module"""

    # ----
    def __init__(self, parent, data):
        wx.Panel.__init__(self, parent, -1)

        self.data = data

        # pack main frame
        if wx.Platform == '__WXMAC__':
            margin = 10
        else:
            margin = 5

        row1Sizer = wx.BoxSizer(wx.HORIZONTAL)
        row1Sizer.Add(self.makeLabelsBox(), 1, wx.EXPAND|wx.ALL, margin)
        row1Sizer.Add(self.makeSpectrumBox(), 1, wx.EXPAND|wx.ALL, margin)

        row2Sizer = wx.BoxSizer(wx.HORIZONTAL)
        row2Sizer.Add(self.makeCursorBox(), 1, wx.EXPAND|wx.LEFT|wx.RIGHT|wx.BOTTOM, margin)
        row2Sizer.Add(self.makePeakPickingBox(), 1, wx.EXPAND|wx.LEFT|wx.RIGHT|wx.BOTTOM, margin)
        
        label = wx.StaticText(self, -1, "Default parameters for mass spectrum")
        mainSizer = wx.BoxSizer(wx.VERTICAL)
        mainSizer.Add(label, 0, wx.ALL|wx.ALIGN_CENTER, 10)
        mainSizer.Add(row1Sizer, 0, wx.EXPAND, 0)
        mainSizer.Add(row2Sizer, 0, wx.EXPAND, 0)
        mainSizer.Add(self.makePeaklistImportBox(), 0, wx.EXPAND|wx.LEFT|wx.RIGHT|wx.BOTTOM, margin)
        self.SetSizer(mainSizer)
    # ----


    # ----
    def getData(self):
        """ Check and get data from page. """

        self.data['importpeaklist'] = 0
        self.data['showlabels'] = 0
        self.data['showannots'] = 0
        self.data['showticks'] = 0
        self.data['showtracker'] = 0
        self.data['showposition'] = 0
        self.data['showdistance'] = 0
        self.data['showgrid'] = 0
        self.data['showlegend'] = 0
        self.data['showposbar'] = 0
        self.data['showgelview'] = 0

        if self.showLabels_check.IsChecked():
            self.data['showlabels'] = 1
        if self.showAnnots_check.IsChecked():
            self.data['showannots'] = 1
        if self.showTicks_check.IsChecked():
            self.data['showticks'] = 1
        if self.showTracker_check.IsChecked():
            self.data['showtracker'] = 1
        if self.showPosition_check.IsChecked():
            self.data['showposition'] = 1
        if self.showDistance_check.IsChecked():
            self.data['showdistance'] = 1
        if self.showGrid_check.IsChecked():
            self.data['showgrid'] = 1
        if self.showLegend_check.IsChecked():
            self.data['showlegend'] = 1
        if self.showPosBar_check.IsChecked():
            self.data['showposbar'] = 1
        if self.showGelView_check.IsChecked():
            self.data['showgelview'] = 1

        labelAngle = self.labelAngle_combo.GetValue()
        try:
            self.data['labelangle'] = int(labelAngle)
        except ValueError:
            pass

        gelViewHeight = self.gelViewHeight_spin.GetValue()
        try:
            self.data['gelviewheight'] = int(gelViewHeight)
        except ValueError:
            pass

        peakHeight = self.peakHeight_value.GetValue()
        try:
            self.data['peakheight'] = int(peakHeight)
        except ValueError:
            pass

        if self.autoImportPeaklist_check.IsChecked():
            self.data['importpeaklist'] = 1

        return self.data
    # ----


    # ----
    def makeLabelsBox(self):
        """ Make box for label view options. """

        # make items
        mainBox = wx.StaticBoxSizer(wx.StaticBox(self, -1, "Label options"), wx.VERTICAL)
        grid = mwx.GridBagSizer()

        self.showTicks_check = wx.CheckBox(self, -1, "Show label ticks")
        self.showLabels_check = wx.CheckBox(self, -1, "Show labels")
        self.showAnnots_check = wx.CheckBox(self, -1, "Show annotations")

        labelAngle_choices = ['0', '45', '90']
        labelAngle_label = wx.StaticText(self, -1, "Label angle: ")
        self.labelAngle_combo = wx.ComboBox(self, -1, str(self.data['labelangle']), size=(60, -1), choices=labelAngle_choices, style=wx.CB_READONLY)
        
        # pack items
        grid.Add(labelAngle_label, (0, 0), flag=wx.ALIGN_CENTER_VERTICAL)
        grid.Add(self.labelAngle_combo, (0, 1))

        mainBox.Add(self.showTicks_check, 0, wx.TOP|wx.LEFT|wx.RIGHT, 5)
        mainBox.Add(self.showLabels_check, 0, wx.TOP|wx.LEFT|wx.RIGHT, 5)
        mainBox.Add(self.showAnnots_check, 0, wx.ALL, 5)
        mainBox.Add(grid, 0, wx.ALL, 5)

        # set defaults
        self.showTicks_check.SetValue(self.data['showticks'])
        self.showLabels_check.SetValue(self.data['showlabels'])
        self.showAnnots_check.SetValue(self.data['showannots'])

        return mainBox
    # ----


    # ----
    def makeSpectrumBox(self):
        """ Make box for spectrum options. """

        # make items
        mainBox = wx.StaticBoxSizer(wx.StaticBox(self, -1, "Spectrum options"), wx.VERTICAL)
        grid = mwx.GridBagSizer()

        self.showGrid_check = wx.CheckBox(self, -1, "Show grid")
        self.showLegend_check = wx.CheckBox(self, -1, "Show legend")
        self.showPosBar_check = wx.CheckBox(self, -1, "Show position bar")
        self.showGelView_check = wx.CheckBox(self, -1, "Show gel view")

        gelViewHeight_label = wx.StaticText(self, -1, "Gel Height: ")
        gelViewHeight_unit = wx.StaticText(self, -1, " px")
        self.gelViewHeight_spin = wx.SpinCtrl(self, -1, str(self.data['gelviewheight']), size=(83, -1), min=5, max=100, style=wx.SP_WRAP|wx.SP_ARROW_KEYS)
        self.gelViewHeight_spin.SetToolTip(wx.ToolTip("Use odd numbers only"))

        # pack items
        grid.Add(gelViewHeight_label, (0, 0), flag=wx.ALIGN_CENTER_VERTICAL)
        grid.Add(self.gelViewHeight_spin, (0, 1))
        grid.Add(gelViewHeight_unit, (0, 2), flag=wx.ALIGN_CENTER_VERTICAL)

        mainBox.Add(self.showGrid_check, 0, wx.TOP|wx.LEFT|wx.RIGHT, 5)
        mainBox.Add(self.showLegend_check, 0, wx.TOP|wx.LEFT|wx.RIGHT, 5)
        mainBox.Add(self.showPosBar_check, 0, wx.TOP|wx.LEFT|wx.RIGHT, 5)
        mainBox.Add(self.showGelView_check, 0, wx.ALL, 5)
        mainBox.Add(grid, 0, wx.ALL, 5)

        # set defaults
        self.showGrid_check.SetValue(self.data['showgrid'])
        self.showLegend_check.SetValue(self.data['showlegend'])
        self.showPosBar_check.SetValue(self.data['showposbar'])
        self.showGelView_check.SetValue(self.data['showgelview'])

        return mainBox
    # ----


    # ----
    def makeCursorBox(self):
        """ Make box for cursor options. """

        # make items
        mainBox = wx.StaticBoxSizer(wx.StaticBox(self, -1, "Cursor options"), wx.VERTICAL)

        self.showTracker_check = wx.CheckBox(self, -1, "Show cursor tracker")
        self.showPosition_check = wx.CheckBox(self, -1, "Show cursor position")
        self.showDistance_check = wx.CheckBox(self, -1, "Show cursor distance")

        # pack items
        mainBox.Add(self.showTracker_check, 0, wx.TOP|wx.LEFT|wx.RIGHT, 5)
        mainBox.Add(self.showPosition_check, 0, wx.TOP|wx.LEFT|wx.RIGHT, 5)
        mainBox.Add(self.showDistance_check, 0, wx.ALL, 5)

        # set defaults
        self.showTracker_check.SetValue(self.data['showtracker'])
        self.showPosition_check.SetValue(self.data['showposition'])
        self.showDistance_check.SetValue(self.data['showdistance'])

        return mainBox
    # ----


    # ----
    def makePeakPickingBox(self):
        """ Make box for peak picking options. """

        # make items
        mainBox = wx.StaticBoxSizer(wx.StaticBox(self, -1, "Peak Picking"), wx.VERTICAL)
        grid = mwx.GridBagSizer()

        peakHeight_label = wx.StaticText(self, -1, "Height: ")
        peakHeight_unit = wx.StaticText(self, -1, " %")
        self.peakHeight_value = wx.SpinCtrl(self, -1, str(self.data['peakheight']), size=(83, -1), min=1, max=100, style=wx.SP_WRAP|wx.SP_ARROW_KEYS)

        # pack items
        grid.Add(peakHeight_label, (0, 0), flag=wx.ALIGN_CENTER_VERTICAL)
        grid.Add(self.peakHeight_value, (0, 1))
        grid.Add(peakHeight_unit, (0, 2), flag=wx.ALIGN_CENTER_VERTICAL)

        mainBox.Add(grid, 0, wx.ALL, 5)

        return mainBox
    # ----


    # ----
    def makePeaklistImportBox(self):
        """ Make box for peaklist importing options. """

        # make items
        mainBox = wx.StaticBoxSizer(wx.StaticBox(self, -1, "Additional Options"), wx.VERTICAL)
        self.autoImportPeaklist_check = wx.CheckBox(self, -1, "Import peaklist to 'Compare Peaklists' tool")

        # pack items
        mainBox.Add(self.autoImportPeaklist_check, 0, wx.ALL, 5)

        # set defaults
        self.autoImportPeaklist_check.SetValue(self.data['importpeaklist'])

        return mainBox
    # ----


class mCutPage(wx.Panel):
    """Controls for mCut module"""

    # ----
    def __init__(self, parent, data, enzymes):
        wx.Panel.__init__(self, parent, -1)

        self.data = data
        self.enzymes = enzymes

        # pack main frame
        if wx.Platform == '__WXMAC__':
            margin = 10
        else:
            margin = 5

        row1Sizer = wx.BoxSizer(wx.HORIZONTAL)
        row1Sizer.Add(self.makeDigestBox(), 1, wx.EXPAND|wx.ALL, margin)
        row1Sizer.Add(self.makeViewBox(), 1, wx.EXPAND|wx.ALL, margin)

        row2Sizer = wx.BoxSizer(wx.HORIZONTAL)
        row2Sizer.Add(self.makeMassBox(), 1, wx.EXPAND|wx.LEFT|wx.RIGHT|wx.BOTTOM, margin)

        # pack main frame
        label = wx.StaticText(self, -1, "Default parameters for protein digest")
        mainSizer = wx.BoxSizer(wx.VERTICAL)
        mainSizer.Add(label, 0, wx.ALL|wx.ALIGN_CENTER, 10)
        mainSizer.Add(row1Sizer, 0, wx.EXPAND, 0)
        mainSizer.Add(row2Sizer, 0, wx.EXPAND, 0)
        self.SetSizer(mainSizer)
    # ----


    # ----
    def getData(self):
        """ Check and get data from page. """

        self.data['optmodif'] = 0
        self.data['notcleavemodif'] = 0
        self.data['uselimits'] = 0
        self.data['highlightmods'] = 0

        # get digest params
        self.data['enzyme'] = self.digestEnzyme_combo.GetValue()
        self.data['partials'] = self.digestPartials_value.GetValue()
        if self.digestOptModif_check.IsChecked():
            self.data['optmodif'] = 1
        if self.digestNotCleaveModif_check.IsChecked():
            self.data['notcleavemodif'] = 1

        # get view params
        sortBy = self.viewSortBy_combo.GetValue()
        if sortBy == 'Range':
            self.data['sortby'] = 1
        elif sortBy == 'Partials':
            self.data['sortby'] = 2
        elif sortBy == 'Mass':
            self.data['sortby'] = 3
        elif sortBy == 'Sequence':
            self.data['sortby'] = 4

        if self.viewHighlightModif_check.IsChecked():
            self.data['highlightmods'] = 1

        # get mass limits
        minmass = self.massMinMass_value.GetValue()
        maxmass = self.massMaxMass_value.GetValue()
        try:
            self.data['minmass'] = int(minmass)
        except ValueError:
            pass
        try:
            self.data['maxmass'] = int(maxmass)
        except ValueError:
            pass

        # get mass params
        self.data['charge'] = self.massCharge_combo.GetValue()
        if self.massUseLimits_check.IsChecked():
            self.data['uselimits'] = 1

        return self.data
    # ----


    # ----
    def makeDigestBox(self):
        """ Make box for digest options. """

        # get enzymes
        choicesEnzyme = []
        for enz in self.enzymes:
            choicesEnzyme.append(enz)
        choicesEnzyme.sort()

        # make items
        mainBox = wx.StaticBoxSizer(wx.StaticBox(self, -1, "Digest Options"), wx.VERTICAL)
        grid = mwx.GridBagSizer()

        digestEnzyme_label = wx.StaticText(self, -1, "Enzyme: ")
        self.digestEnzyme_combo = wx.ComboBox(self, -1, size=(84, -1), choices=choicesEnzyme, style=wx.CB_READONLY)

        digestPartials_label = wx.StaticText(self, -1, "Partials: ")
        self.digestPartials_value = wx.SpinCtrl(self, -1, '', size=(83, -1), min=0, max=self.data['maxpartials'], initial=0, style=wx.SP_WRAP|wx.SP_ARROW_KEYS )
        self.digestPartials_value.SetToolTip(wx.ToolTip("Number of missed digest sites"))

        self.digestOptModif_check = wx.CheckBox(self, -1, "Optional modifications")
        self.digestNotCleaveModif_check = wx.CheckBox(self, -1, "Don't cleave if modified")

        # pack items
        grid.Add(digestEnzyme_label, (0, 0), flag=wx.ALIGN_CENTER_VERTICAL|wx.ALIGN_RIGHT)
        grid.Add(self.digestEnzyme_combo, (0, 1))
        grid.Add(digestPartials_label, (1, 0), flag=wx.ALIGN_CENTER_VERTICAL|wx.ALIGN_RIGHT)
        grid.Add(self.digestPartials_value, (1, 1))

        mainBox.Add(grid, 0, wx.TOP|wx.LEFT|wx.RIGHT, 5)
        mainBox.Add(self.digestOptModif_check, 0, wx.TOP|wx.LEFT|wx.RIGHT, 5)
        mainBox.Add(self.digestNotCleaveModif_check, 0, wx.ALL, 5)

        # set defaults
        try:
            self.digestEnzyme_combo.Select(choicesEnzyme.index(self.data['enzyme']))
        except ValueError:
            self.digestEnzyme_combo.Select(0)
        self.digestPartials_value.SetValue(self.data['partials'])
        self.digestOptModif_check.SetValue(self.data['optmodif'])
        self.digestNotCleaveModif_check.SetValue(self.data['notcleavemodif'])

        return mainBox
    # ----


    # ----
    def makeViewBox(self):
        """ Make box for view options. """

        # make items
        mainBox = wx.StaticBoxSizer(wx.StaticBox(self, -1, "View Options"), wx.VERTICAL)
        sortSizer = wx.BoxSizer(wx.HORIZONTAL)

        choicesSortBy = ('Range', 'Partials', 'Mass', 'Sequence')
        viewSortBy_label = wx.StaticText(self, -1, "Sort by: ")
        self.viewSortBy_combo = wx.ComboBox(self, -1, size=(84, -1), choices=choicesSortBy, style=wx.CB_READONLY)
        self.viewHighlightModif_check = wx.CheckBox(self, -1, "Highlight modified peptides")

        # pack items
        sortSizer.Add(viewSortBy_label, 0, wx.ALIGN_CENTER_VERTICAL)
        sortSizer.Add(self.viewSortBy_combo)

        mainBox.Add(sortSizer, 0, wx.TOP|wx.LEFT|wx.RIGHT, 5)
        mainBox.Add(self.viewHighlightModif_check, 0, wx.ALL, 5)

        # set defaults
        try:
            self.viewSortBy_combo.Select(self.data['sortby']-1)
        except ValueError:
            self.viewSortBy_combo.Select(0)
        self.viewHighlightModif_check.SetValue(self.data['highlightmods'])

        return mainBox
    # ----


    # ----
    def makeMassBox(self):
        """ Make box for mass type and limits. """

        # create items
        mainBox = wx.StaticBoxSizer(wx.StaticBox(self, -1, "Mass Type and Limits"), wx.VERTICAL)
        grid = mwx.GridBagSizer()

        massCharge_choices = ['M', '1+', '2+', '1-', '2-']
        massCharge_label = wx.StaticText(self, -1, "Charge: ")
        self.massCharge_combo = wx.ComboBox(self, -1, size=(85, -1), choices=massCharge_choices, style=wx.CB_READONLY)

        self.massUseLimits_check = wx.CheckBox(self, -1, "Use mass range limits")

        massMinMass_label = wx.StaticText(self, -1, "Low mass: ")
        massMinMass_unit = wx.StaticText(self, -1, " Da")
        self.massMinMass_value = wx.TextCtrl(self, -1, str(self.data['minmass']), size=(85, -1), validator=mwx.txtValidator('digits'))

        massMaxMass_label = wx.StaticText(self, -1, "High mass: ")
        massMaxMass_unit = wx.StaticText(self, -1, " Da")
        self.massMaxMass_value = wx.TextCtrl(self, -1, str(self.data['maxmass']), size=(85, -1), validator=mwx.txtValidator('digits'))

        # pack items
        grid.Add(massCharge_label, (0, 0), flag=wx.ALIGN_CENTER_VERTICAL|wx.ALIGN_RIGHT)
        grid.Add(self.massCharge_combo, (0, 1))
        grid.Add(massMinMass_label, (1, 0), flag=wx.ALIGN_CENTER_VERTICAL|wx.ALIGN_RIGHT)
        grid.Add(self.massMinMass_value, (1, 1))
        grid.Add(massMinMass_unit, (1, 2), flag=wx.ALIGN_CENTER_VERTICAL)
        grid.Add(massMaxMass_label, (2, 0), flag=wx.ALIGN_CENTER_VERTICAL|wx.ALIGN_RIGHT)
        grid.Add(self.massMaxMass_value, (2, 1))
        grid.Add(massMaxMass_unit, (2, 2), flag=wx.ALIGN_CENTER_VERTICAL)

        mainBox.Add(grid, 0, wx.ALL, 5)
        mainBox.Add(self.massUseLimits_check, 0, wx.ALL, 5)

        # set defaults
        try:
            self.massCharge_combo.Select(massCharge_choices.index(self.data['charge']))
        except ValueError:
            self.massCharge_combo.Select(0)
        self.massUseLimits_check.SetValue(self.data['uselimits'])

        return mainBox
    # ----


class mFragPage(wx.Panel):
    """Controls for mFrag module"""

    # ----
    def __init__(self, parent, data):
        wx.Panel.__init__(self, parent, -1)

        self.data = data
        self.ions_check = {}

        # pack main frame
        if wx.Platform == '__WXMAC__':
            margin = 10
        else:
            margin = 5

        row1Sizer = wx.BoxSizer(wx.HORIZONTAL)
        row1Sizer.Add(self.makeMainIonsBox(), 1, wx.EXPAND|wx.ALL, margin)
        row1Sizer.Add(self.makeDoublyChargedIonsBox(), 1, wx.EXPAND|wx.ALL, margin)

        row2Sizer = wx.BoxSizer(wx.HORIZONTAL)
        row2Sizer.Add(self.makeInternalIonsBox(), 1, wx.EXPAND|wx.LEFT|wx.RIGHT|wx.BOTTOM, margin)
        row2Sizer.Add(self.makeMatchingBox(), 1, wx.EXPAND|wx.LEFT|wx.RIGHT|wx.BOTTOM, margin)

        label = wx.StaticText(self, -1, "Default parameters for peptide fragmentation")
        mainSizer = wx.BoxSizer(wx.VERTICAL)
        mainSizer.Add(label, 0, wx.ALL|wx.ALIGN_CENTER, 10)
        mainSizer.Add(row1Sizer, 0, wx.EXPAND, 0)
        mainSizer.Add(row2Sizer, 0, wx.EXPAND, 0)
        self.SetSizer(mainSizer)

        # set default ions
        self.setDefaultIons()
    # ----


    # ----
    def getData(self):
        """ Check and get data from page. """

        self.data['pref-default'] = ''
        self.data['matchfiltered'] = 0
        self.data['annotfiltered'] = 0

        # get ions
        for ion in self.ions_check:
            if self.ions_check[ion].IsChecked():
                self.data['pref-default'] += ion+';'

        # match filtered
        if self.matchMatchFiltered_check.IsChecked():
            self.data['matchfiltered'] = 1
        if self.matchAnnotFiltered_check.IsChecked():
            self.data['annotfiltered'] = 1

        return self.data
    # ----


    # ----
    def makeMainIonsBox(self):
        """ Make box for main ions. """

        # make items
        mainBox = wx.StaticBoxSizer(wx.StaticBox(self, -1, "Main Ions"), wx.VERTICAL)
        grid = wx.GridBagSizer(4, 4)

        self.ions_check['a0'] = wx.CheckBox(self, -1, "a")
        self.ions_check['a1'] = wx.CheckBox(self, -1, "a*")
        self.ions_check['a2'] = wx.CheckBox(self, -1, "a~")
        self.ions_check['a1'].SetToolTip(wx.ToolTip("a-NH3"))
        self.ions_check['a2'].SetToolTip(wx.ToolTip("a-H2O"))

        self.ions_check['b0'] = wx.CheckBox(self, -1, "b")
        self.ions_check['b1'] = wx.CheckBox(self, -1, "b*")
        self.ions_check['b2'] = wx.CheckBox(self, -1, "b~")
        self.ions_check['b1'].SetToolTip(wx.ToolTip("b-NH3"))
        self.ions_check['b2'].SetToolTip(wx.ToolTip("b-H2O (same as N-term ladder)"))

        self.ions_check['y0'] = wx.CheckBox(self, -1, "y")
        self.ions_check['y1'] = wx.CheckBox(self, -1, "y*")
        self.ions_check['y2'] = wx.CheckBox(self, -1, "y~")
        self.ions_check['y0'].SetToolTip(wx.ToolTip("same as C-term ladder"))
        self.ions_check['y1'].SetToolTip(wx.ToolTip("y-NH3"))
        self.ions_check['y2'].SetToolTip(wx.ToolTip("y-H2O"))

        self.ions_check['c0'] = wx.CheckBox(self, -1, "c")
        self.ions_check['x0'] = wx.CheckBox(self, -1, "x")
        self.ions_check['z0'] = wx.CheckBox(self, -1, "z")

        self.ions_check['im'] = wx.CheckBox(self, -1, "imm.")

        # pack items
        grid.Add(self.ions_check['a0'], (0, 0))
        grid.Add(self.ions_check['a1'], (1, 0))
        grid.Add(self.ions_check['a2'], (2, 0))

        grid.Add(self.ions_check['b0'], (0, 1))
        grid.Add(self.ions_check['b1'], (1, 1))
        grid.Add(self.ions_check['b2'], (2, 1))

        grid.Add(self.ions_check['y0'], (0, 2))
        grid.Add(self.ions_check['y1'], (1, 2))
        grid.Add(self.ions_check['y2'], (2, 2))

        grid.Add(self.ions_check['c0'], (0, 3))
        grid.Add(self.ions_check['x0'], (1, 3))
        grid.Add(self.ions_check['z0'], (2, 3))

        grid.Add(self.ions_check['im'], (3, 0), (1, 2))

        mainBox.Add(grid, 0, wx.TOP|wx.LEFT|wx.RIGHT, 5)

        return mainBox
    # ----


    # ----
    def makeDoublyChargedIonsBox(self):
        """ Make box for doubly charged ions. """

        # make items
        mainBox = wx.StaticBoxSizer(wx.StaticBox(self, -1, "Doubly Charged Ions (2+)"), wx.VERTICAL)
        grid = wx.GridBagSizer(4, 4)

        self.ions_check['a3'] = wx.CheckBox(self, -1, "a")
        self.ions_check['a4'] = wx.CheckBox(self, -1, "a*")
        self.ions_check['a5'] = wx.CheckBox(self, -1, "a~")
        self.ions_check['a4'].SetToolTip(wx.ToolTip("a-NH3"))
        self.ions_check['a5'].SetToolTip(wx.ToolTip("a-H2O"))

        self.ions_check['b3'] = wx.CheckBox(self, -1, "b")
        self.ions_check['b4'] = wx.CheckBox(self, -1, "b*")
        self.ions_check['b5'] = wx.CheckBox(self, -1, "b~")
        self.ions_check['b4'].SetToolTip(wx.ToolTip("b-NH3"))
        self.ions_check['b5'].SetToolTip(wx.ToolTip("b-H2O"))

        self.ions_check['y3'] = wx.CheckBox(self, -1, "y")
        self.ions_check['y4'] = wx.CheckBox(self, -1, "y*")
        self.ions_check['y5'] = wx.CheckBox(self, -1, "y~")
        self.ions_check['y4'].SetToolTip(wx.ToolTip("y-NH3"))
        self.ions_check['y5'].SetToolTip(wx.ToolTip("y-H2O"))

        self.ions_check['c1'] = wx.CheckBox(self, -1, "c")
        self.ions_check['x1'] = wx.CheckBox(self, -1, "x")
        self.ions_check['z1'] = wx.CheckBox(self, -1, "z")

        # pack items
        grid.Add(self.ions_check['a3'], (0, 0))
        grid.Add(self.ions_check['a4'], (1, 0))
        grid.Add(self.ions_check['a5'], (2, 0))

        grid.Add(self.ions_check['b3'], (0, 1))
        grid.Add(self.ions_check['b4'], (1, 1))
        grid.Add(self.ions_check['b5'], (2, 1))

        grid.Add(self.ions_check['y3'], (0, 2))
        grid.Add(self.ions_check['y4'], (1, 2))
        grid.Add(self.ions_check['y5'], (2, 2))

        grid.Add(self.ions_check['c1'], (0, 3))
        grid.Add(self.ions_check['x1'], (1, 3))
        grid.Add(self.ions_check['z1'], (2, 3))

        mainBox.Add(grid, 0, wx.TOP|wx.LEFT|wx.RIGHT, 5)

        return mainBox
    # ----


    # ----
    def makeInternalIonsBox(self):
        """ Make box for internal ions. """

        # make items
        mainBox = wx.StaticBoxSizer(wx.StaticBox(self, -1, "Internal Ions"), wx.VERTICAL)
        grid = wx.GridBagSizer(4, 10)

        self.ions_check['int0'] = wx.CheckBox(self, -1, "main")
        self.ions_check['int1'] = wx.CheckBox(self, -1, "-CO")
        self.ions_check['int2'] = wx.CheckBox(self, -1, "-NH3")
        self.ions_check['int3'] = wx.CheckBox(self, -1, "-H2O")

        # pack items
        grid.Add(self.ions_check['int0'], (0, 0))
        grid.Add(self.ions_check['int1'], (1, 0))
        grid.Add(self.ions_check['int2'], (0, 1))
        grid.Add(self.ions_check['int3'], (1, 1))

        mainBox.Add(grid, 0, wx.TOP|wx.LEFT, 5)

        return mainBox
    # ----


    # ----
    def makeMatchingBox(self):
        """ Make box for matching. """

        # make items
        mainBox = wx.StaticBoxSizer(wx.StaticBox(self, -1, "Match Options"), wx.VERTICAL)

        self.matchMatchFiltered_check = wx.CheckBox(self, -1, "Match filtered ions")
        self.matchAnnotFiltered_check = wx.CheckBox(self, -1, "Annotate filtered ions")

        # pack items
        mainBox.Add(self.matchMatchFiltered_check, 0, wx.TOP|wx.LEFT|wx.RIGHT, 5)
        mainBox.Add(self.matchAnnotFiltered_check, 0, wx.ALL, 5)

        # set events
        self.matchMatchFiltered_check.Bind(wx.EVT_CHECKBOX, self.onMatchFiltered)

        # set defaults
        self.matchMatchFiltered_check.SetValue(self.data['matchfiltered'])
        self.matchAnnotFiltered_check.SetValue(self.data['annotfiltered'])
        self.matchAnnotFiltered_check.Enable(self.data['matchfiltered'])

        return mainBox
    # ----


    # ----
    def setDefaultIons(self):
        """ Check default ions. """

        # check ions
        for ion in self.data['pres-default'].split(';'):
            try:
                self.ions_check[ion].SetValue(True)
            except KeyError:
                pass
    # ----


    # ----
    def onMatchFiltered(self, evt):
        """ Enable/disable 'annotate filtered'. """
        if evt.IsChecked():
            self.matchAnnotFiltered_check.Enable(True)
        else:
            self.matchAnnotFiltered_check.Enable(False)
    # ----


class mDiffPage(wx.Panel):
    """Controls for mDiff module"""

    # ----
    def __init__(self, parent, data):
        wx.Panel.__init__(self, parent, -1)

        self.data = data

        # pack main frame
        if wx.Platform == '__WXMAC__':
            margin = 10
        else:
            margin = 5

        row1Sizer = wx.BoxSizer(wx.HORIZONTAL)
        row1Sizer.Add(self.makeHighlightBox(), 1, wx.EXPAND|wx.TOP|wx.LEFT|wx.RIGHT, margin)
        row1Sizer.Add(self.makeModGroupsBox(), 1, wx.EXPAND|wx.TOP|wx.LEFT|wx.RIGHT, margin)

        label = wx.StaticText(self, -1, "Default parameters for difference checking")
        mainSizer = wx.BoxSizer(wx.VERTICAL)
        mainSizer.Add(label, 0, wx.ALL|wx.ALIGN_CENTER, 10)
        mainSizer.Add(row1Sizer, 0, wx.EXPAND, 0)
        mainSizer.Add(self.makeMassLimitsBox(), 0, wx.EXPAND|wx.ALL, margin)
        self.SetSizer(mainSizer)
    # ----


    # ----
    def getData(self):
        """ Check and get data from page. """

        self.data['checkaa'] = 0
        self.data['checkdip'] = 0
        self.data['checkmod'] = 0
        self.data['checkdiff'] = 0
        self.data['usemaxdiff'] = 0
        self.data['modgroups'] = ''

        # highlights
        if self.highlightAA_check.IsChecked():
            self.data['checkaa'] = 1
        if self.highlightDip_check.IsChecked():
            self.data['checkdip'] = 1
        if self.highlightMod_check.IsChecked():
            self.data['checkmod'] = 1
        if self.highlightDiff_check.IsChecked():
            self.data['checkdiff'] = 1

        # selected modification groups
        if self.modGroupPTM_check.IsChecked():
            self.data['modgroups'] += 'ptm;'
        if self.modGroupChemical_check.IsChecked():
            self.data['modgroups'] += 'chemical;'
        if self.modGroupGlyco_check.IsChecked():
            self.data['modgroups'] += 'glyco;'
        if self.modGroupArtefact_check.IsChecked():
            self.data['modgroups'] += 'artefact;'
        if self.modGroupOther_check.IsChecked():
            self.data['modgroups'] += 'other;'

        # mass limits
        maxdiff = self.massLimitsMaxDiff_value.GetValue()
        maxdiff = maxdiff.replace(',', '.')
        try:
            self.data['maxdiff'] = int(maxdiff)
        except ValueError:
            pass
        if self.massLimitsUse_check.IsChecked():
            self.data['usemaxdiff'] = 1

        return self.data
    # ----


    # ----
    def makeHighlightBox(self):
        """ Make box for default highlights. """

        # make items
        mainBox = wx.StaticBoxSizer(wx.StaticBox(self, -1, "Check and Highlight"), wx.VERTICAL)

        self.highlightAA_check = wx.CheckBox(self, -1, "Amino acids")
        self.highlightDip_check = wx.CheckBox(self, -1, "Dipeptides")
        self.highlightMod_check = wx.CheckBox(self, -1, "Modifications")
        self.highlightDiff_check = wx.CheckBox(self, -1, "Difference")

        # pack items
        mainBox.Add(self.highlightAA_check, 0, wx.TOP|wx.LEFT|wx.RIGHT, 5)
        mainBox.Add(self.highlightDip_check, 0, wx.TOP|wx.LEFT|wx.RIGHT, 5)
        mainBox.Add(self.highlightMod_check, 0, wx.TOP|wx.LEFT|wx.RIGHT, 5)
        mainBox.Add(self.highlightDiff_check, 0, wx.ALL, 5)

        # set defaults
        self.highlightAA_check.SetValue(self.data['checkaa'])
        self.highlightDip_check.SetValue(self.data['checkdip'])
        self.highlightMod_check.SetValue(self.data['checkmod'])
        self.highlightDiff_check.SetValue(self.data['checkdiff'])

        return mainBox
    # ----


    # ----
    def makeModGroupsBox(self):
        """ Make box for modification groups. """

        # make items
        mainBox = wx.StaticBoxSizer(wx.StaticBox(self, -1, "Modification Groups"), wx.VERTICAL)

        self.modGroupPTM_check = wx.CheckBox(self, -1, "PTM")
        self.modGroupPTM_check.SetToolTip(wx.ToolTip("Post Translational Modifications"))

        self.modGroupChemical_check = wx.CheckBox(self, -1, "Chemical")
        self.modGroupChemical_check.SetToolTip(wx.ToolTip("Chemical derivative"))

        self.modGroupGlyco_check = wx.CheckBox(self, -1, "Glyco")
        self.modGroupGlyco_check.SetToolTip(wx.ToolTip("Glycosylation"))

        self.modGroupArtefact_check = wx.CheckBox(self, -1, "Artefact")
        self.modGroupArtefact_check.SetToolTip(wx.ToolTip("Preparation artefacts"))

        self.modGroupOther_check = wx.CheckBox(self, -1, "Other")
        self.modGroupOther_check.SetToolTip(wx.ToolTip("All others"))

        # pack items
        mainBox.Add(self.modGroupPTM_check, 0, wx.TOP|wx.LEFT|wx.RIGHT, 5)
        mainBox.Add(self.modGroupChemical_check, 0, wx.TOP|wx.LEFT|wx.RIGHT, 5)
        mainBox.Add(self.modGroupGlyco_check, 0, wx.TOP|wx.LEFT|wx.RIGHT, 5)
        mainBox.Add(self.modGroupArtefact_check, 0, wx.TOP|wx.LEFT|wx.RIGHT, 5)
        mainBox.Add(self.modGroupOther_check, 0, wx.ALL, 5)

        # set defaults
        if 'ptm' in self.data['modgroups']:
            self.modGroupPTM_check.SetValue(True)
        if 'chemical' in self.data['modgroups']:
            self.modGroupChemical_check.SetValue(True)
        if 'glyco' in self.data['modgroups']:
            self.modGroupGlyco_check.SetValue(True)
        if 'artefact' in self.data['modgroups']:
            self.modGroupArtefact_check.SetValue(True)
        if 'other' in self.data['modgroups']:
            self.modGroupOther_check.SetValue(True)

        return mainBox
    # ----


    # ----
    def makeMassLimitsBox(self):
        """ Make box for mass limits. """

        # make items
        mainBox = wx.StaticBoxSizer(wx.StaticBox(self, -1, "Match Options"), wx.VERTICAL)
        massLimitsGrid = mwx.GridBagSizer()

        massLimitsMaxDiff_label = wx.StaticText(self, -1, "Max difference: ")
        massLimitsMaxDiff_units = wx.StaticText(self, -1, " Da")
        self.massLimitsMaxDiff_value = wx.TextCtrl(self, -1, str(self.data['maxdiff']), size=(70, -1), validator=mwx.txtValidator('digits'))
        self.massLimitsMaxDiff_value.SetToolTip(wx.ToolTip("Max difference counted"))
        self.massLimitsUse_check = wx.CheckBox(self, -1, "Use max difference")

        # pack items
        massLimitsGrid.Add(massLimitsMaxDiff_label, (0, 0), flag=wx.ALIGN_CENTER_VERTICAL)
        massLimitsGrid.Add(self.massLimitsMaxDiff_value, (0, 1))
        massLimitsGrid.Add(massLimitsMaxDiff_units, (0, 2), flag=wx.ALIGN_CENTER_VERTICAL)

        mainBox.Add(massLimitsGrid, 0, wx.ALL, 5)
        mainBox.Add(self.massLimitsUse_check, 0, wx.ALL, 5)

        # set defaults
        self.massLimitsUse_check.SetValue(self.data['usemaxdiff'])

        return mainBox
    # ----
