# -------------------------------------------------------------------------
#     This file is part of mMass - the spectrum analysis tool for MS.
#     Copyright (C) 2005-07 Martin Strohalm <mmass@biographics.cz>

#     This program is free software; you can redistribute it and/or modify
#     it under the terms of the GNU General Public License as published by
#     the Free Software Foundation; either version 2 of the License, or
#     (at your option) any later version.

#     This program is distributed in the hope that it will be useful,
#     but WITHOUT ANY WARRANTY; without even the implied warranty of
#     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#     GNU General Public License for more details.

#     Complete text of GNU GPL can be found in the file LICENSE in the
#     main directory of the program
# -------------------------------------------------------------------------

# Function: Base class for calibration dialog.

# load libs
import wx

# load modules
import count
from dlg_calibplot import calibPlot
from nucleus import mwx


class dlgCalibration(wx.Dialog):
    """ Base class for calibration dialog. """

    # ----
    def __init__(self, parent, title, calibType, grayColour):
        wx.Dialog.__init__(self, parent, -1, title, style=wx.DEFAULT_DIALOG_STYLE|wx.RESIZE_BORDER)

        self.parent = parent
        self.calibType = calibType
        self.grayColour = grayColour
        self.peaklist = []
        self.fit = []
        self.selected = -1
    # ----


    # ----
    def makeCalibrationBox(self):
        """ Make calibration box. """

        # make items
        mainBox = wx.StaticBoxSizer(wx.StaticBox(self, -1, "Calibration Type"), wx.VERTICAL)

        self.calibLiner_radio = wx.RadioButton(self, -1, "Linear fitting", style=wx.RB_GROUP)
        self.calibQuadratic_radio = wx.RadioButton(self, -1, "Quadratic fitting")

        # pack items
        if wx.Platform == '__WXMAC__':
            mainBox.Add(self.calibLiner_radio, 0, 0)
            mainBox.Add(self.calibQuadratic_radio, 0, wx.TOP, 5)
        else:
            mainBox.Add(self.calibLiner_radio, 0, wx.TOP|wx.LEFT|wx.RIGHT, 5)
            mainBox.Add(self.calibQuadratic_radio, 0, wx.ALL, 5)

        # set defaults
        self.calibQuadratic_radio.SetValue(True)

        return mainBox
    # ----


    # ----
    def makeResultsBox(self):
        """ Make result box. """

        # make items
        mainBox = wx.StaticBoxSizer(wx.StaticBox(self, -1, "Results"), wx.HORIZONTAL)
        grid = mwx.GridBagSizer()

        calibC0_label = wx.StaticText(self, -1, "Const. c0: ")
        self.calibC0_value = wx.TextCtrl(self, -1, size=(85, -1), style=wx.TE_READONLY|wx.TE_RIGHT)
        self.calibC0_value.Enable(False)

        calibC1_label = wx.StaticText(self, -1, "Const. c1: ")
        self.calibC1_value = wx.TextCtrl(self, -1, size=(85, -1), style=wx.TE_READONLY|wx.TE_RIGHT)
        self.calibC1_value.Enable(False)

        calibC2_label = wx.StaticText(self, -1, "Const. c2: ")
        self.calibC2_value = wx.TextCtrl(self, -1, size=(85, -1), style=wx.TE_READONLY|wx.TE_RIGHT)
        self.calibC2_value.Enable(False)

        calibPeaks_label = wx.StaticText(self, -1, "Peaks: ")
        self.calibPeaks_value = wx.TextCtrl(self, -1, size=(85, -1), style=wx.TE_READONLY|wx.TE_RIGHT)
        self.calibPeaks_value.Enable(False)

        # pack items
        grid.Add(calibC0_label, (0, 0), flag=wx.ALIGN_CENTER_VERTICAL|wx.ALIGN_RIGHT)
        grid.Add(self.calibC0_value, (0, 1))
        grid.Add(calibC1_label, (1, 0), flag=wx.ALIGN_CENTER_VERTICAL|wx.ALIGN_RIGHT)
        grid.Add(self.calibC1_value, (1, 1))
        grid.Add(calibC2_label, (2, 0), flag=wx.ALIGN_CENTER_VERTICAL|wx.ALIGN_RIGHT)
        grid.Add(self.calibC2_value, (2, 1))
        grid.Add(calibPeaks_label, (3, 0), flag=wx.ALIGN_CENTER_VERTICAL|wx.ALIGN_RIGHT)
        grid.Add(self.calibPeaks_value, (3, 1))

        if wx.Platform == '__WXMAC__':
            mainBox.Add(grid, 0, wx.EXPAND, 0)
        else:
            mainBox.Add(grid, 0, wx.EXPAND|wx.ALL, 5)

        return mainBox
    # ----


    # ----
    def makeZoomBox(self):
        """ Make zoom box. """

        # make items
        mainBox = wx.StaticBoxSizer(wx.StaticBox(self, -1, "Spectrum Zoom"), wx.VERTICAL)
        grid = mwx.GridBagSizer()

        zoom_choices = ['Current', '0.1', '0.5', '1', '5', '10', '20', '50', '100']
        zoom_label = wx.StaticText(self, -1, "Zoom: ")
        self.zoom_combo = wx.ComboBox(self, -1, '10', size=(80, -1), choices=zoom_choices, style=wx.CB_READONLY)
        zoom_units = wx.StaticText(self, -1, ' %')
        self.zoom_combo.SetToolTip(wx.ToolTip("Zoom spectrum"))

        # pack items
        grid.Add(zoom_label, (0, 0), flag=wx.ALIGN_CENTER_VERTICAL|wx.ALIGN_RIGHT)
        grid.Add(self.zoom_combo, (0, 1))
        grid.Add(zoom_units, (0, 2), flag=wx.ALIGN_CENTER_VERTICAL)

        if wx.Platform == '__WXMAC__':
            mainBox.Add(grid, 0, wx.EXPAND, 0)
        else:
            mainBox.Add(grid, 0, wx.EXPAND|wx.ALL, 5)

        return mainBox
    # ----


    # ----
    def makeButtonBox(self):
        """ Make box for main buttons. """

        # make items
        self.Clear_button = wx.Button(self, -1, "Clear")
        Calculate_button = wx.Button(self, -1, "Calculate")
        Plot_button = wx.Button(self, -1, "View plot")
        self.OK_button = wx.Button(self, wx.ID_OK, "Apply")
        Cancel_button = wx.Button(self, wx.ID_CANCEL, "Cancel")

        # pack items
        buttonBox = wx.BoxSizer(wx.HORIZONTAL)
        buttonBox.Add(self.Clear_button, 0, wx.ALL, 5)
        buttonBox.Add(Calculate_button, 0, wx.ALL, 5)
        buttonBox.Add(Plot_button, 0, wx.ALL, 5)
        buttonBox.Add(self.OK_button, 0, wx.ALL, 5)
        buttonBox.Add(Cancel_button, 0, wx.ALL, 5)

        # set events
        self.Clear_button.Bind(wx.EVT_BUTTON, self.onClear)
        Calculate_button.Bind(wx.EVT_BUTTON, self.onCalculate)
        Plot_button.Bind(wx.EVT_BUTTON, self.onPlot)
        self.OK_button.Bind(wx.EVT_BUTTON, self.onOK)

        # set defaults
        self.Clear_button.Enable(False)
        self.OK_button.Enable(False)

        return buttonBox
    # ----


    # ----
    def updateResults(self):
        """ Show calculated parameters in results box. """

        # get parameters
        if self.fit:
            c0 = round(self.fit[0][0], 6)
            c1 = round(self.fit[0][1], 6)
            if self.fit[2] == 'quadratic':
                c2 = round(self.fit[0][2], 6)
            else:
                c2 = ''
        else:
            c0 = ''
            c1 = ''
            c2 = ''

        # update fields
        self.calibC0_value.SetValue(str(c0))
        self.calibC1_value.SetValue(str(c1))
        self.calibC2_value.SetValue(str(c2))
    # ----


    # ----
    def updatePeaksCount(self):
        """ Show number of peak for calculation. """

        # get count
        peaks = 0
        for peak in self.peaklist:
            if peak[0]:
                peaks += 1

        # update field
        self.calibPeaks_value.SetValue(str(peaks))
    # ----


    # ----
    def onPeakActivated(self, evt):
        """ Change status of selected peak. """

        # get selected peak
        index = evt.m_itemIndex
        self.selected = index
        if index == -1:
            return

        # change status
        status = self.peaklist[index][0]
        reference = self.peaklist[index][2]
        if reference != None:
            if status:
                self.peaklist[index][0] = False
                self.peaklist_list.SetStringItem(index, 1, "skip")
                self.peaklist_list.SetItemTextColour(index, self.grayColour)
            else:
                self.peaklist[index][0] = True
                self.peaklist_list.SetStringItem(index, 1, "use")
                self.peaklist_list.SetItemTextColour(index, (0, 0, 0))

        # update peaks' counter
        self.updatePeaksCount()
    # ----


    # ----
    def onCalculate(self, evt):
        """ Calculate calibration parameters. """

        # get equation
        equation = 'linear'
        if self.calibQuadratic_radio.GetValue():
            equation = 'quadratic'

        # get selected peaks
        peaklist = []
        for peak in self.peaklist:
            if peak[0]:
               peaklist.append([peak[1], peak[2]])

        # check peaks' count
        if (equation == 'linear' and len(peaklist) < 2) \
            or (equation == 'quadratic' and len(peaklist) < 3):

            dlg = wx.MessageDialog(self, "Not enough peaks for selected calibration type!", "Calibration Error", wx.OK|wx.ICON_ERROR)
            dlg.ShowModal()
            dlg.Destroy()
            return

        # check peaks' count
        if (equation == 'linear' and len(peaklist) < 3) \
            or (equation == 'quadratic' and len(peaklist) < 4):

            dlg = wx.MessageDialog(self, "Not enough peaks for relevant calibration!\nContinue anyway?", "Calibration Warning", wx.YES_NO|wx.NO_DEFAULT|wx.ICON_EXCLAMATION)
            button = dlg.ShowModal()
            dlg.Destroy()
            if button != wx.ID_YES:
                return

        # calculate calibration params - fit data to curve
        if self.calibType == 'statistical':
            self.fit = count.statisticalCalibration(peaklist, equation=equation)
        else:
            self.fit = count.referenceCalibration(peaklist, equation=equation)
        self.fit = list(self.fit)
        self.fit.append(equation)

        # recalculate data
        for peak in self.peaklist:

            # recalibrate
            if equation == 'quadratic':
                peak[3] = peak[1] - count.quadraticEq(self.fit[0], peak[1])
            else:
                peak[3] = peak[1] - count.linearEq(self.fit[0], peak[1])

            # calculate error
            if peak[2] == None:
                peak[4] = None
            elif self.calibType == 'statistical':
                peak[4] = count.calcDecimalDiff(peak[3], peak[2])
            else:
                peak[4] = peak[3] - peak[2]

        # update results and peaklist
        self.updateResults()
        self.updatePeaklist()

        # enable buttons
        self.Clear_button.Enable(True)
        self.OK_button.Enable(True)
    # ----


    # ----
    def onPlot(self, evt):
        """ View plot with current data and calibration. """

        # format measured data
        measured = []
        for peak in self.peaklist:
            if peak[0]:
                if self.calibType == 'statistical':
                    diff = count.calcDecimalDiff(peak[1], peak[2])
                else:
                    diff = peak[1] - peak[2]
                measured.append([peak[1], diff])

        if measured == []:
            return

        # make calibration curve
        calibration = []
        if self.fit:
            calibration = count.makeCalibrationCurve(min(measured)[0], max(measured)[0], self.fit)

        # check peaks' count
        if not measured:
            dlg = wx.MessageDialog(self, "Not enough peaks selected!", "Not Enough Peaks", wx.OK|wx.ICON_ERROR)
            dlg.ShowModal()
            dlg.Destroy()
            return

        # raise plot dialog
        else:
            dlg = calibPlot(self, [measured, calibration])
            dlg.ShowModal()
            dlg.Destroy()
    # ----


    # ----
    def onOK(self, evt):
        """ Get and return calibration parameters. """

        if self.fit:
            self.EndModal(wx.ID_OK)
    # ----


    # ----
    def getData(self):
        """ Return calibration parameters. """

        return self.fit
    # ----


    # ----
    def highlightPoint(self, point):
        """ Highlight selected point in the spectrum. """

        # get zoom
        zoom = self.zoom_combo.GetValue()
        if zoom == 'Current':
            zoom = None
        else:
            zoom = float(zoom)

        # highlight point
        self.parent.onSpectrumHighlightPoint(point, zoom)
    # ----
