# -------------------------------------------------------------------------
#     This file is part of mMass - the spectrum analysis tool for MS.
#     Copyright (C) 2005-07 Martin Strohalm <mmass@biographics.cz>

#     This program is free software; you can redistribute it and/or modify
#     it under the terms of the GNU General Public License as published by
#     the Free Software Foundation; either version 2 of the License, or
#     (at your option) any later version.

#     This program is distributed in the hope that it will be useful,
#     but WITHOUT ANY WARRANTY; without even the implied warranty of
#     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#     GNU General Public License for more details.

#     Complete text of GNU GPL can be found in the file LICENSE in the
#     main directory of the program
# -------------------------------------------------------------------------

# Function: Spectrum and peaklist processing functions.

# load libs
import wx

# load modules
import count
from dlg_intercalib import dlgInterCalib
from dlg_statcalib import dlgStatCalib
from dlg_cropparams import dlgCropParams


class mProc:
    """ Spectrum and peaklist processing. """

    # ----
    def __init__(self, parent, document):

        self.parent = parent
        self.config = parent.config
        self.docMonitor = parent.docMonitor
        self.docData = document
    # ----


    # ----
    def undo(self):
        """ Revert spectrum and peaklist to init data. """

        # set application working
        self.docMonitor('setAppStatus', "Undo processings...")

        # revert document data
        self.docData.revertSpectrum()
        self.docData.revertPeaklist()

        # update application
        self.docMonitor('onSpectrumChanged')
        self.docMonitor('onPeaklistChanged', 'processing')
        self.docMonitor('setAppStatus', 0)
    # ----


    # ----
    def calibrate(self, calType):
        """ Recalibrate spectrum and peaklist data. """

        # get data
        spectrum = self.docData.getSpectrum()
        peaklist = self.docData.getPeaks()

        # select calibration type
        if calType == 'internal':
            dlg = dlgInterCalib(self.parent, peaklist, self.config)
        elif calType == 'statistic':
            dlg = dlgStatCalib(self.parent, peaklist, self.config)
        else:
            return

        # raise calibration dialog and calculate constants
        if dlg.ShowModal() == wx.ID_OK:
            fit = dlg.getData()
            dlg.Destroy()
        else:
            dlg.Destroy()
            return

        # set application working
        self.docMonitor('setAppStatus', "Applying calibration...")

        # store old data for undo
        self.docData.storeSpectrum()
        self.docData.storePeaklist()

        # recalculate data
        spectrum = count.calibrateData(spectrum, fit[0], fit[2])
        peaklist = count.calibrateData(peaklist, fit[0], fit[2])

        # set all peaks to "point"
        for x in range(len(peaklist)):
            peaklist[x][3] = 0

        # update document
        self.docData.setProcessingStatus(True)
        self.docData.setSpectrum(spectrum)
        self.docData.setPeaklist(peaklist, undo=False)

        # update application
        self.docMonitor('setAppStatus', 0)
        self.docMonitor('onSpectrumChanged')
        self.docMonitor('onPeaklistChanged', 'processing')
    # ----


    # ----
    def baseline(self):
        """ Do spectrum baseline correction. """
        pass
    # ----


    # ----
    def smooth(self):
        """ Smooth spectrum by Savitzky-Golay filter. """
        pass
    # ----


    # ----
    def crop(self):
        """ Crop spectrum and peaklist data to selected range. """

        # get data
        spectrum = self.docData.getSpectrum()
        XRange = self.docData.getSpectrumXRange()
        peaklist = self.docData.getPeaks()

        # raise crop params dialog
        dlg = dlgCropParams(self.parent, int(round(XRange[0])), int(round(XRange[1])))
        if dlg.ShowModal() == wx.ID_OK:
            XRange = dlg.data
            dlg.Destroy()
        else:
            dlg.Destroy()
            return

        # set application working
        self.docMonitor('setAppStatus', "Cropping spectrum...")

        # store old data for undo
        self.docData.storeSpectrum()
        self.docData.storePeaklist()

        # crop spectrum
        spectrum = count.cropData(spectrum, XRange[0], XRange[1])
        peaklist = count.cropData(peaklist, XRange[0], XRange[1])

        # update document
        self.docData.setProcessingStatus(True)
        self.docData.setSpectrum(spectrum)
        self.docData.setPeaklist(peaklist, undo=False)

        # update application
        self.docMonitor('onSpectrumChanged')
        self.docMonitor('onPeaklistChanged', 'processing')
        self.docMonitor('setAppStatus', 0)
    # ----
