/*
 *  mod_bt - Making Things Better For Seeders
 *  Copyright 2004, 2005, 2006 Tyler MacDonald <tyler@yi.org>
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/* other libs */
#include <db.h>
#include <apr.h>
#include <apr_pools.h>
/* local */
#include <libbttracker.h>

/* all callbacks return 0 for success, BT_CALLBACK_STOP for success, but no more callbacks should run,
 * and any other value for failure. */
int btt_txn_iterate_callbacks(
    apr_pool_t* p, DB_TXN* txn, DBC* cursor, DBT* key, DBT* val,
    const btt_txn_iterator callbacks[]
) {
    int ret;
    int i;
    for(i=0; callbacks[i].callback; i++) {
        ret = (callbacks[i].callback)(
            p, txn, cursor, key, val, callbacks[i].data
        );
        if(ret == BT_CALLBACK_STOP)
            return 0;
        else if(ret)
            return ret;
    }

    return 0;
}


int btt_txn_iterate_peerlist(
    btt_tracker* tracker, apr_pool_t* p, btt_infohash* hash, DB_TXN* txn,
    int c_flags, int cc_flags, const btt_txn_iterator callbacks[]
) {
    int ret = 0;
    DBT index_key;
    DBT peer_key;
    DBT peer_val;
    DBC* index_cur = NULL;

    if(
        (ret = tracker->db.index->cursor(
            tracker->db.index, txn, &index_cur, c_flags
        )) != 0
    ) {
        tracker->db.index->err(
            tracker->db.index, ret,
            "bt_txn_iterate_peerlist(%s): index->cursor()",
            bt_str_infohash(p, hash->infohash)
        );
        return ret;
    }

    bzero(&index_key, sizeof(index_key));
    bzero(&peer_key, sizeof(peer_key));
    bzero(&peer_val, sizeof(peer_val));
 
    index_key.data = apr_palloc(p, BT_INFOHASH_LEN);
    index_key.size = index_key.ulen = BT_INFOHASH_LEN;
    index_key.flags = DB_DBT_USERMEM;
    memcpy(index_key.data, hash->infohash, BT_INFOHASH_LEN);
 
    peer_key.data = apr_palloc(p, BT_PEERID_LEN + BT_INFOHASH_LEN);
    peer_key.size = 0;
    peer_key.ulen = BT_PEERID_LEN + BT_INFOHASH_LEN;
    peer_key.flags = DB_DBT_USERMEM;
 
    peer_val.data = apr_palloc(p, sizeof(btt_peer));
    peer_val.size = 0;
    peer_val.ulen = sizeof(btt_peer);
    peer_val.flags = DB_DBT_USERMEM;

    if(
        (ret = index_cur->c_pget(
            index_cur, &index_key, &peer_key, &peer_val, DB_SET | cc_flags
        )) != 0
    )
        if(ret != DB_NOTFOUND)
            tracker->db.index->err(
                tracker->db.index, ret,
                "bt_txn_iterate_peerlist(%s): first peersindex c_pget()",
                bt_str_infohash(p, hash->infohash)
            );

    while(!ret)
        if((ret = btt_txn_iterate_callbacks(
            p, txn, index_cur, &peer_key, &peer_val, callbacks)
        ) == 0)
            if((ret = index_cur->c_pget(
                index_cur, &index_key, &peer_key, &peer_val,
                DB_NEXT_DUP | cc_flags)
            ) != 0)
                if(ret != DB_NOTFOUND)
                    tracker->db.index->err(
                        tracker->db.index, ret,
                        "bt_txn_iterate_peerlist(%s): iterative "
                        "index_cur->c_pget()",
                        bt_str_infohash(p, hash->infohash)
                    );
 
    if(ret == BT_CALLBACK_STOP_ALL || ret == DB_NOTFOUND)
        ret = 0;

    index_cur->c_close(index_cur);
    return ret;
}
