/*
** Copyright (C) 1998 Steve Borho <steve@borho.myip.org>
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
** 
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
** 
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software 
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/


#define FSTAB     "/etc/fstab"

#include "mount.h"

int             autoDetect(char *);

/* Attempt to glean pertinent information from /etc/fstab before even
 * looking at config files */
void            parseFstab(allInfo * ai)
{
    FILE           *fs;
    char            buf[256];
    char           *tok, *nm_ptr;
    int             nm = ai->numMounts;
    char            devStrSave[MP_LEN];

    if (access(FSTAB, R_OK) != 0 || (fs = fopen(FSTAB, "r")) == 0)
    {
        fprintf(stderr, "Unable to process fstab\n");
        return;
    }

/*  sample line from /etc/fstab
   /dev/cdrom   /mnt/cdrom   iso9660 noauto,ro,user  0 0
*/
    while (fgets(buf, sizeof(buf), fs))
    {
        /* Skip comments */
        if (*buf == '#')
            continue;

        /* skip NFS mounts */
        if (ignoreNFS && strncmp(buf, "/dev/", 5))
            continue;

        /* Skip proc mounts */
        if (strncmp(buf, "proc", 4) == 0 || strncmp(buf, "none", 4) == 0)
            continue;

        /* Truncate device name in buf */
        strtok(buf, " \t");

        safeCopy(devStrSave, buf, sizeof(devStrSave));

        /* Truncate mount name from tok */
        tok = strtok(NULL, " \t");

        if (!tok)
            continue;

        /* Skip swap partitions */
        if (strcmp(tok, "swap") == 0 || strcmp(tok, "none") == 0)
            continue;

        if (nm >= MAX_MOUNT_POINTS - 1)
            break;

        ai->mpi[nm] = (struct MPInfo *)malloc(sizeof(struct MPInfo));
        
        if (ai->mpi[nm] == 0)
            break;

        bzero(ai->mpi[nm], sizeof(struct MPInfo));
        safeCopy(ai->mpi[nm]->mountStr, tok, MP_LEN);
        safeCopy(ai->mpi[nm]->devStr, devStrSave, MP_LEN);

        /* Default the name to be the last dirname in mount point */
        if ((nm_ptr = strrchr(tok, '/')) != NULL && nm_ptr != tok)
            safeCopy(ai->mpi[nm]->name, nm_ptr + 1, NAME_LEN);
        else
            safeCopy(ai->mpi[nm]->name, tok, NAME_LEN);

        ai->mpi[nm]->isMounted = 0;
        ai->mpi[nm]->currentCap = 0;
        ai->mpi[nm]->showCap = CAPACITY_PFULL;
        ai->mpi[nm]->icon = autoDetect(buf + 5);

        /* Make wild guess that this is a zip drive */
        if (strcmp(ai->mpi[nm]->name, "zip") == 0 ||
            strcmp(ai->mpi[nm]->name, "/zip") == 0)
            ai->mpi[nm]->icon = ICO_ZIP;

        nm++;

        /* Truncate fs type from tok */
        tok = strtok(NULL, " \t");
        if (!tok)
            continue;

        /* Identify probable icon type by filesystem */
#ifdef __FreeBSD__
        if (strcmp(tok, "cd9660") == 0)
#else
        if (strcmp(tok, "iso9660") == 0)
#endif
            ai->mpi[nm - 1]->icon = ICO_CD;

        if (ai->mpi[nm - 1]->icon == ICO_HD)
        {
          if (  !strcmp(tok, "vfat" ) ||
                !strcmp(tok, "fat"  ) ||
                !strcmp(tok, "msdos")    )
              ai->mpi[nm - 1]->icon = ICO_DOS; /* Wintel filesystem */
          else if (!strcmp(tok, "hfs"))
              ai->mpi[nm - 1]->icon = ICO_MAC; /* Macintosh filesystem */
        }

        /* Truncate mount optons from tok */
        tok = strtok(NULL, " \t");
        if (!tok)
            continue;

        /* If the drive is user mountable, don't show capacity */
        if (strstr(tok, "user") != NULL)
            ai->mpi[nm - 1]->showCap = CAPACITY_NONE;
        else if (userOnly)
            ai->mpi[nm - 1]->ignored = TRUE;

        /* If the drive is not automounted, don't show capacity */
        if (strstr(tok, "noauto") != NULL)
            ai->mpi[nm - 1]->showCap = CAPACITY_NONE;
    }

    ai->numMounts = nm;
    fclose(fs);
}

int             autoDetect(char *devType)
{
#ifdef __linux__
    if (strcmp(devType, "cdrom") == 0)
        return ICO_CD;
    if (strncmp(devType, "fd", 2) == 0)
        return ICO_FLOPPY;
    if (strncmp(devType, "hd", 2) == 0)
    {
        if (isdigit(devType[3]))
            return ICO_HD;
        else
            return ICO_CD;
    }
    if (strncmp(devType, "sd", 2) == 0)
        return ICO_HD;
    if (strncmp(devType, "sr", 2) == 0)
        return ICO_CD;
#endif
#ifdef __FreeBSD__
    if (strncmp(devType, "wcd", 3) == 0)
        return ICO_CD;
    if (strncmp(devType, "scd", 3) == 0)
        return ICO_CD;
    if (strncmp(devType, "wd", 2) == 0)
    {
        if (isdigit(devType[2]))
            return ICO_HD;
        else
            return ICO_CD;
    }
    if (strncmp(devType, "sd", 2) == 0)
        return ICO_HD;
#endif
    if (strncmp(devType, "fd", 2) == 0)
        return ICO_FLOPPY;
    if (strncmp(devType, "zip", 3) == 0)
        return ICO_ZIP;
    return ICO_OTHER;
}
