# -*- coding: utf-8 -*-

# Copyright (c) 2011, 2012 Jack Kaliko <efrim@azylum.org>
#
#  This file is part of MPD_sima
#
#  MPD_sima is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  MPD_sima is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with MPD_sima.  If not, see <http://www.gnu.org/licenses/>.
#
#

import sys
from select import select

try:
    from mpd import (MPDClient, MPDError, CommandError)
except ImportError, err:
    print 'ERROR: "%s"\n\nPlease install python-mpd module.\n' % err
    sys.exit(1)

if 'idle' not in MPDClient()._commands:
    print 'ERROR: Please upgrade to at least python-mpd 0.3 (idle command required)'
    sys.exit(1)

from lib.track import Track


class PlayerError(Exception):
    """Fatal error in poller."""

class PlayerCommandError(Exception):

    """Command error"""


class Player(object):
    """
    WARNING: MPD only deals with UTF-8 strings
    Here is a possible track return by MPD
            >>> {'album': 'Kollaps Tradixionales',
            >>>  'albumartist': 'Thee Silver Mt. Zion Memorial Orchestra',
            >>>  'title': 'Collapse Traditional (For Darling)',
            >>>  'artist': 'A Silver Mt. Zion',
            >>>  'track': '5/7', 'pos': '26', 'id': '40',
            >>>  'file': 'muse/A Silver Mt. Zion/2010-Kollaps Tradixionales [MP3]/05-A Silver Mt. Zion - Collapse Traditional (For Darling).mp3',
            >>>  'time': '89', 'date': '2010-02-08',
            >>>  'disc': '1/1'}
    """
    def __init__(self, host="localhost", port="6600", password=None):
        self._host = host
        self._port = port
        self._password = password
        self._client = MPDClient()
        self._client.iterate = True

    def connect(self):
        self.disconnect()
        try:
            self._client.connect(self._host, self._port)

        # Catch socket errors
        except IOError as (errno, strerror):
            raise PlayerError('Could not connect to "%s:%s": %s' %
                              (self._host, self._port, strerror))

        # Catch all other possible errors
        # ConnectionError and ProtocolError are always fatal.  Others may not
        # be, but we don't know how to handle them here, so treat them as if
        # they are instead of ignoring them.
        except MPDError as err:
            raise PlayerError('Could not connect to "%s:%s": %s' %
                              (self._host, self._port, err))

        if self._password:
            try:
                self._client.password(self._password)

            # Catch errors with the password command (e.g., wrong password)
            except CommandError as err:
                raise PlayerError("Could not connect to '%s': "
                                  "password commmand failed: %s" %
                                  (self._host, err))

            # Catch all other possible errors
            except (MPDError, IOError) as err:
                raise PlayerError("Could not connect to '%s': "
                                  "error with password command: %s" %
                                  (self._host, err))
        # Controls we have sufficient rights for MPD_sima
        needed_cmds = ['status', 'stats', 'add', 'find', \
                       'search', 'currentsong', 'ping']

        available_cmd = self._client.commands()
        for nddcmd in needed_cmds:
            if nddcmd not in available_cmd:
                self.disconnect()
                raise PlayerError('Could connect to "%s", '
                                  'but command "%s" not available' %
                                  (self._host, nddcmd))

    def disconnect(self):
        # Try to tell MPD we're closing the connection first
        try:
            self._client.close()

        # If that fails, don't worry, just ignore it and disconnect
        except (MPDError, IOError):
            pass

        try:
            self._client.disconnect()

        # Disconnecting failed, so use a new client object instead
        # This should never happen.  If it does, something is seriously broken,
        # and the client object shouldn't be trusted to be re-used.
        except (MPDError, IOError):
            self._client = MPDClient()

    def find_artist(self, artist=None):
        return [Track(**track) for track in self._client.find('artist', artist.encode('utf-8'))]

    def find_album(self, album=None):
        return [Track(**track) for track in self._client.find('album', album.encode('utf-8'))]

    def find_artist_title(self, artist=None, title=None):
        art_enc = artist.encode('utf-8')
        tit_enc = title.encode('utf-8')
        return [Track(**track) for track in self._client.find('artist', art_enc,
            'title', tit_enc)]

    def find_artist_album(self, artist=None, album=None):
        art_enc = artist.encode('utf-8')
        alb_enc = album.encode('utf-8')
        return [Track(**track) for track in self._client.find('artist', art_enc,
            'album', alb_enc)]

    def list_album(self, artist=None):
        if artist:  # for a specific artist
            art_enc = artist.encode('utf-8')
            albums_list = self._client.list('album', 'artist', art_enc)
        else:
            albums_list = self._client.list('album')
        return [unicode(a, 'UTF-8') for a in albums_list]  # get unicode

    def list(self, what='artist'):
        listing = self._client.list(what)
        return [unicode(a, 'UTF-8') for a in listing]  # get unicode

    def _add(self, ftrack=None):
        if not isinstance(ftrack, list):
            ftrack = [ftrack,]
        for trk in ftrack:
            self._client.add(trk)

    def add(self, ftrack=None):
        try:
            self._add(ftrack)
        except CommandError as err:
            raise PlayerCommandError('MPD command error: %s' % err)

    def remove(self, position=0):
        self._client.delete(position)

    def idle(self):
        try:
            self._client.send_idle('database', 'playlist', 'player', 'options')
            select([self._client], [], [], 60)
            return self._client.fetch_idle()
        except (MPDError, IOError) as err:
            raise PlayerError("Couldn't init idle: %s" % err)

    def currentsong(self):
        try:
            song = self._client.currentsong()
        except (MPDError, IOError) as err:
            raise PlayerError("Couldn't retrieve current song: %s" % err)
        return Track(**song)

    def get_state(self):
        return str(self._client.status().get('state'))

    def get_status(self):
        """
        Attributes:
            * state: string in stop/pause/play
            * playlistlength: int
            * single: int (as boolean for the play mode)
            * repeat: int (as boolean for the play mode)
            * song: int (current  playlist song number)
        """
        return self._client.status()

    def get_playlist(self):
        """
        Returns a list of tracks
        """
        return [Track(**track) for track in self._client.playlistinfo()]


def main():
    from time import sleep

    poller = Player(host='localhost')
    poller.connect()
    while True:
        song = poller.currentsong()
        print song
        sleep(3)


if __name__ == "__main__":
    import sys
    try:
        main()
    # Catch fatal poller errors
    except PlayerError as e:
        print >> sys.stderr, "Fatal poller error: %s" % e
        sys.exit(1)

    # Catch all other non-exit errors
    except Exception as e:
        print >> sys.stderr, "Unexpected exception: %s" % e
        sys.exit(1)

    # Catch the remaining exit errors
    except:
        sys.exit(0)

# VIM MODLINE
# vim: ai ts=4 sw=4 sts=4 expandtab
