/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2000-2001 CodeFactory AB
 * Copyright (C) 2000-2001 Richard Hult <rhult@codefactory.se>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Richard Hult
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <gtk/gtksignal.h>
#include <gdk/gdkkeysyms.h>
#include <math.h>
#include <libgnomeui/gnome-canvas.h>
#include <libgnomeui/gnome-canvas-util.h>
#include <gal/e-table/e-tree.h>
#include <gal/e-table/e-table-subset.h>
#include <gal/e-table/e-tree-model.h>
#include <gal/e-table/e-tree-memory.h>
#include <gal/widgets/e-canvas.h>
#include <gal/widgets/e-canvas-utils.h>
#include <gal/util/e-util.h>
#include <gal/widgets/e-font.h>
#include <libgnomeprint/gnome-print.h>
#include <gal/widgets/e-unicode.h>
#include "util/corba-utils.h"
#include "util/type-utils.h"
#include "util/time-utils.h"
#include "util/id-map.h"
#include "gantt-model.h"
#include "gantt-scale.h"
#include "gantt-util.h"
#include "gantt-item.h"
#include "gantt-row-item.h"
#include "gantt-arrow-item.h"
#include "gantt-print.h"

#define d(x)

enum {
	ROW_CLICKED,
	ROW_DOUBLE_CLICKED,
	ROW_RIGHT_CLICKED,
	LAST_SIGNAL
};

/* Every row is as heigh as the font used, + this padding. */
#define TEXT_PAD 4

#define DRAW_H_LINES 0

static gint signals [LAST_SIGNAL] = { 0, };

enum {
	ARG_0,
	ARG_GANTT_MODEL,
	ARG_GANTT_SCALE,
	ARG_TABLE_MODEL,
	ARG_X1,
	ARG_Y1,
	ARG_X2,
	ARG_Y2
};

struct _GanttItemPriv {
	GanttModel      *gantt_model;
	GanttScale      *gantt_scale;
	ETableModel     *table_model;

	IdMap           *row_map;
	IdMap	        *dependency_map;
	EFont		*font;
	gint		 font_height;

	gdouble          x1, y1;
	gdouble          x2, y2;

	gint             rows;
	GArray		*row_items;
	
	/* Ids for the signals we connect to. */
	int              table_model_change_id;
	int              table_model_row_change_id;
	int              table_model_row_inserted_id;
	int              table_model_row_deleted_id;

	GdkGC           *grid_gc;

	gint             reflow_idle_id;
};

static void gantt_item_class_init       (GtkObjectClass       *object_class);
static void gantt_item_init             (GnomeCanvasItem      *item);

static int gantt_item_get_height        (GanttItem            *gantt);
static void table_model_rows_inserted   (ETableModel          *table_model,
					 int                   row,
					 int                   count,
					 GanttItem            *gantt);
static void table_model_rows_deleted    (ETableModel          *table_model,
				         int                   row,
				         int                   count,
				         GanttItem            *gantt);
static void task_linked                 (GanttModel           *gantt_model,
					 GNOME_MrProject_Dependency *dependency,
					 GanttItem            *item);
static void task_unlinked               (GanttModel           *gantt_model,
					 GNOME_MrProject_Dependency *dependency,
					 GanttItem            *item);

static void allocation_assigned         (GanttModel               *gantt_model,
					 GNOME_MrProject_Resource *resource,
					 GNOME_MrProject_Id        task_id,
					 GanttItem                *item);

static void allocated_resource_changed  (GanttModel               *gantt_model,
					 GM_Id                     task_id,
					 GM_Resource              *resource,
					 GanttItem                *item);

GNOME_CLASS_BOILERPLATE (GanttItem,
			 gantt_item,
			 GnomeCanvasItem,
			 gnome_canvas_item);

static void
gantt_item_bounds_item_coordinates (GnomeCanvasItem *item,
				    double	    *x1,
				    double          *y1,
				    double          *x2,
				    double          *y2)
{
	GanttItem     *gantt;
	GanttItemPriv *priv;

	gantt = GANTT_ITEM (item);
	priv = gantt->priv;
	
	*x1 = priv->x1 - 1;
	*y1 = priv->y1 - 1;
	*x2 = priv->x2 + 1;
	*y2 = priv->y2 + 1;
}

static void
gantt_item_bounds_canvas_coordinates (GnomeCanvasItem *item,
				      double          *x1,
				      double          *y1,
				      double          *x2,
				      double          *y2)
{
	GanttItem     *gantt;
	GanttItemPriv *priv;
	gdouble        i2c[6];
	ArtPoint       c1, c2, i1, i2;

	gantt = GANTT_ITEM (item);
	priv = gantt->priv;

	gantt_item_bounds_item_coordinates (item,
					    &i1.x,
					    &i1.y,
					    &i2.x,
					    &i2.y);
	
	gnome_canvas_item_i2c_affine (item, i2c);
	art_affine_point (&c1, &i1, i2c);
	art_affine_point (&c2, &i2, i2c);
	
	*x1 = c1.x;
	*y1 = c1.y;
	*x2 = c2.x;
	*y2 = c2.y;
}

static gboolean
real_reflow (GanttItem *gantt)
{
	GanttItemPriv   *priv;
	GnomeCanvasItem *row_item;
	gint             row, rows;
	gint             font_height;
	gint             yd;
	gdouble          new_height;

	priv = gantt->priv;
	rows = priv->rows;
	font_height = priv->font_height;

	yd = 0;
	for (row = 0; row < rows; row++) {
		row_item = g_array_index (priv->row_items, GnomeCanvasItem *, row);
		gtk_object_set (GTK_OBJECT (row_item),
				"y", (double) yd,
				"height", (double) priv->font_height + TEXT_PAD + 1,
				NULL);

		yd += font_height + TEXT_PAD + 1;
	}

	new_height = gantt_item_get_height (gantt);
	if (new_height != (priv->y2 - priv->y1)) {
		priv->y2 = priv->y1 + new_height;
	}

	gnome_canvas_item_request_update (GNOME_CANVAS_ITEM (gantt));
	
	priv->reflow_idle_id = 0;
	return FALSE;
}

static void
gantt_item_reflow (GnomeCanvasItem *item)
{
	GanttItem     *gantt;
	GanttItemPriv *priv;

	gantt = GANTT_ITEM (item);
	priv = gantt->priv;

	if (priv->reflow_idle_id != 0) {
		return;
	}

	priv->reflow_idle_id = gtk_idle_add_priority (GTK_PRIORITY_DEFAULT,
						      (GtkFunction) real_reflow,
						      gantt);
}
	
static void
units_changed (GanttScale *gantt_scale, GanttItem *item)
{
	/*gnome_canvas_item_request_update (GNOME_CANVAS_ITEM (item));*/
}

static void
scale_changed (GanttScale *gantt_scale, GanttItem *item)
{
	GanttItemPriv *priv;
	GnomeCanvas   *canvas;
	GtkAllocation *allocation;
	gdouble        width, height;
	gint           i, rows;

	priv = item->priv;
	canvas = GNOME_CANVAS_ITEM (item)->canvas;
	allocation = &GTK_WIDGET (canvas)->allocation;

	priv->x1 = gantt_scale->x1;
	priv->x2 = gantt_scale->x2;

	width = priv->x2 - priv->x1;
	height = priv->y2 - priv->y1;
	
	width = MAX ((int) width, allocation->width);
	height = MAX ((int) height, allocation->height);

	gnome_canvas_set_scroll_region (
		canvas,
		priv->x1,
		priv->y1,
		priv->x1 + width - 1,
		priv->y1 + height - 1);

	rows = priv->rows;
	for (i = 0; i < rows; i++) {
		gdouble               x1, x2;
		GNOME_MrProject_Task *task;
		GtkObject            *row_item;

		row_item = g_array_index (priv->row_items, GtkObject *, i);
		gtk_object_get (GTK_OBJECT (row_item),
				"task", &task,
				NULL);

		x1 = gantt_scale_t2w (gantt_scale, task->start);
		x2 = gantt_scale_t2w (gantt_scale, task->end);

		gnome_canvas_item_set (GNOME_CANVAS_ITEM (row_item),
				       "x", x1,
				       "width", x2 - x1,
				       NULL);
	}
	
	gantt_item_reflow (GNOME_CANVAS_ITEM (item));
}

static void
viewport_changed (GanttScale *gantt_scale, GanttItem *item)
{
	GanttItemPriv *priv;
	GnomeCanvas   *canvas;
	GtkAllocation *allocation;
	gdouble        width, height;

	priv = item->priv;
	canvas = GNOME_CANVAS_ITEM (item)->canvas;
	allocation = &GTK_WIDGET (canvas)->allocation;

	priv->x1 = gantt_scale->x1;
	priv->x2 = gantt_scale->x2;

	width = priv->x2 - priv->x1;
	height = priv->y2 - priv->y1;
	
	width = MAX ((int) width, allocation->width);
	height = MAX ((int) height, allocation->height);

	gnome_canvas_set_scroll_region (
		canvas,
		priv->x1,
		priv->y1,
		priv->x1 + width - 1,
		priv->y1 + height - 1);

	gantt_item_reflow (GNOME_CANVAS_ITEM (item));
}

static void
gantt_item_update (GnomeCanvasItem *item,
		   double          *affine,
		   ArtSVP          *clip_path,
		   int              flags)
{
	gdouble x1, y1, x2, y2;

	GNOME_CALL_PARENT_HANDLER (GNOME_CANVAS_ITEM_CLASS,
				   update,
				   (item, affine, clip_path, flags));
	gantt_item_bounds_canvas_coordinates (item, &x1, &y1, &x2, &y2);
	gnome_canvas_update_bbox (item, x1, y1, x2, y2);
}

static void
gantt_item_remove_table_model (GanttItem *gantt)
{
	GanttItemPriv *priv;

	priv = gantt->priv;

	if (!priv->table_model) {
		return;
	}

	gtk_signal_disconnect (GTK_OBJECT (priv->table_model),
			       priv->table_model_change_id);
	gtk_signal_disconnect (GTK_OBJECT (priv->table_model),
			       priv->table_model_row_change_id);
	gtk_signal_disconnect (GTK_OBJECT (priv->table_model),
			       priv->table_model_row_inserted_id);
	gtk_signal_disconnect (GTK_OBJECT (priv->table_model),
			       priv->table_model_row_deleted_id);
	gtk_object_unref (GTK_OBJECT (priv->table_model));

	priv->table_model_change_id       = 0;
	priv->table_model_row_change_id   = 0;
	priv->table_model_row_inserted_id = 0;
	priv->table_model_row_deleted_id  = 0;
	priv->table_model                 = NULL;
}

static int
gantt_item_get_height (GanttItem *gantt)
{
	return (gantt->priv->font_height + TEXT_PAD + 1) * gantt->priv->rows;
}

static void
gantt_item_item_region_redraw (GanttItem *gantt, int x0, int y0, int x1, int y1)
{
	GnomeCanvasItem *item;
	ArtDRect         rect;
	double           i2c[6];

	item = (GnomeCanvasItem *) gantt;
	
	rect.x0 = x0;
	rect.y0 = y0;
	rect.x1 = x1;
	rect.y1 = y1;

	gnome_canvas_item_i2c_affine (item, i2c);
	art_drect_affine_transform (&rect, &rect, i2c);

	gnome_canvas_request_redraw (item->canvas, rect.x0, rect.y0, rect.x1, rect.y1);
}

static void
row_item_clicked (GanttRowItem *row_item, GdkEvent *event, GanttItem *gantt)
{

	switch (event->type) {
	case GDK_BUTTON_PRESS:
		gtk_signal_emit (GTK_OBJECT (gantt),
				 signals[ROW_CLICKED],
				 row_item,
				 event);
		break;
	case GDK_2BUTTON_PRESS:
		gtk_signal_emit (GTK_OBJECT (gantt),
				 signals[ROW_DOUBLE_CLICKED],
				 row_item,
				 event);
		break;
	default:
		break;
	}
	
}


/* small helper function */
static void
create_dependency_map (IdMap *map, GSList *dep_list)
{
	GSList *node;

	g_return_if_fail (IS_ID_MAP (map));

	for (node = dep_list; node; node = node->next) {
		GM_Dependency *dep;

		dep = (GM_Dependency *)node->data;
		g_assert (dep != NULL); /* just in case */

		id_map_insert_id (map, dep->depId, dep);
	}
}

static void
unlink_dependencies (gpointer key, GM_Dependency *dependency, GanttItem *gantt)
{
	task_unlinked (gantt->priv->gantt_model, dependency, gantt);
}

static void
link_dependencies (gpointer key, GM_Dependency *dependency, GanttItem *gantt)
{
	task_linked (gantt->priv->gantt_model, dependency, gantt);
}

static void
insert_rows (ETableModel *table_model,
	     int          row,
	     int          count,
	     GanttItem   *gantt)
{
	GanttItemPriv   *priv;
	GanttModel      *gantt_model;
	ETreeModel      *etm;
	ETreePath        node;
	GM_Task         *task;
	GnomeCanvasItem *item;
	gdouble          x1, x2;
	gint             i;
	GSList          *dependencies, *dependencies_list;
	GSList          *list;
	IdMap           *dep_map;

	priv = gantt->priv;
	gantt_model = priv->gantt_model;
	etm = gantt_model->etm;

	gantt->priv->rows = e_table_model_row_count (table_model);
	
	dep_map = id_map_new (0);

	dependencies_list = NULL;
	for (i = row; i < row + count; i++) {
		node = e_tree_node_at_row (gantt_model->e_tree, i);
		task = e_tree_memory_node_get_data (E_TREE_MEMORY (etm), node);

		x1 = gantt_scale_t2w (priv->gantt_scale, task->start);
		x2 = gantt_scale_t2w (priv->gantt_scale, task->end);
		
		item = gnome_canvas_item_new (
			gnome_canvas_root (GNOME_CANVAS_ITEM (gantt)->canvas),
			TYPE_GANTT_ROW_ITEM,
			"task", task,
			"leaf", gantt_model_task_is_leaf (gantt_model, task->taskId),
			"x", x1,
			"width", x2 - x1,
			NULL);

		/* Proxy click signals. */
		gtk_signal_connect (GTK_OBJECT (item),
				    "clicked",
				    row_item_clicked,
				    gantt);

		/* Get dependencies here, add them after all tasks are added back. */
		dependencies = gantt_model_task_get_predecessors (gantt_model, task->taskId);
		create_dependency_map (dep_map, dependencies);

		dependencies = gantt_model_task_get_successors (gantt_model, task->taskId);
		create_dependency_map (dep_map, dependencies);
		
		g_array_insert_val (priv->row_items, i, item);
		id_map_insert_id (priv->row_map, task->taskId, item);

		/* Add assigned resources. */
		list = gantt_model_task_get_resources (gantt_model, task->taskId);
		for (; list; list = list->next) {
			GM_Resource *resource;
			
			resource = list->data;
			allocation_assigned (gantt_model,
					     resource,
					     task->taskId,
					     gantt);
		}
	}

	id_map_foreach (dep_map, (GHFunc)link_dependencies, gantt);

	/* Free the memory used by depmap - the dependencies should not be freed. */
	gtk_object_unref (GTK_OBJECT (dep_map));

	gantt_item_reflow (GNOME_CANVAS_ITEM (gantt));
}



static void
delete_rows (ETableModel *table_model,
	     int          row,
	     int          count,
	     GanttItem   *gantt)
{
	GtkObject            *item;
	GNOME_MrProject_Task *task;
	gint                  i;
	GSList               *deps;
	IdMap                *dep_map;

	gantt->priv->rows = e_table_model_row_count (table_model);

	dep_map = id_map_new (0);

	/* collect all dependencies */
	for (i = row + count - 1; i >= row; i--) {
		item = g_array_index (gantt->priv->row_items, GtkObject *, i);
		gtk_object_get (item, "task", &task, NULL);

		deps = gantt_model_task_get_predecessors (gantt->priv->gantt_model, task->taskId);
		create_dependency_map (dep_map, deps);

		deps = gantt_model_task_get_successors (gantt->priv->gantt_model, task->taskId);
		create_dependency_map (dep_map, deps);
	} 
	id_map_foreach (dep_map, (GHFunc)unlink_dependencies, gantt);
	/* free the memory used by the dep_map, the dependencies are freed elsewhere */
	gtk_object_destroy (GTK_OBJECT (dep_map));

	/* destroy the items */
	for (i = row + count - 1; i >= row; i--) {
		item = g_array_index (gantt->priv->row_items, GtkObject*, i);
		gtk_object_get (item, "task", &task, NULL);

		/* FIXME: destroy allocated resources */

		g_array_remove_index (gantt->priv->row_items, i);
		id_map_remove (gantt->priv->row_map, task->taskId);
		gtk_object_destroy (item);
	}

	gantt_item_reflow (GNOME_CANVAS_ITEM (gantt));
}

static void
table_model_rows_inserted (ETableModel *table_model,
			   int          row,
			   int          count,
			   GanttItem   *gantt)
{
	insert_rows (table_model, row, count, gantt);
}

static void
table_model_rows_deleted (ETableModel *table_model,
			  int          row,
			  int          count,
			  GanttItem   *gantt)
{
	delete_rows (table_model, row, count, gantt);
}

static void
table_model_changed (ETableModel *table_model, GanttItem *gantt)
{
	gint rows;

	/* Remove all the row items. */
	rows = gantt->priv->row_items->len;
	if (rows > 0) {
		delete_rows (table_model, 0, rows, gantt);
	}

	gantt->priv->rows = 0;
	
	/* Rebuild the row items. */
	rows = e_table_model_row_count (gantt->priv->table_model);
	if (rows > 0) {
		insert_rows (table_model, 0, rows, gantt);
	}
	
	gnome_canvas_item_request_update (GNOME_CANVAS_ITEM (gantt));
}

/* Computes the distance between two rows in pixels. */
static int
gantt_item_row_diff (GanttItem *gantt, int start_row, int end_row)
{
	return (gantt->priv->font_height + TEXT_PAD + 1) * (end_row - start_row);
}

static void
gantt_item_request_region_redraw (GanttItem *gantt, int start_row, int end_row)
{
	GanttItemPriv *priv;
	int            y1;
	int            height;

	priv = gantt->priv;

	if (priv->rows > 0) {
		y1 = gantt_item_row_diff (gantt, 0, start_row);
		height = gantt_item_row_diff (gantt, start_row, end_row + 1);

		gantt_item_item_region_redraw (gantt, 
					       priv->x1,
					       priv->y1 + y1,
					       priv->x2 + 1,
					       priv->y1 + height + 1 + TEXT_PAD);
	}
}

static void
table_model_row_changed (ETableModel *table_model, int row, GanttItem *gantt)
{
	ETreeModel           *etm;
	GanttModel           *gantt_model;
	GNOME_MrProject_Task *task;
	ETreePath             node;
	GnomeCanvasItem      *row_item;
	gdouble               x1, x2;

	gantt_model = gantt->priv->gantt_model;
	etm = gantt_model->etm;

	node = e_tree_node_at_row (gantt_model->e_tree, row);
	task = e_tree_memory_node_get_data (E_TREE_MEMORY (etm), node);

	x1 = gantt_scale_t2w (gantt->priv->gantt_scale, task->start);
	x2 = gantt_scale_t2w (gantt->priv->gantt_scale, task->end);

	row_item = g_array_index (gantt->priv->row_items, GnomeCanvasItem*, row);
	gtk_object_set (GTK_OBJECT (row_item),
			"x", x1,
			"width", x2 - x1,
			"leaf", gantt_model_task_is_leaf (gantt_model, task->taskId),
			NULL);
}

/* This function assumes that both linked tasks are added and exist.
 */
static void
task_linked (GanttModel    *gantt_model,
	     GM_Dependency *dependency,
	     GanttItem     *item)
{
	GanttRowItem    *row1, *row2;
	GnomeCanvasItem *dependency_item;
	
	d(puts(__FUNCTION__));

	row1 = id_map_lookup (item->priv->row_map, dependency->predecessorId);
	row2 = id_map_lookup (item->priv->row_map, dependency->taskId);

	if (!row1 || !row2) {
		g_warning ("Linked task items do not exist.");
		return;
	}
 	
	g_return_if_fail (IS_GANTT_ROW_ITEM (row1));
	g_return_if_fail (IS_GANTT_ROW_ITEM (row2));
	
/*	dependency_item = gnome_canvas_item_new (GNOME_CANVAS_GROUP (GNOME_CANVAS_ITEM (item)->canvas->root),
					   TYPE_GANTT_LINK_ITEM,
					   "row1", row1,
					   "row2", row2,
					   NULL);
*/

	dependency_item = GNOME_CANVAS_ITEM (gantt_arrow_item_new (row2, row1));

	/* Some hackery: if we already have the dependency in the map, that's
	 * because we are hidden and just got showed again.
	 */
	if (!id_map_lookup (item->priv->dependency_map, dependency->depId)) {
		id_map_insert_id (item->priv->dependency_map, dependency->depId, dependency_item);
	}
}

/* This function assumes that both linked tasks are added and exist.
 */
static void
task_unlinked (GanttModel    *gantt_model,
	       GM_Dependency *dependency,
	       GanttItem     *item)
{
	GanttRowItem    *row1, *row2;
	GnomeCanvasItem *dependency_item;

	d(puts(__FUNCTION__));
	
	row1 = id_map_lookup (item->priv->row_map, dependency->predecessorId);
	row2 = id_map_lookup (item->priv->row_map, dependency->taskId);

	if (!row1 || !row2) {
		g_warning ("Linked task items do not exist.");
		return;
	}
 	
	g_return_if_fail (IS_GANTT_ROW_ITEM (row1));
	g_return_if_fail (IS_GANTT_ROW_ITEM (row2));

	dependency_item = id_map_lookup (item->priv->dependency_map, dependency->depId);
	if (!dependency_item) {
		g_warning ("Could not find dependency item.");
		return;
	}

	gtk_object_destroy (GTK_OBJECT (dependency_item));
	id_map_remove (item->priv->dependency_map, dependency->depId);
	/* FIXME: corba free here? */
}

static void
allocation_assigned (GanttModel               *gantt_model,
		     GNOME_MrProject_Resource *resource,
		     GNOME_MrProject_Id        task_id,
		     GanttItem                *item)
{
	GanttRowItem *row;
	
	row = id_map_lookup (item->priv->row_map, task_id);
	if (!row) {
		g_warning ("Assigned task does not exist.");
		return;
	}

	gantt_row_item_add_resource (row, resource->resourceId, resource->name);
}

static void
allocation_unassigned (GanttModel         *gantt_model,
		       GNOME_MrProject_Id  resource_id,
		       GNOME_MrProject_Id  task_id,
		       GanttItem          *item)
{
	GanttRowItem    *row;
	
	row = id_map_lookup (item->priv->row_map, task_id);
	if (!row) {
		g_warning ("Assigned task does not exist.");
		return;
	}

	gantt_row_item_remove_resource (row, resource_id);
}

static void
allocated_resource_changed (GanttModel    *gantt_model,
			    GM_Id          task_id,
			    GM_Resource   *resource,
			    GanttItem     *item) 
{
	GanttRowItem   *row;
	
	row = id_map_lookup (item->priv->row_map, task_id);
	
	if (!row) {
		g_warning ("Assigned task does not exist.");
		return;
	}

	gantt_row_item_update_resource (row, 
					resource->resourceId, 
					resource->name);
}

void
gantt_item_redraw_range (GanttItem *gantt, int start_row, int end_row)
{
	g_return_if_fail (gantt != NULL);
	g_return_if_fail (IS_GANTT_ITEM (gantt));
	
	gantt_item_request_region_redraw (gantt, start_row, end_row);
}

static void
gantt_item_set_table_model (GanttItem *gantt, ETableModel *table_model)
{
	GanttItemPriv *priv;

	g_return_if_fail (table_model != NULL);
	g_return_if_fail (E_IS_TABLE_MODEL (table_model));
	
	priv = gantt->priv;
	g_assert (priv->table_model == NULL);

	priv->table_model = table_model;
	gtk_object_ref (GTK_OBJECT (table_model));

	priv->table_model_change_id = gtk_signal_connect (
		GTK_OBJECT (table_model), "model_changed",
		GTK_SIGNAL_FUNC (table_model_changed), gantt);

	priv->table_model_row_change_id = gtk_signal_connect (
		GTK_OBJECT (table_model), "model_row_changed",
		GTK_SIGNAL_FUNC (table_model_row_changed), gantt);

	priv->table_model_row_inserted_id = gtk_signal_connect (
		GTK_OBJECT (table_model), "model_rows_inserted",
		GTK_SIGNAL_FUNC (table_model_rows_inserted), gantt);

 	priv->table_model_row_deleted_id = gtk_signal_connect (
		GTK_OBJECT (table_model), "model_rows_deleted",
		GTK_SIGNAL_FUNC (table_model_rows_deleted), gantt);

	table_model_changed (table_model, gantt);
}

static void
gantt_item_destroy (GtkObject *object)
{
	GanttItem *gantt = GANTT_ITEM (object);

	gantt_item_remove_table_model (gantt);

	GNOME_CALL_PARENT_HANDLER (GTK_OBJECT_CLASS, destroy, (object));
}

static void
gantt_item_set_arg (GtkObject *object, GtkArg *arg, guint arg_id)
{
	GnomeCanvasItem *item;
	GanttItem       *gantt;
	GanttItemPriv   *priv;

	item = GNOME_CANVAS_ITEM (object);
	gantt = GANTT_ITEM (object);
	priv = gantt->priv;
	
	switch (arg_id){
	case ARG_GANTT_MODEL:
		/* We don't handle changing models after setting it (yet?). */
		if (priv->gantt_model != NULL) {
			g_warning ("Changing gantt model is not allowed.");
			return;
		}
		priv->gantt_model = GANTT_MODEL (GTK_VALUE_OBJECT (*arg));

		gtk_signal_connect (GTK_OBJECT (priv->gantt_model),
				    "task_linked",
				    task_linked,
				    gantt);
		gtk_signal_connect (GTK_OBJECT (priv->gantt_model),
				    "task_unlinked",
				    task_unlinked,
				    gantt);
		gtk_signal_connect (GTK_OBJECT (priv->gantt_model),
				    "allocation_assigned",
				    allocation_assigned,
				    gantt);
		gtk_signal_connect (GTK_OBJECT (priv->gantt_model),
				    "allocation_unassigned",
				    allocation_unassigned,
				    gantt);
		gtk_signal_connect (GTK_OBJECT (priv->gantt_model),
				    "allocated_resource_changed",
				    allocated_resource_changed,
				    gantt);
		break;

	case ARG_GANTT_SCALE:
		/* We don't handle changing scales after setting it (yet?). */
		if (priv->gantt_scale != NULL) {
			g_warning ("Changing gantt scale is not allowed.");
			return;
		}
		priv->gantt_scale = GANTT_SCALE (GTK_VALUE_OBJECT (*arg));
		
		priv->x1 = priv->gantt_scale->x1;
		priv->x2 = priv->gantt_scale->x2;

		scale_changed (priv->gantt_scale, gantt);
		
		gtk_signal_connect (GTK_OBJECT (priv->gantt_scale),
				    "scale_changed",
				    scale_changed,
				    gantt);

		gtk_signal_connect (GTK_OBJECT (priv->gantt_scale),
				    "viewport_changed",
				    viewport_changed,
				    gantt);

		gtk_signal_connect (GTK_OBJECT (priv->gantt_scale),
				    "units_changed",
				    units_changed,
				    gantt);
		break;
		
	case ARG_TABLE_MODEL:
		gantt_item_remove_table_model (gantt);
		gantt_item_set_table_model (gantt, E_TABLE_MODEL (GTK_VALUE_OBJECT (*arg)));
		break;

	case ARG_X1:
		priv->x1 = GTK_VALUE_DOUBLE (*arg);
		break;

	case ARG_Y1:
		priv->y1 = GTK_VALUE_DOUBLE (*arg);
		break;

	case ARG_X2:
		priv->x2 = GTK_VALUE_DOUBLE (*arg);
		break;

	case ARG_Y2:
		priv->y2 = GTK_VALUE_DOUBLE (*arg);
		break;

	default:
		break;
	}

	gnome_canvas_item_request_update (GNOME_CANVAS_ITEM (gantt));
}

static void
gantt_item_get_arg (GtkObject *o, GtkArg *arg, guint arg_id)
{
	GnomeCanvasItem *item;
	GanttItem       *gantt;
	GanttItemPriv   *priv;

	item = GNOME_CANVAS_ITEM (o);
	gantt = GANTT_ITEM (o);
	priv = gantt->priv;

	switch (arg_id){
	case ARG_X1:
		GTK_VALUE_DOUBLE (*arg) = priv->x1;
		break;

	case ARG_Y1:
		GTK_VALUE_DOUBLE (*arg) = priv->y1;
		break;

	case ARG_X2:
		GTK_VALUE_DOUBLE (*arg) = priv->x2;
		break;

	case ARG_Y2:
		GTK_VALUE_DOUBLE (*arg) = priv->y2;
		break;

	default:
		arg->type = GTK_TYPE_INVALID;
	}
}

static void
gantt_item_init (GnomeCanvasItem *item)
{
	GanttItem     *gantt;
	GanttItemPriv *priv;

	gantt = GANTT_ITEM (item);

	priv = g_new0 (GanttItemPriv, 1);
	gantt->priv = priv;

	priv->row_map = id_map_new (0);
	priv->dependency_map = id_map_new (0);

	priv->x1 = 0;
	priv->y1 = 0;
	priv->x2 = 0;
	priv->y2 = 0;

	priv->row_items = g_array_new (FALSE, FALSE, sizeof (GnomeCanvasItem*));
}

static void
gantt_item_realize (GnomeCanvasItem *item)
{
	GanttItem     *gantt;
	GanttItemPriv *priv;
	GtkWidget     *canvas_widget;

	gantt = GANTT_ITEM (item);
	priv = gantt->priv;
	canvas_widget = GTK_WIDGET (item->canvas);

	GNOME_CALL_PARENT_HANDLER (GNOME_CANVAS_ITEM_CLASS, realize, (item));

	priv->font = e_font_from_gdk_font (canvas_widget->style->font);
	priv->font_height = e_font_height (priv->font);	

	/* Gdk Resource allocation. */
	priv->grid_gc = gdk_gc_new (canvas_widget->window);
	gdk_gc_set_foreground (priv->grid_gc, &canvas_widget->style->bg [GTK_STATE_NORMAL]);
	gdk_gc_set_background (priv->grid_gc, &canvas_widget->style->fg [GTK_STATE_NORMAL]);

	gnome_canvas_item_request_update (GNOME_CANVAS_ITEM (gantt));
}

static void
gantt_item_unrealize (GnomeCanvasItem *item)
{
	GanttItem     *gantt;
	GanttItemPriv *priv;

	gantt = GANTT_ITEM (item);
	priv = gantt->priv;
	
	gdk_gc_unref (priv->grid_gc);
	priv->grid_gc = NULL;

	e_font_unref (priv->font);

	GNOME_CALL_PARENT_HANDLER (GNOME_CANVAS_ITEM_CLASS, unrealize, (item));
}

#if 0
static GNOME_MrProject_Task *
gantt_item_row_at_y_pos (GanttItem *gantt, gint y, gint *row)
{
	gint r;

	r = y / (gantt->priv->font_height + TEXT_PAD + 1);
	if (r >= gantt->priv->rows || r < -1) {
		r = -1;
	}
	
	if (row) {
		*row = r;
	}

	if (r == -1) {
		return NULL;
	}

	return gantt_model_get_task_at_row (gantt->priv->gantt_model, r);
}
#endif

static void
gantt_item_draw (GnomeCanvasItem *item,
		 GdkDrawable     *drawable,
		 int              x,
		 int	          y,
		 int              width,
		 int              height)
{
	GanttItem     *gantt;
	GanttItemPriv *priv;
	GanttModel    *gantt_model;
	gint           font_height;
	gint           rows;
	gint           xd, cx1, cy1, cx2, cy2, tick_width;
	ArtPoint       i1, i2, c1, c2;
	double         i2c[6], c2i[6];
	time_t         first_time, last_time, time, tick_time;
	GanttUnitType  unit;

	gantt = GANTT_ITEM (item);
	priv = gantt->priv;
	gantt_model = priv->gantt_model;
	rows = priv->rows;
	font_height = priv->font_height;

	if (rows == 0) {
		return;
	}

	gnome_canvas_item_i2c_affine (item, i2c);

	c1.x = x;
	c1.y = y;
	c2.x = x + width;
	c2.y = y + height;
	art_affine_invert (c2i, i2c);
	art_affine_point (&i1, &c1, c2i);
	art_affine_point (&i2, &c2, c2i);

	/* i is where we should start drawing, in item coordinates
	 * ( == world coordinates since we never scale the canvas).
	 */
	first_time = gantt_scale_w2t (priv->gantt_scale, i1.x);
	last_time = gantt_scale_w2t (priv->gantt_scale, i2.x);

	if (first_time < 0) {
		first_time = 0;
	}
	if (first_time >= last_time) {
		return;
	}
	
	i1.x = priv->x1;
	i1.y = priv->y1;
	i2.x = priv->x2;
	i2.y = priv->y2;
	art_affine_point (&c1, &i1, i2c); 
	art_affine_point (&c2, &i2, i2c); 

	cx1 = c1.x - x;
	cy1 = c1.y - y;
	cx2 = c2.x - x;
	cy2 = c2.y - y;

	/* Get the pixel width of a minor tick. Use it to decide if to
	 * draw tick marks.
	 */
	i1.x = gantt_scale_t2w (priv->gantt_scale, priv->gantt_scale->t1);
	i1.y = 0;
	art_affine_point (&c1, &i1, i2c);
	tick_width = floor (c1.x + 0.5) - x;

	tick_time = gantt_scale_increase_one_tick (priv->gantt_scale,
						   GANTT_UNIT_MINOR,
						   priv->gantt_scale->t1);

	i1.x = gantt_scale_t2w (priv->gantt_scale, tick_time);
	i1.y = 0;
	art_affine_point (&c1, &i1, i2c);
	tick_width = (floor (c1.x + 0.5) - x) - tick_width;
	
	if (tick_width > 20) {
		unit = GANTT_UNIT_MINOR;
	}
	else {
		unit = GANTT_UNIT_MAJOR;
	}
	
	time = gantt_scale_snap_time (priv->gantt_scale,
				      unit,
				      first_time);
	
	do {
		/* Time to canvas coord. */
		i1.x = gantt_scale_t2w (priv->gantt_scale, time);
		i1.y = 0;
		art_affine_point (&c1, &i1, i2c);
		xd = floor (c1.x + 0.5) - x;
		
		/* Don't draw a grid line at the left edge, since it looks bad. */
		if (xd + x > 0) {
			gdk_draw_line (drawable, priv->grid_gc,
				       xd, cy1, xd, cy2);
		}

		time = gantt_scale_increase_one_tick (priv->gantt_scale,
						      unit,
						      time);
	} while (time < last_time);
}

static double
gantt_item_point (GnomeCanvasItem   *item,
		  double             x,
		  double             y,
		  int                cx,
		  int                cy,
		  GnomeCanvasItem  **actual_item)
{
	GanttItem     *gantt;	GanttItemPriv *priv;

	gantt = GANTT_ITEM (item);
	priv = gantt->priv;

	if (x >= priv->x1 && x <= priv->x2 &&
	    y >= priv->y1 && y <= priv->y2) {
		*actual_item = item;
		return 0.0;
	} else {
		return 1.0;
	}
}

static int
gantt_item_event (GnomeCanvasItem *item, GdkEvent *e)
{
	GanttItem                   *gantt;
	GanttItemPriv               *priv;
	GanttModel                  *gantt_model;

	gantt = GANTT_ITEM (item);
	priv = gantt->priv;
	gantt_model = priv->gantt_model;

	switch (e->type) {
	case GDK_BUTTON_PRESS: {
	}
	break;

	case GDK_BUTTON_RELEASE: {
	}
	break;

	case GDK_2BUTTON_PRESS: {
	}
	break;

	case GDK_LEAVE_NOTIFY: {
	}
	break;

	case GDK_MOTION_NOTIFY: {
	}
	break;

	default:
		break;
	}

	return FALSE;
}

/* FIXME: this will need optimizing later on, like keeping the selected
 * rows in a list and only deselected those instead of all.
 */
void
gantt_item_unselect_all (GanttItem *gantt)
{
	GtkObject *row_item;
	guint      i, rows;

	rows = gantt->priv->row_items->len;
	
	for (i = 0; i < rows; i++) {
		row_item = g_array_index (gantt->priv->row_items,
					  GtkObject *,
					  i);
		if (GTK_IS_OBJECT (row_item)) {
			gtk_object_set (row_item,
					"selected", FALSE,
					NULL);
		}
	}
}

void
gantt_item_select_row (GanttItem *gantt, guint row)
{
	GtkObject *row_item;

	if (row >= gantt->priv->row_items->len) {
		return;
	}
	
	row_item = g_array_index (gantt->priv->row_items,
				  GtkObject *,
				  row);
	if (GTK_IS_OBJECT (row_item)) {
		gtk_object_set (row_item,
				"selected", TRUE,
				NULL);
	}
}

void
gantt_item_select_row_item (GanttItem       *gantt,
			    GnomeCanvasItem *row_item)
{
	gtk_object_set (GTK_OBJECT (row_item),
			"selected", TRUE,
			NULL);
}

static void
gantt_item_class_init (GtkObjectClass *object_class)
{
	GnomeCanvasItemClass *item_class = (GnomeCanvasItemClass *) object_class;
	GanttItemClass       *gantt_item_class = (GanttItemClass *) object_class;
	
	object_class->destroy = gantt_item_destroy;
	object_class->set_arg = gantt_item_set_arg;
	object_class->get_arg = gantt_item_get_arg;

	item_class->update    = gantt_item_update;
	item_class->realize   = gantt_item_realize;
	item_class->unrealize = gantt_item_unrealize;
	item_class->bounds    = gantt_item_bounds_item_coordinates;
	item_class->draw      = gantt_item_draw;
	item_class->point     = gantt_item_point;
	item_class->event     = gantt_item_event;
	
	gantt_item_class->cursor_change = NULL;
	gantt_item_class->double_click  = NULL;
	gantt_item_class->right_click   = NULL;
	gantt_item_class->click         = NULL;
	gantt_item_class->key_press     = NULL;

	gtk_object_add_arg_type ("GanttItem::gantt_model", GTK_TYPE_OBJECT, GTK_ARG_WRITABLE, ARG_GANTT_MODEL);
	gtk_object_add_arg_type ("GanttItem::gantt_scale", GTK_TYPE_OBJECT, GTK_ARG_WRITABLE, ARG_GANTT_SCALE);
	gtk_object_add_arg_type ("GanttItem::table_model", GTK_TYPE_OBJECT, GTK_ARG_WRITABLE, ARG_TABLE_MODEL);
	gtk_object_add_arg_type ("GanttItem::x1", GTK_TYPE_DOUBLE, GTK_ARG_READWRITE, ARG_X1);
	gtk_object_add_arg_type ("GanttItem::y1", GTK_TYPE_DOUBLE, GTK_ARG_READWRITE, ARG_Y1);
	gtk_object_add_arg_type ("GanttItem::x2", GTK_TYPE_DOUBLE, GTK_ARG_READWRITE, ARG_X2);
	gtk_object_add_arg_type ("GanttItem::y2", GTK_TYPE_DOUBLE, GTK_ARG_READWRITE, ARG_Y2);

	signals [ROW_RIGHT_CLICKED] =
		gtk_signal_new ("row-right-clicked",
				GTK_RUN_LAST,
				object_class->type,
				GTK_SIGNAL_OFFSET (GanttItemClass, right_click),
				e_marshal_INT__INT_INT_POINTER,
				GTK_TYPE_INT, 3, GTK_TYPE_INT, GTK_TYPE_INT, GTK_TYPE_POINTER);

	signals [ROW_CLICKED] =
		gtk_signal_new ("row-clicked",
				GTK_RUN_LAST,
				object_class->type,
				0,
				gtk_marshal_NONE__POINTER_POINTER,
				GTK_TYPE_NONE,
				2, GTK_TYPE_POINTER, GTK_TYPE_POINTER);

	signals [ROW_DOUBLE_CLICKED] =
		gtk_signal_new ("row-double-clicked",
				GTK_RUN_LAST,
				object_class->type,
				0,
				gtk_marshal_NONE__POINTER_POINTER,
				GTK_TYPE_NONE,
				2, GTK_TYPE_POINTER, GTK_TYPE_POINTER);

	gtk_object_class_add_signals (object_class, signals, LAST_SIGNAL);
}

/*
 * Experimental printing support.
 */

static gdouble
print_get_row_height (GanttPrintInfo *print_info)
{
	return 1.25 * gnome_font_get_size (print_info->font) + 2;
}

static void
gantt_item_print_gantt_row (GanttItem         *gantt,
			    GanttPrintInfo    *print_info,
			    ArtDRect          *bbox,
			    gdouble            hscale,
			    gdouble            vscale,
			    time_t             t1,
			    time_t             t2,
			    gint               row_nr,
			    gint               row_offset)
{
	GnomeCanvasItem      *row_item;
	GanttItemPriv        *priv;
	GnomePrintContext    *ctx;
	GNOME_MrProject_Task *task;
	gdouble               x1, y1, x2, y2, height;
	gdouble               name_x;
	gchar                *name;
	gdouble               clipped_x1, clipped_x2;
	gboolean              clipped_left = FALSE, clipped_right = FALSE;
	GSList               *list;

	priv = gantt->priv;
	ctx = print_info->context;

	row_item = g_array_index (priv->row_items, GnomeCanvasItem*, row_nr);
	gtk_object_get (GTK_OBJECT (row_item),
			"task", &task,
			NULL);

	y1 = print_get_row_height (print_info) * row_offset;
	height = print_get_row_height (print_info);

	y2 = y1 + height;

	y1 += height * 0.2;
	y2 -= height * 0.2;

	y1 += bbox->y0;
	y2 += bbox->y0;
	
	x1 = hscale * (gantt_scale_t2w (priv->gantt_scale, task->start) -
		       gantt_scale_t2w (priv->gantt_scale, t1)) + bbox->x0;
	x2 = hscale * (gantt_scale_t2w (priv->gantt_scale, task->end) -
		       gantt_scale_t2w (priv->gantt_scale, t1)) + bbox->x0;
	
	if (x2 < bbox->x0 || x1 > bbox->x1) {
		return;
	}

	/* Workaround for b0rked clipping in the print preview... */
	clipped_x1 = MAX (x1, bbox->x0);
	clipped_x2 = MIN (x2, bbox->x1);
	
	if (clipped_x1 != x1) {
		clipped_left = TRUE;
		x1 = clipped_x1;
	}
	
	if (clipped_x2 != x2) {
		clipped_right = TRUE;
		x2 = clipped_x2;
	}

	/* Print the task name to the left of the task. 
	 * FIXME: when the gantt tree is printed, we
	 * don't need this anymore.
	 */
	/* FIXME 2: gnome-print 0.25 doesn't have the _utf8 variant
	 * that 0.26 has. 
	 */
	if (!clipped_left) {
		name = g_strconcat (e_utf8_to_locale_string (task->name), " ", NULL);
		name_x = x1 - gnome_font_get_width_string (print_info->font, name);
		gantt_print_text (print_info, name_x, y2, name);
	}
	
	if (gantt_model_task_is_leaf (priv->gantt_model, task->taskId)) {
		/* Light green: 144 238 144, doesn't work??  */
		gnome_print_newpath (ctx);
		gnome_print_setlinewidth (ctx, 0);
		gnome_print_moveto (ctx, x1, y1);
		gnome_print_lineto (ctx, x1, y2);
		gnome_print_lineto (ctx, x2, y2);
		gnome_print_lineto (ctx, x2, y1);
		gnome_print_lineto (ctx, x1, y1);
/*		gnome_print_gsave (ctx);
		gnome_print_setrgbcolor (ctx,
					 144,
					 238,
					 144);
		gnome_print_fill (ctx);
		gnome_print_grestore (ctx);
*/
		gnome_print_closepath (ctx);
		gnome_print_stroke (ctx);
	} else {
		/* Summary task. */
		gint     top, bottom, mid;

		top = y1 + TEXT_PAD / 2 + 2;
		mid = top + TEXT_PAD / 2;
		bottom = mid + TEXT_PAD / 2 + 1;

		if (!clipped_left) {
			gnome_print_newpath (ctx);
			gnome_print_moveto (ctx, x1, top);
			gnome_print_lineto (ctx, x1, bottom);
			gnome_print_lineto (ctx, x1 + 1, bottom);
			gnome_print_lineto (ctx, x1 + 4, mid);
			gnome_print_lineto (ctx, x1 + 4, top);
			gnome_print_closepath (ctx);
			gnome_print_fill (ctx);
		}

		if (!clipped_right) {
			gnome_print_newpath (ctx);
			gnome_print_moveto (ctx, x2, top);
			gnome_print_lineto (ctx, x2, bottom);
			gnome_print_lineto (ctx, x2 - 1, bottom);
			gnome_print_lineto (ctx, x2 - 4, mid);
			gnome_print_lineto (ctx, x2 - 4, top);
			gnome_print_closepath (ctx);
			gnome_print_fill (ctx);
		}

		gnome_print_newpath (ctx);
		gnome_print_moveto (ctx, x1, top);
		gnome_print_lineto (ctx, x2, top);
		gnome_print_lineto (ctx, x2, mid);
		gnome_print_lineto (ctx, x1, mid);
		gnome_print_lineto (ctx, x1, top);
		gnome_print_closepath (ctx);
		gnome_print_fill (ctx);
	}

	/* FIXME: Print dependency arrows as well. */
	list = gantt_model_task_get_successors (priv->gantt_model, task->taskId);
	for (; list; list = list->next) {
		GNOME_MrProject_Dependency *dependency;

		dependency = list->data;
		/*g_print ("Successor: %d\n", dependency->taskId);*/
		/*gantt_print_arrow (print_info, x1, y1, x2, y2);*/
	}
}

static void
gantt_item_print_page (GanttPrintable    *printable,
		       GanttPrintInfo    *print_info,
		       ArtDRect          *bbox,
		       gdouble            hscale,
		       gdouble            vscale,
		       time_t             t1,
		       time_t             t2,
		       gint               r1,
		       gint               r2,
		       GanttItem         *item)
{
	gint i;

	g_return_if_fail (r1 <= r2);

	r2 = MIN (r2, item->priv->rows - 1);
	
	for (i = r1; i <= r2; i++) {
		gantt_item_print_gantt_row (item,
					    print_info,
					    bbox,
					    hscale,
					    vscale,
					    t1,
					    t2,
					    i,
					    i - r1);
	}
}

static gdouble
gantt_item_printable_get_width (GanttPrintable  *printable,
				GanttPrintInfo  *print_info,
				time_t           t1,
				time_t           t2,
				GanttItem       *item)
{
	gdouble x1, x2;

	x1 = gantt_scale_t2w (item->priv->gantt_scale, t1);
	x2 = gantt_scale_t2w (item->priv->gantt_scale, t2);

	return x2 - x1;
}

static gdouble
gantt_item_printable_get_height (GanttPrintable  *printable,
				 GanttPrintInfo  *print_info,
				 GanttItem       *item)
{
	gdouble y1, y2;

	y1 = 0;
	y2 = item->priv->rows * print_get_row_height (print_info);
	
	return y2 - y1;
}

static gdouble
gantt_item_printable_get_row_height (GanttPrintable  *printable,
				     GanttPrintInfo  *print_info,
				     GanttItem       *item)
{
	return print_get_row_height (print_info);
}

GanttPrintable *
gantt_item_get_printable (GanttItem *item)
{
	GanttPrintable *printable;
		
	g_return_val_if_fail (item != NULL, NULL);
	g_return_val_if_fail (GANTT_ITEM (item), NULL);

	printable = gantt_printable_new ();

	gtk_signal_connect (GTK_OBJECT (printable),
			    "print_page",
			    gantt_item_print_page,
			    item);

	gtk_signal_connect (GTK_OBJECT (printable),
			    "get_width",
			    GTK_SIGNAL_FUNC (gantt_item_printable_get_width),
			    item);

	gtk_signal_connect (GTK_OBJECT (printable),
			    "get_height",
			    GTK_SIGNAL_FUNC (gantt_item_printable_get_height),
			    item);

	gtk_signal_connect (GTK_OBJECT (printable),
			    "get_row_height",
			    GTK_SIGNAL_FUNC (gantt_item_printable_get_row_height),
			    item);
	
	return printable;
}
	



