/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2001 CodeFactory AB
 * Copyright (C) 2001 Mikael Hallendal <micke@codefactory.se>
 * Copyright (C) 2001 Richard Hult <rhult@codefactory.se>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Mikael Hallendal <micke@codefactory.se>
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdlib.h>
#include <gnome.h>
#include <glade/glade.h>
#include <gtk/gtkclist.h>
#include <gtk/gtkcontainer.h>
#include <gtk/gtkentry.h>
#include <gtk/gtksignal.h>
#include <gal/widgets/e-unicode.h>
#include "util/id-map.h"
#include "util/type-utils.h"
#include "group-dialog.h"

#define DEBUG 0
#include "util/debug.h"

static void      group_dialog_init         (GroupDialog      *gd);
static void      group_dialog_class_init   (GtkObjectClass   *klass);

static void      gd_destroy                (GtkObject        *object);

static void      gd_new_group_cb           (GtkObject        *object,
                                            gpointer          user_data);

static void      gd_delete_group_cb        (GtkObject        *object,
                                            gpointer          user_data);

static gboolean  gd_update_group_cb        (GtkWidget        *widget,
                                            GdkEventFocus    *focus_event,
                                            gpointer          user_data);

static void      gd_group_select_cb        (GtkCList         *clist,
                                            gint              row,
                                            gint              column,
                                            GdkEvent         *event,
                                            gpointer          user_data);

static void      gd_group_unselect_cb      (GtkCList         *clist,
                                            gint              row,
                                            gint              column,
                                            GdkEvent         *event,
                                            gpointer          user_data);

static void      gd_group_default_set_cb   (GtkWidget        *button,
					    gpointer          user_data);

static void      gd_update_entries_text    (GroupDialog      *gd);


static gint      gd_get_row                (GroupDialog      *gd, 
					    GM_Id             group_i);
enum {
        GROUP_ADD,
        GROUP_DELETE,
        GROUP_UPDATED,
	GROUP_DEFAULT_SET,
        LAST_SIGNAL
};

#define COL_ID      0
#define COL_NAME    1
#define COL_DEFAULT 2

static gint signals[LAST_SIGNAL] = { 0 };

GNOME_CLASS_BOILERPLATE (GroupDialog,  group_dialog, 
                         GtkWindow,    gtk_window);

struct _GroupDialogPriv {
	GtkWidget        *list_groups;
	GtkWidget        *entry_name;
	GtkWidget        *entry_adm_name;
	GtkWidget        *entry_adm_phone;
	GtkWidget        *entry_adm_email;
        GtkWidget        *button_default;
        GtkWidget        *button_delete;

	IdMap            *groups;

	GM_Id             default_group_id;
        GM_ResourceGroup *selected_group;
};

static void  
group_dialog_init (GroupDialog *gd)
{
	GroupDialogPriv *priv;
        GladeXML        *glade;
        GtkWidget       *button_new;
        GtkWidget       *button_close;
        GtkWidget       *contents;
        GtkWidget       *window;
	GtkWidget       *pixmap;
	GtkWidget       *buttonbox;
                
        g_return_if_fail (gd != NULL);
        g_return_if_fail (IS_GROUP_DIALOG (gd));

	priv     = g_new0 (GroupDialogPriv, 1);
	gd->priv = priv;
	
	gtk_window_set_wmclass (GTK_WINDOW (gd), "Group Editor", "MrProject");

	gtk_window_set_title (GTK_WINDOW (gd), _("Group Editor - MrProject"));
	
        priv->groups = id_map_new (0);

        glade = glade_xml_new (MRPROJECT_DATADIR "group-dialog.glade",
                               "group_dialog");
        
        window = glade_xml_get_widget (glade, "group_dialog");

        contents = glade_xml_get_widget (glade, "dialog_contents"); 
        gtk_widget_reparent (contents, GTK_WIDGET(gd));

        gtk_widget_destroy (window);
        
        /* Get references to all widgets */
        priv->list_groups     = glade_xml_get_widget (glade, "list_groups");
        priv->entry_name      = glade_xml_get_widget (glade, "entry_name");
        priv->entry_adm_name  = glade_xml_get_widget (glade, 
						      "entry_mgr_name");
        priv->entry_adm_phone = glade_xml_get_widget (glade, 
						      "entry_mgr_phone");
        priv->entry_adm_email = glade_xml_get_widget (glade, 
						      "entry_mgr_email");
        priv->button_default  = glade_xml_get_widget (glade, 
						      "button_default");

	button_close          = glade_xml_get_widget (glade, "button_close");
	buttonbox             = glade_xml_get_widget (glade, "hbuttonbox");

	pixmap = gnome_stock_pixmap_widget (window, GNOME_STOCK_MENU_NEW);
	button_new  = gnome_pixmap_button (pixmap, _("Add"));
	
	pixmap = gnome_stock_pixmap_widget (window, GNOME_STOCK_PIXMAP_TRASH);
	priv->button_delete = gnome_pixmap_button (pixmap, _("Remove"));
	gtk_widget_set_sensitive (priv->button_delete, FALSE);
	
	gtk_box_pack_start_defaults (GTK_BOX (buttonbox), button_new);
	gtk_box_pack_start_defaults (GTK_BOX (buttonbox), priv->button_delete);
	
	gtk_widget_show (button_new);
	gtk_widget_show (priv->button_delete);

        /* Configure the widgets */
        gtk_clist_set_sort_column (GTK_CLIST (priv->list_groups), COL_NAME);
	gtk_clist_set_column_visibility (GTK_CLIST (priv->list_groups), 
					 COL_ID, FALSE);
	gtk_clist_set_column_visibility (GTK_CLIST (priv->list_groups), 
					 COL_DEFAULT, TRUE);

	gtk_clist_set_column_width (GTK_CLIST (priv->list_groups), 
				    COL_DEFAULT,
				    gtk_clist_optimal_column_width (GTK_CLIST (priv->list_groups), COL_DEFAULT));

	/* Initialize state of widgets. */
        gd_update_entries_text (gd);

        /* Connect callbacks to the widgets */
        
        gtk_signal_connect_object (GTK_OBJECT (button_new),
                                   "clicked",
                                   GTK_SIGNAL_FUNC (gd_new_group_cb),
                                   GTK_OBJECT (gd));
        
        gtk_signal_connect_object (GTK_OBJECT (priv->button_delete),
                                   "clicked",
                                   GTK_SIGNAL_FUNC (gd_delete_group_cb),
                                   GTK_OBJECT (gd));
        
        gtk_signal_connect        (GTK_OBJECT (priv->list_groups),
                                   "select_row",
                                   GTK_SIGNAL_FUNC (gd_group_select_cb),
                                   GTK_OBJECT (gd));

        gtk_signal_connect        (GTK_OBJECT (priv->list_groups),
                                   "unselect_row",
                                   GTK_SIGNAL_FUNC (gd_group_unselect_cb),
                                   GTK_OBJECT (gd));
        
        gtk_signal_connect        (GTK_OBJECT (priv->entry_name),
                                   "focus_out_event",
                                   GTK_SIGNAL_FUNC (gd_update_group_cb),
                                   GTK_OBJECT (gd));

        gtk_signal_connect        (GTK_OBJECT (priv->entry_adm_name),
                                   "focus_out_event",
                                   GTK_SIGNAL_FUNC (gd_update_group_cb),
                                   GTK_OBJECT (gd));

        gtk_signal_connect        (GTK_OBJECT (priv->entry_adm_phone),
                                   "focus_out_event",
                                   GTK_SIGNAL_FUNC (gd_update_group_cb),
                                   GTK_OBJECT (gd));

        gtk_signal_connect        (GTK_OBJECT (priv->entry_adm_email),
                                   "focus_out_event",
                                   GTK_SIGNAL_FUNC (gd_update_group_cb),
                                   GTK_OBJECT (gd));

        gtk_signal_connect        (GTK_OBJECT (priv->button_default),
                                   "clicked",
                                   GTK_SIGNAL_FUNC (gd_group_default_set_cb),
                                   GTK_OBJECT (gd));

        gtk_signal_connect_object (GTK_OBJECT (button_close),
                                   "clicked",
                                   GTK_SIGNAL_FUNC (gtk_object_destroy),
                                   GTK_OBJECT (gd));


        gtk_object_unref (GTK_OBJECT (glade));
}

static void  
group_dialog_class_init (GtkObjectClass *klass)
{
        g_return_if_fail (klass != NULL);
        g_return_if_fail (IS_GROUP_DIALOG_CLASS (klass));

        /* GtkObject */
        klass->destroy = gd_destroy;
        
	/* Signals */
        signals[GROUP_ADD] = 
                gtk_signal_new ("group_add",
                                GTK_RUN_LAST,
                                klass->type,
                                GTK_SIGNAL_OFFSET (GroupDialogClass,
                                                   group_add),
                                gtk_marshal_NONE__NONE,
                                GTK_TYPE_NONE,
                                0, GTK_TYPE_NONE);
        
        signals[GROUP_DELETE] = 
                gtk_signal_new ("group_delete",
                                GTK_RUN_LAST,
                                klass->type,
                                GTK_SIGNAL_OFFSET (GroupDialogClass, 
                                                   group_delete),
                                gtk_marshal_NONE__INT,
                                GTK_TYPE_NONE,
                                1, GTK_TYPE_INT);
        
        signals[GROUP_UPDATED] =
                gtk_signal_new ("group_updated",
                                GTK_RUN_LAST,
                                klass->type,
                                GTK_SIGNAL_OFFSET (GroupDialogClass, 
                                                   group_updated),
                                gtk_marshal_NONE__POINTER,
                                GTK_TYPE_NONE,
                                1, GTK_TYPE_POINTER);

        signals[GROUP_DEFAULT_SET] =
                gtk_signal_new ("group_default_set",
                                GTK_RUN_LAST,
                                klass->type,
                                GTK_SIGNAL_OFFSET (GroupDialogClass, 
                                                   group_default_set),
                                gtk_marshal_NONE__INT,
                                GTK_TYPE_NONE,
                                1, GTK_TYPE_INT);
        
        gtk_object_class_add_signals (klass, signals, LAST_SIGNAL);
}

static void  
gd_destroy (GtkObject *object)
{
        GroupDialog     *gd;
        GroupDialogPriv *priv;
	
        g_return_if_fail (object != NULL);
        g_return_if_fail (IS_GROUP_DIALOG (object));

        gd   = GROUP_DIALOG (object);
	priv = gd->priv;
}

static void  
gd_new_group_cb (GtkObject *object, gpointer user_data)
{
        GroupDialog *gd;
        
        g_return_if_fail (object != NULL);
        g_return_if_fail (IS_GROUP_DIALOG (object));

        gd = GROUP_DIALOG (object);

        gtk_signal_emit (GTK_OBJECT (gd), signals[GROUP_ADD]);
}

static void  
gd_delete_group_cb (GtkObject *object, gpointer user_data)
{
        GroupDialog      *gd;
        GM_ResourceGroup *group;
        
        g_return_if_fail (object != NULL);
        g_return_if_fail (IS_GROUP_DIALOG (object));

        gd = GROUP_DIALOG (object);

        if ( !(gd->priv->selected_group) ) {
                /* Should never occur since the delete button is non  *
                 * sensitive if no row is selected                    */
                return;
        }

        group = gd->priv->selected_group;

        gtk_signal_emit (GTK_OBJECT (gd), signals[GROUP_DELETE], 
			 group->groupId);
}

static gboolean
gd_update_group_cb (GtkWidget     *widget, 
                    GdkEventFocus *focus_event,
                    gpointer       user_data) 
{ 
        GroupDialog      *gd;
	GroupDialogPriv  *priv;
        GM_ResourceGroup *group;
        gchar            *text, *text_utf8;
        gboolean          changed = FALSE;
        gint              row;
        
        g_return_val_if_fail (widget != NULL, FALSE);
        g_return_val_if_fail (GTK_IS_ENTRY (widget), FALSE);
        g_return_val_if_fail (user_data != NULL, FALSE);
        g_return_val_if_fail (IS_GROUP_DIALOG (user_data), FALSE);

        gd   = GROUP_DIALOG (user_data);
	priv = gd->priv;
	
        if (!(priv->selected_group)) {
                return FALSE;
        }

        group     = priv->selected_group;
        row       = gd_get_row (gd, group->groupId);
        text      = gtk_entry_get_text (GTK_ENTRY (widget));
        text_utf8 = e_utf8_from_locale_string (text);
	
        if (widget == priv->entry_name) {
                if (strcmp (text_utf8, "") && strcmp (text_utf8, group->name)) {
                        changed = TRUE;
                        CORBA_free (group->name);
			
			group->name = CORBA_string_dup (text_utf8);
			gtk_clist_set_text (GTK_CLIST (priv->list_groups),
					    row, COL_NAME, text);
			gtk_clist_sort (GTK_CLIST (priv->list_groups));
                }
        }
        else if (widget == priv->entry_adm_name) {
                if (strcmp (text_utf8, group->adminName)) {
                        changed = TRUE;
                        CORBA_free (group->adminName);

                        group->adminName = CORBA_string_dup (text_utf8);
                }
                
        }
        else if (widget == priv->entry_adm_phone) {
                if (strcmp (text_utf8, group->adminPhone)) {
                        changed = TRUE;
                        CORBA_free (group->adminPhone);

                        group->adminPhone = CORBA_string_dup (text_utf8);
                }
        }
        else if (widget == priv->entry_adm_email) {
                if (strcmp (text_utf8, group->adminEmail)) {
                        changed = TRUE;
                        CORBA_free (group->adminEmail);

                        group->adminEmail = CORBA_string_dup (text_utf8);
                }
        }
        
        if (changed) {
                gtk_signal_emit (GTK_OBJECT (gd), 
				 signals[GROUP_UPDATED],
				 group);
        }

	g_free (text_utf8);

        return FALSE;
} 

static void
gd_group_select_cb (GtkCList *clist, 
                    gint      row, 
                    gint      column, 
                    GdkEvent *event, 
                    gpointer  user_data)
{
        GroupDialog      *gd; 
	GroupDialogPriv  *priv;
        gchar            *text; 
        GM_ResourceGroup *group = NULL; 
        gint              id;
        
        g_return_if_fail (clist != NULL); 
        g_return_if_fail (GTK_IS_CLIST (clist));
        g_return_if_fail (user_data != NULL); 
        g_return_if_fail (IS_GROUP_DIALOG (user_data)); 

        gd   = GROUP_DIALOG (user_data); 
	priv = gd->priv;
	
        gtk_clist_get_text (clist, row, COL_ID, &text);
        
        id    = atoi (text);
        group = id_map_lookup (priv->groups, id);

        priv->selected_group = group;

        gd_update_entries_text (gd);

        gtk_widget_set_sensitive (priv->button_delete, TRUE);
	gtk_editable_select_region (GTK_EDITABLE (priv->entry_name),
				    0, -1);
 	gtk_widget_grab_focus (priv->entry_name); 
}

static void
gd_group_unselect_cb (GtkCList *clist,
                      gint      row,
                      gint      column,
                      GdkEvent *event,
                      gpointer  user_data)
{
        GroupDialog     *gd;
	GroupDialogPriv *priv;
        
        g_return_if_fail (clist != NULL);
        g_return_if_fail (GTK_IS_CLIST (clist));
        g_return_if_fail (user_data != NULL);
        g_return_if_fail (IS_GROUP_DIALOG (user_data));
        
        gd   = GROUP_DIALOG (user_data);
        priv = gd->priv;
	
        priv->selected_group = NULL;
        
        gd_update_entries_text (gd);
        gtk_widget_set_sensitive (priv->button_delete, FALSE);
}

static void
gd_group_default_set_cb (GtkWidget *button, gpointer user_data)
{
	GroupDialog     *gd;
	GroupDialogPriv *priv;
	
	g_return_if_fail (user_data != NULL);
	g_return_if_fail (IS_GROUP_DIALOG (user_data));
	
	gd   = GROUP_DIALOG (user_data);
	priv = gd->priv;

	gtk_signal_emit (GTK_OBJECT (gd),
			 signals[GROUP_DEFAULT_SET],
			 priv->selected_group->groupId);
}

static void
gd_update_entries_text (GroupDialog *gd)
{
	GroupDialogPriv  *priv;
        GM_ResourceGroup *group;
	gboolean          is_default;
        
        g_return_if_fail (gd != NULL);
        g_return_if_fail (IS_GROUP_DIALOG (gd));

	priv = gd->priv;

        if (priv->selected_group) {
                group = priv->selected_group;
		is_default = (group->groupId == priv->default_group_id);

                e_utf8_gtk_entry_set_text (GTK_ENTRY (priv->entry_name), 
					   group->name);
                e_utf8_gtk_entry_set_text (GTK_ENTRY (priv->entry_adm_name),
					   group->adminName);
                e_utf8_gtk_entry_set_text (GTK_ENTRY (priv->entry_adm_phone),
					   group->adminPhone);
                e_utf8_gtk_entry_set_text (GTK_ENTRY (priv->entry_adm_email),
					   group->adminEmail);

		gtk_widget_set_sensitive (priv->entry_name, TRUE);
		gtk_widget_set_sensitive (priv->entry_adm_name, TRUE);
		gtk_widget_set_sensitive (priv->entry_adm_phone, TRUE);
		gtk_widget_set_sensitive (priv->entry_adm_email, TRUE);
		gtk_widget_set_sensitive (priv->button_default, TRUE);
        } else {
                gtk_entry_set_text (GTK_ENTRY (priv->entry_name), "");
                gtk_entry_set_text (GTK_ENTRY (priv->entry_adm_name), "");
                gtk_entry_set_text (GTK_ENTRY (priv->entry_adm_phone), "");
                gtk_entry_set_text (GTK_ENTRY (priv->entry_adm_email), "");

		gtk_widget_set_sensitive (priv->entry_name, FALSE);
		gtk_widget_set_sensitive (priv->entry_adm_name, FALSE);
		gtk_widget_set_sensitive (priv->entry_adm_phone, FALSE);
		gtk_widget_set_sensitive (priv->entry_adm_email, FALSE);
		gtk_widget_set_sensitive (priv->button_default, FALSE);
        }
}

static gint
gd_get_row (GroupDialog *gd, GM_Id group_id)
{
        GtkCList *clist;
        gint      i, rows;
        gchar    *text;
        gint      id;
                
	d(puts(__FUNCTION__));
	
        clist = GTK_CLIST (gd->priv->list_groups);
        rows  = clist->rows;
        
        for (i = 0; i < rows; ++i) {
                gtk_clist_get_text (clist, i, COL_ID, &text);
                id = atoi (text);
                
                if (id == group_id) {
                        return i;
                }
        }

        return -1;
}

GtkWidget *
group_dialog_new (GSList *groups)
{
        GroupDialog *gd;
        GSList      *tmp_list;
	
        gd = gtk_type_new (GROUP_DIALOG_TYPE);

	if (!groups) {
		return GTK_WIDGET (gd);
	}
	
	tmp_list = groups;
	
	for (tmp_list = groups; tmp_list; tmp_list = tmp_list->next) {
		group_dialog_add_group (gd, 
					(GM_ResourceGroup *) tmp_list->data);
	}
        
        return GTK_WIDGET (gd);
}

void        
group_dialog_add_group (GroupDialog *gd, GM_ResourceGroup *grp)
{
	GroupDialogPriv  *priv;
        gchar            *c_text[3];
        GM_ResourceGroup *group;
	gint              row;
	
	priv  = gd->priv;
        group = corba_util_resource_group_duplicate (grp);

	g_print ("add group\n");
	
        c_text[COL_NAME]    = e_utf8_to_locale_string (group->name);
	c_text[COL_DEFAULT] = "";
        c_text[COL_ID]      = g_strdup_printf ("%d", group->groupId);
                      
        gtk_clist_append (GTK_CLIST (priv->list_groups), c_text);
        gtk_clist_sort (GTK_CLIST (priv->list_groups));
	
        id_map_insert_id (priv->groups, group->groupId, group);

        g_free (c_text[COL_NAME]);
        g_free (c_text[COL_ID]);

	row = gd_get_row (gd, grp->groupId);

	gtk_clist_select_row (GTK_CLIST (priv->list_groups),
			      row, COL_NAME);
}

void        
group_dialog_remove_group (GroupDialog *gd, GM_Id gid)
{
	GroupDialogPriv  *priv;
        GM_ResourceGroup *group;
        gint              row;
        gint              select_row;
	
        g_return_if_fail (gd != NULL);
        g_return_if_fail (IS_GROUP_DIALOG (gd));
        
	priv  = gd->priv;
        group = id_map_lookup (priv->groups, gid);
	
        if (!group) {
                return;
        }

        id_map_remove (priv->groups, gid);
        
        row        = gd_get_row (gd, group->groupId);
	select_row = row;
	
	if (select_row >= GTK_CLIST (priv->list_groups)->rows - 1) {
		select_row = select_row - 1;
	}

        gtk_clist_remove (GTK_CLIST (priv->list_groups), row);
        
	if (select_row >= 0) {
		gtk_clist_select_row (GTK_CLIST (priv->list_groups),
				      select_row, COL_NAME);
	}

        CORBA_free (group);
}

void
group_dialog_update_group (GroupDialog *dialog, GM_ResourceGroup *group)
{
        d(puts (__FUNCTION__));
}


void
group_dialog_set_default_group (GroupDialog *dialog, GM_Id gid)
{
	GroupDialogPriv *priv;
	gint             row;
	
	g_return_if_fail (dialog != NULL);
	g_return_if_fail (IS_GROUP_DIALOG (dialog));

	d(puts (__FUNCTION__));
	
	priv = dialog->priv;
	
	if (id_map_lookup (priv->groups, gid)) {
		if (priv->default_group_id > 0) {
			row = gd_get_row (dialog, priv->default_group_id);
			gtk_clist_set_text (GTK_CLIST (priv->list_groups),
					    row, COL_DEFAULT, NULL);
		}
		
		priv->default_group_id = gid;
		row = gd_get_row (dialog, gid);

		if (row >= 0) {
			gtk_clist_set_text (GTK_CLIST (priv->list_groups),
					    row, COL_DEFAULT, "*");
		}
		
	}
}

