/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2001 Richard Hult <rhult@codefactory.se>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Richard Hult
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <glib.h>
#include <liboaf/liboaf.h>
#include <bonobo/bonobo-xobject.h>
#include <bonobo/bonobo-shlib-factory.h>
#include <libgnomevfs/gnome-vfs-utils.h>
#include <libgnomevfs/gnome-vfs-result.h>
#include <libgnomevfs/gnome-vfs-mime.h>
#include <liboaf/liboaf.h>
#include <liboaf/oaf-async.h>

#include "util/corba-utils.h"
#include "util/file-sel.h"
#include "util/eel-vfs-extensions.h"
#include "client/widgets/mr-message-box.h"

#include "libmrproject/GNOME_MrProject.h"

#ifdef HAVE_EVOLUTION_COMPOSER_IDL
#include "libmrproject/Composer.h"
#endif

#include "client/manager-clients/task-manager-client.h"
#include "send-component.h"

#define d(x)

#define OAFIID "OAFIID:GNOME_MrProject_SendComponent"
#define COMPOSER_IID "OAFIID:GNOME_Evolution_Mail_Composer"

static void send_component_init       (SendComponent       *component);
static void send_component_class_init (SendComponentClass  *klass);
static void send_project_cb           (BonoboUIComponent   *component,
				       gpointer             data,
				       const char          *cname);
static void set_pixmap                (BonoboUIComponent   *uic,
				       const char          *xml_path,
				       const char          *icon);

#define PARENT_TYPE SHELL_COMPONENT_TYPE
static GtkObjectClass *parent_class;

static BonoboUIVerb verbs[] = {
	BONOBO_UI_VERB ("FileSendProject",      send_project_cb),
	BONOBO_UI_VERB_END
};

static gboolean
send_component_add_to_shell (ShellComponent      *component, 
			     GM_Shell             shell)
{
	SendComponent      *send_component;
	CORBA_Environment   ev;
	Bonobo_UIContainer  ui_container;
	BonoboUIComponent  *ui_component;
	
	g_return_val_if_fail (component != NULL, FALSE);
	g_return_val_if_fail (IS_SEND_COMPONENT (component), FALSE);

	d(puts(__FUNCTION__));

	send_component = SEND_COMPONENT (component);
	
	if (!corba_util_evolution_composer_exist ()) {
		g_print ("No composer!!\n");
		return FALSE;
	}

	CORBA_exception_init (&ev);
	
	ui_container = GNOME_MrProject_Shell_getUIContainer (component->shell, &ev);

	ui_component = bonobo_ui_component_new ("MrProjectSendComponent");
	bonobo_ui_component_set_container (ui_component, ui_container);
	bonobo_ui_component_add_verb_list_with_data (ui_component,
						     verbs,
						     component);

	bonobo_ui_util_set_ui (ui_component, GNOME_DATADIR,
			       "GNOME_MrProject_SendComponent.ui",
			       "mrproject");

	set_pixmap (ui_component,
		    "/menu/File/Send/FileSendProject",
		    "16_send_to.png");
	
	CORBA_exception_free (&ev);
	
	return TRUE;
}

static void
send_component_remove_from_shell (ShellComponent *component)
{
	g_return_if_fail (component != NULL);
	g_return_if_fail (IS_SEND_COMPONENT (component));

	d(puts(__FUNCTION__));
}

static void
set_pixmap (BonoboUIComponent *uic,
	    const char        *xml_path,
	    const char        *icon)
{
	char      *path;
	GdkPixbuf *pixbuf;

	path = g_concat_dir_and_file (MRPROJECT_IMAGEDIR, icon);

	pixbuf = gdk_pixbuf_new_from_file (path);
	if (pixbuf == NULL) {
		g_warning ("Cannot load image -- %s", path);
		g_free (path);
		return;
	}

	bonobo_ui_util_set_pixbuf (uic, xml_path, pixbuf);

	gdk_pixbuf_unref (pixbuf);

	g_free (path);
}

static void
send_component_destroy (GtkObject *object)
{
	SendComponent       *component;

	d(puts(__FUNCTION__));

	component = SEND_COMPONENT (object);

	(* GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

static void
send_component_class_init (SendComponentClass *klass)
{
	GtkObjectClass        *object_class;
	ShellComponentClass   *shell_component_class;

	object_class                             = (GtkObjectClass *) klass;
	shell_component_class                    = SHELL_COMPONENT_CLASS (klass);
	parent_class                             = gtk_type_class (PARENT_TYPE);

	object_class->destroy                    = send_component_destroy;

	shell_component_class->add_to_shell      = send_component_add_to_shell;
	shell_component_class->remove_from_shell = send_component_remove_from_shell;
}

static void
send_component_init (SendComponent *component)
{
}

static BonoboObject *
send_component_factory (BonoboGenericFactory   *this,
			   const char             *object_id,
			   void                   *data)
{
	g_return_val_if_fail (object_id != NULL, NULL);

	if (!strcmp (object_id, OAFIID)) {
		return BONOBO_OBJECT (gtk_type_new (SEND_COMPONENT_TYPE));
	} else {
		g_warning ("Failing to manufacture a '%s'", object_id);
	}
	
	return NULL;
}

BONOBO_OAF_SHLIB_FACTORY_MULTI ("OAFIID:GNOME_MrProject_SendComponentFactory",
				"Mr Project send component factory",
				send_component_factory,
				NULL);

BONOBO_X_TYPE_FUNC (SendComponent, 
		    PARENT_TYPE,
		    send_component);


#ifdef HAVE_EVOLUTION_COMPOSER_IDL
static void
composer_component_loaded (CORBA_Object  composer,
			   const gchar  *error_reason,
			   gpointer      user_data)
{
	gchar                                   *uri;
	gint                                     file_size;
	gchar                                   *file_contents;
	gchar                                   *file_basename;
	GnomeVFSResult                           result;
	GNOME_MrProject_Project                  project_co;
	CORBA_Environment                        ev;
	GNOME_Evolution_Composer_RecipientList  *list;
	GNOME_Evolution_Composer_AttachmentData *attachment_data;
	ShellComponent                          *shell_component;

	shell_component = user_data;

	CORBA_exception_init (&ev);
		
	/*gdk_window_set_cursor (GTK_WIDGET (window)->window, NULL);*/
	project_co = GNOME_MrProject_Shell_getProject (shell_component->shell, &ev);
	if (BONOBO_EX (&ev)) {
		CORBA_exception_free (&ev);
		return;
	}

	/*bonobo_ui_component_set_status (window->priv->component, "", NULL);*/

	if (composer == CORBA_OBJECT_NIL) {
		mr_message_box_show_message (GNOME_MESSAGE_BOX_ERROR,
					     "Unable to start mail composer component!");
		return;
	}

	/* Get the URI */
	uri = GNOME_MrProject_Project_getURI (project_co, &ev);
	if (BONOBO_EX (&ev)) {
		g_warning ("Could not get URI (%s).", 
			   bonobo_exception_get_text (&ev));
		CORBA_exception_free (&ev);
		return;
	}

	result = eel_read_entire_file (uri, &file_size, &file_contents);
	file_basename = eel_uri_get_basename (uri);

	CORBA_free (uri);

	if (result != GNOME_VFS_OK) {
		g_warning ("There was an error reading the file (%s).",
			   gnome_vfs_result_to_string (result));
		return;
	}

	/* Set the subject */
	list = GNOME_Evolution_Composer_RecipientList__alloc ();
	list->_length = 0;
	list->_maximum = 0;
	list->_buffer = CORBA_sequence_GNOME_Evolution_Composer_Recipient_allocbuf (0);

	GNOME_Evolution_Composer_setHeaders (composer, list, list, list,
					     file_basename, &ev);

	if (BONOBO_EX (&ev)) {
		g_warning ("Could not attach data (%s).", 
			   bonobo_exception_get_text (&ev));
		CORBA_exception_free (&ev);

		CORBA_free (list);
		
		g_free (file_contents);
		g_free (file_basename);

		return;
	}

	attachment_data = GNOME_Evolution_Composer_AttachmentData__alloc ();
	attachment_data->_length = file_size;
	attachment_data->_buffer = file_contents;
	CORBA_sequence_set_release (attachment_data, CORBA_FALSE);
	
	/* Attach the file */
	GNOME_Evolution_Composer_attachData (
		composer,
		gnome_vfs_mime_type_from_name (file_basename),
		file_basename,
		"",
		FALSE,
		attachment_data,
		&ev);
	
	g_free (file_contents);
	g_free (file_basename);
	CORBA_free (attachment_data);
	
	if (BONOBO_EX (&ev)) {
		g_warning ("Could not attach data (%s).", 
			   bonobo_exception_get_text (&ev));
		CORBA_exception_free (&ev);

		return;
	}

	/* Show the file */
	GNOME_Evolution_Composer_show (composer, &ev);
	CORBA_exception_free (&ev);

	CORBA_Object_release (project_co, NULL);
}
#endif

static void
send_project_cb (BonoboUIComponent *component, gpointer data, const char *cname)
{
#ifdef HAVE_EVOLUTION_COMPOSER_IDL
	GNOME_MrProject_Project       project_co;
	GNOME_MrProject_ProjectState  state;
	CORBA_Environment             ev;
	ShellComponent               *shell_component;

	shell_component = SHELL_COMPONENT (data);

	CORBA_exception_init (&ev);
		
	project_co = GNOME_MrProject_Shell_getProject (shell_component->shell, &ev);
	if (BONOBO_EX (&ev)) {
		g_log_exception (&ev, "");
		CORBA_exception_free (&ev);
		return;
	}
	
	state = GNOME_MrProject_Project_getState (project_co, &ev);
	if (BONOBO_EX (&ev)) {
		g_log_exception (&ev, "");
		CORBA_exception_free (&ev);
		return;
	}
	
	if (state == GNOME_MrProject_PROJECT_STATE_EMPTY ||
	    state == GNOME_MrProject_PROJECT_STATE_DIRTY) {
		GtkWidget *dialog;
		gint       result;

		/* FIXME: would be nicer to save the project to a tmp location and then send it. */
		
		dialog = gnome_message_box_new (_("The project has not been saved,\n"
                                                  "you must save your project before sending it.\n"
						  "Do you wish to do so?"),
						GNOME_MESSAGE_BOX_QUESTION,
						_("Save"),
						GNOME_STOCK_BUTTON_CANCEL,
						NULL);

		gnome_dialog_set_default (GNOME_DIALOG (dialog), 0);
		result = gnome_dialog_run (GNOME_DIALOG (dialog));

		if (result == 0) {
			CORBA_char *uri;

			uri = GNOME_MrProject_Project_getURI (project_co, &ev);
			if (BONOBO_EX (&ev)) {
				CORBA_exception_free (&ev);
				return;
			}

			if (uri == NULL || strlen (uri) == 0) {
				gchar *tmp, *new_uri;
				
				tmp = file_sel_run (_("Save as"));
				if (tmp) {
					new_uri = gnome_vfs_get_uri_from_local_path (tmp);
					g_free (tmp);
					
					GNOME_MrProject_Project_save (project_co, new_uri, &ev);
					
					if (BONOBO_EX (&ev)) {
						CORBA_exception_free (&ev);
					}

					g_free (new_uri);
				}
			}

			CORBA_free (uri);
		}
	}

	state = GNOME_MrProject_Project_getState (project_co, &ev);

	/* Check if the user saved the project */
	if (state != GNOME_MrProject_PROJECT_STATE_SAVED) {
		return;
	}

	oaf_activate_from_id_async (COMPOSER_IID,
				    0,
				    composer_component_loaded,
				    shell_component,
				    &ev);

	if (ev._major != CORBA_NO_EXCEPTION) {
		CORBA_exception_free (&ev);
		mr_message_box_show_message (GNOME_MESSAGE_BOX_ERROR,
					     "Unable to start mail composer component!");
		return;
	}

	/* FIXME: It would be nice to be able to do this: */
#if 0
	{
		GdkCursor *cursor = gdk_cursor_new (GDK_WATCH);
		gdk_window_set_cursor (GTK_WIDGET (window)->window, cursor);
		gdk_cursor_destroy (cursor);
	}
#endif

	/*bonobo_ui_component_set_status (window->priv->component,
					_("Loading composer ..."),
					NULL);*/

#endif /* HAVE_EVOLUTION_COMPOSER_IDL */
}

