/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2001 CodeFactory AB
 * Copyright (C) 2001 Mikael Hallendal <micke@codefactory.se>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Mikael Hallendal <micke@codefactory.se>
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <bonobo.h>
#include <glib.h>
#include "allocation-manager.h"
#include "util/id-map.h"
#include "util/type-utils.h"
#include "util/corba-utils.h"

#define d(x)

/* GtkObject. */
static void     allocation_manager_init       (AllocationManager      *manager);
static void     allocation_manager_class_init (AllocationManagerClass *klass);
static void     am_destroy                    (GtkObject              *object);
static void     am_get_arg                    (GtkObject              *object, 
					       GtkArg                 *arg, 
					       guint                   arg_id);
static void     am_set_arg                    (GtkObject              *object, 
					       GtkArg                 *arg, 
					       guint                   arg_id);
static gboolean am_does_task_exist            (AllocationManager      *manager,
					       GM_Id                   id);

#define PARENT_TYPE BONOBO_X_OBJECT_TYPE
static GtkObjectClass   *parent_class;

struct _AllocationManagerPriv {
	Bonobo_EventSource_ListenerId   listener_id;
	BonoboEventSource              *event_source;
	GNOME_MrProject_ProjectState    state;
};

/* Object arguments */
enum {
	ARG_0,
	ARG_STATE
};

static void
impl_AllocationManager__set_state (PortableServer_Servant  servant,
				   const GM_ProjectState   state,
				   CORBA_Environment      *ev)
{
	AllocationManager *manager;

	manager = ALLOCATION_MANAGER (bonobo_x_object (servant));
	manager->priv->state = state;
}

static GM_ProjectState
impl_AllocationManager__get_state (PortableServer_Servant    servant,
				   CORBA_Environment        *ev)
{
	AllocationManager *manager;

	manager = ALLOCATION_MANAGER (bonobo_x_object (servant));
	return manager->priv->state;
}

static GM_Allocation
impl_AllocationManager_allocate (
	PortableServer_Servant    servant,
	GM_Id                     taskId,
	GM_Id                     resourceId,
	CORBA_Environment        *ev)
{
	AllocationManager        *am;
	AllocationManagerPriv    *priv;
	AllocationManagerClass   *klass;
	CORBA_any                *any;
	CORBA_Environment         e;
	GM_Allocation            *allocation;
	GM_Allocation             ret_allocation;
	
	am    = ALLOCATION_MANAGER (bonobo_x_object (servant));
	klass = ALLOCATION_MANAGER_CLASS (GTK_OBJECT (am)->klass);
	priv  = am->priv;

	ret_allocation.taskId = -1;
	ret_allocation.resourceId = -1;

	/* Perform some sanity checking. */
	if (!am_does_task_exist (am, taskId)) {
		CORBA_exception_set (ev, CORBA_USER_EXCEPTION,
				     ex_GNOME_MrProject_AllocationManager_NoSuchTask,
				     NULL);
	
		return ret_allocation;
	}
		
	/* FIX: Fon't hardcode units. */
	/* 100 it number of units to allocate, currently only 100% */
	allocation = klass->allocate (am, taskId, resourceId, 100);
	
	d(puts(__FUNCTION__));
	
	if (!allocation) {
		CORBA_exception_set (ev, CORBA_USER_EXCEPTION,
				     ex_GNOME_MrProject_AllocationManager_AlreadyAllocated, 
				     NULL);

		d(g_print ("AllocationManager::allocat: Already allocated [ %d <-> %d ]\n",
			   taskId, resourceId));

		return ret_allocation;
	}
	
	corba_util_allocation_copy (&ret_allocation, 
				    allocation);

	any         = CORBA_any__alloc ();
	any->_type  = TC_GNOME_MrProject_Allocation;
	any->_value = allocation;
	CORBA_any_set_release (any, CORBA_FALSE);

	CORBA_exception_init (&e);
	
	if (priv->event_source) {
		bonobo_event_source_notify_listeners (
			priv->event_source,
			"GNOME/MrProject:allocation:added",
			any,
			&e);
	} else {
		g_warning ("Couldn't get EventSource interface.");
	}
	
	priv->state = GNOME_MrProject_PROJECT_STATE_DIRTY;

	CORBA_free (any);
	CORBA_exception_free (&e);

	return ret_allocation;
}

static void
impl_AllocationManager_deallocate (
	PortableServer_Servant    servant,
	const GM_Allocation      *allocation, 
	CORBA_Environment        *ev)
{
	AllocationManager        *am;
	AllocationManagerClass   *klass;
	AllocationManagerPriv    *priv;
	CORBA_any                *any;
	CORBA_Environment         e;
	GM_AllocationSeq         *allocations;
	
	am    = ALLOCATION_MANAGER (bonobo_x_object (servant));
	klass = ALLOCATION_MANAGER_CLASS (GTK_OBJECT (am)->klass);
	priv  = am->priv;

	d(puts(__FUNCTION__));
	
	klass->deallocate (am, allocation);


	allocations           = GNOME_MrProject_AllocationSeq__alloc ();
	allocations->_buffer  = 
		CORBA_sequence_GNOME_MrProject_Allocation_allocbuf (1);
	allocations->_length  = 1;
	allocations->_maximum = 1;
	CORBA_sequence_set_release (allocations, CORBA_FALSE);

	corba_util_allocation_copy (&allocations->_buffer[0], allocation);
	
	any         = CORBA_any__alloc ();
	any->_type  = TC_GNOME_MrProject_AllocationSeq;
	any->_value = allocations;
	CORBA_any_set_release (any, CORBA_TRUE);

	CORBA_exception_init (&e);
	
	d(g_print ("removing\n"));
	
	bonobo_event_source_notify_listeners (
		priv->event_source,
		"GNOME/MrProject:allocation:removed",
		any,
		&e);
	
	priv->state = GNOME_MrProject_PROJECT_STATE_DIRTY;
	
	CORBA_exception_free (&e);
	CORBA_free (any);
}

static GM_Allocation 
impl_AllocationManager_getAllocation (
	PortableServer_Servant    servant,
	GM_Id                     taskId,
	GM_Id                     resourceId,
	CORBA_Environment        *ev)
{
	AllocationManager        *am;
	AllocationManagerClass   *klass;
	GSList                   *list, *l;
	GM_Allocation            *allocation;
	GM_Allocation             ret_allocation = { 0, };

	am    = ALLOCATION_MANAGER (bonobo_x_object (servant));
	klass = ALLOCATION_MANAGER_CLASS (GTK_OBJECT (am)->klass);

	list  = klass->get_allocations_by_task (am, taskId);

	for (l = list; l; l = l->next) {
		allocation = (GM_Allocation *) l->data;

		if (allocation->taskId == taskId &&
		    allocation->resourceId == resourceId) {
			g_slist_free (list);

			ret_allocation = *allocation;

			return ret_allocation;
		}								
	}

	g_slist_free (list);

	CORBA_exception_set (ev, CORBA_USER_EXCEPTION,
			     ex_GNOME_MrProject_AllocationManager_NotAllocated,
			     NULL);

	return ret_allocation;
}

static GM_AllocationSeq *
impl_AllocationManager_getAllocationsByTask (
	PortableServer_Servant    servant,
	GM_Id                     taskId,	
	CORBA_Environment        *ev)
{
	AllocationManager        *am;
	AllocationManagerClass   *klass;
	GSList                   *list;
	GM_AllocationSeq         *allocations;

	am    = ALLOCATION_MANAGER (bonobo_x_object (servant));
	klass = ALLOCATION_MANAGER_CLASS (GTK_OBJECT (am)->klass);

	list  = klass->get_allocations_by_task (am, taskId);

        allocations = corba_util_allocation_seq_from_list (list);

	g_slist_free (list);
	
	return allocations;
}

static GM_AllocationSeq *
impl_AllocationManager_getAllocationsByResource (
	PortableServer_Servant    servant,
	GM_Id                     resourceId,
	CORBA_Environment        *ev)
{
	AllocationManager        *am;
	AllocationManagerClass   *klass;
	GSList                   *list;
	GM_AllocationSeq         *allocations;
	
	am    = ALLOCATION_MANAGER (bonobo_x_object (servant));
	klass = ALLOCATION_MANAGER_CLASS (GTK_OBJECT (am)->klass);
	list  = klass->get_allocations_by_resource (am, resourceId);
	
	allocations = corba_util_allocation_seq_from_list (list);
	
	g_slist_free (list);
	
	return allocations;
}

static GM_AllocationSeq *
impl_AllocationManager_getAllAllocations (
	PortableServer_Servant    servant,
	CORBA_Environment        *ev)
{
	AllocationManager        *am;
	AllocationManagerClass   *klass;
	GSList                   *list;
	GM_AllocationSeq         *allocations;
	
	am    = ALLOCATION_MANAGER (bonobo_x_object (servant));
	klass = ALLOCATION_MANAGER_CLASS (GTK_OBJECT (am)->klass);
	list  = klass->get_all_allocations (am);
	allocations = corba_util_allocation_seq_from_list (list);
	
	g_slist_free (list);
	
	return allocations;
}

static void
impl_AllocationManager_deallocateAllResources (
	PortableServer_Servant    servant,
	GM_Id                     taskId,	
	CORBA_Environment        *ev)
{
	AllocationManager        *am;
	AllocationManagerClass   *klass;
	AllocationManagerPriv    *priv;
	GSList                   *list;
	GM_IdSeq                 *allocations;
	CORBA_any                *any;
	CORBA_Environment         e;
	
	am          = ALLOCATION_MANAGER (bonobo_x_object (servant));
	klass       = ALLOCATION_MANAGER_CLASS (GTK_OBJECT (am)->klass);
	priv        = am->priv;
        list        = klass->deallocate_all_resources (am, taskId);

	if (!list) {
		return;
	}
	
	allocations = corba_util_id_seq_from_list (list);
	
	any         = CORBA_any__alloc ();
	any->_type  = TC_GNOME_MrProject_IdSeq;
	any->_value = allocations;
	CORBA_any_set_release (any, CORBA_TRUE);

	CORBA_exception_init (&e);
	
	bonobo_event_source_notify_listeners (
		priv->event_source,
		"GNOME/MrProject:allocation:removed",
		any,
		&e);

	priv->state = GNOME_MrProject_PROJECT_STATE_DIRTY;

	CORBA_exception_free (&e);
        g_slist_free (list);
}

static void
impl_AllocationManager_deallocateAllTasks (
	PortableServer_Servant    servant,
	GM_Id                     resourceId,
	CORBA_Environment        *ev)
{
	AllocationManager        *am;
	AllocationManagerClass   *klass;
	AllocationManagerPriv    *priv;
	GM_IdSeq                 *allocations;
	GSList                   *list;
	CORBA_any                *any;
	CORBA_Environment         e;
	
	am          = ALLOCATION_MANAGER (bonobo_x_object (servant));
	klass       = ALLOCATION_MANAGER_CLASS (GTK_OBJECT (am)->klass);
	priv        = am->priv;
        list        = klass->deallocate_all_tasks (am, resourceId);

	if (!list) {
		return;
	}

	allocations = corba_util_id_seq_from_list (list);
	
	any         = CORBA_any__alloc ();
	any->_type  = TC_GNOME_MrProject_IdSeq;
	any->_value = allocations;
	CORBA_any_set_release (any, CORBA_TRUE);

	CORBA_exception_init (&e);
	
	bonobo_event_source_notify_listeners (
		priv->event_source,
		"GNOME/MrProject:allocation:removed",
		any,
		&e);

	priv->state = GNOME_MrProject_PROJECT_STATE_DIRTY;

	CORBA_exception_free (&e);
        g_slist_free (list);
}

static gboolean
am_does_task_exist (AllocationManager *manager, GM_Id id)
{
	CORBA_Environment  ev;
	GM_TaskManager     tm;
	GM_Task           *task;

	CORBA_exception_init (&ev);
	
	tm = Bonobo_Unknown_queryInterface (BONOBO_OBJREF (manager),
					    "IDL:GNOME/MrProject/TaskManager:1.0",
					    &ev);
	if (BONOBO_EX (&ev)) {
		CORBA_exception_free (&ev);
		return FALSE;
	}

	task = GNOME_MrProject_TaskManager_getTask (tm, id, &ev);
	if (!task || BONOBO_EX (&ev)) {
			CORBA_exception_free (&ev);
			CORBA_free (task);
			bonobo_object_release_unref (tm, NULL);

			return FALSE;
	}

	CORBA_exception_free (&ev);
	CORBA_free (task);
	bonobo_object_release_unref (tm, NULL);

	return TRUE;
}
	
static void 
allocation_manager_init (AllocationManager *am)
{
	AllocationManagerPriv   *priv;
	
	priv        = g_new0 (AllocationManagerPriv, 1);
	priv->state = GNOME_MrProject_PROJECT_STATE_EMPTY;
	am->priv    = priv;
}

static void 
allocation_manager_class_init (AllocationManagerClass   *klass)
{
        POA_GNOME_MrProject_AllocationManager__epv   *epv;
        GtkObjectClass                               *object_class;

        epv           = &klass->epv;
        object_class  = (GtkObjectClass *) klass;
        parent_class  = gtk_type_class (PARENT_TYPE);

        /* GtkObject */
        object_class->destroy  = am_destroy;
	object_class->get_arg  = am_get_arg;
	object_class->set_arg  = am_set_arg;
        
        /* Allocation Management */
        epv->_set_state             = impl_AllocationManager__set_state;
        epv->_get_state             = impl_AllocationManager__get_state;
        epv->allocate               = impl_AllocationManager_allocate;
	epv->deallocate             = impl_AllocationManager_deallocate;
	epv->getAllocation          = impl_AllocationManager_getAllocation;
	epv->getAllocationsByTask   = impl_AllocationManager_getAllocationsByTask;
	epv->getAllocationsByResource = impl_AllocationManager_getAllocationsByResource;
	epv->getAllAllocations      = impl_AllocationManager_getAllAllocations;
	epv->deallocateAllResources = impl_AllocationManager_deallocateAllResources;
	epv->deallocateAllTasks     = impl_AllocationManager_deallocateAllTasks;
	
	/* Object arguments */
	gtk_object_add_arg_type ("AllocationManager::state", 
				 GTK_TYPE_INT, 
				 GTK_ARG_READWRITE, 
				 ARG_STATE);
}

static void
am_get_arg (GtkObject   *object, 
	    GtkArg      *arg, 
	    guint        arg_id)
{
	AllocationManager       *manager;
	AllocationManagerPriv   *priv;
	
	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_ALLOCATION_MANAGER (object));

	manager = ALLOCATION_MANAGER (object);
	priv = manager->priv;

	switch (arg_id) {
	case ARG_STATE:
		GTK_VALUE_INT (*arg) = priv->state;
		break;
		
	default:
		arg->type = GTK_TYPE_INVALID;
		break;
	}
}

static void
am_set_arg (GtkObject   *object, 
	    GtkArg      *arg, 
	    guint        arg_id)
{
	AllocationManager       *manager;
	AllocationManagerPriv   *priv;
	
	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_ALLOCATION_MANAGER (object));

	manager = ALLOCATION_MANAGER (object);
	priv = manager->priv;

	switch (arg_id) {
	case ARG_STATE:
		priv->state = GTK_VALUE_INT (*arg);
		break;
	default:
		return;
		break;
	}
}

static void 
am_destroy (GtkObject *object)
{
	AllocationManager       *manager;
	AllocationManagerPriv   *priv;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_ALLOCATION_MANAGER (object));
	
	manager  = ALLOCATION_MANAGER (object);
	priv     = manager->priv;

	if (priv->event_source) {
		if (priv->listener_id) {
			bonobo_event_source_client_remove_listener (BONOBO_OBJREF (priv->event_source),
								    priv->listener_id,
								    NULL);
			priv->listener_id = 0;
		}

		priv->event_source = NULL;
	}

        GNOME_CALL_PARENT_HANDLER (GTK_OBJECT_CLASS, destroy, (object));
}

static void
am_tasks_removed (AllocationManager *am, CORBA_any *any)
{
	GM_AllocationManager    manager_co;
	CORBA_Environment       ev;
	/*guint                   i;*/
	CORBA_char             *ids;
	GSList                 *list, *l;
	/*GM_IdSeq               *ids;*/
	
	ids = BONOBO_ARG_GET_STRING (any);
	list = corba_util_id_string_to_list (ids);

	manager_co = bonobo_object_corba_objref (BONOBO_OBJECT (am));
	
	if (!manager_co) {
		g_warning ("Couldn't get the CORBA object from manager");
		return;
	}
	
	CORBA_exception_init (&ev);

	/*for (i = 0; i < ids->_length; ++i) {*/
	for (l = list; l; l = l->next) {
		/*GNOME_MrProject_AllocationManager_deallocateAllResources (manager_co,
									  ids->_buffer[i],
									  &ev);*/
		GNOME_MrProject_AllocationManager_deallocateAllResources (manager_co,
									  GPOINTER_TO_INT (l->data),
									  &ev);
	}
	
	CORBA_exception_free (&ev);
}

static void      
am_resource_removed (AllocationManager *am, CORBA_any *any)
{
	GM_AllocationManager    manager_co;
	GM_Resource            *resource;
	CORBA_Environment       ev;
	
	resource    = (GM_Resource *) any->_value;
 	manager_co  = bonobo_object_corba_objref (BONOBO_OBJECT (am)); 
	
	if (!manager_co) {
		g_warning ("Couldn't get the CORBA object from manager");
		return;
	}

	CORBA_exception_init (&ev);

	GNOME_MrProject_AllocationManager_deallocateAllTasks (manager_co,
							      resource->resourceId,
							      &ev);
	
	CORBA_exception_free (&ev);
}

static void
listener_callback (BonoboListener      *listener,
		   char                *event_name,
		   CORBA_any           *any,
		   CORBA_Environment   *ev,
		   gpointer             user_data)
{
	AllocationManager   *am;

	g_return_if_fail (user_data != NULL);
	g_return_if_fail (IS_ALLOCATION_MANAGER (user_data));
	
	am = ALLOCATION_MANAGER (user_data);
	
	if (strstr (event_name, "task:removed_seq")) {
		am_tasks_removed (am, any);
	}
	else if (strstr (event_name, "resource:removed")) {
		am_resource_removed (am, any);
	} else {
		g_print ("AllocationManager: got unhandled event: %s\n", 
			 event_name);
	}
}


void
allocation_manager_start_listen (AllocationManager *am)
{
	AllocationManagerPriv   *priv;
	CORBA_Environment        ev;

	priv = am->priv;

	g_return_if_fail (priv->event_source != NULL);
	
	CORBA_exception_init (&ev);

	priv->listener_id =
		bonobo_event_source_client_add_listener (BONOBO_OBJREF (priv->event_source),
							 listener_callback,
							 "GNOME/MrProject:resource:removed,"
							 "GNOME/MrProject:task:removed_seq",
							 &ev,
							 am);
	
	CORBA_exception_free (&ev);
}

void
allocation_manager_foreach_allocation (AllocationManager *am,
				       GFunc              func,
				       gpointer           user_data)
{
	AllocationManagerClass           *klass;

	g_return_if_fail (am != NULL);
	g_return_if_fail (IS_ALLOCATION_MANAGER (am));
	
	klass = ALLOCATION_MANAGER_CLASS (GTK_OBJECT (am)->klass);
	
	if (klass->foreach_allocation) {
		klass->foreach_allocation (am, func, user_data);
	}
}

void
allocation_manager_construct (AllocationManager  *am,
			      BonoboEventSource  *event_source)
{
	AllocationManagerPriv   *priv;

	priv = am->priv;
	
	priv->event_source = event_source;
}



BONOBO_X_TYPE_FUNC_FULL (AllocationManager,
			 GNOME_MrProject_AllocationManager,
			 PARENT_TYPE,
			 allocation_manager);
