/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2000-2001 CodeFactory AB
 * Copyright (C) 2000-2001 Richard Hult <rhult@codefactory.se>
 * Copyright (C) 2001      Mikael Hallendal <micke@codefactory.se>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Richard Hult
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h>
#include <time.h>
#include <bonobo.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-i18n.h>
#include "util/type-utils.h"
#include "util/corba-utils.h"
#include "libmrproject/GNOME_MrProject.h"
#include "file-task-model.h"


/* GtkObject. */
static void     file_task_model_init       (FileTaskModel      *model);
static void     file_task_model_class_init (FileTaskModelClass *klass);


GNOME_CLASS_BOILERPLATE (FileTaskModel, file_task_model,
			 TaskModel, task_model);

static void
ftm_destroy (GtkObject *object)
{
	FileTaskModel *ftm;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_FILE_TASK_MODEL (object));
	
	ftm = FILE_TASK_MODEL (object);

	if (ftm->task) {
		CORBA_free (ftm->task);
	}
	
	g_node_destroy (ftm->node);

/*	g_slist_foreach (ftm->predecessors, (GFunc) CORBA_free, NULL);
	g_slist_free (ftm->predecessors);
	ftm->predecessors = NULL;

	g_slist_foreach (ftm->successors, (GFunc) CORBA_free, NULL);
	g_slist_free (ftm->successors);
	ftm->successors = NULL;
*/
	ftm->task = NULL;
	ftm->node = NULL;

	GNOME_CALL_PARENT_HANDLER (GTK_OBJECT_CLASS, destroy, (object));
}

static void
ftm_add_predecessor (TaskModel                  *task,
		     const GNOME_MrProject_Dependency *dependency)
{
	FileTaskModel        *ftm;
	GNOME_MrProject_Dependency *copy;

	ftm = FILE_TASK_MODEL (task);
	
	copy = corba_util_dependency_duplicate (dependency);
	ftm->predecessors = g_slist_prepend (ftm->predecessors, copy);
}

static void
ftm_add_successor (TaskModel                  *task,
		   const GNOME_MrProject_Dependency *dependency)
{
	FileTaskModel        *ftm;
	GNOME_MrProject_Dependency *copy;

	ftm = FILE_TASK_MODEL (task);

	copy = corba_util_dependency_duplicate (dependency);
	ftm->successors = g_slist_prepend (ftm->successors, copy);
}

static void
ftm_remove_predecessor (TaskModel          *task,
			GNOME_MrProject_Id  id)
{
	FileTaskModel *ftm;
	GSList        *list;

	ftm = FILE_TASK_MODEL (task);

	for (list = ftm->predecessors; list; list = list->next) {
		GNOME_MrProject_Dependency *dependency;

		dependency = list->data;

		if (dependency->predecessorId == id) {
			ftm->predecessors = g_slist_remove_link (
				ftm->predecessors, list);
			g_slist_free (list);
			CORBA_free (dependency);
			break;
		}
	}
}

static void
ftm_remove_successor (TaskModel          *task,
		      GNOME_MrProject_Id  id)
{
	FileTaskModel *ftm;
	GSList        *list;

	ftm = FILE_TASK_MODEL (task);
	
	for (list = ftm->successors; list; list = list->next) {
		GNOME_MrProject_Dependency *dependency;

		dependency = list->data;

		if (dependency->taskId == id) {
			ftm->successors = g_slist_remove_link (
				ftm->successors, list);
			g_slist_free (list);
			CORBA_free (dependency);
			break;
		}
	}
}

static GSList *
ftm_get_predecessors (TaskModel *task)
{
	FileTaskModel *ftm;

	ftm = FILE_TASK_MODEL (task);
	return ftm->predecessors;
}

static GSList *
ftm_get_successors (TaskModel *task)
{
	FileTaskModel *ftm;

	ftm = FILE_TASK_MODEL (task);
	return ftm->successors;
}

static gboolean
ftm_has_predecessor (TaskModel          *task,
		     GNOME_MrProject_Id  id,
		     GNOME_MrProject_Id *dependency_id)
{
	FileTaskModel *ftm;
	GSList        *list;

	ftm = FILE_TASK_MODEL (task);

	for (list = ftm->predecessors; list; list = list->next) {
		GNOME_MrProject_Dependency *dependency;

		dependency = list->data;

		if (dependency->predecessorId == id) {
			if (dependency_id) {
				*dependency_id = dependency->depId;
			}
			return TRUE;
		}
	}
		
	return FALSE;
}

static gboolean
ftm_has_successor (TaskModel          *task,
		   GNOME_MrProject_Id  id,
		   GNOME_MrProject_Id *dependency_id)
{
	FileTaskModel *ftm;
	GSList        *list;

	ftm = FILE_TASK_MODEL (task);

	for (list = ftm->successors; list; list = list->next) {
		GNOME_MrProject_Dependency *dependency;

		dependency = list->data;

		if (dependency->taskId == id) {
			if (dependency_id) {
				*dependency_id = dependency->depId;
			}
			return TRUE;
		}
	}
		
	return FALSE;
}

static gboolean
ftm_has_ancestor (TaskModel *task,
		  TaskModel *ancestor)
{
	FileTaskModel *ftm;
	FileTaskModel *ancestor_ftm;
	GNode         *node;

	ftm = FILE_TASK_MODEL (task);
	ancestor_ftm = FILE_TASK_MODEL (ancestor);

	node = ftm->node;
	while (node) {
		if (node == ancestor_ftm->node) {
			return TRUE;
		}

		node = node->parent;
	}
	return FALSE;
}

static GNOME_MrProject_Task *
ftm_get_task (TaskModel *task_model)
{
	FileTaskModel *ftm;

	ftm = FILE_TASK_MODEL (task_model);
	return ftm->task;
}

static gint
ftm_get_num_children (TaskModel *task_model)
{
	FileTaskModel *ftm;

	ftm = FILE_TASK_MODEL (task_model);
	return (gint) g_node_n_children (ftm->node);
}

static void
file_task_model_class_init (FileTaskModelClass *klass)
{
	GtkObjectClass *object_class;
	TaskModelClass *task_model_class;
	
	task_model_class = (TaskModelClass*) klass;
	object_class     = (GtkObjectClass*) klass;

	task_model_class->add_predecessor    = ftm_add_predecessor;
	task_model_class->add_successor      = ftm_add_successor;
	task_model_class->remove_predecessor = ftm_remove_predecessor;
	task_model_class->remove_successor   = ftm_remove_successor;
	task_model_class->get_predecessors   = ftm_get_predecessors;
	task_model_class->get_successors     = ftm_get_successors;
	task_model_class->has_predecessor    = ftm_has_predecessor;
	task_model_class->has_successor      = ftm_has_successor;
	task_model_class->has_ancestor       = ftm_has_ancestor;
	task_model_class->get_task           = ftm_get_task;
	task_model_class->get_num_children   = ftm_get_num_children;
	
	object_class->destroy                = ftm_destroy;
}

static void
file_task_model_init (FileTaskModel *ftm)
{
	ftm->task= NULL;
	ftm->node = g_node_new (ftm);
}

TaskModel *
file_task_model_new (const GNOME_MrProject_Task *task)
{
	FileTaskModel        *ftm;
	
	ftm = FILE_TASK_MODEL (gtk_type_new (TYPE_FILE_TASK_MODEL));
	ftm->task = corba_util_task_duplicate (task);
	
	return TASK_MODEL (ftm);
}

