/* d4libh.c 
 * Copyright (C) 2001 Jean-Jacques Sarton jj.sarton@t-online.de
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* this file will be a part of a library, which will allow to use
 * the EPSON Stylus Scanner.
 *
 * The functions contained here will be replacement for:
 *  open()
 *  close()
 *  write()
 *  read()
 *
 * I dont own a Stylus Scanner and I am not able to test this
 * code.
 *
 * The best way to get the Stylus Scanner workimg is to test
 * this and also correct my possibly errors.
 * Programming knowledge will be helpfull for this.
 *
 * The implemented function shall work as close as possible
 * as the open(), ... functions.
 *
 * In order to make the scanner part working. it will be
 * necessary to replace the calls for opening, writing to,
 * reading from and closing the device by the funxtions
 * included here.
 * 
 * Please take care fron the fact that closing of the
 * connection is NECESSARY.
 *
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <signal.h>
#include <fcntl.h>
#include <sys/time.h>
#include <unistd.h>
#include <errno.h>

#include "d4lib.h"
#include "d4libh.h"

/* the size for control and data (printer) buffers are OK */
#define CTRL_WR_SZ 0x0200
#define CTRL_RD_SZ 0x0200

#define DATA_WR_SZ 0x1006
#define DATA_RD_SZ 0x0200

#define SCAN_WR_SZ 0x0200  /* is this OK ? */
#define SCAN_RD_SZ 0x1006  /* include 6 bytes for IEEE header */

typedef struct channel_s 
{
  uc  socketID;
  int wrSz;
  int rdSz;
  int creditWr;
  int creditRd;
  uc  *rdBuffer;
  int mark;
  int count;
} channel_t;

static channel_t channelData[3] =
{
   { 0, 0, 0, 0, 0, NULL, 0, 0 },
   { 0, 0, 0, 0, 0, NULL, 0, 0 },
   { 0, 0, 0, 0, 0, NULL, 0, 0 }
};

# define CTRL 0
# define DATA 1
# define SCAN 2

static int internalFD  =  0;
static int mode        = -1;
static int idx         =  0;


/*******************************************************************/
/* Function sigKill(int code)                                      */
/*                                                                 */
/* terminate gacefully if a sigkill occured                        */
/*                                                                 */
/*******************************************************************/

static void sigKill(int code)
{
   //terminate = 1;
   //signr     = code;
}

/*******************************************************************/
/* Function Open()                                                 */
/*    Open the given port and initialize the IEEE protokoll        */
/*                                                                 */
/* Input:  char *path the name of device file                      */
/*         int   mode  PRINTER, SCANNER or CONTROL the channel     */
/*                     to be opened                                */
/*                                                                 */
/* Return: fatal = -1 or the file handle                           */
/*                                                                 */
/*******************************************************************/

int Open(char *path, int mode)
{
   int fd;
   int sockId = 0;
#if DEBUG
   fprintf(stderr,"Open %s\n",path);
#endif
   if ( (fd = open(path, O_RDWR)) <= 0 )
   {
      return fd;
   }

   channelData[CTRL].wrSz     = CTRL_WR_SZ;
   channelData[CTRL].rdSz     = CTRL_RD_SZ;
   channelData[DATA].wrSz     = DATA_WR_SZ;
   channelData[DATA].rdSz     = DATA_RD_SZ;
   channelData[SCAN].wrSz     = SCAN_WR_SZ;
   channelData[SCAN].rdSz     = SCAN_RD_SZ;
   channelData[DATA].creditWr =      0;
   channelData[DATA].creditRd =      0;
   channelData[SCAN].creditWr =      0;
   channelData[SCAN].creditRd =      0;
   channelData[CTRL].creditWr =      0;
   channelData[CTRL].creditRd =      0;
   channelData[CTRL].mark     =     -1;
   channelData[CTRL].count    =      0;
   channelData[DATA].mark     =     -1;
   channelData[DATA].count    =      0;
   channelData[SCAN].mark     =     -1;
   channelData[SCAN].count    =      0;
  
   signal(SIGTERM, sigKill);
   signal(SIGINT,  sigKill);
   signal(SIGQUIT, sigKill);
   signal(SIGSTOP, sigKill);

   if ( ! EnterIEEE(fd) )
   {
#if DEBUG
      fprintf(stderr,"Terminate with error at Enter IEEE\n");
#endif
      return -1;
   }
   if ( ! Init(fd) )
   {
#if DEBUG
      fprintf(stderr,"Terminate with error at Init IEEE\n");
#endif
      return -1;
   }

   if ( mode == PRINTER )
   {
      sockId = GetSocketID(fd, "EPSON-DATA");
      idx = DATA;
   }
   else if ( mode == SCANNER )
   {
      sockId = GetSocketID(fd, "EPSON-SCAN");
      idx = SCAN;
   }
   else if ( mode == CONTROL )
   {
      sockId = 2 /*GetSocketID(fd, "EPSON-CTRL")*/;
      idx = CTRL;
   }

   if ( sockId == 0 )
   {
      return -1;
   }

   /* data / scan channel */
   channelData[idx].socketID = sockId;
   switch ( OpenChannel(fd, sockId,
                        &channelData[idx].wrSz,
                        &channelData[idx].rdSz) )
   {
      case -1:
         /* unrecoverable error */
#if DEBUG
         fprintf(stderr,"Fatal Error at Open channel\n");
#endif
         close(fd);
         return -1;
         /* unrecoverable error */
         break;
      case  0:
#if DEBUG
         fprintf(stderr, "Error at Open channel\n"); /* recoverable error ? */
#endif
         close(fd);
         return -1;
   }

   if ( idx != CTRL )
   {
      /* allocate the own buffers */
      channelData[idx].rdBuffer = (uc*) malloc(channelData[idx].rdSz);
      if ( channelData[idx].rdBuffer == NULL )
      {
#if DEBUG
         perror("Open: ");
#endif
         close(fd);
         return -1;
      }
   }
   else
   {
      if ( channelData[CTRL].socketID )
      {
         if ( channelData[CTRL].rdSz == 0 )
             channelData[CTRL].rdSz    = 0x0200;
         channelData[CTRL].rdBuffer = (uc*) malloc(channelData[CTRL].rdSz);
         if ( channelData[CTRL].rdBuffer == NULL )
         {
            channelData[CTRL].rdSz = 0;
         }
      }
   }

   internalFD = fd;
   return fd;
}

/*******************************************************************/
/* Function Close()                                                */
/*    Close the IEEE channel and the opened file                   */
/*                                                                 */
/* Input:  int fd the file handle                                  */
/*                                                                 */
/* Return: fatal = -1 or 0                                         */
/*                                                                 */
/*******************************************************************/

int Close(int fd)
{
   if ( fd != internalFD )
   {
      return close(fd);
   }

   if ( mode == PRINTER )
   {
      /* send an EOJ command */
      writeData(fd, channelData[idx].socketID,
                channelData[idx].rdBuffer,
                0, 1);
   }
   
   /* close the channels */
   CloseChannel(fd, channelData[idx].socketID);
   channelData[idx].socketID  = 0;
   close(fd);

   /* free the memory */
   if ( channelData[idx].rdBuffer != NULL )
   {
      free(channelData[idx].rdBuffer );
      channelData[idx].rdBuffer = NULL;
      channelData[idx].rdSz    = 0;
   }

   internalFD = 0;
   channelData[idx].count  =  0;
   channelData[idx].mark   = -1;

   return 0;
}

/*******************************************************************/
/* Function Read()                                                 */
/*    Read from a previously opened channel                        */
/*                                                                 */
/* Input:  int   fd     the file handle                            */
/*         char *buf    the data buffer                            */
/*         size_t count the number of bytes to read                */
/*                                                                 */
/* Return: fatal = -1 on error or the number of bytes read         */
/*                                                                 */
/* Remark: The number of file read may be less as the required #   */
/*         This function will only return as many bytes as the     */
/*         device is able to send (limited by the declared IEEE    */
/*         buffer size)                                            */
/*                                                                 */
/*******************************************************************/

int Read(int fd, char *buf, size_t count)
{
   int rd = 0;
   int in;
   int got;

   if ( fd != internalFD )
   {
      errno = EBADF;
      return -1;
   }
   /* the number of bytes required has to be reached, dont   */
   /* leave the loop if there is no time out (return = -1)    */
   {
      /* take account from the 6 IEEE protocol bytes while calculating */
      /* the number of bytes we can read                               */
      in = (count - rd >= ( channelData[idx].rdSz - 6 ) ) ?
                                channelData[idx].rdSz - 6 :
                                count - rd;
      got = readData(fd,channelData[idx].socketID, buf+rd, in);
      if( got < 0 )
      {
         return got;
      }
      else if ( got == 0 )
      {
         return rd;
      }
      else
      {
         rd += got;
      }
   }
   return rd;
}

/*******************************************************************/
/* Function Write()                                                */
/*    Write to a previously opened channel                         */
/*                                                                 */
/* Input:  int   fd     the file handle                            */
/*         char *buf    the data buffer                            */
/*         size_t count the number of bytes to write               */
/*                                                                 */
/* Return: fatal = -1 on error or the number of bytes write        */
/*                                                                 */
/*******************************************************************/

int Write(int fd, char *buf, size_t count)
{
   int wr = 0;
   int toWrite;
   int out;
   int eof = (mode == SCANNER || mode == CONTROL) ? 1 : 0;

   if ( fd != internalFD )
   {
      errno = EBADF;
      return -1;
   }
   
   while ( wr < count )
   {
       if ( channelData[idx].creditWr == 0 )
       {
          channelData[idx].creditWr = askForCredit(fd,
                                                  channelData[idx].socketID,
                                                  &channelData[idx].wrSz,
                                                  &channelData[idx].rdSz);
         if ( channelData[idx].creditWr < 1 )
         {
            return -1;
         }
      }
      /* take account of the 6 IEEE protokol bytes incl. in wrSz */
      toWrite = ((count-wr) <= (channelData[idx].wrSz-6)) ?
                                          (count-wr) :
                                          channelData[idx].wrSz-6;
      out = writeData(fd, channelData[idx].socketID, buf + wr, toWrite, eof);
      if ( out > 0 )
      {
         wr += out;
      }
      else
      {
         return -1;
      }
      channelData[idx].creditWr--;
   }
   return wr;
}
