//=========================================================
//  MusE
//  Linux Music Editor
//  $Id: s1.cpp,v 1.1.1.1 2003/10/29 10:06:00 wschweer Exp $
//
//    S1  - simple mono demo synthesizer
//
//  (C) Copyright 2001 Werner Schweer (ws@seh.de)
//=========================================================

#include <cmath>
#include <stdio.h>
#include <sys/time.h>
#include <pthread.h>

#include "mess.h"
#include "ladspa.h"

#define RESOLUTION   16384

//---------------------------------------------------------
//   S1 - simple mono demo synthesizer
//---------------------------------------------------------

class S1 : public MessMono {
      static int useCount;

      static float *wave_table;
      bool gate;
      float velocity;
      float freq;
      unsigned accu;

      void noteon(int channel, int pitch, int velo);
      void noteoff(int channel, int pitch);
      virtual void write(int n, float** buffer, int);

   public:
      S1(const char* classname);
      ~S1();
      virtual bool init();
      };

float* S1::wave_table;
int S1::useCount = 0;

//---------------------------------------------------------
//   S1
//---------------------------------------------------------

S1::S1(const char* cn)
   : MessMono(cn, 1)
      {
      if (useCount++ == 0) {
            wave_table = new float[RESOLUTION];
            for (int i = 0; i < RESOLUTION; i++)
                  wave_table[i] = sin ((i * 2.0 * M_PI) / RESOLUTION) / 6.0;
            }
      }

//---------------------------------------------------------
//   ~S1
//---------------------------------------------------------

S1::~S1()
      {
      if (--useCount == 0)
            delete[] wave_table;
      }

//---------------------------------------------------------
//   init
//    return true on error
//---------------------------------------------------------

bool S1::init()
      {
      gate     = false;
      velocity = 0.0;
      accu     = 0;
      return false;
      }

//---------------------------------------------------------
//   noteon
//    process note on
//---------------------------------------------------------

void S1::noteon(int /*channel*/, int pitch, int velo)
      {
      gate     = true;
      velocity = velo / 128.0;
      freq     = 8.176 * exp(float(pitch)*log(2.0)/12.0);
      }

//---------------------------------------------------------
//   noteoff
//    process note off
//---------------------------------------------------------

void S1::noteoff(int, int)
      {
      gate     = false;
      velocity = 0;
      accu     = 0;
      }

//---------------------------------------------------------
//   write
//    synthesize n samples into buffer+offset
//---------------------------------------------------------

void S1::write(int n, float** buffer, int offset)
      {
      if (!gate)
            return;
      float* p = buffer[0] + offset;
      unsigned freq_256 = (int) (freq * ((double) RESOLUTION) / sampleRate() * 256.0);
      for (int i = 0; i < n; i++) {
            accu += freq_256;
            while (accu >= RESOLUTION * 256)
                  accu -= RESOLUTION * 256;
            p[i] += wave_table[accu >> 8];
            }
      }

//---------------------------------------------------------
//   inst
//---------------------------------------------------------

static Mess* inst(const char* name)
      {
      return new S1(name);
      }

//---------------------------------------------------------
//   initMess
//---------------------------------------------------------

void initMess()
      {
      Mess::initMess(
          3921,
         "S1",
         "Werner Schweer",
         "S1 demo soft synth",
         "None",
         1,
         inst);
      }

