// conv_config.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/

// Methods for protected RealConverter::ConfigRequester class and its subclasses
// These internal classes request user input for converter parameters such as
// playback level and output port.

#include "localdefs.h"
#include "converter.h"
#include "query_templates.h"
#include "request.h"
#include "sndconfig.h"

RealConverter::ConfigRequester::ConfigRequester(RealConverter* converter)
        : TitledRequester("Converter Settings:"),
          myConverter(converter) {}

void
RealConverter::ConfigRequester::configureRequest(Request* request) {
    request->appendValue(
#if defined( __GNUG__ ) || !defined(TEMPLATES_WORK_REALLY_WELL)
        new Query<int>(
            "Record Gain:",
            createSetterGetter(
                myConverter,
                &RealConverter::setRecordLevel,
                &RealConverter::currentRecordLevel),
            Range(0, 100),
            true
        )
#else
        createQuery(
            "Record Gain:",
            myConverter,
            &RealConverter::setRecordLevel,
            &RealConverter::currentRecordLevel,
            Range(0, 100),
            true
        )
#endif
    );
    request->appendValue(
#if defined( __GNUG__ ) || !defined( TEMPLATES_WORK_REALLY_WELL )
        new Query<int>(
            "Playback Gain:",
            createSetterGetter(
                myConverter,
                &RealConverter::setPlayLevel,
                &RealConverter::currentPlayLevel),
            Range(0, 100),
            true
        )
#else
        createQuery(
            "Playback Gain:",
            myConverter,
            &RealConverter::setPlayLevel,
            &RealConverter::currentPlayLevel,
            Range(0, 100),
            true
        )
#endif
    );
}

#ifdef NeXT

#include "next_dac.h"

void
NextConverter::NextConfigRequester::configureRequest(Request* request) {
    ConfigRequester::configureRequest(request);
}

boolean
NextConverter::NextConfigRequester::confirmValues() {
    return ConfigRequester::confirmValues();
}

#endif /* Next */

#ifdef SGI_DACS

#include "sgi_dac.h"
#include "MyString.h"

void
SGIConverter::SGIConfigRequester::configureRequest(Request* request) {
    ConfigRequester::configureRequest(request);
    SGIConverter* sc = (SGIConverter *) myConverter;

    inputDeviceValue = sc->inputDeviceToChoice(sc->currentInputDevice());

    // assemble string of available input device names
    String devNames = "|";
    for(int i=0; i < sc->inputDeviceCount; i++) {
	devNames += sc->inputDevices[i].label;
	devNames += "|";
    }
    request->appendChoice("Input Device:", devNames, &inputDeviceValue, true);

#ifdef AL_SYSTEM
    // assemble string of available output device names
    devNames = "|";
    for(i=0; i < sc->outputDeviceCount; i++) {
	devNames += sc->outputDevices[i].label;
	devNames += "|";
    }
    outputDeviceValue = sc->outputDeviceToChoice(sc->currentOutputDevice());
    request->appendChoice("Output Device:", devNames, &outputDeviceValue, true);
#endif
    polite = sc->dontInterrupt ? True : False;
    request->appendChoice("Converter resource access mode:",
        "|rude|polite|",
        &polite,
        true
    );
}

boolean
SGIConverter::SGIConfigRequester::confirmValues() {
    if(ConfigRequester::confirmValues()) {
        SGIConverter* sc = (SGIConverter *) myConverter;
        sc->dontInterrupt = (polite == True);
        sc->setOutputDevice(sc->outputChoiceToDevice(outputDeviceValue));    
        return sc->setInputDevice(sc->inputChoiceToDevice(inputDeviceValue));    
    }
    return false;
}

#endif /* SGI_DACS */

#ifdef SPARC_DACS

#include "sun_dac.h"

void
SunConverter::SunConfigRequester::configureRequest(Request* request) {
    SunConverter* sc = (SunConverter *) myConverter;
    ChoiceValue inputPort = 0;
    ChoiceValue outputPort = 0;
    if(sc->reOpen()) {
        ConfigRequester::configureRequest(request);
        inputPort = sc->currentInputPort();
        outputPort = sc->currentOutputPort();
        sc->close();
        request->appendChoice("Input Port:", "|mike|line|",
                     &inputPort, true);
        request->appendChoice("Output Port:", "|speaker|headphone|",
                             &outputPort, true);
    }
    else
        request->appendLabel("Unable to open converter device");
}

boolean
SunConverter::SunConfigRequester::confirmValues() {
    SunConverter* sc = (SunConverter *) myConverter;
    boolean status = false;
    if(sc->reOpen()) {
        if(ConfigRequester::confirmValues()) {
            status = sc->setInputPort(SunConverter::InputPort(inputPort));
            status = status && sc->setOutputPort(SunConverter::OutputPort(outputPort));
        }
        sc->close();
    }
    return status;
}

#endif /* SPARC_DACS */

#ifdef OSS

#include "oss_converter.h"
#include "MyString.h"

void
OSS_Converter::OSS_ConfigRequester::configureRequest(Request* request) {
    ConfigRequester::configureRequest(request);
    OSS_Converter* oc = (OSS_Converter *) myConverter;

    // assemble string of available input device names
    String devNames;
    oc->getInputDeviceLabels(devNames);
    
    inputDeviceValue = oc->inputDeviceToChoice(oc->currentInputDevice());

    request->appendChoice("Input Device:", devNames, &inputDeviceValue, true);

#ifdef SOME_FUTURE_LINUX
    // assemble string of available output device names
    oc->getOutputDeviceLabels(devNames);

    outputDeviceValue = oc->outputDeviceToChoice(oc->currentOutputDevice());

    request->appendChoice("Output Device:", devNames, &outputDeviceValue, true);
#endif
}

boolean
OSS_Converter::OSS_ConfigRequester::confirmValues() {
    if(ConfigRequester::confirmValues()) {
        OSS_Converter* oc = (OSS_Converter *) myConverter;
#ifdef SOME_FUTURE_LINUX
        oc->setOutputDevice(oc->outputChoiceToDevice(outputDeviceValue));    
#endif
        return oc->setInputDevice(oc->inputChoiceToDevice(inputDeviceValue));
    }
    return false;
}

#endif /* OSS */

#ifdef ALSA

#include "alsa_converter.h"
#include "MyString.h"

void
ALSA_Converter::ALSA_ConfigRequester::configureRequest(Request* request) {
    ConfigRequester::configureRequest(request);
    ALSA_Converter* ac = (ALSA_Converter *) myConverter;

    // assemble string of available input device names
    String devNames;
    ac->getInputDeviceLabels(devNames);
    
    inputDeviceValue = ac->inputDeviceToChoice(ac->currentInputDevice());

    request->appendChoice("Input Device:", devNames, &inputDeviceValue, true);

#ifdef SOME_FUTURE_LINUX
    // assemble string of available output device names
    ac->getOutputDeviceLabels(devNames);

    outputDeviceValue = ac->outputDeviceToChoice(ac->currentOutputDevice());

    request->appendChoice("Output Device:", devNames, &outputDeviceValue, true);
#endif
}

boolean
ALSA_Converter::ALSA_ConfigRequester::confirmValues() {
    if(ConfigRequester::confirmValues()) {
        ALSA_Converter* ac = (ALSA_Converter *) myConverter;
#ifdef SOME_FUTURE_LINUX
        ac->setOutputDevice(ac->outputChoiceToDevice(outputDeviceValue));    
#endif
        return ac->setInputDevice(ac->inputChoiceToDevice(inputDeviceValue));
    }
    return false;
}

#endif /* ALSA */
#ifdef MACOSX
#include "osx_dac.h"

void
OSXConverter::OSXConfigRequester::configureRequest(Request* request) {
    ConfigRequester::configureRequest(request);
    OSXConverter* oc = (OSXConverter *) myConverter;
}

boolean
OSXConverter::OSXConfigRequester::confirmValues() {
    return ConfigRequester::confirmValues();
}

#endif	/* MACOSX */
