// oss_converter.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/

#ifdef OSS

#ifdef __GNUG__
#pragma implementation
#endif

#undef DEBUG_MIXER
#undef DEBUG

#include <assert.h>
#include <errno.h>
#include <fcntl.h>
#ifdef DEBUG_MIXER
#include <string.h>
#endif
#include <sys/soundcard.h>
#include "MyString.h"
#include "localdefs.h"
#include "application.h"
#include "oss_converter.h"
#include "progressaction.h"
#include "typeconvert.h"

// convert (1 << x) into x

int maskToIndex(int mask) {
    int idx = 0;
    assert(mask > 0);
    while (!(mask & (1 << idx))) idx++;
    return idx;
}

class HWMixer : public Device {
public:
	HWMixer();
	virtual ~HWMixer() {}
	int recordLevel() const;
	int playLevel() const;
	boolean setRecordLevel(int);
	boolean setPlayLevel(int);
	int recordDevice();
	int playDevice();
	boolean setRecordDevice(int);
	boolean setPlayDevice(int);
	boolean isAvailableInputDevice(int);
	boolean isAvailableOutputDevice(int);
	boolean present() { return _status != NotPresent; }
	boolean good() { return _status == Good; }
private:
	enum { NotPresent, Error, Good } _status;
	int _devMask;
	int _recMask;
	int _playMask;
	int _currentRecDevice, _currentPlayDevice;	// indices
};

static const char *getOSSMixerDevice()
{
    const char *theDevice = Application::getGlobalResource("OSSMixerDevice");
	return (theDevice != NULL) ? theDevice : "/dev/mixer";
}

HWMixer::HWMixer() : _status(NotPresent), _devMask(0), _recMask(0), _playMask(0) {
    if (!open(getOSSMixerDevice(), O_RDWR)) {
        if (errno == ENODEV) {
	    _status = NotPresent;
	}
	else
	    _status = Error;
    }
    else {
	if (!ioctl(SOUND_MIXER_READ_DEVMASK, &_devMask)) {
	    _status = (errno == ENXIO) ? NotPresent : Error;
	}
	else {
	    int mask = 0;
	    int status;
	    ioctl(SOUND_MIXER_READ_RECMASK, &_recMask);
            status = ioctl(SOUND_MIXER_READ_RECSRC, &mask);
#ifdef DEBUG_MIXER
	    printf("Mixer: _recMask: 0x%x, recsrc mask: 0x%x\n", _recMask, mask);
#endif
	    // set defaults in case of failure to query
	    if (!status || mask <= 0)
		_currentRecDevice = SOUND_MIXER_MIC;
	    else
		_currentRecDevice = maskToIndex(mask);

	    // set defaults in case of failure to query
	    if (!ioctl(MIXER_READ(SOUND_MIXER_OUTMASK), &_playMask)
	            || _playMask <= 0)
		_playMask = SOUND_MASK_SPEAKER;
            if (!ioctl(MIXER_READ(SOUND_MIXER_OUTSRC), &mask) || mask <= 0)
		_currentPlayDevice = SOUND_MIXER_SPEAKER;
	    else
		_currentPlayDevice = maskToIndex(mask);
#ifdef DEBUG_MIXER
	    printf("Mixer: current rec device: %d  current play device: %d\n",
		   _currentRecDevice, _currentPlayDevice);
#endif
	    _status = Good;
	}
    }
}

int
HWMixer::recordLevel() const {
    int level = 0 << 8 | 0;
    Device *d = (Device *) this;	// cast away const
    if (d->ioctl(MIXER_READ(_currentRecDevice), &level) && level >= 0) {
#ifdef DEBUG_MIXER
	printf("HWMixer::recordLevel(): device = 0x%x, level = 0x%x\n",
	       _currentRecDevice, level);
#endif
	return level & 0xff;
    }
    else {
#ifdef DEBUG_MIXER
	printf("HWMixer::recordLevel(): ioctl failed: %s\n", strerror(errno));
#endif
	return 0;
    }
}

int
HWMixer::playLevel() const {
    int level = 0 << 8 | 0;
    Device *d = (Device *) this;	// cast away const
    if (d->ioctl(MIXER_READ(_currentPlayDevice), &level) && level >= 0) {
#ifdef DEBUG_MIXER
	printf("HWMixer::playLevel(): device = 0x%x, level = 0x%x\n",
	       _currentPlayDevice, level);
#endif
	return level & 0xff;
    }
    else {
#ifdef DEBUG_MIXER
	printf("HWMixer::playLevel(): ioctl failed: %s\n", strerror(errno));
#endif
	return 0;
    }
}

boolean
HWMixer::setRecordLevel(int rLevel) {
    int level = rLevel << 8 | rLevel;
#ifdef DEBUG_MIXER
    printf("HWMixer::setRecordLevel(): device = 0x%x, level = 0x%x\n",
	   _currentRecDevice, level);
#endif
    boolean status = ioctl(MIXER_WRITE(_currentRecDevice), &level);
#ifdef DEBUG_MIXER
    if (!status) printf("\tioctl failed: %s\n", strerror(errno));
#endif
    return status;
}

boolean
HWMixer::setPlayLevel(int pLevel) {
    int level = pLevel << 8 | pLevel;
#ifdef DEBUG_MIXER
    printf("HWMixer::setPlayLevel(): device = 0x%x level = 0x%x\n",
	   _currentPlayDevice, level);
#endif
    boolean status = ioctl(MIXER_WRITE(_currentPlayDevice), &level);
#ifdef DEBUG_MIXER
    if (!status) printf("\tioctl failed: %s\n", strerror(errno));
#endif
    return status;
}

int
HWMixer::recordDevice() {
    int mask = 0;
    if (ioctl(SOUND_MIXER_READ_RECSRC, &mask) && mask > 0)
	_currentRecDevice = maskToIndex(mask);
#ifdef DEBUG_MIXER
    printf("HWMixer::recordDevice(): mask: 0x%x, device = %d\n",
           mask, _currentRecDevice);
#endif
    return _currentRecDevice;
}

int
HWMixer::playDevice() {
    int mask = 0;
    if (ioctl(MIXER_READ(SOUND_MIXER_OUTSRC), &mask) && mask > 0)
        _currentPlayDevice = maskToIndex(mask);
    return _currentPlayDevice;
}

boolean
HWMixer::setRecordDevice(int dev) {
    int val = 1 << dev;
#ifdef DEBUG_MIXER
    printf("HWMixer::setRecordDevice(): requesting device %d (0x%x)\n",
           dev, val);
#endif
    boolean status = ioctl(SOUND_MIXER_WRITE_RECSRC, &val);
    if (status && val > 0)
        _currentRecDevice = dev;
    else {
#ifdef DEBUG_MIXER
        if (val < 0) errno = -val;
        printf("HWMixer::setRecordDevice(): ioctl failed: %s", strerror(errno));
#endif
    }
#ifdef DEBUG_MIXER
    printf("HWMixer::setRecordDevice(): device now %d, val = 0x%x\n",
           _currentRecDevice, val);
#endif
    return status;
}

boolean
HWMixer::setPlayDevice(int dev) {
    int val = 1 << dev;
#ifdef DEBUG_MIXER
    printf("HWMixer::setPlayDevice(): requesting device %d (0x%x)\n",
           dev, val);
#endif
    boolean status = ioctl(MIXER_WRITE(SOUND_MIXER_OUTSRC), &val);
    if (status && val > 0)
        _currentPlayDevice = dev;
#ifdef DEBUG_MIXER
    printf("HWMixer::setPlayDevice(): device now 0x%x\n", _currentPlayDevice);
#endif
    return status;
}

// this is hacked because many drivers give bogus record channel masks - D.S.

boolean
HWMixer::isAvailableInputDevice(int dev) {
    return (dev > SOUND_MIXER_TREBLE
            && dev != SOUND_MIXER_RECLEV
	    && dev != SOUND_MIXER_IGAIN
	    && dev != SOUND_MIXER_OGAIN
	    && ((1 << dev) & _recMask));
}

boolean
HWMixer::isAvailableOutputDevice(int dev) {
    return (1 << dev) & _playMask;
}

// ********

int
OSS_Converter::_openFlags = 0;

char *
OSS_Converter::_deviceLabels[] = SOUND_DEVICE_NAMES;

static const char *getOSSAudioDevice()
{
    const char *theDevice = Application::getGlobalResource("OSSAudioDevice");
	return (theDevice != NULL) ? theDevice : "/dev/dsp";
}

OSS_Converter::OSS_Converter() : ConverterDevice(getOSSAudioDevice()),
                               _mixer(nil),
			       audioFormatBits(0), audioBufferSize(0) {
    BUG("OSS_Converter::OSS_Converter()");
    _mixer = new HWMixer;
    _mixer->ref();
    int devfd = ::open(deviceName(), O_RDWR);
    if (devfd < 0 || ::ioctl(devfd, SNDCTL_DSP_GETFMTS, (char *) &audioFormatBits) < 0)
        audioFormatBits = AFMT_U8 | AFMT_S16_LE;
    ::close(devfd);
}

OSS_Converter::~OSS_Converter() {
	if (isOpen())
	    stop();
	Resource::unref(_mixer);
	_mixer = nil;
}

int
OSS_Converter::extraOpenFlags() { return _openFlags; }

int
OSS_Converter::currentPlayLevel() const {
    return _mixer->playLevel();
}

int
OSS_Converter::currentRecordLevel() const {
    return _mixer->recordLevel();
}

boolean
OSS_Converter::setPlayLevel(int volume) {
    return _mixer->setPlayLevel(volume);
}

boolean
OSS_Converter::setRecordLevel(int volume) {
    return _mixer->setRecordLevel(volume);
}

int
OSS_Converter::currentInputDevice() {
   return _mixer->recordDevice();
}

int
OSS_Converter::currentOutputDevice() {
    return _mixer->playDevice();
}

boolean
OSS_Converter::setInputDevice(int dev) {
    return _mixer->setRecordDevice(dev);
}

boolean
OSS_Converter::setOutputDevice(int dev) {
    return _mixer->setPlayDevice(dev);
}

ChoiceValue
OSS_Converter::inputDeviceToChoice(int iDev) {
    ChoiceValue choice = 0x1;
    for(int d=0; d < SOUND_MIXER_NRDEVICES && d != iDev; d++) {
        if (_mixer->isAvailableInputDevice(d))
	    choice <<= 1;
    }
    return choice;
}

ChoiceValue
OSS_Converter::outputDeviceToChoice(int oDev) {
    ChoiceValue choice = 0x1;
    for(int d=0; d < SOUND_MIXER_NRDEVICES && d != oDev; d++) {
        if (_mixer->isAvailableOutputDevice(d))
	    choice <<= 1;
    }
    return choice;
}

int
OSS_Converter::inputChoiceToDevice(ChoiceValue val) {
    ChoiceValue current = 0x1;
    int dev = 0;
    for(; dev < SOUND_MIXER_NRDEVICES; dev++) {
        if (_mixer->isAvailableInputDevice(dev)) {
	    if (current == val)
		break;
	    current <<= 1;
	}
    }
#ifdef DEBUG_MIXER
    printf("OSS_Converter::inputChoiceToDevice(): choice = 0x%x, dev = %d\n",
           val, dev);
#endif
    return dev;
}

int
OSS_Converter::outputChoiceToDevice(ChoiceValue val) {
    ChoiceValue current = 0x1;
    int dev = 0;
    for(; dev < SOUND_MIXER_NRDEVICES; dev++) {
        if (_mixer->isAvailableOutputDevice(dev)) {
	    if (current == val)
		break;
	    current <<= 1;
	}
    }
#ifdef DEBUG_MIXER
    printf("OSS_Converter::outputChoiceToDevice(): choice = 0x%x, dev = %d\n",
           val, dev);
#endif
    return dev;
}

void
OSS_Converter::getInputDeviceLabels(String &labels) {
    labels = "|";
    for(int d=0; d < SOUND_MIXER_NRDEVICES; d++) {
        if (_mixer->isAvailableInputDevice(d)) {
	    labels += _deviceLabels[d];
	    labels += "|";
	}
    }
}

void
OSS_Converter::getOutputDeviceLabels(String &labels) {
    labels = "|";
    for(int d=0; d < SOUND_MIXER_NRDEVICES; d++) {
        if (_mixer->isAvailableOutputDevice(d)) {
	    labels += _deviceLabels[d];
	    labels += "|";
	}
    }
}

boolean
OSS_Converter::isPlayableFormat(DataType type) {
    switch(type)
    {
        case MuLawData:
	    return (audioFormatBits & AFMT_MU_LAW) != 0;
        case UnsignedCharData:
	    return (audioFormatBits & AFMT_U8) != 0;
        case SignedCharData:
	    return (audioFormatBits & AFMT_S8) != 0;
        case ShortData:
	    return (audioFormatBits & AFMT_S16_LE) != 0;
	default:
	    return false;
    }
	return false;
}

// what is best format to play (if given choice)

DataType
OSS_Converter::bestPlayableType() {
    return (audioFormatBits & AFMT_S16_LE) ? ShortData : UnsignedCharData;
}

// for future use -- no real way to do this yet

int
OSS_Converter::pause() {
	return false;
}

int
OSS_Converter::stop() {
	return ioctl(SNDCTL_DSP_RESET, 0) && Super::stop();
}

Requester *
OSS_Converter::configRequester() {
    Requester *req = nil;
    if (!_mixer->present()) {
	Application::alert("No mixer is available on this system,",
	      "and so no configuration is possible.");
    }
    else if (!_mixer->good()) {
	// avoiding call to Converter::error() cuz this is not conv. failure
	Application::error("Failed to access mixer hardware.");
    }
    else
	req = new OSS_ConfigRequester(this);
    return req;
}

int
OSS_Converter::checkChannels(int chans) {
	int status = false;
	if(chans != 1 && chans != 2)
		Application::alert("Converter only handles monaural and stereo files.");
	else
		status = true;
	return status;
}

int
OSS_Converter::doConfigure() {
    BUG("OSS_Converter::doConfigure()");
	int status = false;
	if(reOpen()) {
		int dsp_speed = sampleRate();
		int dsp_stereo = (channels() == 2);
		int sampleFormat = 0;
		switch(dataType()) {
			case UnsignedCharData: sampleFormat = AFMT_U8; break;
			case SignedCharData: sampleFormat = AFMT_S8; break;
//			case ALawData: sampleFormat = AFMT_A_LAW; break;
			case MuLawData: sampleFormat = AFMT_MU_LAW; break;
			case ShortData: sampleFormat = AFMT_S16_LE; break;
			default: break;
		};
		int confirmedFormat = sampleFormat;

		int sizeCode = 0x1;
		// default fragment size is 0.1 second's worth of sound
		float bufferTime = 0.1;
		const char *res = Application::getGlobalResource("OSSBufferTime");
		if (res)
		    bufferTime = atof(res);
		// We cannot set the frag size larger than the segment we wish to play.
		if (duration() < bufferTime)
			bufferTime = duration();
		int fragSize = type_to_sampsize(dataType()) *
		               iround(float(channels()) * sampleRate() * bufferTime);
#ifdef DEBUG
		fprintf(stderr, "data type = %d, data size = %d bytes\n",
			(int) dataType(), type_to_sampsize(dataType()));

		fprintf(stderr, "requesting frag size %d\n", fragSize);
#endif
		static const double L10_2 = log10(2.0);
		sizeCode = int(log10(fragSize) / L10_2);
		sizeCode |= (4 << 16);

		// sizeCode = 0x0004XXXX where XXXX is (int) log2(fragSize)
		// and 0004 is 4 max number of fragments

#ifdef DEBUG
		fprintf(stderr, "attempting to set frag size code to 0x%x\n", sizeCode);
#endif
		// we do not check for failure on this
		ioctl (SNDCTL_DSP_SETFRAGMENT, (char *) &sizeCode);
		if (!ioctl(SNDCTL_DSP_SETFMT, (char *) &confirmedFormat))
			error("Unable to set sample format.");
		else if(confirmedFormat != sampleFormat)
			error("This sample format not supported by hardware.");
		else if (!ioctl(SNDCTL_DSP_STEREO, (char *) &dsp_stereo))
			error("Unable to set channel attribute.");
		else if (!ioctl (SNDCTL_DSP_SPEED, (char *) &dsp_speed))
			error("Unable to set sample rate.");
		else if (abs(dsp_speed - sampleRate()) > (sampleRate() / 50))
		    Application::alert("Hardware does not support this sample rate");
		else if(!ioctl(SNDCTL_DSP_GETBLKSIZE, (char *) &audioBufferSize))
			error("Unable to get audio buffer size.");
		else status = true;
#ifdef DEBUG
		fprintf(stderr, "audio buffer size is %d\n", audioBufferSize);
		fprintf(stderr, "SR is %d\n", dsp_speed);
#endif
	}
	return status;
}

int
OSS_Converter::waitForStop(ProgressAction* askedToStop) {
    BUG("OSS_Converter::waitForStop()");
	ioctl(SNDCTL_DSP_SYNC, 0);
	int countdown = 2;	// ms count
    while(countdown-- > 0) {
        if((*askedToStop)(1.0))
            break;
        usleep(1000);        // allow audio buffer to drain
    }
    return true;
}

// return size of buffer, in bytes, to be written to the device during play

int
OSS_Converter::writeSize() {
	return audioBufferSize;
}

// return size of buffer, in bytes, to be read from the device during record

int
OSS_Converter::readSize() {
	return audioBufferSize;
}

#endif	/* OSS */
