/* (C) Copyright MySQL AB
**
** CURSOR.C - This is the MyODBC sample driver code for implementing
** client side cursors
**
** With a non-keyset client-side cursor, the server sends the entire
** result set across the network to the client machine. The client
** machine provides and manages the temporary resources needed by
** the cursor and result set. The client-side application can browse
** through the entire result set to determine which rows it requires.
**
** Static and keyset-driven client-side cursors may place a significant
** load on your workstation if they include too many rows. While all of
** the cursor libraries are capable of building cursors with thousands
** of rows, applications designed to fetch such large rowsets may perform
** poorly. There are exceptions, of course. For some applications, a large
** client-side cursor may be perfectly appropriate and performance may not
** be an issue.
**
** One obvious benefit of the client-side cursor is quick response. After
** the result set has been downloaded to the client machine, browsing
** through the rows is very fast. Your application is generally more
** scalable with client-side cursors because the cursor's resource
** requirements are placed on each separate client and not on the server.
**
** @author : MySQL AB
** @date   : 2001-June-01
*/

#include "myodbc.h"

static const char *find_used_table(STMT *stmt);

void my_param_bind(STMT FAR *srcStmt, STMT FAR *dstStmt,
                   SQLUSMALLINT nSrcCol,SQLUSMALLINT nDstCol);

SQLRETURN my_if_pk_exits(STMT FAR *stmt);
SQLRETURN my_pk_param_bind(STMT FAR *stmtNew,STMT FAR *stmt,
                           SQLUSMALLINT irow,SQLUSMALLINT nColParam);
my_bool my_build_where_clause(STMT FAR *stmt,
                              DYNAMIC_STRING *dynQuery);
void my_set_cursor_data(STMT FAR *stmt);
SQLRETURN my_pos_add(STMT FAR *stmt, SQLUSMALLINT irow, 
                     DYNAMIC_STRING dynQuery);
/*
   SQLSetCursorName associates a cursor name with an active statement.
*/

SQLRETURN SQL_API SQLSetCursorName(SQLHSTMT  hstmt,
                                   SQLCHAR FAR *szCursor,
                                   SQLSMALLINT cbCursor)
{
  STMT FAR *stmt=(STMT FAR*) hstmt;

  DBUG_ENTER("SQLSetCursorName");

  /* if null string passed as the cursor name */
  if (!szCursor)
    DBUG_RETURN(set_stmt_error(stmt,"S1009",
                               "Invalid use of null pointer",0));
  
  if (cbCursor == SQL_NTS)
    cbCursor = (SQLSMALLINT) strlen(szCursor);

  /*
    This will be handled by the DM, but when the application is linked
    with the client lib(myodbc.lib), then driver need to through the error
  */
  if (cbCursor < 0) 
    DBUG_RETURN(set_stmt_error(stmt,"S1090",
                               "Invalid string or buffer length",0));
  /*
    Through an error, when trying to set the cursor in the middle of
    execution/ fetch
  */
  if (stmt->cursor_state == MYSQL_CURSOR_IN_EXECUTION ||
      stmt->cursor_state == MYSQL_CURSOR_NEED_DATA) 
    DBUG_RETURN(set_stmt_error(stmt,"24000","Invalid cursor state",0));
  
  /* validate for cursor names */
  if ( (cbCursor == 0) ||
       (my_casecmp(szCursor, "SQLCUR", 6) == 0)  ||
       (my_casecmp(szCursor, "SQL_CUR", 7) == 0)) 
    DBUG_RETURN(set_stmt_error(stmt,"34000","Invalid cursor name",0));
  
  stmt->cursor_state = MYSQL_CURSOR_DEFINED;
  /*
    if length is larger than the supported, then truncate and
    return warning
  */
  if (cbCursor > MYSQL_MAX_CURSOR_LEN) 
  {
    /* set the warning ...by truncating the cursor name to max length */
    strmake( stmt->cursor_name, szCursor, MYSQL_MAX_CURSOR_LEN);
    set_stmt_error(stmt,"01004","String data, right truncated",01004);
    DBUG_RETURN(SQL_SUCCESS_WITH_INFO);
  }
  strmake( stmt->cursor_name, szCursor, cbCursor);
  DBUG_RETURN(SQL_SUCCESS);
}


/*
   SQLGetCursorName returns the cursor name associated with a specified
   statement.
*/

SQLRETURN SQL_API SQLGetCursorName(SQLHSTMT hstmt,
                                   SQLCHAR FAR *szCursor,
                                   SQLSMALLINT cbCursorMax,
                                   SQLSMALLINT FAR *pcbCursor)
{
  STMT FAR   *stmt=(STMT FAR*) hstmt;
  SQLINTEGER nLength;
  SQLINTEGER nDummyLength;
  DBUG_ENTER("SQLGetCursorName");

  LINT_INIT(nLength);
  LINT_INIT(nDummyLength);

  /*
    If no prior cursor set by the application, as 2.x driver doesn't
    handle any default cursors, give an error
  */
  if (stmt->cursor_state == MYSQL_CURSOR_UNDEFINED)
    DBUG_RETURN(set_stmt_error(stmt,"S1015","No cursor name available",0));

  /* if invalid buffer length */
  if (cbCursorMax < 0)
    DBUG_RETURN(set_stmt_error(stmt,"S1090",
                               "Invalid string or buffer length",0));

  /* if the stmt is in the middle of execution... */
  if (stmt->cursor_state == MYSQL_CURSOR_NEED_DATA ) 
    DBUG_RETURN(set_stmt_error(stmt,"S1010","Function sequence error",0));

  if (pcbCursor == NULL)
    pcbCursor = (SQLSMALLINT *) &nDummyLength;

  *pcbCursor = strlen(stmt->cursor_name);
  if ( cbCursorMax )
    cbCursorMax -= sizeof(char); /* NULL termination */

  if ( szCursor && cbCursorMax > 0)
    strmake( szCursor, stmt->cursor_name, cbCursorMax);

  nLength = min( *pcbCursor , cbCursorMax );
  if ( nLength != *pcbCursor )
  {
    set_stmt_error(stmt,"01004","String data, right truncated",01004);
    DBUG_RETURN(SQL_SUCCESS_WITH_INFO);
  }
  DBUG_RETURN(SQL_SUCCESS);
}


/*
 SQLSetPos sets the cursor position in a rowset and allows an application
 to refresh data in the rowset or to update or delete data in the result
 set.

 As the server doesn't give the BEST_ROWID always..better to manage
 all the rows in the where clause to avoid duplicate data delete/update
*/

SQLRETURN SQL_API SQLSetPos(SQLHSTMT hstmt,
                            SQLUSMALLINT irow,
                            SQLUSMALLINT fOption,
                            SQLUSMALLINT fLock)
{
  STMT FAR  *stmt=(STMT FAR*) hstmt;
  RETCODE   sqlRet;
  DYNAMIC_STRING dynQuery;
  MYSQL_RES *result = stmt->result;

  DBUG_ENTER("SQLSetPos");
  DBUG_PRINT("enter",("irow, refresh: %d   Lock: %d",irow,fOption,fLock));

  /*
    if no result set, no meaning for SQLSetPos call, so just
    through an error
  */
  if (!result)
    DBUG_RETURN(set_stmt_error(stmt,"S1010",
                              "Function sequence error, no result set",0));

  /* if irow > maximum rows in the resultset */
  if ( fOption != SQL_ADD && irow > mysql_num_rows(result))
    DBUG_RETURN(set_stmt_error(stmt,"S1107","Row value out of range",0));
  
  if ( fLock > SQL_LOCK_UNLOCK)
    DBUG_RETURN(set_stmt_error(stmt,"S1009","Invalid argument value",0));

  switch(fOption) {
  case SQL_POSITION:
  {
    /* The RowNumber argument was 0, and the Operation argument was SQL_POSITION
       through an error as per the spec
    */
    if( irow == 0)
      DBUG_RETURN(set_stmt_error(stmt,"S1109","Invalid cursor position",0)); 

    irow--; sqlRet = SQL_SUCCESS;
    stmt->position_in_set=irow;
    stmt->current_row = irow;
    mysql_data_seek(result,irow);
    break;
  }/* END OF SQL_POSITION */

  case SQL_DELETE:
  {
    const char *table_name=find_used_table(stmt);
    
    if (!table_name)
      DBUG_RETURN(SQL_ERROR);

    if(irow && (stmt->current_row != (uint)irow-1))
      DBUG_RETURN(set_stmt_error(stmt,"S1109","Invalid cursor position",0));
		
    /*
      dynamic string, to build and store the SQL Query for
      DELETE statement
    */    
    if (init_dynamic_string(&dynQuery, "DELETE FROM ", 1024, 1024))
      DBUG_RETURN(set_stmt_error(stmt,"S1001","Not enough memory",4001));
   
    dynstr_append(&dynQuery,table_name);    
    sqlRet = my_pos_delete(stmt,irow,dynQuery);
    dynstr_free(&dynQuery);
    break;
  } /* END of CASE SQL_DELETE */

  case SQL_UPDATE:
  {
    MYSQL_FIELD *field, *end;
    const char  *table_name=find_used_table(stmt);
    
    if (!table_name)
      DBUG_RETURN(SQL_ERROR);

    if(irow && stmt->current_row != (uint)irow-1)
      DBUG_RETURN(set_stmt_error(stmt,"S1109","Invalid cursor position",0));

    /*
      dynamic string, to build and store the SQL Query for
      UPDATE statement
    */    
    if (init_dynamic_string(&dynQuery, "UPDATE ", 1024,1024))
      DBUG_RETURN(set_stmt_error(stmt,"S1001","Not enough memory",4001));
    
    dynstr_append(&dynQuery,table_name);
    dynstr_append_mem(&dynQuery," SET ",5);

    /*
      SET clause manipulation

      TODO:  as per spec: driver should retrieve the lengths of the data from
             the number-of-bytes buffers (the pcbValue argument in SQLBindCol).
             If the length of any column is SQL_IGNORE, the column is not 
             updated.
    */
    for (field=result->fields, end=field+ result->field_count;
         field < end ;
         field++)
    {
      if (field->table)
      {
        dynstr_append(&dynQuery,field->name);
        dynstr_append(&dynQuery,"=?,");
      }
    }
    /* Remove last ',' */    
    dynQuery.str[--dynQuery.length]=0;
    sqlRet = my_pos_update(stmt,irow,dynQuery,1);
    dynstr_free(&dynQuery);
    break;
  } /* END of CASE SQL_UPDATE */

  case SQL_REFRESH:
  {
    /* The driver positions the cursor on the row specified by irow and 
       refreshes data in the rowset buffers for that row.

       Note that, data in the buffers is refreshed but not refetched,
       the membership in the rowset is fixed.
       
       so it needs to refresh the specified row buffers, as this does't 
       need refetches the data
    */
    sqlRet = SQLExtendedFetch(hstmt,SQL_FETCH_ABSOLUTE,irow,
			      NULL,stmt->rgfRowStatus);
    break;
  }
  case SQL_ADD:
  {
    MYSQL_FIELD *field, *end;
    const char  *table_name=find_used_table(stmt);
    SQLUINTEGER index;
    
    if (!table_name)
      DBUG_RETURN(SQL_ERROR);

    if (init_dynamic_string(&dynQuery, "INSERT INTO ", 1024,1024))
      DBUG_RETURN(set_stmt_error(stmt,"S1001","Not enough memory",4001));
    
    dynstr_append(&dynQuery,table_name);
    dynstr_append_mem(&dynQuery,"(",1);
		
    /*
      INSERT column list
    */
    for (field=result->fields, end=field+ result->field_count;
         field < end ;
         field++)
    {
        dynstr_append(&dynQuery,field->name);
        dynstr_append(&dynQuery,",");
    }
    /* Remove last ',' */    
    dynQuery.length--;dynstr_append(&dynQuery,") VALUES (");
    for (index=0; index < result->field_count; index++)
      dynstr_append(&dynQuery,"?,");
   
    dynQuery.length--;dynstr_append(&dynQuery,")");
    sqlRet = my_pos_add(stmt,irow,dynQuery);
    dynstr_free(&dynQuery);
    break;
  }
  default:
  {
    DBUG_RETURN(set_stmt_error(stmt,"S1009","Invalid argument value",0));
  }
  }/* END OF switch */ 	
  DBUG_RETURN(sqlRet);
}


/*
   my_pos_delete deletes the positioned row
*/

SQLRETURN my_pos_delete(STMT FAR *stmt, SQLUSMALLINT irow, 
                        DYNAMIC_STRING dynQuery)
{
  SQLHSTMT     hstmtNew;
  SQLUSMALLINT ncol; 
  MYSQL_RES    *result = stmt->result;
  SQLRETURN    sqlRet;  
  STMT FAR     *stmtNew;
  my_bool      pk_exists; 
  
  /* if irow = 0, then DELETE all rows, else delete the current row */
  if (irow) 
    pk_exists = my_build_where_clause(stmt,&dynQuery);
  
  /*
    allocate new stmt handle, and execute the DELETE query in
    that stmt, without disturbing the original stmt
  */

  /* Make sure the cursor is still in the same position..
     Fix to MS Access making wrong calls
  */
  my_set_cursor_data(stmt);

  sqlRet = my_SQLAllocStmt(stmt->dbc, &hstmtNew);
  if(sqlRet != SQL_SUCCESS)
    return(sqlRet);

  stmtNew = (STMT FAR *)hstmtNew;
    
  sqlRet = my_SQLPrepare(stmtNew,dynQuery.str,SQL_NTS);
  if(sqlRet != SQL_SUCCESS)
    goto my_time_to_return_error; 

  /* if only the current row to be deleted, then supply param data */
  if (irow) 
  {
    /* check if PK column exists in the result set, else
       use the temporary result set to get the PK column data
    */
    if (pk_exists)
      my_pk_param_bind(stmtNew,stmt,irow,0);
    else
    {
      for (ncol = 0; ncol < result->field_count; ncol++)
      {
        if (result->fields[ncol].table)
          my_param_bind(stmtNew,stmt,ncol,ncol);
      }
      stmtNew->query = insert_params(stmtNew);
    }
  }

  /* now execute the real DELETE query... */
  DBUG_PRINT("SQL_DELETE:",("%s",stmtNew->query));  
  sqlRet = do_query(stmtNew,stmtNew->query);
  if( sqlRet == SQL_SUCCESS || sqlRet == SQL_SUCCESS_WITH_INFO )
  {
    /* if irow=0, server returns affected rows as 0, this is a bug, so
       manually do this from driver side, but change this when server
       supports it.
    */
    if(irow == 0)
    {
      stmt->affected_rows=result->row_count;
      stmt->dbc->mysql.affected_rows = result->row_count;
    }
    else
      stmt->affected_rows=mysql_affected_rows(&stmtNew->dbc->mysql);
    
    /* no rows updated/deleted, return warning */
    if( stmt->affected_rows == 0)
    {
      sqlRet = SQL_SUCCESS_WITH_INFO;
      set_stmt_error(stmt,"01S03","No rows updated/deleted",0);
    }
    /* if irow !=0 and stmt->affected_rows > 1, then return warning */
    else if( irow != 0 && stmt->affected_rows > 1)
    {
      sqlRet = SQL_SUCCESS_WITH_INFO;
      set_stmt_error(stmt,"01S04","More than one row updated/deleted",0);
    }
    else if( stmt->rgfRowStatus) /* update the status */
    {
      for(ncol = 0; ncol < stmt->affected_rows; ncol++)
        stmt->rgfRowStatus[ncol+stmt->current_row] = SQL_ROW_DELETED;
    }
  }
  else
  {
    DBUG_PRINT("error",("%s:%s",stmtNew->sqlstate,stmtNew->last_error));
    set_stmt_error(stmt,stmtNew->sqlstate,stmtNew->last_error,
                   stmt->last_errno);
  } 
my_time_to_return_error:
  my_SQLFreeStmt(hstmtNew,SQL_DROP);
  return(sqlRet);
}


/*
 my_pos_update updates the positioned row
*/

SQLRETURN my_pos_update(STMT FAR *stmt,
                        SQLUSMALLINT irow,
                        DYNAMIC_STRING dynQuery,
                        SQLUSMALLINT set_param_exists)
{
  SQLHSTMT     hstmtNew;
  SQLUSMALLINT ncol;
  MYSQL_RES    *result = stmt->result;
  SQLRETURN    sqlRet;
  STMT FAR     *stmtNew;
  my_bool      pk_exists; 
  
  /* if irow = 0, then UPDATE all rows, else update the current row */
  if (irow)
    pk_exists = my_build_where_clause(stmt,&dynQuery);

  /* Make sure the cursor is still in the same position..
     Fix to MS Access making wrong calls
  */
  my_set_cursor_data(stmt);

  /*  now...prepare to execute   */
  sqlRet = my_SQLAllocStmt(stmt->dbc, &hstmtNew);
  if(sqlRet != SQL_SUCCESS)
    return(sqlRet);

  stmtNew = (STMT FAR *)hstmtNew;
  sqlRet = my_SQLPrepare(stmtNew,dynQuery.str,SQL_NTS);
  if(sqlRet != SQL_SUCCESS)
    goto my_time_to_return_error;

  if (set_param_exists)
  {
    /*
      copy row buffers to param buffers to set the update data i.e.
      param for SET clause
    */    
    if(!stmt->bind){
      /* The argument fOption was SQL_ADD or SQL_UPDATE and no columns were
         bound with SQLBindCol.
      */
        sqlRet = SQL_ERROR;
        set_stmt_error(stmt,"21S02",
                       "Degree of derived table does not match column list",0);
        goto my_time_to_return_error;
    }
    for (ncol = 0; ncol < stmt->result->field_count; ncol++)
    {
      PARAM_BIND *param = dynamic_element(&stmtNew->params,ncol,PARAM_BIND*);
      ulong transfer_length,precision,display_size;
      MYSQL_FIELD *field = mysql_fetch_field_direct(result,ncol);
      BIND *bind = stmt->bind+ncol; 

      param->used= 1;
      param->SqlType = unireg_to_sql_datatype(stmt,field,0,
                                              &transfer_length,&precision,
                                              &display_size);
      param->CType = bind->fCType;
      param->buffer = (SQLCHAR *)bind->rgbValue;
      param->ValueMax = bind->cbValueMax;
      param->actual_len= bind->pcbValue;
      param->real_param_done = TRUE;

      set_dynamic(&stmtNew->params,(gptr)param,ncol);
    }
  } 

  /*  if only the current row to be updated, then supply param data */
  if (irow ) 
  {
    if (set_param_exists)
    {
      SQLUSMALLINT dstCol;
      if(pk_exists)
      {
        dstCol = result->field_count;
        my_pk_param_bind(stmtNew,stmt,irow,dstCol);
      }
      else
      {
        for (ncol = 0; ncol < result->field_count; ncol++)
        {
          dstCol = ncol+result->field_count;
          my_param_bind(stmtNew,stmt,ncol,dstCol);
        }
        stmtNew->query = insert_params(stmtNew);
      }
    }
    else
    {
      if(pk_exists)
        my_pk_param_bind(stmtNew,stmt,irow,0);
      else
      {
        for (ncol = 0; ncol < result->field_count; ncol++)
          my_param_bind(stmtNew,stmt,ncol,ncol);
        stmtNew->query = insert_params(stmtNew);
      }
    }
  }/* END of if (iRow) */

  /*  now execute the real UPDATE query...  */
  DBUG_PRINT("SQL_UPDATE:",("%s",stmtNew->query));    
  sqlRet = do_query(stmtNew,stmtNew->query);
  if( sqlRet == SQL_SUCCESS || sqlRet == SQL_SUCCESS_WITH_INFO )
  {
    stmt->affected_rows=mysql_affected_rows(&stmtNew->dbc->mysql);

    /* no rows updated/deleted, return warning */
    if( stmt->affected_rows == 0)
    {
      sqlRet = SQL_SUCCESS_WITH_INFO;
      set_stmt_error(stmt,"01S03","No rows updated/deleted",0);
    }
    /* if irow !=0 and stmt->affected_rows > 1, then return warning */
    else if( irow != 0 && stmt->affected_rows > 1)
    {
      sqlRet = SQL_SUCCESS_WITH_INFO;
      set_stmt_error(stmt,"01S04","More than one row updated/deleted",0);
    }
    else if( stmt->rgfRowStatus) /* update the status */
    {
      for(ncol = 0; ncol < stmt->affected_rows; ncol++)
        stmt->rgfRowStatus[ncol+stmt->current_row] = SQL_ROW_UPDATED;
    }
  }
  else 
  {
    DBUG_PRINT("error",("%s:%s",stmtNew->sqlstate,stmtNew->last_error));
    set_stmt_error(stmt,stmtNew->sqlstate,stmtNew->last_error,
                   stmt->last_errno);
  }    

my_time_to_return_error:
  my_SQLFreeStmt(hstmtNew,SQL_DROP); 
  return(sqlRet);
}

/*
 my_pos_add inserts a positioned row of data
*/

SQLRETURN my_pos_add(STMT FAR *stmt,
                     SQLUSMALLINT irow,
                     DYNAMIC_STRING dynQuery)
{
  SQLHSTMT     hstmtNew;
  SQLUSMALLINT ncol;
  MYSQL_RES    *result = stmt->result;
  SQLRETURN    sqlRet;
  STMT FAR     *stmtNew;

  /*  now...prepare to execute   */
  sqlRet = my_SQLAllocStmt(stmt->dbc, &hstmtNew);
  if(sqlRet != SQL_SUCCESS)
    return(sqlRet);

  stmtNew = (STMT FAR *)hstmtNew;
  sqlRet = my_SQLPrepare(stmtNew,dynQuery.str,SQL_NTS);
  if(sqlRet != SQL_SUCCESS)
    goto my_time_to_return_error;

  /*
    copy row buffers to param buffers to set the update data i.e.
    param for SET clause
  */    
  if(!stmt->bind){
    /* The argument fOption was SQL_ADD or SQL_UPDATE and no columns were
       bound with SQLBindCol.
    */
    sqlRet = SQL_ERROR;
    set_stmt_error(stmt,"21S02",
                   "Degree of derived table does not match column list",0);
    goto my_time_to_return_error;
  }
  for (ncol = 0; ncol < stmt->result->field_count; ncol++)
  {
    PARAM_BIND *param = dynamic_element(&stmtNew->params,ncol,PARAM_BIND*);
    ulong transfer_length,precision,display_size;
    MYSQL_FIELD *field = mysql_fetch_field_direct(result,ncol);
    BIND *bind = stmt->bind+ncol; 

    param->used= 1;

    param->SqlType = unireg_to_sql_datatype(stmt,field,0,
                                              &transfer_length,&precision,
                                              &display_size);
    param->CType = bind->fCType;
    param->buffer = (SQLCHAR *)bind->rgbValue;
    param->ValueMax = bind->cbValueMax;
    param->actual_len= bind->pcbValue;
    param->real_param_done = TRUE;

    set_dynamic(&stmtNew->params,(gptr)param,ncol);
  }
  /* This is a buggy code..it doesn't handle DATA_LEN_AT_EXEC...
     better to use my_SQLExecute, so that it will take care of it, but 
     it reduces the performance
  */
  stmtNew->query = insert_params(stmtNew);/* INSERT param DATA */
  /*  now execute the real INSERT query...  */
  DBUG_PRINT("SQL_ADD:",("%s",stmtNew->query));    
  sqlRet = do_query(stmtNew,stmtNew->query);
  if( sqlRet == SQL_SUCCESS || sqlRet == SQL_SUCCESS_WITH_INFO )
  {
    stmt->affected_rows=mysql_affected_rows(&stmtNew->dbc->mysql);
    if( stmt->rgfRowStatus) /* update the INSERT status */
       stmt->rgfRowStatus[stmt->current_row] = SQL_ROW_ADDED;
  }
  else 
  {
    DBUG_PRINT("error",("%s:%s",stmtNew->sqlstate,stmtNew->last_error));
    set_stmt_error(stmt,stmtNew->sqlstate,stmtNew->last_error,
                   stmt->last_errno);
  }    

my_time_to_return_error:
  my_SQLFreeStmt(hstmtNew,SQL_DROP); 
  return(sqlRet);
}

/*
 my_param_bind, does the param binding for the specific stmt by taking the
 data from the current row of the result set
*/

void my_param_bind(STMT FAR *dstStmt, STMT FAR *srcStmt,
                   SQLUSMALLINT nSrcCol, SQLUSMALLINT nDstCol)
{
  PARAM_BIND  *param;
  MYSQL_RES   *result = srcStmt->result;
  ulong       transfer_length,precision,display_size;
  MYSQL_FIELD *field = mysql_fetch_field_direct(result,nSrcCol);
  MYSQL_ROW   row_data = result->data_cursor->data + nSrcCol;
  
  param=dynamic_element(&dstStmt->params,nDstCol,PARAM_BIND*);
  param->used= 1;
  param->SqlType = unireg_to_sql_datatype(srcStmt,field,0,
        &transfer_length,&precision,
        &display_size);;
  param->CType = SQL_C_CHAR;
  param->buffer = (SQLCHAR *)*row_data;
  param->ValueMax = strlen(*row_data);
  param->actual_len = NULL;
  param->real_param_done = TRUE;
  set_dynamic(&dstStmt->params,(gptr)param,nDstCol);
}

/**
** checks whether the SQL statement contains WHERE CURRENT OF CURSOR
*/

my_bool check_if_positioned_cursor_exists(STMT FAR *stmt,STMT FAR **stmtNew)
{
  if (stmt->query && stmt->query_end)
  {
    char *szQueryEnd = stmt->query_end;
    char *szQuery = stmt->query;
    const char *szCursor = mystr_get_prev_token((const char **)&szQueryEnd,
						 stmt->query);

    if (!my_casecmp(mystr_get_prev_token((const char **)&szQueryEnd,
					  stmt->query),"OF",2) &&
        !my_casecmp(mystr_get_prev_token((const char **)&szQueryEnd,
					  stmt->query),"CURRENT",7) &&
        !my_casecmp(mystr_get_prev_token((const char **)&szQueryEnd,
					  stmt->query),"WHERE",5))
    {

      /*
        Driver needs to setup the active stmt which has the cursor 
        specified in the query.

        Find out the cursor from all the stmts and make that as
        the active stmt..becuase this cursor is applicable
        for all stmts in the current DBC level
      */
      LIST *list_element,*next_element;
      DBC FAR *dbc=(DBC FAR*) stmt->dbc;

      for (list_element=dbc->statements ; list_element ;
           list_element=next_element)
      {
        next_element=list_element->next;
        *stmtNew = (HSTMT)list_element->data;
				
	/* might have the cursor in the stmt without any resultset, so 
	   avoid crashes, by keeping check on (*stmtNew)->result)
	*/
        if (!my_strcasecmp((*stmtNew)->cursor_name,szCursor) &&
			   (*stmtNew)->result)
        {
          *szQueryEnd= '\0';
          return(TRUE);
        }
      }
      if (!list_element) 
      {
	char buff[100];
	strxmov(buff,"Cursor '",szCursor,"' does not exist",NullS);
        set_stmt_error(stmt,"3400",buff,ER_INVALID_CURSOR_NAME);
      }
      return(TRUE);
    }
  } /* END OF if (stmt->query && stmt->query_end) */
  return(FALSE);
} 

/*
** checks whether the Primary Key column exists in the table
** if it exists, returns the PK column name
*/

SQLRETURN my_if_pk_exits(STMT FAR *stmt)
{
  char buff[100];
  MYSQL_ROW row;
  SQLHSTMT hStmtTemp;
  STMT FAR *stmtTemp;
  
  DBUG_ENTER("my_if_pk_exists");

  if(stmt->my_pk_validated)
    DBUG_RETURN(stmt->pk_count);
  
  if(my_SQLAllocStmt(stmt->dbc, &hStmtTemp) != SQL_SUCCESS)
      DBUG_RETURN(0);

  stmtTemp = (STMT FAR *)hStmtTemp;

  strxmov(buff,"show keys from ",stmt->result->fields->table,NullS);
  pthread_mutex_lock(&stmtTemp->dbc->lock);
  if( mysql_query(&stmtTemp->dbc->mysql,buff) ||
    !(stmtTemp->result=mysql_store_result(&stmtTemp->dbc->mysql)))
  {
    set_stmt_error(stmt,"S1000",mysql_error(&stmtTemp->dbc->mysql),
       mysql_errno(&stmtTemp->dbc->mysql));
    pthread_mutex_unlock(&stmtTemp->dbc->lock);
    my_SQLFreeStmt(hStmtTemp,SQL_DROP);
    DBUG_RETURN(0);
  }
  pthread_mutex_unlock(&stmtTemp->dbc->lock);

  /* right now take care of only PK's */
  while ((row = mysql_fetch_row(stmtTemp->result)) && 
      !(my_casecmp(row[2],"PRIMARY",7)) &&
       (stmt->pk_count < MY_MAX_PK_PARTS))
  {
    strmov(stmt->pk_col[stmt->pk_count++].name,row[4]);
  }
  stmt->my_pk_validated = TRUE;
  my_SQLFreeStmt(hStmtTemp,SQL_DROP);
  DBUG_RETURN(stmt->pk_count);
}


/*
** checks whether the Primary Key column exists in the result
** if it exists, takes that data and binds it,else does gets the 
** temporary resultset and takes the data
*/

SQLRETURN my_pk_param_bind(STMT FAR *stmtNew,STMT FAR *stmt,
                           SQLUSMALLINT irow,SQLUSMALLINT nColParam)
{
  MYSQL_RES   *result = stmt->result;
  MYSQL_FIELD *field;
  SQLUSMALLINT ncol;  
  SQLUSMALLINT index;
  DBUG_ENTER("my_pk_param_bind");

  /* check for any pk columns exists in the current resultset,
     if exists, take data from it and bind it
  */
  for (ncol = 0; ncol < result->field_count; ncol++)
  {
    field=result->fields+ncol;
    for (index=0; index < stmt->pk_count; index++)
    {
      if(!my_strcasecmp(stmt->pk_col[index].name,field->name))
      {     
        my_param_bind(stmtNew,stmt,index,(SQLUSMALLINT)(nColParam+index));
        stmt->pk_col[index].bind_done = TRUE; 
        break;
      }
    }
  }/* END OF for(ncol = 1 ; ncol <= stmt->result->field_count; ncol++)*/
  
  /* if PK doesn't exists in the current result set means, create a temp result
     set, and take the data from it.
  */
  {
    SQLHSTMT hStmtTemp;
    STMT FAR *stmtTemp; 
    DYNAMIC_STRING query;
    my_bool pk_not_in_set = FALSE;
    
    if (init_dynamic_string(&query, "SELECT ", 1024,1024))
      DBUG_RETURN(set_stmt_error(stmt,"S1001","Not enough memory",4001));

    for(index=0; index < stmt->pk_count; index++)
    {
      if(!stmt->pk_col[index].bind_done)
      {
        dynstr_append(&query,stmt->pk_col[index].name);   
        dynstr_append_mem(&query,",",1);
        pk_not_in_set = TRUE;
      }
    }
    /* if all the PK columns exists in the result set itself,
       return back, by inserting
    */
    if(!pk_not_in_set)
    {
      stmtNew->query = insert_params(stmtNew);
      dynstr_free(&query);
      DBUG_RETURN(0);
    }
    /* else get the data for other PK columns */
    query.length-=1;        /* Remove last  ', ' */
    dynstr_append(&query," FROM ");    
    dynstr_append(&query,stmt->table_name);    

    if(my_SQLAllocStmt(stmt->dbc, &hStmtTemp) != SQL_SUCCESS)
      DBUG_RETURN(0);

    stmtTemp = (STMT FAR *)hStmtTemp;
    pthread_mutex_lock(&stmtTemp->dbc->lock);
    if( mysql_query(&stmtTemp->dbc->mysql,query.str) ||
      !(stmtTemp->result=mysql_store_result(&stmtTemp->dbc->mysql)))
    {
      set_stmt_error(stmt,"S1000",mysql_error(&stmtTemp->dbc->mysql),
         mysql_errno(&stmtTemp->dbc->mysql));
      pthread_mutex_unlock(&stmtTemp->dbc->lock);
      my_SQLFreeStmt(hStmtTemp,SQL_DROP);
      DBUG_RETURN(0);
    }
    pthread_mutex_unlock(&stmtTemp->dbc->lock);       

    for(index=1;index <irow;index++)
      stmtTemp->result->data_cursor = stmtTemp->result->data_cursor->next;

    for(index=0; index < stmt->pk_count; index++)
    {
      if(!stmt->pk_col[index].bind_done)
      {
        my_param_bind(stmtNew,stmtTemp,0,(SQLUSMALLINT)(nColParam+index));
      }
    }    
    stmtNew->query = insert_params(stmtNew);
    my_SQLFreeStmt(hStmtTemp,SQL_DROP);
  } 
  DBUG_RETURN(1);
}

/**
* Building where search clause
**/
my_bool my_build_where_clause(STMT FAR *stmt,
                              DYNAMIC_STRING *dynQuery)
{
  MYSQL_RES *result = stmt->result;
  my_bool   pk_exists = FALSE;   
  uint      index;

  /* WHERE clause building.. */    
  dynstr_append(dynQuery," WHERE "); 
    
  /*  build the WHERE caluse... , if Primary Key exists
     take that column for searching, else use all columns
  */
  if (my_if_pk_exits(stmt))
  {
    for(index=0; index < stmt->pk_count; index++)
    {
      dynstr_append(dynQuery,stmt->pk_col[index].name);   
      dynstr_append_mem(dynQuery,"=? AND ",7);
    }
    pk_exists = TRUE;
  }
  else
  {
    MYSQL_FIELD *field, *end;
    for (field=result->fields, end=field+ result->field_count;
         field < end ; field++)
    {
      dynstr_append(dynQuery,field->name);   
      dynstr_append_mem(dynQuery,"=? AND ",7);
    }
  }

  dynQuery->length-=5;        /* Remove last  ' AND ' */
  dynstr_append(dynQuery," LIMIT 1");    
  return(pk_exists);
}

/*
** Returns the table used by this query.
** Ensures that all columns are from the same table
*/

static const char *find_used_table(STMT *stmt)
{
  MYSQL_FIELD  *field, *end;  
  const char *table_name;
  MYSQL_RES *result = stmt->result;
  DBUG_ENTER("find_used_table");

  if (stmt->table_name)
    DBUG_RETURN(stmt->table_name);    /* Return cached name */

  table_name=0;
  for (field=result->fields, end=field+ result->field_count;
       field < end ; field++)
  {
    if (field->table)
    {
      if (!table_name)
        table_name=field->table;
      if (strcmp(field->table, table_name))
      {
        set_stmt_error(stmt,"34000",
              "Can't modify a row from a statement that uses more than one table",
             ER_MULTIPLE_TABLE_IN_RESULT);
        DBUG_RETURN(NULL);
      }
    }
  }
  stmt->table_name=table_name;
  DBUG_RETURN(stmt->table_name);
}

/*
** position the data cursor to appropriate row
*/
void my_set_cursor_data(STMT FAR *stmt)
{
  SQLUINTEGER irow;
  MYSQL_RES *result=stmt->result;
  MYSQL_ROWS	*dcursor = result->data->data;

  if(!stmt->data_cursor)
  {
    for(irow=0; irow < stmt->current_row; irow++)
      dcursor = dcursor->next;

    stmt->data_cursor = TRUE;
    result->data_cursor = dcursor;   
  }
}


