--------------------------------------------------------------------------
--                                                                      --
--           Copyright: Copyright (C) 2000-2010 CNRS/IN2P3              --
--                                                                      --
-- Narval framework is free  software; you can redistribute  it and/or  --
-- modify  it   under  terms  of  the  GNU General  Public  License as  --
-- published  by  the  Free Software Foundation; either version  2, or  --
-- (at your option) any later version. Narval framework is distributed  --
-- in the hope  that  they will  be useful, but  WITHOUT ANY WARRANTY;  --
-- without even the implied warranty of  MERCHANTABILITY or FITNESS FOR --
-- A PARTICULAR PURPOSE. See the  GNU. General Public License for more  --
-- details. You should have received  a copy of the GNU General Public  --
-- License distributed with Narval; see file COPYING. If not, write to  --
-- the Free Software  Foundation,  Inc., 51 Franklin St,  Fifth Floor,  --
-- Boston, MA 02110-1301 USA.                                           --
--------------------------------------------------------------------------
with Ada.Unchecked_Deallocation;
with System;
with System.Storage_Elements;

package Narval.Protected_Memory is

   type Acquisition_Kind_Type is (Event_By_Event, Multiple_Events);
   type Start_Addresses is array (Buffer_Enumeration_Type) of System.Address;
   type Buffer_State_Type is (Memory_Available,
                              Events_Available,
                              Full,
                              Lock_For_Filling,
                              Lock_For_Filling_Bis,
                              Lock_For_Dump);
   type Buffer_State_Array is array (Buffer_Enumeration_Type)
     of Buffer_State_Type;
   type Buffer_Occupancy_Rates_Type is array (Buffer_Enumeration_Type)
     of System.Storage_Elements.Storage_Count;

   type Raw_Access_Type is record
      Address : System.Address;
      Size : System.Storage_Elements.Storage_Count;
   end record;

   type Developed_Access_Type is record
      Buffer_Number : Buffer_Enumeration_Type;
      Size : System.Storage_Elements.Storage_Count;
   end record;

   type Full_State_Type is record
      Buffers_State : Buffer_State_Array;
      Buffers_Occupancy : Buffer_Occupancy_Rates_Type;
   end record;

   protected type Buffered_Memory_Type is
      function Get_State return Full_State_Type;
      function Buffer_Size return System.Storage_Elements.Storage_Count;
      procedure Initialise (Buffer_Size :
                               System.Storage_Elements.Storage_Count;
                             Mode : Acquisition_Kind_Type := Multiple_Events);
      procedure Free;
      procedure Toggle_Dumping_Mode (Event_Based_Dumping : Boolean := True);
      entry Get_Memory_For_Filling
        (Bytes_Number : in System.Storage_Elements.Storage_Count;
         Address : out System.Address;
         Duplicate : Boolean := False);
      entry Get_Memory_For_Dumping (Bytes_Number : in
                                     System.Storage_Elements.Storage_Count;
                                   Address : out System.Address);
      procedure Release_Memory (Duplicate : Boolean := False);
      entry Get_Memory
        (Address : out System.Address;
         Available_Memory_Size : out System.Storage_Elements.Storage_Count);
      procedure Release_Memory
        (Used_Memory_Size : in System.Storage_Elements.Storage_Count;
         Duplicate : Boolean := False);
      entry Lock_Buffer
        (Manipulation_Buffer : out Buffer_Enumeration_Type);
      function Get_Buffer_Structure
        (Manipulation_Buffer : in Buffer_Enumeration_Type)
        return Developed_Access_Type;
      function Get_Buffer_Structure
        (Manipulation_Buffer : in Buffer_Enumeration_Type)
        return Raw_Access_Type;
      procedure Unlock_Buffer
        (Locked_Buffer : in Buffer_Enumeration_Type);
      procedure Check (Duplicate : Boolean := False);
      procedure Notify (Data : Developed_Access_Type;
                        Duplicate : Boolean := False);
      entry Available_Event
        (Remaining_Size : out System.Storage_Elements.Storage_Count);
      function Is_Empty return Boolean;
      function Dumping_Event_By_Event_Mode return Boolean;
      entry Get_Duplicated_Data
        (Size : out System.Storage_Elements.Storage_Count;
         Data : out System.Address);
      procedure Release_Duplicated_Data;
   private
      Acquisition_Mode : Acquisition_Kind_Type := Multiple_Events;
      Memory_Addresses : Start_Addresses;
      Buffer_State : Buffer_State_Array := (others => Memory_Available);
      Memory_Occupation_Rate : Buffer_Occupancy_Rates_Type := (others => 0);
      Memory_Dumping_Rate : Buffer_Occupancy_Rates_Type := (others => 0);
      Bytes_Sizes : System.Storage_Elements.Storage_Count;
      Buffer_In_Filling : Buffer_Enumeration_Type := First;
      First_Buffer_To_Send : Buffer_Enumeration_Type := First;
      Number_Of_Buffer_To_Send : Buffer_Numbering_Type := 0;
      Dumping_Event_By_Event : Boolean := False;
      Full_Value : Buffer_State_Type := Full;
      Duplicated_Address : System.Address;
      Duplicated_Data_Size : System.Storage_Elements.Storage_Count;
      Duplicated_Data_Available : Boolean := False;
   end Buffered_Memory_Type;
   type Memory_Access is access all Buffered_Memory_Type;
   procedure Free is new Ada.Unchecked_Deallocation
     (Buffered_Memory_Type, Memory_Access);
end Narval.Protected_Memory;
