!**************************************************************************
!*
!* Boot-ROM-Code to load an operating system across a TCP/IP network.
!*
!* Module:  packet.S
!* Purpose: Interface to the packet driver
!* Entries: _reg_type, _write_packet, _pktpoll, _init_packet, _empty_buf,
!*          term_packet, reg_packet, unreg_packet
!*
!**************************************************************************
!*
!* Copyright (C) 1995-1998 Gero Kuhlmann <gero@gkminix.han.de>
!*
!*  This program is free software; you can redistribute it and/or modify
!*  it under the terms of the GNU General Public License as published by
!*  the Free Software Foundation; either version 2 of the License, or
!*  any later version.
!*
!*  This program is distributed in the hope that it will be useful,
!*  but WITHOUT ANY WARRANTY; without even the implied warranty of
!*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!*  GNU General Public License for more details.
!*
!*  You should have received a copy of the GNU General Public License
!*  along with this program; if not, write to the Free Software
!*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
!*


!
!**************************************************************************
!
! Include assembler macros:
!
#include <macros.inc>
#include <memory.inc>
#include "./pktpriv.inc"


!
!**************************************************************************
!
! Miscelleneous equates:
!
MIN_INT		equ	$60		! lowest usable packet driver interrupt
MAX_INT		equ	$80		! highest usable packet driver interrupt
DRIVER_LEN	equ	8		! length of ID string
DRIVER_OFS	equ	3		! offset of ID string


!
!**************************************************************************
!
! Define entry for packet type table:
!
type_id		equ	0		! ID for packet type
type_driver	equ	2		! pointer to type event routine
type_handle	equ	4		! packet driver handle for type

type_size	equ	6		! size of packet type table entry


!
!**************************************************************************
!
! BSS segment. We can safely include static read and write buffers here
! as the BSS segment will not be stored in the final bootrom and therefore
! doesnt occupy any space.
!
	.bss

	extrn	_myhwaddr

	.lcomm	pktoldint,2		! packet driver interrupt no.
	.lcomm	readtype,2		! type of packet in buffer
	.lcomm	readsize,2		! actual size of data in buffer
	.lcomm	readbuf,ETH_BUF_SIZE+4	! pointer to read buffer
	.lcomm	writebuf,ETH_BUF_SIZE+4	! pointer to write buffer

	.lcomm	typetab,TYPENUM * type_size


!
!**************************************************************************
!
! Start code segment.
!
	.text

	public	_reg_type		! define entry points
	public	_write_packet
	public	_pktpoll
	public	_empty_buf
	public	_init_packet
	public	term_packet
	public	reg_packet
	public	unreg_packet

! General library functions and data within text segment

	extrn	datseg
	extrn	prnstr
	extrn	prnbyte
	extrn	_fatal


!
!**************************************************************************
!
! Register a new packet type.
! Input:  1. arg  -  type identifier
!         2. arg  -  near pointer to poll callback routine
! Output: Pointer to write buffer, NULL if error
!
_reg_type:

	penter				! setup standard stack frame
	push	es
	push	di
	push	si
	mov	si,#typetab		! find a free slot (no overrun check!)
regt1:	mov	cx,type_id[si]
	jcxz	regt3			! is the slot available?
	add	si,#type_size
	jmp	regt1

regt3:	getcarg	(ax,1)			! get pointer to handler routine
	mov	type_driver[si],ax
	getcarg	(ax,0)			! get type ID
	mov	bx,si
	call	reg_handle		! register type with packet driver
	jnc	regt4
regt5:	xor	ax,ax
	jmp	regt9

regt4:	mov	type_handle[si],ax	! save access type handle
	cmp	si,#typetab		! check if hardware address already
	jnz	regt7			! initialized
	mov	bx,ax
	mov	ax,ds
	mov	es,ax
	mov	di,#_myhwaddr
	mov	cx,#ETH_ALEN		! get hardware address for current
	mov	ah,#GET_ADDRESS		! handle from packet driver
	int	PKTINT
	jc	regt5
regt7:	mov	ax,#writebuf + ETH_HLEN	! return write buffer pointer
regt9:	pop	si
	pop	di
	pop	es
	pleave
	ret


!
!**************************************************************************
!
! Send a packet using the packet driver
! Input:  1. arg  -  number of bytes in write buffer
!         2. arg  -  packet type value
!         3. arg  -  near pointer to destination hardware address
! Output: error code
!
_write_packet:

	penter				! setup standard stack frame
	cld
	push	es
	push	si
	push	di
	mov	ax,ds
	mov	es,ax
	getcarg	(ax,1)			! get packet type
	mov	bx,#typetab
	mov	cx,#TYPENUM
wrtp1:	cmp	ax,type_id[bx]		! find packet type in type table
	je	wrtp2
	add	bx,#type_size
	loop	wrtp1
	mov	ax,#-ERR_BAD_TYPE	! type not found
	jmp	wrtp9

wrtp2:	mov	di,#writebuf
	mov	cx,#ETH_ALEN		! copy destination address into
	getcarg	(si,2)			! send buffer
	rep
	movsb
	mov	cx,#ETH_ALEN		! copy source address into send
	mov	si,#_myhwaddr		! buffer
	rep
	movsb
	stosw				! copy packet type

	getcarg	(ax,0)			! get number of characters to write
	cmp	ax,#ETH_FRAME_MAX
	jbe	wrtp4			! check if maximum number of data
	mov	ax,#ETH_FRAME_MAX	! bytes exceeded
wrtp4:	cmp	ax,#ETH_DATA_MIN
	jae	wrtp5			! check if minimum number of data
	mov	cx,#ETH_DATA_MIN	! bytes copied
	sub	cx,ax
	add	di,ax
	xor	al,al			! pad with zeroes if not
	rep
	stosb
	mov	ax,#ETH_DATA_MIN

wrtp5:	mov	si,#writebuf		! set pointer to write buffer
	mov	bx,type_handle[bx]	! set packet driver handle
	mov	cx,ax
	add	cx,#ETH_HLEN		! set packet buffer size
	mov	ah,#SEND_PKT		! now send the packet
	int	PKTINT
	jnc	wrtp8
#ifdef IS386
	movzx	ax,dh			! return with error
#else
	mov	al,dh
	xor	ah,ah			! return with error
#endif
	neg	ax
	jmp	wrtp9

wrtp8:	xor	ax,ax			! no error
wrtp9:	pop	di
	pop	si
	pop	es
	pleave
	ret


!
!**************************************************************************
!
! This routine gets called by the packet driver when a new packet arrives.
! Input:  AX  -  function number: 0 = return buffer pointer, 1 = buffer filled
!         BX  -  handle
!         CX  -  length of data received (only when AX = 1)
!         DS:SI  -  pointer to receive buffer (only when AX = 1)
! Output: CX     -  length of buffer (only when AX = 0)
!         ES:DI  -  pointer to receive buffer (only when AX = 0)
! Registers changed: AX, BX, CX, DX, ES, SI, DI
!
receiver:

	push	ds
	seg	cs
	mov	ds,datseg
	or	ax,ax			! check function code
	jnz	rec3

! Return address of receive buffer in ES:DI. Discard the packet if the
! receive buffer is not empty.

	mov	es,ax
	mov	di,ax			! discard packet if no free buffer
	mov	ax,readsize		! buffer free?
	or	ax,ax
	jnz	rec9
	seg	cs
	mov	es,datseg
	mov	di,#readbuf		! return pointer to read buffer
	jmp	rec9

! The buffer has been read. Now mark it as busy, so that the polling routine
! can give it up one level.

rec3:	mov	si,#typetab
	mov	dx,#TYPENUM
rec4:	cmp	bx,type_handle[si]	! find handle in type table
	je	rec5
	add	si,#type_size
	dec	dx
	jnz	rec4			! discard packet if handle not found
	jmp	rec9

rec5:	mov	dx,type_id[si]
	cli
	mov	readsize,cx		! save buffer size and mark buffer
	mov	readtype,dx		! busy
	sti
rec9:	pop	ds
	retf


!
!**************************************************************************
!
! Get a received packet out of the read buffer, and call the upper layer
! handler, which gets the following parameters on its stack:
!	char *buf    -  pointer to receive buffer
!	int bufsize  -  number of data bytes in receive buffer
!	char *addr   -  pointer to senders address
!
! Input:  none
! Output: none
!
_pktpoll:

	mov	cx,readsize		! immediately terminate if no
	jcxz	pktp9			! packet received
	mov	dx,readtype
	or	dx,dx			! dont do anything if we already
	jz	pktp9			! processed this buffer

	mov	bx,#readbuf
	lea	ax,ETH_ALEN[bx]		! push pointer to source hardware
	push	ax			! address
	sub	cx,#ETH_HLEN
	push	cx			! push buffer size (without header)
	add	bx,#ETH_HLEN
	push	bx			! push buffer pointer (without header)

	xor	ax,ax
	mov	bx,#typetab
	mov	cx,#TYPENUM
pktp1:	cmp	dx,type_id[bx]
	je	pktp2
	add	bx,#type_size		! search type in type list
	loop	pktp1
	jmp	pktp3

pktp2:	mov	bx,type_driver[bx]	! call type handler
	or	bx,bx
	jz	pktp3
	call	bx
pktp3:	add	sp,#6			! cleanup stack
	cli
	or	ax,ax			! dont discard packet if higher level
	jnz	pktp8			! doesnt want to
	mov	word ptr readsize,#0	! discard packet
pktp8:	mov	word ptr readtype,#0	! mark buffer as processed
	sti
pktp9:	ret


!
!**************************************************************************
!
! Mark read buffer empty
! Input:  None
! Output: None
!
_empty_buf:

	cli
	mov	word ptr readsize,#0	! discard packet
	mov	word ptr readtype,#0	! mark buffer as processed
	sti
	ret


!
!**************************************************************************
!
! Initialize packet driver interface.
! Input:  None
! Output: None
!
_init_packet:

	push	es
	push	di

! First determine the drivers interrupt, and redirect it to our own
! interrupt. Using an interrupt which is known at assembly time allows
! the use of the int instruction.

	call	pktfind			! try to find packet driver int
	or	ax,ax
	jnz	init1
	mov	bx,#intmsg		! print interrupt error message
initF:	jmp	near pktfatal

init1:	mov	pktoldint,ax		! save packet driver interrupt
	cli
	xor	ax,ax
	mov	es,ax			! save packet driver interrupt vector
#ifdef IS386
	seg	es
	mov	eax,[bx]
	seg	es
	mov	[PKTINT * 4],eax
#else
	seg	es
	mov	ax,[bx+0]
	seg	es
	mov	word ptr [PKTINT * 4 + 0],ax
	seg	es
	mov	ax,[bx+2]
	seg	es
	mov	word ptr [PKTINT * 4 + 2],ax
#endif
	sti

! Setup the packet driver interface. This involves checking the maximum
! transfer data size, the network class and the address size. Then allocate
! memory for the send and receive buffers.

	mov	ah,#GET_PARAMETERS	! get parameters from packet
	int	PKTINT			! driver. if driver is too old,
	jc	init4			! assume default value
	seg	es
	mov	ax,[di + 4]		! get MTU from driver
	cmp	ax,#ETH_FRAME_MAX
	jae	init3
init2:	mov	bx,#pktmsg		! invalid packet driver
	jmp	initF

init3:	seg	es
	mov	dl,[di + 3]		! get size of MAC address
	cmp	dl,#ETH_ALEN		! check if ethernet address length
	jne	init2			! is correct

init4:	push	ds
	push	si
	mov	ax,#(DRIVER_INFO << 8) + $FF
	xor	bx,bx			! dummy handle
	int	PKTINT			! get packet driver info
	pop	si
	pop	ds
	jc	init2			! packet driver is too old
	cmp	ch,#CL_DEFAULT		! packet driver is not for ethernet
	jne	init2

! Tell the user that we found something.

	mov	bx,#fndmsg
	call	near prnstr
	mov	ax,pktoldint
	call	near prnbyte
	mov	bx,#crlf
	call	near prnstr
	pop	di
	pop	es
	ret


! Error messages:

intmsg:	.asciz	"No int"
pktmsg:	.asciz	"Invalid PKTDRVR"


! Message to tell user about packet driver interrupt:

fndmsg:	.byte	$0D,$0A
	.asciz	"Found packet driver at int "
crlf:	.byte	$0D,$0A,0


!
!**************************************************************************
!
! Find packet driver interrupt
! Input:  none
! Output: AX  -  Interrupt number, or 0 if not found
!         BX  -  pointer to interrupt vector in segment 0
! Registers changed: AX, BX, CX, DI, ES
!
pktfind:

	cld
	push	ds
	push	si
	mov	ax,cs
	mov	ds,ax
	mov	bx,#MIN_INT * 4		! start looking at interrupt 60h
pktf1:	xor	ax,ax
	mov	es,ax
	seg	es
	les	di,[bx]			! get interrupt vector
	mov	ax,es
	or	ax,di			! ignore null pointers
	jz	pktf2
	add	di,#DRIVER_OFS		! compute address of ID string
	mov	si,#pktdrv_id
	mov	cx,#DRIVER_LEN
	repe
	cmpsb				! compare both strings
	jz	pktf3			! found it
pktf2:	add	bx,#4			! advance to next interrupt vector
	cmp	bx,#MAX_INT * 4		! at the end of the list?
	jb	pktf1
	xor	ax,ax
	jmp	pktf4

pktf3:	mov	ax,bx
#ifdef IS186
	shr	ax,#2			! return interrupt number
#else
	shr	ax,#1
	shr	ax,#1			! return interrupt number
#endif
pktf4:	pop	si
	pop	ds
	ret

! ID string at the beginning of the packet driver code:

pktdrv_id:	.ascii	"PKT DRVR"


!
!**************************************************************************
!
! Print fatal error message and exit
! Input:  BX  -  pointer to error message
! Output: routine does not return
!
pktfatal:

	push	bx
	mov	bx,#fatmsg
	call	near prnstr
	pop	bx
	call	near prnstr
	jmp	near _fatal

! Preceding error message:

fatmsg:	.byte	$0D,$0A
	.asciz	"PKTDRV ERROR: "


!
!**************************************************************************
!
! Register one handle with the packet driver
! Input:  AX  -  type ID
!         BX  -  pointer into handle table
! Output: AX  -  handle as returned by packet driver
!                Carry flag set if error
! Registers changed: AX, CX, DX
!
reg_handle:

	mov	type_id[bx],ax		! save type id
	push	es
	push	di
	push	si
	push	bx
	mov	di,#receiver
	mov	ax,cs
	mov	es,ax			! ES:DI  -  pointer to receive handler
	lea	si,type_id[bx]		! DS:SI  -  pointer to type ID
	mov	cx,#2			! CX     -  size of type ID
	mov	al,#CL_DEFAULT		! AL     -  default class
	mov	bx,#TYPE_DEFAULT	! BX     -  default type
	xor	dl,dl			! DL     -  default interface number
	mov	ah,#ACCESS_TYPE
	int	PKTINT			! setup access type
	pop	bx
	pop	si
	pop	di
	pop	es
	ret


!
!**************************************************************************
!
! Register all previously unregistered handles with the packet driver
! Input:  none
! Output: Carry flag set if error
! Registers changed: AX, BX, CX, DX
!
reg_packet:

	push	es
	push	si
	push	di
	mov	bx,#typetab
	mov	cx,#TYPENUM
regp1:	mov	ax,type_id[bx]
	or	ax,ax
	jz	regp9			! end of list reached?
	mov	dx,type_handle[bx]
	cmp	dx,#$FFFF		! check if handle already used
	jne	regp2
	call	reg_handle		! register handle with packet driver
	jc	regp9			! check for error
	mov	word ptr type_handle[bx],ax
regp2:	add	bx,#type_size
	loop	regp1
	clc				! return without error
regp9:	pop	di
	pop	si
	pop	es
	ret


!
!**************************************************************************
!
! Unregister all handles with the packet driver
! Input:  AX  -  non zero if all but the first handle have to be released
! Output: none
! Registers changed: AX, BX, CX, DX
!
unreg_packet:

	mov	bx,#typetab
	mov	cx,#TYPENUM
	or	ax,ax
	jz	unrgp1
	add	bx,#type_size		! leave the first handle untouched
	dec	cx
unrgp1:	mov	ax,type_id[bx]
	or	ax,ax
	jz	unrgp9			! end of list reached?
	push	bx
	push	cx
	mov	ah,#RELEASE_TYPE
	mov	bx,type_handle[bx]	! check if handle already released
	cmp	bx,#$FFFF
	je	unrgp2
	int	PKTINT			! release packet handle
unrgp2:	pop	cx
	pop	bx
	mov	word ptr type_handle[bx],#$FFFF
	add	bx,#type_size
	loop	unrgp1
unrgp9:	ret


!
!**************************************************************************
!
! Terminate the packet driver
! Input:  none
! Output: none
! Registers changed: AX, BX, CX, DX
!
term_packet:

	mov	ax,#1
	call	unreg_packet		! unregister all but the first handle
	mov	ah,#TERMINATE
	mov	bx,#$FFFF
	xchg	bx,typetab + type_handle
	int	PKTINT			! terminate packet driver
	ret


!
!**************************************************************************
!
	end

