!**************************************************************************
!*
!* PKTWATCH - Log packets going through a packet driver
!*
!* Module:  pktwatch.S
!* Purpose: Log packets going through a packet driver
!* Entries: start
!*
!**************************************************************************
!*
!* Copyright (C) 1995-1998 Gero Kuhlmann <gero@gkminix.han.de>
!*
!*  This program is free software; you can redistribute it and/or modify
!*  it under the terms of the GNU General Public License as published by
!*  the Free Software Foundation; either version 2 of the License, or
!*  any later version.
!*
!*  This program is distributed in the hope that it will be useful,
!*  but WITHOUT ANY WARRANTY; without even the implied warranty of
!*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!*  GNU General Public License for more details.
!*
!*  You should have received a copy of the GNU General Public License
!*  along with this program; if not, write to the Free Software
!*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
!*
!**************************************************************************
!*


!
!**************************************************************************
!
! The purpose of this program is to log all packets onto the screen, which
! go through a packet driver. The packet driver interrupt has to be specified
! on the command line in hex. It then splits the screen into two halves by
! using the 43x80 text resolution on a EGA/VGA card. The upper 25 lines
! receive the unmodified output from the original screen, and the lower
! lines receive the packets in hex dump.
! We dont use any of the packet driver assembler include modules here, as
! this is a completely independent program. No fancy 386 optimization here.
!


!
!**************************************************************************
!
! Definition of BIOS data area:
!
BIOSSEG		equ	$0040		! BIOS segment address

b_columns	equ	$004A		! number of screen columns
b_vidsize	equ	$004C		! size of video buffer in bytes
b_lines		equ	$0084		! number of screen lines - 1


!
!**************************************************************************
!
! Miscellaneous defines:
!
CMDLINE		equ	$0080		! offset of command line into PSP
VIDEO_SEG	equ	$B800		! segment of video buffer
DEACTIVATE	equ	$CF45		! magic key for deactivation

MAX_BYTES	equ	36		! max num of buffer bytes to display
IP_TYPE		equ	$0008		! IP network type in network order
ADDR_SIZE	equ	6		! size of hardware address
PKTTYPE_SIZE	equ	2		! size of packet type
DEST_OFS	equ	6		! offset to dest addr in MAC header
TYPE_OFS	equ	12		! offset to packet type in MAC header
DATA_OFS	equ	14		! offset to data area in ethernet pkt


!
!**************************************************************************
!
! There is no other than a code segment. All data will be placed in it
! as well. The resident section of the program starts here.
!
	.text

	org	$0100			! we have to start at $0100

	entry	start			! define startup point

start:	jmp	near start1


!
!**************************************************************************
!
! Data segment:
!

scnlines:	.word	0		! number of screen lines
scncolumns:	.word	0		! number of screen columns
scnbytes:	.word	0		! number of bytes per screen line
vidbufsize:	.word	0		! video buffer size
vidoffset:	.word	0		! offset into second half of screen
curoffset:	.word	0		! current character offset

pkttype:	.word	IP_TYPE		! packet type to display

pktint:		.word	0		! packet driver interrupt number
old2f_vect:	.long	0		! old $2F interrupt vector
oldpkt_vect:	.long	0		! old packet driver interrupt vector
oldreceiver:	.long	0		! pointer to old receiver routine


!
!**************************************************************************
!
! Interrupt routine to terminate pktwatch services. This will wait for
! a key press, and then restore the screen.
! Input:  AX  -  Magic key $0CF45
! Output: none
! Registers used: none
!
new_int2f:

	cmp	ax,#DEACTIVATE		! check if we have to deactivate
	je	int2f1
	seg	cs
	jmp	far ptr [old2f_vect]

int2f1:	cli
	push	ax
	push	si
	push	ds
	cli
	xor	ax,ax			! first restore all interrupt vectors
	mov	ds,ax
	seg	cs
	mov	si,pktint
	shl	si,1
	shl	si,1
	seg	cs
	mov	ax,word ptr [oldpkt_vect + 0]
	mov	word ptr [si + 0],ax
	seg	cs
	mov	ax,word ptr [oldpkt_vect + 2]
	mov	word ptr [si + 2],ax
	seg	cs
	mov	ax,word ptr [old2f_vect + 0]
	mov	word ptr [$2F * 4 + 0],ax
	seg	cs
	mov	ax,word ptr [old2f_vect + 2]
	mov	word ptr [$2F * 4 + 2],ax
	pushf
	seg	cs
	call	far ptr [old2f_vect]	! daisy chain old interrupt
	sti

	mov	ax,cs			! let DS point to data segment
	mov	ds,ax
	call	scrollup		! scroll screen up one line
	mov	si,#waitms
	call	printstr		! print waiting message
	xor	ah,ah
	int	$16			! wait for key press
	mov	ax,#$0003		! restore screen to mode 3
	int	$10
	pop	ds
	pop	si
	pop	ax
	iret

waitms:	.asciz	"Terminating pktwatch. Press a key to continue..."


!
!**************************************************************************
!
! New packet driver interrupt.
! Input:  none
! Output: none
! Registers changed: none
!
new_pkt:

	jmp	newpk0

	.byte	0
	.ascii	"PKT DRVR"		! packet driver signature

newpk0:	cmp	ah,#4			! send packet?
	je	newpk2
	cmp	ah,#2			! initialize access type?
	je	newpk4
newpk1:	seg	cs
	jmp	far ptr [oldpkt_vect]	! call old packet vector

! Requested to send a packet.

newpk2:	push	ax
	seg	cs
	mov	ax,pkttype
	cmp	word ptr [si + TYPE_OFS],ax
	jne	newpk3
	mov	al,#$53			! character indicating 'send'
	call	printbuf
newpk3:	pop	ax
	jmp	newpk1

! Requested to set new packet access type. We only handle packets with
! a specified packet type, e.g. 'typelen' has to be exactly 2 which is
! the size of word. The receiver routine is then replaced by a new one
! which prints every received packet.

newpk4: push	ax
	cmp	cx,#PKTTYPE_SIZE	! check for correct packet type size
	jne	newpk3
	mov	ax,word ptr [si]	! check for correct packet type
	seg	cs
	cmp	ax,pkttype
	jne	newpk3
	seg	cs
	mov	word ptr [oldreceiver + 0],di
	seg	cs
	mov	word ptr [oldreceiver + 2],es
	mov	ax,cs
	mov	es,ax
	mov	di,#new_receiver	! replace old receiver routine with
	jmp	newpk3			! new one


!
!**************************************************************************
!
! New receiver routine. This routine gets called by the packet driver
! whenever a new packet arrives.
! Input:  BX     -  handle
!         AX     -  flag indicating first or second call
!         CX     -  buffer length
!         DS:SI  -  pointer to receive buffer (only if AX = 1)
! Output: none
! Registers changed: none
!
new_receiver:

	or	ax,ax			! check if this is the correct call
	jz	newrc9
	push	ax
	mov	al,#$52			! character indicating 'receive'
	call	printbuf		! print buffer contents
	pop	ax
newrc9:	seg	cs
	jmp	far ptr [oldreceiver]


!
!**************************************************************************
!
! Print a packet buffer into the last screen line
! Input:  DS:SI  -  pointer to buffer
!         CX     -  size of buffer
!         AL     -  character in first column
! Output: none
! Registers changed: AX
!
printbuf:

	push	es
	push	ds
	pop	es			! point ES:SI to buffer
	push	cs
	pop	ds
	push	bx
	xor	bx,bx
	cmp	al,#$53
	jne	prnb0			! select source or destination address
	mov	bx,#DEST_OFS		! to print
prnb0:	call	scrollup		! scroll screen up one line
	call	printchr
	mov	al,#$20			! first print the character indicating
	call	printchr		! direction of packet

prnb1:	seg	es
	mov	al,[si + bx]
	call	printbyte		! print next byte
	cmp	bx,#DEST_OFS - 1
	jne	prnb4
	add	bx,#ADDR_SIZE		! skip destination addr if source addr
prnb4:	cmp	bx,#TYPE_OFS - 1	! printed
	jne	prnb5
	add	bx,#DATA_OFS - TYPE_OFS	! skip packet type
prnb5:	cmp	bx,#DATA_OFS - 1
	jb	prnb3
prnb2:	mov	al,#$20			! print blanks between data bytes
	call	printchr
prnb3:	inc	bx
	cmp	bx,cx			! check if at the end of the buffer
	jae	prnb9
	cmp	bx,#MAX_BYTES		! cant write more than one line
	jb	prnb1

prnb9:	pop	bx
	push	es
	pop	ds			! restore DS:SI
	pop	es
	ret


!
!**************************************************************************
!
! Print a byte in hex form onto screen
! Input:  AL  -  byte value
! Output: none
! Registers changed: AX
!
printbyte:

	push	ax
	shr	al,#1
	shr	al,#1
	shr	al,#1			! first print upper nibble
	shr	al,#1
	call	printnib
	pop	ax

printnib:

	and	al,#$0F
	add	al,#$30
	cmp	al,#$39			! convert nibble and print it
	jbe	prnn1
	add	al,#7
prnn1:	call	printchr
	ret


!
!**************************************************************************
!
! Scroll the lower half of the screen up one line.
! Input:  none
! Output: none
! Registers used: none
!
scrollup:

	cld
	push	es
	push	di
	push	ds
	push	si
	push	cx
	mov	cx,#VIDEO_SEG
	mov	es,cx			! set video buffer segment
	mov	ds,cx
	seg	cs
	mov	si,scnbytes
	seg	cs			! compute start of source and
	mov	di,vidoffset		! destination area
	add	si,di
	seg	cs
	mov	cx,vidbufsize		! compute number of words to move
	sub	cx,si
	shr	cx,#1
	rep
	movsw				! now scroll the screen
	mov	si,ax			! save AX
	mov	ax,#$0720		! fill last line with blanks
	seg	cs
	mov	cx,scncolumns
	rep
	stosw
	mov	ax,si			! restore AX
	pop	cx
	pop	si
	pop	ds
	sub	di,scnbytes
	mov	curoffset,di		! set current character position to
	pop	di			! first column in last line
	pop	es
	ret


!
!**************************************************************************
!
! Print a string into the last screen line.
! Input:  DS:SI  -  pointer to zero terminated string
! Output: none
! Registers used: AX, SI
!
printstr:

	cld
	push	es
	push	di
	mov	di,#VIDEO_SEG		! set video buffer segment
	mov	es,di
	mov	di,curoffset
	mov	ah,#$07			! white on black
prnst1:	lodsb				! now print the string
	or	al,al			! check for end of string
	jz	prnst2
	stosw				! store character into video buffer
	jmp	prnst1
prnst2:	pop	di
	pop	es
	ret


!
!**************************************************************************
!
! Print a character into the last screen line.
! Input:  AL  -  character to print
! Output: none
! Registers used: AX
!
printchr:

	cld
	push	es
	push	di
	mov	di,#VIDEO_SEG		! set video buffer segment
	mov	es,di
	mov	di,curoffset
	mov	ah,#$07			! white on black
	stosw				! write character onto screen
	mov	curoffset,di		! save new character position
	pop	di
	pop	es
	ret


!
!**************************************************************************
!
! Start the non-resident section here:
!
start1:	mov	ax,cs
	mov	ds,ax

! First analyze the command line. The only arguments for this program
! are the interrupt vector and the packet type in hex. Then initialize
! the screen.

	mov	si,#CMDLINE + 1
	call	skipblank
	cmp	al,#$0D			! must be hex number
	je	start5
	call	gethex			! get hex number
	cmp	bx,#$0060		! should be in the range of $60
	jb	start2			! and $7F
	cmp	bx,#$007F
	jbe	start4
start2:	mov	dx,#interr		! print error message
start3:	mov	ah,#$09
	int	$21
	mov	ax,#$4C01		! terminate program
	int	$21

start4:	mov	pktint,bx		! save packet driver interrupt number
	call	skipblank
	cmp	al,#$0D
	je	start6			! packet type is optional
	call	gethex
	xchg	al,ah			! convert from host into network order
	mov	pkttype,ax
	call	skipblank		! nothing should follow anymore
	cmp	al,#$0D
	je	start6
start5:	mov	dx,#useerr		! print usage message
	jmp	start3

start6:	push	es
	mov	ax,pktint
	mov	ah,#$35			! determine packet driver vector
	int	$21
	mov	word ptr [oldpkt_vect + 0],bx
	mov	word ptr [oldpkt_vect + 2],es
	pop	es
	call	checkpkt		! check if there really is a packet drvr
	jnc	start7
	mov	dx,#pkterr		! print error message
	jmp	start3

start7:	call	setvideo		! setup new video mode
	jnc	start8
	mov	dx,#viderr		! print error message
	jmp	start3

start8:	mov	dx,#cpymsg		! print copyright
	mov	ah,#$09
	int	$21

! From this point on we are going to manipulate interrupt vectors, so
! there is no way back now.

	mov	dx,#new_pkt
	mov	ax,pktint
	mov	ah,#$25			! set new packet driver interrupt
	int	$21
	push	es
	mov	ax,#$352F		! determine $2F interrupt vector
	int	$21
	mov	word ptr [old2f_vect + 0],bx
	mov	word ptr [old2f_vect + 2],es
	pop	es
	mov	dx,#new_int2f
	mov	ax,#$252F		! set new interrupt $2F vector
	int	$21

	mov	dx,#start1
	mov	cl,#4			! determine number of resident
	shr	dx,cl			! paragraphs
	inc	dx
	mov	ax,#$3100		! terminate and stay resident
	int	$21
	ret

cpymsg:	.byte	$0D,$0A
	.ascii	"pktwatch - Packet driver logging utility"
	.byte	$0D,$0A
	.ascii	"Copyright (C) G. Kuhlmann, 1996,1997"
	.byte	$0D,$0A,$0D,$0A
	.ascii	"$"

interr:	.ascii	"Illegal interrupt number"
	.byte	$0D,$0A
	.ascii	"$"
useerr:	.ascii	"usage: pktwatch <pkt-int-no> [<pkt type>]"
	.byte	$0D,$0A
	.ascii	"$"
pkterr:	.ascii	"No packet driver at given interrupt number"
	.byte	$0D,$0A
	.ascii	"$"
viderr:	.ascii	"Cannot switch video mode"
	.byte	$0D,$0A
	.ascii	"$"


!
!**************************************************************************
!
! Skip blanks on command line.
! Input:  DS:SI  -  pointer to command line
! Output: DS:SI  -  pointer to first non-blank
!         AL     -  first non-blank character
! Registers changed: AX, SI
!
skipblank:

	cld
skip1:	lodsb
	cmp	al,#$20			! skip blanks
	je	skip1
	cmp	al,#$09			! skip tabs
	je	skip1
	dec	si			! point SI to first non-blank
	ret


!
!**************************************************************************
!
! Return hex number from string pointed to by DS:SI.
! Input:  DS:SI  -  pointer to string
! Output: DS:SI  -  pointer to first character after hex number
!         BX     -  hex number
! Registers changed: AX, BX, CX, SI
!
gethex:

	xor	bx,bx
geth1:	lodsb
	cmp	al,#$30
	jb	geth9			! check for valid hex digit
	cmp	al,#$39
	jbe	geth3
	cmp	al,#$41
	jb	geth9
	cmp	al,#$61
	jb	geth2
	sub	al,#$20			! convert character to upper case
geth2:	cmp	al,#$46
	ja	geth9
	sub	al,#7
geth3:	sub	al,#$30			! convert character to digit
	and	al,#$0F
	mov	cl,#4
	shl	bx,cl			! include digit into result
	or	bl,al
	jmp	geth1			! get next character

geth9:	dec	si			! point SI to last non-digit
	ret


!
!**************************************************************************
!
! Check if there really is a packet driver at the given interrupt.
! Input:  none
! Output: Carry set, if no packet driver
! Registers changed: AX, CX, SI, DI
!
checkpkt:

	push	es
	les	di,oldpkt_vect
	mov	ax,es			! check if the interrupt vector
	or	ax,di			! points to somewhere
	jz	chkp8
	add	di,#3			! DI points to start of signature
	mov	si,#pktsig
	mov	cx,#8
	repe
	cmpsb				! check for packet driver
	clc
	jz	chkp9			! found packet driver
chkp8:	stc
chkp9:	pop	es
	ret

pktsig:	.ascii	"PKT DRVR"		! packet driver signature


!
!**************************************************************************
!
! Setup video mode to 43x80 (or 50x80 on VGA). This is done by using the
! 8x8 character set.
! Input:  none
! Output: Carry flag set if error
! Registers changed: AX, BX, CX, DX, DI
!
setvideo:

	mov	ah,#$0F
	int	$10			! get current video mode
	cmp	al,#7			! monochrome not supported
	je	setv8
	cmp	al,#3			! if its mode 3 we dont have
	je	setv1			! to change the mode
	mov	ax,#$0003		! set video mode to 3
	int	$10

setv1:	mov	ah,#$12			! determine current EGA configuration
	mov	bx,#$0110		! this is used to check if we really
	int	$10			! have an EGA or VGA card
	or	bh,bh			! BH has to be set to 0 by BIOS
	jnz	setv8

	push	es
	mov	ax,#BIOSSEG
	mov	es,ax
	seg	es
	mov	al,[b_lines]
	xor	ah,ah
	mov	scnlines,ax		! save current screen size
	seg	es
	mov	bx,[b_vidsize]
	mov	vidbufsize,bx
	seg	es
	mov	bx,[b_columns]
	mov	scncolumns,bx
	inc	ax
	shl	ax,#1
	mul	bx			! compute address of first character
	or	dx,dx			! after first half of screen
	jz	setv2			! it should not be larger than 64kB
	pop	es
setv8:	stc				! return with error
	jmp	setv9

setv2:	mov	vidoffset,ax		! save offset to second half
	mov	ax,#$1112		! load 8x8 character set into
	xor	bl,bl			! block no. 0 and reset screen
	int	$10
	seg	es
	mov	al,[b_lines]
	xor	ah,ah			! restore original screen size
	xchg	scnlines,ax		! this will let the BIOS output
	seg	es			! routines just access the upper 25
	mov	[b_lines],al		! scnlines is number of lines - 1
	inc	word ptr scnlines
	seg	es
	mov	ax,[b_vidsize]
	xchg	vidbufsize,ax
	seg	es
	mov	[b_vidsize],ax
	seg	es
	mov	ax,[b_columns]
	mov	cx,ax			! save number of columns for later
	xchg	scncolumns,ax
	seg	es
	mov	[b_columns],ax
	pop	es

	shl	cx,#1			! compute number of bytes per line
	mov	scnbytes,cx
	mov	ax,scnlines
	mul	cx			! compute real size of video buffer
	or	dx,dx
	jz	setv3
	mov	ax,#$FFFF		! if screen buffer too large, limit it
setv3:	mov	vidbufsize,ax		! to 64kB

	cld
	push	es
	shr	cx,#1
	mov	ax,#VIDEO_SEG
	mov	es,ax
	mov	di,vidoffset
	mov	ax,#$70C4		! print horizontal line
	rep
	stosw
	pop	es
	mov	vidoffset,di
	clc				! return without error
setv9:	ret

	end

